<?php

declare(strict_types=1);

namespace Ig\IgGooglemaps\Command;

use TYPO3\CMS\Core\Configuration\ExtensionConfiguration;
use PDO;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Http\RequestFactory;

class GeoCodeCommand extends Command
{
     
    protected $api_url = 'https://maps.googleapis.com/maps/api/geocode/json?';
    
    protected function configure()
    {
        $this
            ->setDescription('Geocode entries of tables')
            //->setHelp('')
            ->addArgument(
                'tablename',
                InputArgument::REQUIRED,
                'Tabellen Name'
            )
            
            ->addArgument(
                'Adress',
                InputArgument::OPTIONAL,
                'PLZ,ort,Strasse,StrassenNr,Land Spaltennamen in der Tabelle'
            )
            /*
              ->addArgument(
              'ids',
              InputArgument::OPTIONAL,
              'von id bis id (1,100)'
              )
            */
            ->addOption(
                'max',
                'm',
                InputOption::VALUE_OPTIONAL,
                'max entries to process'
            )
            ->addOption(
                'latlng',
                'l',
                InputOption::VALUE_OPTIONAL,
                'latitude,longitude attribute names',
                'latitude,longitude'
            )

            ->addOption(
                'uid',
                'u',
                InputOption::VALUE_OPTIONAL,
                'primary key attribute name',
                'uid'
            )
            ->addOption(
                'force',
                'f',
                InputOption::VALUE_NONE,
                'override existing latitude/longitude value',
            );
    }


    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $extConf = GeneralUtility::makeInstance(
            ExtensionConfiguration::class
        )->get('ig_googlemaps');
                                                          
        if(isset($extConf['googleServerGeocodingApiKey']) && $extConf['googleServerGeocodingApiKey'] != null){
            $key = $extConf['googleServerGeocodingApiKey'];
        }
        else{
            $io->writeln('ig_googlemaps API Key wurde nicht gefunden');
            return 1;
        }

        $tablename = $input->getArgument('tablename');
        $addressAttributeList = $input->getArgument('Adress') ?? 'zip, city, address, country';

        $latLng = $input->getOption('latlng');
        $force = (bool)$input->getOption('force');
        $max = (int)$input->getOption('max');
        $uidAttribute = $input->getOption('uid') ?? 'uid';


        if(isset($latLng) && $latLng != null){
            [$latitudeAttribute, $longitudeAttribute] = GeneralUtility::trimExplode(',' ,$latLng);
        }
        else{
            $latitudeAttribute = 'latitude';
            $longitudeAttribute = 'longitude';
        }
        
        $addressAttributes = GeneralUtility::trimExplode(',' ,$addressAttributeList);
        
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tablename);

        
        $queryBuilder
            ->select('*')
            ->from($tablename);

        if (!$force) {
            $queryBuilder->where(
                $queryBuilder->expr()->or(
                    $queryBuilder->expr()->isNull($latitudeAttribute),
                    $queryBuilder->expr()->eq(
                        $latitudeAttribute, 
                        $queryBuilder->createNamedParameter('')
                    ),
                    $queryBuilder->expr()->isNull($longitudeAttribute),
                    $queryBuilder->expr()->eq(
                        $longitudeAttribute, 
                        $queryBuilder->createNamedParameter('')
                    ),
                )
            );
        }
        
        if ($max) {
            $queryBuilder->setMaxResults($max);
        }

        $result = $queryBuilder
                ->executeQuery()
                ->fetchAllAssociative();
        $entriesUpdate = 0;
        $entriesTotal = 0;
        foreach ($result as $row) {
            ++$entriesTotal;
            $address = '';
            foreach($addressAttributes as $col){
                $address .= trim((string) $row[$col]) . ' ';
            }

            $queryData = http_build_query(
                [
                    'key'     => $key,
                    'address' => $address,
                ]
            );
            $url = $this->api_url . $queryData;
            $res = GeneralUtility::makeInstance(RequestFactory::class)->request($url, 'GET', ['headers' => ['accept' => 'application/json']]);
            $content = json_decode($res->getBody()->getContents(), true);

            if (isset($content['error_message'])) {                
                $io->writeln($content['error_message']);
                $io->writeln('used API key: ' . $key);
                return 1;
            }

            $location = $content['results'][0]['geometry']['location'];
            $latitude =  $location['lat'];
            $longitude = $location['lng'];
            $text = $row[$uidAttribute] . ': ' . $address;
            if ($latitude && $longitude) {
                $io->writeln($text . ' latLng: <fg=green>' . $latitude . ', ' . $longitude .'</>');
            } else {
                $io->writeln($text . ' latLng: <fg=red>NOT FOUND</>');
            }

            $row[$latitudeAttribute] = $latitude;
            $row[$longitudeAttribute] = $longitude;
            if($row[$latitudeAttribute] != null && $row[$longitudeAttribute] != null){
                $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tablename);
                $queryBuilder->update($tablename)
                             ->where(
                                 $queryBuilder->expr()->eq($uidAttribute, $queryBuilder->createNamedParameter($row[$uidAttribute], PDO::PARAM_INT))
                             )
                             ->set($latitudeAttribute, $row[$latitudeAttribute])
                             ->set($longitudeAttribute, $row[$longitudeAttribute])
                             ->executeStatement();

                ++$entriesUpdate;
            }
        }

        if ($entriesUpdate == $entriesTotal) {
            $io->success('Es wurden ' . $entriesUpdate . ' von ' . $entriesTotal . ' Einträge aktualisiert');
            return 0;
        } else {
            $io->warning('Es wurden ' . $entriesUpdate . ' von ' . $entriesTotal . ' Einträge aktualisiert');
            return 1;
        }
    }
}