<?php
namespace Ig\IgJobs\Controller;

use Ig\IgJobs\Domain\Model\Application;
use Ig\IgJobs\Domain\Model\Position;
use Ig\IgJobs\Domain\Repository\ApplicationRepository;
use Ig\IgJobs\Domain\Repository\AttentionRepository;
use Ig\IgJobs\PageTitle\JobPageTitleProvider;
use Internetgalerie\IgDynval\Controller\DynamicValidationActionController;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use \TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Mail\FluidEmail;
use TYPO3\CMS\Core\Mail\Mailer;
use TYPO3\CMS\Core\MetaTag\MetaTagManagerRegistry;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Fluid\View\StandaloneView;

/***
 *
 * This file is part of the "SBK Job Database" Extension for TYPO3 CMS.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 *  (c) 2018 Simon Häsler <sh@internetgalerie.ch>, Internetgalerie AG
 *
 ***/

/**
 * ApplicationController
 */
class ApplicationController extends \Internetgalerie\IgDatapoolFe\Controller\ActionController
{
    use DynamicValidationActionController;

    protected ApplicationRepository $applicationRepository;
    protected AttentionRepository $attentionRepository;

    public function __construct(
        ApplicationRepository $applicationRepository,
        AttentionRepository $attentionRepository
    ) {
        $this->applicationRepository = $applicationRepository;
        $this->attentionRepository = $attentionRepository;
    }

    public function initializeAction(): void
    {
        parent::initializeAction();
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_jobs/Resources/Public/JavaScript/ig_jobs.js');
    }

    /**
     * action new
     *
     * @param \Ig\IgJobs\Domain\Model\Position $position
     * @return void
     */
    public function newAction(Position $position): ResponseInterface
    {
        /*$titleProvider = GeneralUtility::makeInstance(JobPageTitleProvider::class);
        $titleProvider->setTitle('Bewerbung Stelle ' . $position->getTitle());*/

        if($position->isOnline()) {
            $metaTagManager = GeneralUtility::makeInstance(MetaTagManagerRegistry::class)->getManagerForProperty('description');
            $metaTagManager->addProperty('description', 'Bewerbung für Stelle Nr. ' . $position->getFullTitle());

            $attentions = $this->attentionRepository->findAll();

            $this->view->assign('position', $position);
            $this->view->assign('attentions', $attentions);
        } else {
            return $this->redirect('list', 'Position');
        }

        return $this->htmlResponse();
    }

    /**
     * action create
     *
     * @param \Ig\IgJobs\Domain\Model\Application $newApplication
     * @return void
     * @TYPO3\CMS\Extbase\Annotation\Validate(param="newApplication", validator="\Internetgalerie\IgDynval\Validation\Validator\DynamicValidator")
     */
    public function createAction(Application $newApplication): ResponseInterface
    {
        if($newApplication->getPosition()->isOnline()) {
            $this->applicationRepository->add($newApplication);

            $vars = [
                'application' => $newApplication,
                'layout' => 'Email'
            ];
            $attachments = [
                $newApplication->getCurriculumVitae(),
                $newApplication->getApplicationLetterFile(),
                $newApplication->getDiplomas(),
                $newApplication->getJobReferences(),
                $newApplication->getFile1(),
                $newApplication->getFile2(),
                $newApplication->getFile3(),
                $newApplication->getFile4(),
            ];

            foreach($newApplication->getDocuments() as $document) {
                $attachments[] = $document;
            }

            $mailFrom = [$this->settings['adminMailFrom']];
            $adminMailTo = [$this->settings['adminMailTo']];

            if($newApplication->getPosition()->getLocation()) {
                $locations = $newApplication->getPosition()->getLocation();
                $locationMailFrom = [];
                $locationMailTo = [];
                foreach($locations as $location) {
                    if($location->getApplicationMail() && GeneralUtility::validEmail($location->getApplicationMail())) {
                        $locationMailFrom[] = $location->getApplicationMail();
                        $locationMailTo[] = $location->getApplicationMail();
                    }
                }

                if(count($locationMailFrom)) {
                    $mailFrom = $locationMailFrom;
                }

                if(count($locationMailTo)) {
                    $adminMailTo = $locationMailTo;
                }
            }

            if($this->settings['sendUserMailAfterApplication']) {
                $this->sendMailWithAttachments(
                    'Ihre Bewerbung: ' . $newApplication->getPosition()->getFullTitle(),
                    $mailFrom,
                    [$newApplication->getEmail()],
                    'IgJobs/Application/User',
                    $vars
                );
            }

            $this->sendMailWithAttachments(
                'Bewerbung für Stelle ' . $newApplication->getPosition()->getFullTitle(),
                $mailFrom,
                $adminMailTo,
                'IgJobs/Application/Admin',
                $vars,
                $attachments
            );

            GeneralUtility::makeInstance(PersistenceManager::class)->persistAll();

            return $this->redirect('thankYou', null, null, ['application' => $newApplication]);
        } else {
            return $this->redirect('list', 'Position');
        }
    }

    /**
     * action thankYou
     *
     * @param \Ig\IgJobs\Domain\Model\Application $application
     * @return void
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("application")
     */
    public function thankYouAction(Application $application): ResponseInterface
    {
        $this->view->assign('layout', 'Default');
        $this->view->assign('application', $application);


        return $this->htmlResponse();
    }

    protected function sendMailWithAttachments(string $subject, array $from, array $to, string $template, array $variables = [], array $attachments = []): void
    {
        $email = GeneralUtility::makeInstance(FluidEmail::class)
            ->subject($subject)
            ->from(...$from)
            ->to(...$to)
            ->setTemplate($template)
            ->format('html')
            ->assign('settings', $this->settings)
            ->assignMultiple($variables);

        foreach($attachments as $attachment) {
            if(is_array($attachment) || $attachment instanceof \Iterator) {
                foreach($attachment as $att) {
                    $email->attachFromPath(Environment::getPublicPath() . $att->getOriginalResource()->getPublicUrl());
                }
            } else if($attachment) {
                $email->attachFromPath(Environment::getPublicPath() . $attachment->getOriginalResource()->getPublicUrl());
            }
        }

        if ($GLOBALS['TYPO3_REQUEST'] instanceof ServerRequestInterface) {
            $email->setRequest($GLOBALS['TYPO3_REQUEST']);
        }

        GeneralUtility::makeInstance(Mailer::class)->send($email);
    }
}
