<?php
namespace Ig\IgJobs\Controller;

use GeorgRinger\NumberedPagination\NumberedPagination;
use Ig\IgJobs\Domain\Model\Position;
use Ig\IgJobs\Domain\Repository\CategoryRepository;
use Ig\IgJobs\Domain\Repository\EmploymentRepository;
use Ig\IgJobs\Domain\Repository\LocationRepository;
use Ig\IgJobs\Domain\Repository\PositionRepository;
use Ig\IgJobs\Domain\Repository\TagRepository;
use Ig\IgJobs\PageTitle\JobPageTitleProvider;
use Internetgalerie\IgRender\Utility\PdfUtility;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Http\PropagateResponseException;
use TYPO3\CMS\Core\Http\Request;
use TYPO3\CMS\Core\Http\ServerRequest;
use TYPO3\CMS\Core\MetaTag\MetaTagManagerRegistry;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Pagination\QueryResultPaginator;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;

/***
 *
 * This file is part of the "SBK Job Database" Extension for TYPO3 CMS.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 *  (c) 2018 Simon Häsler <sh@internetgalerie.ch>, Internetgalerie AG
 *
 ***/

/**
 * PositionController
 */
class PositionController extends \TYPO3\CMS\Extbase\Mvc\Controller\ActionController
{
    protected PositionRepository $positionRepository;
    protected CategoryRepository $categoryRepository;
    protected TagRepository $tagRepository;
    protected EmploymentRepository $employmentRepository;
    protected LocationRepository $locationRepository;

    public function __construct(
        PositionRepository $positionRepository,
        CategoryRepository $categoryRepository,
        TagRepository $tagRepository,
        EmploymentRepository $employmentRepository,
        LocationRepository $locationRepository
    ) {
        $this->positionRepository = $positionRepository;
        $this->categoryRepository = $categoryRepository;
        $this->tagRepository = $tagRepository;
        $this->employmentRepository = $employmentRepository;
        $this->locationRepository = $locationRepository;
    }

    public function initializeAction(): void
    {
        parent::initializeAction();
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_jobs/Resources/Public/JavaScript/wNumb.min.js');
        $pageRenderer->addJsFooterFile('EXT:ig_jobs/Resources/Public/JavaScript/ig_jobs.js');
    }

    /**
     * action list
     *
     * @param array $search
     * @return void
     */
    public function listAction(array $search = []): ResponseInterface
    {
        $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument('currentPage') : 1;

        $minAndMaxLevelOfEmployment = $this->positionRepository->findMinAndMaxLevelOfEmployment();
        $locations = $this->locationRepository->findAll();
        $positions = $this->positionRepository->findBySearch($search, (bool)($this->settings['showTopJobs'] ?? false),$locations);
        $categories = $this->categoryRepository->findAll();
        $tags = $this->tagRepository->findAll();
        $employmentTypes = $this->employmentRepository->findAll();

        $paginator = new QueryResultPaginator($positions, $currentPage, 20);
        $pagination = new NumberedPagination($paginator, 10); // zweiter Argument: maximal Anzahl Links

        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);

        $this->view->assign('positions', $positions);
        $this->view->assign('categories', $categories);
        $this->view->assign('tags', $tags);
        $this->view->assign('employmentTypes', $employmentTypes);
        $this->view->assign('locations', $locations);
        $this->view->assign('search', $search);
        $this->view->assign('minAndMaxLevelOfEmployment', $minAndMaxLevelOfEmployment);
        $GLOBALS['TSFE']->addCacheTags(['tx_igjobs_position_list']);

        return $this->htmlResponse();
    }

    /**
     * action show
     *
     * @param \Ig\IgJobs\Domain\Model\Position $position
     * @return void
     */
    public function showAction(Position $position): ResponseInterface
    {
        $titleProvider = GeneralUtility::makeInstance(JobPageTitleProvider::class);
        $titleProvider->setTitle($position->getCompleteTitle());
        $locations = $this->locationRepository->findAll();

        //var_dump($position->isOnline());die();
        if($position->isOnline()) {
            $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);

            $metaTagManager = GeneralUtility::makeInstance(MetaTagManagerRegistry::class)->getManagerForProperty('description');
            $metaTagManager->addProperty('description', 'Bewerbung für Stelle Nr. ' . $position->getFullTitle());
            $this->view->assign('locations', $locations);
            $this->view->assign('position', $position);
        } else {
            $this->redirect('list');
        }
        $GLOBALS['TSFE']->addCacheTags(['tx_igjobs_position_show']);

        return $this->htmlResponse();
    }

    /**
     * action search
     *
     * @return void
     */
    public function searchAction(): ResponseInterface
    {
        /** @var ServerRequest */
        $request = $GLOBALS['TYPO3_REQUEST'];
        $params = $request->getParsedBody()['tx_igjobs_list'] ?? $request->getQueryParams()['tx_igjobs_list'] ?? null;

        if($params && !empty($params)) {
            if(isset($params['search'])) {
                $search = $params['search'];
            }
        }
        $categories = $this->categoryRepository->findAll();
        $employmentTypes = $this->employmentRepository->findAll();

        $this->view->assign('categories', $categories);
        $this->view->assign('employmentTypes', $employmentTypes);
        $this->view->assign('search', $search);

        return $this->htmlResponse();
    }

    public function pdfAction(Position $position): ResponseInterface
    {
        $pdf = GeneralUtility::makeInstance(PdfUtility::class);
        $pdf->setFilename('ig-jobs-position-'.strftime('%Y-%m-%d').'.pdf');
        $pdf->setTitle($position->getTitle());
        $pdf->setCssFile('EXT:ig_jobs/Resources/Public/Css/ig_jobs_pdf.css');

        if($this->settings['pdfAdditionalCssFile']) {
            $pdf->addCssFile($this->settings['pdfAdditionalCssFile']);
        }

        $pdf->setConverter(PdfUtility::CONVERTER_WEASYPRINT);

        $this->view->assign('position', $position);

        $html = $this->view->render();

        $pdfContent = $pdf->get($html);

        $response = $this->responseFactory->createResponse()
            ->withHeader('Expires', '0')
            ->withHeader('Cache-Control', 'private')
            ->withHeader('Content-Type', 'application/pdf')
            ->withHeader('Content-Disposition', 'attachment; filename="' . $pdf->getFilename() . '"')
            ->withHeader('Content-Length', (string)strlen($pdfContent))
            ->withBody($this->streamFactory->createStream($pdfContent));

        throw new PropagateResponseException($response, 200);
    }

    /**
     * action new
     *
     * @return void
     */
    public function newAction()
    {

    }

    /**
     * action create
     *
     * @param \Ig\IgJobs\Domain\Model\Position $newPosition
     * @return void
     */
    public function createAction(Position $newPosition): ResponseInterface
    {
        $this->addFlashMessage('The object was created. Please be aware that this action is publicly accessible unless you implement an access check. See https://docs.typo3.org/typo3cms/extensions/extension_builder/User/Index.html', '', ContextualFeedbackSeverity::INFO);
        $this->positionRepository->add($newPosition);
        return $this->redirect('list');
    }

    /**
     * action edit
     *
     * @param \Ig\IgJobs\Domain\Model\Position $position
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("position")
     * @return void
     */
    public function editAction(Position $position): ResponseInterface
    {
        $this->view->assign('position', $position);

        return $this->htmlResponse();
    }

    /**
     * action update
     *
     * @param \Ig\IgJobs\Domain\Model\Position $position
     * @return void
     */
    public function updateAction(Position $position): ResponseInterface
    {
        $this->addFlashMessage('The object was updated. Please be aware that this action is publicly accessible unless you implement an access check. See https://docs.typo3.org/typo3cms/extensions/extension_builder/User/Index.html', '', ContextualFeedbackSeverity::WARNING);
        $this->positionRepository->update($position);
        return $this->redirect('list');
    }

    /**
     * action delete
     *
     * @param \Ig\IgJobs\Domain\Model\Position $position
     * @return void
     */
    public function deleteAction(Position $position): ResponseInterface
    {
        $this->addFlashMessage('The object was deleted. Please be aware that this action is publicly accessible unless you implement an access check. See https://docs.typo3.org/typo3cms/extensions/extension_builder/User/Index.html', '', ContextualFeedbackSeverity::WARNING);
        $this->positionRepository->remove($position);
        return $this->redirect('list');
    }

}
