<?php
namespace Ig\IgJobs\Domain\Repository;

use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;

/***
 *
 * This file is part of the "SBK Job Database" Extension for TYPO3 CMS.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 *  (c) 2018
 *
 ***/

/**
 * The repository for Positions
 */
class PositionRepository extends \TYPO3\CMS\Extbase\Persistence\Repository
{
    /**
     * @var array
     */
    protected $defaultOrderings = [
        'sorting' => \TYPO3\CMS\Extbase\Persistence\QueryInterface::ORDER_ASCENDING
    ];

    /**
     * Find positions by search
     *
     * @param array $search
     * @param bool $showTopJobs
     * @return \TYPO3\CMS\Extbase\Persistence\Generic\QueryResult $result
     */
    public function findBySearch(array $search = [], $showTopJobs, $locations = []) {
        $query = $this->createQuery();

        $constraints = [];

        $now = new \DateTime();
        $constraints[] = $query->logicalOr(
            $query->logicalAnd(
                $query->lessThanOrEqual('startdate', $now->format('Y-m-d')),
                $query->greaterThanOrEqual('enddate', $now->format('Y-m-d'))
            ),
            $query->logicalAnd(
                $query->equals('startdate', null),
                $query->greaterThanOrEqual('enddate', $now->format('Y-m-d'))
            ),
            $query->logicalAnd(
                $query->lessThanOrEqual('startdate', $now->format('Y-m-d')),
                $query->equals('enddate', null)
            ),
            $query->logicalAnd(
                $query->equals('startdate', null),
                $query->equals('enddate', null)
            )
        );

        if(is_array($search) && !empty($search)) {
            if(isset($search['searchterm']) && $search['searchterm'] != '') {
                $searchterms = GeneralUtility::trimExplode(' ', $search['searchterm']);
                $searchtermConstraints = [];

                foreach($searchterms as $searchterm) {
                    $searchtermConstraints[] = $query->logicalOr(
                        $query->like('title', '%' . $searchterm . '%'),
                        $query->like('positionNo', '%' . $searchterm . '%'),
                        $query->like('levelOfEmployment', '%' . $searchterm . '%'),
                        $query->like('region', '%' . $searchterm . '%'),
                        $query->like('description', '%' . $searchterm . '%')
                    );
                }

                if(!empty($searchtermConstraints)) {
                    $constraints[] = $query->logicalAnd(...$searchtermConstraints);
                }
            }

            if(isset($search['category']) && $search['category'] != '') {
                $constraints[] = $query->contains('categories', $search['category']);
            }

            if(isset($search['categories']) && !empty($search['categories'])) {
                $categoryConstraints = [];
                foreach($search['categories'] as $categoryUid) {
                    $categoryConstraints[] = $query->contains('categories', $categoryUid);
                }

                $constraints[] = $query->logicalOr(...$categoryConstraints);
            }

            if(isset($search['employmentType']) && $search['employmentType'] != '') {
                $constraints[] = $query->equals('employment', $search['employmentType']);
            }

            if(isset($search['employmentTypes']) && !empty($search['employmentTypes'])) {
                $constraints[] = $query->in('employment', $search['employmentTypes']);
            }

            if(isset($search['location']) && $search['location'] != '' && $search['location'] != 'overarching') {
                //$constraints[] = $query->equals('location', $search['location']);
                $constraints[] = $query->contains('location', $search['location']);
            }
            // overarching if every location is set
            if(isset($search['location']) && $search['location'] == 'overarching') {
                $categoryConstraints = [];
                foreach ($locations as $location) {
                    $categoryConstraints[] = $query->contains('location', $location);
                }
                $constraints[] = $query->logicalAnd(...$categoryConstraints);
            }
            if(isset($search['locations']) && !empty($search['locations'])) {
                //$constraints[] = $query->in('location', $search['locations']);
                $categoryConstraints = [];
                foreach ($search['locations'] as $categoryUid) {
                    $categoryConstraints[] = $query->contains('location', $categoryUid);
                }

                $constraints[] = $query->logicalOr(...$categoryConstraints);
            }
            if(isset($search['tags']) && !empty($search['tags'])) {
                //$constraints[] = $query->in('location', $search['locations']);
                $tagConstraints = [];
                foreach ($search['tags'] as $tagUid) {
                    $tagConstraints[] = $query->contains('tags', $tagUid);
                }

                $constraints[] = $query->logicalOr(...$tagConstraints);
            }

            if(isset($search['minAndMaxLevelOfEmployment']) && !empty($search['minAndMaxLevelOfEmployment'])) {
                $min = (int)$search['minAndMaxLevelOfEmployment']['from'];
                $max = (int)$search['minAndMaxLevelOfEmployment']['to'];
                $constraints[] = $query->logicalOr(
                    $query->logicalAnd(
                        $query->equals('levelOfEmploymentFrom', 0),
                        $query->equals('levelOfEmploymentTo', 0),
                    ),
                    $query->logicalAnd(
                        $query->equals('levelOfEmploymentFrom', 0),
                        $query->greaterThanOrEqual('levelOfEmploymentTo', $min),
                    ),
                    $query->logicalAnd(
                        $query->lessThanOrEqual('levelOfEmploymentFrom', $max),
                        $query->equals('levelOfEmploymentTo', 0),
                    ),
                    $query->logicalAnd(
                        $query->greaterThanOrEqual('levelOfEmploymentTo', $min),
                        $query->lessThanOrEqual('levelOfEmploymentFrom', $max),
                    )
                );
            }
        }

        if($showTopJobs) {
            $constraints[] = $query->equals('isTopPosition', true);
        }

        $query->matching($query->logicalAnd(...$constraints));

        return $query->execute();
    }

    public function findMinAndMaxLevelOfEmployment()
    {
        $qb = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getQueryBuilderForTable('tx_igjobs_domain_model_position');

        $row = $qb->selectLiteral('min(level_of_employment_from) AS min,
                CASE WHEN level_of_employment_to > level_of_employment_from
                    THEN max(level_of_employment_to)
                    ELSE max(level_of_employment_from)
                END AS max')
            ->from('tx_igjobs_domain_model_position')
            ->where('(level_of_employment_from > 0 AND level_of_employment_to > 0)
                OR (level_of_employment_from = 0 AND level_of_employment_to > 0)
                OR (level_of_employment_from > 0 AND level_of_employment_to = 0)')
            ->executeQuery()
            ->fetchAssociative();

        return $row;
    }
}
