/**
 * A modern, accessible, and feature-rich lightbox component.
 */
class IgLightbox {
    /**
     * @param {object} config - The configuration object for the lightbox.
     */
    constructor(config) {
        this._initProperties(config);
        this._initEventListeners();
    }

    /**
     * Initializes component properties and state.
     * @param {object} config - The user-provided configuration.
     * @private
     */
    _initProperties(config) {
        const defaults = {
            id: 1,
            sources: null,
            dragPxLimit: 200,
            draggable: true,
            download: false,
            autoPlay: true,
            autoPlayTime: 3000,
            fullscreen: true,
            zoom: true,
            zoomConfig: { min: 1, max: 5, scale: 1, speed: 1 },
            preload: 2,
            visibleSlidesAmount: 2,
            loadingOffset: Math.min(20, Math.floor(window.innerWidth / 2 / 80)),
            book: false,
            thumbnails: true,
        };
        this.config = { ...defaults, ...config };
        this.isOpen = false;
        this.isPlaying = false;
        this.isZoomed = false;
        this.containers = {};
        this.current = { index: 0, slide: null, thumbnailSlide: null };
        this.secondPage = { index: 1, slide: null, thumbnailSlide: null };
        this.wasCurrent = { index: 0, slide: null, thumbnailSlide: null };
        this.wasSecondPage = { index: 0, slide: null, thumbnailSlide: null };
        this.isForward = true;
        this.drag = {
            isDragging: false,
            startX: null,
            startY: null,
            currentX: null,
            currentY: null,
            mX: null,
            mY: null,
            translateX: null,
        };
        this.autoPlayInterval = null;
        this.max = this.config.sources ? this.config.sources.length : 0;
        this.containerElement = null;
        this.cache = {
            slides: {},
            thumbnails: {},
        };
        this.visibleSlides = new Set();
        this.fixedProperties = {
            "--ig-lightbox-translateX": 0,
            "--ig-lightbox-scale": 0,
            "--ig-lightbox-track-transition-multiplier": 0,
        };
        this.updateConsents = false;
        this.availableEvents = {
            open: "lightbox:open",
            close: "lightbox:close",
            goTo: "lightbox:goTo",
        };
        this.eventListeners = [];
    }

    /**
     * Attaches an event listener and tracks it for later removal.
     * @param {EventTarget} element - The DOM element to attach the listener to.
     * @param {string} event - The event name.
     * @param {Function} handler - The event handler function.
     * @private
     */
    _addManagedEventListener(element, event, handler) {
        element.addEventListener(event, handler);
        this.eventListeners.push({ element, event, handler });
    }

    /**
     * Binds initial event listeners to image elements.
     * @private
     */
    _initEventListeners() {
        if (this.max > 0) {
            this._addManagedEventListener(document, "click", (e) => {
                const anchor = e.target.closest('a.lightbox, a[data-fslightbox="gallery"]');
                if (anchor) {
                    e.preventDefault();
                    const imagesArr = Array.from(this.config.sources);
                    const index = imagesArr.indexOf(anchor);
                    if (index !== -1) {
                        this._handleImageClick(e, index, anchor);
                    }
                }
            });
        }
    }

    /**
     * Removes all managed event listeners to prevent memory leaks.
     * @private
     */
    _removeEventListeners() {
        this.eventListeners.forEach(({ element, event, handler }) => {
            element.removeEventListener(event, handler);
        });
        this.eventListeners = [];
    }

    /**
     * Handles the click event on a gallery image.
     * @param {Event} e - The click event.
     * @param {number} index - The index of the clicked image.
     * @param {HTMLElement} el - The clicked element.
     * @private
     */
    _handleImageClick(e, index, el) {
        e.preventDefault();
        this.open();
        this.goTo(index);
        this.containers.triggerElement = el;
    }

    /**
     * Sets up the lightbox DOM and initial event listeners.
     * @private
     */
    _init() {
        this._addOverlay();
        this._loadContainers();
        this._setupDrag();
        this._setUpKeyBoardEvents();
        this._setupActionButtons();
        if (this.config.zoom) {
            this._setupScrollZoom();
        }
    }

    /**
     * Injects the lightbox element into the DOM.
     * @private
     */
    _addOverlay() {
        this.containerElement = this._createContainerElement();
        document.body.appendChild(this.containerElement);
    }

    /**
     * Caches references to key DOM elements.
     * @private
     */
    _loadContainers() {
        this.containers = {
            container: document.querySelector(`[data-id="${this.config.id}"].ig-lightbox-container`),
            actionsContainer: document.querySelector(`[data-id="${this.config.id}"] .ig-lightbox-action-container`),
            actions: document.querySelectorAll(`[data-id="${this.config.id}"] .ig-lightbox-action`),
            slidesTrack: document.querySelector(`[data-id="${this.config.id}"] .ig-lightbox-slides .ig-lightbox-track`),
            thumbnailsTrack: document.querySelector(`[data-id="${this.config.id}"] .ig-lightbox-thumbnails .ig-lightbox-track`),
        };
    }

    /**
     * Binds event listeners for drag/swipe functionality.
     * @private
     */
    _setupDrag() {
        if (this.config.draggable) {
            const dragHandler = (e) => this._initDrag(e);
            this._addManagedEventListener(document, "mousedown", dragHandler);
            this._addManagedEventListener(document, "mousemove", dragHandler);
            this._addManagedEventListener(document, "mouseup", dragHandler);
            this._addManagedEventListener(document, "touchstart", dragHandler, { passive: false });
            this._addManagedEventListener(document, "touchmove", dragHandler, { passive: false });
            this._addManagedEventListener(document, "touchend", dragHandler, { passive: false });
        }
    }

    /**
     * Handles the start of a drag/swipe interaction.
     * @param {Event} e - The mousedown or touchstart event.
     * @private
     */
    _initDrag(e) {
        const touchEvent = e.type.startsWith("touch");
        if (!touchEvent) {
            e.preventDefault();
        }

        if (touchEvent && e.touches.length > 1) {
            e.preventDefault();
        }

        const clientX = touchEvent ? e.touches[0]?.clientX : e.clientX;
        const clientY = touchEvent ? e.touches[0]?.clientY : e.clientY;

        const isValidTarget = !!e.target.closest(".ig-lightbox-slides");
        if (!isValidTarget) return;

        switch (e.type) {
            case "mousedown":
            case "touchstart":
                this._startDrag(clientX, clientY);
                break;
            case "mousemove":
            case "touchmove":
                this._duringDrag(clientX, clientY);
                break;
            case "mouseup":
            case "touchend":
                this._endDrag(this.drag.mX, this.drag.mY);
                break;
        }
    }

    /**
     * Sets initial state at the start of a drag operation.
     * @param {number} clientX - The horizontal coordinate of the pointer.
     * @param {number} clientY - The vertical coordinate of the pointer.
     * @private
     */
    _startDrag(clientX, clientY) {
        if (this.isOpen) {
            this.drag.isDragging = true;
            this.drag.startX = clientX;
            this.drag.startY = clientY;
            this.drag.mX = clientX;
            this.drag.mY = clientY;
            this.containers.slidesTrack.classList.add("is-dragging");
            this._setTransitionMultiplier(1);
        }
    }

    /**
     * Updates slide position during a drag operation.
     * @param {number} clientX - The horizontal coordinate of the pointer.
     * @param {number} clientY - The vertical coordinate of the pointer.
     * @private
     */
    _duringDrag(clientX, clientY) {
        if (!this.isOpen || !this.drag.isDragging) return;

        this._updateDrag(clientX, clientY);
        this.drag.mX = clientX;
        this.drag.mY = clientY;

        if (this.isZoomed) {
            this._updateZoomPosition(this.drag.currentX, this.drag.currentY);
        } else {
            const newTranslateX = Math.max(0, Math.min(this.max - 1, this.drag.translateX));
            this._updateTrackPosition(newTranslateX);
        }
    }

    /**
     * Updates the drag state based on the current pointer position.
     * @param {number} clientX - The clientX position of the pointer.
     * @param {number} clientY - The clientY position of the pointer.
     * @private
     */
    _updateDrag(clientX, clientY) {
        if (this.isOpen && this.drag.isDragging) {
            this.drag.currentX = clientX - this.drag.startX;
            this.drag.currentY = clientY - this.drag.startY;
            if (!this.isZoomed) {
                const maxDragLimit = this.config.dragPxLimit;
                const containerWidth = this.containers.container.clientWidth;
                const dragThresholdPx = containerWidth < maxDragLimit * 3 ? containerWidth / 3 : maxDragLimit;
                const normalizedShift = Math.max(-1, Math.min(-this.drag.currentX / dragThresholdPx, 1));
                this.drag.translateX = this.current.index + normalizedShift;

                if (normalizedShift > 0 && this.config.book) {
                    this.drag.translateX += 1;
                }
                this.drag.index = Math.round(this.drag.translateX);
            }
        }
    }

    /**
     * Finalizes the slide position at the end of a drag operation.
     * @param {number} clientX - The horizontal coordinate of the pointer.
     * @param {number} clientY - The vertical coordinate of the pointer.
     * @private
     */
    _endDrag(clientX, clientY) {
        if (!this.isOpen || !this.drag.isDragging) return;

        if (!this.drag.index) {
            this._updateDrag(clientX, clientY);
        }
        this.drag.isDragging = false;
        this.containers.slidesTrack.classList.remove("is-dragging");
        if (this.isZoomed) {
            this.current.lastStaticX = this.current.currentX;
            this.current.lastStaticY = this.current.currentY;
            this.current.wasDragged = true;
        } else {
            const targetIndex = (this.drag.index + this.max) % this.max;
            const isLooping = this.drag.index < 0 || this.drag.index >= this.max;
            if (isLooping) {
                this._setTransitionMultiplier(0);
            } else {
                this._setTransitionMultiplier(1);
            }
            this.goTo(targetIndex);
        }
    }

    /**
     * Binds keyboard event listeners for navigation.
     * @private
     */
    _setUpKeyBoardEvents() {
        if (this.max > 1) {
            this._addManagedEventListener(document, "keydown", (event) => this._handleKeyboardEvents(event));
        }
    }

    /**
     * Handles keyboard shortcuts for lightbox control.
     * @param {KeyboardEvent} event - The keyboard event.
     * @private
     */
    _handleKeyboardEvents(event) {
        if (this.isOpen) {
            switch (event.key) {
                case "Escape":
                    this.close();
                    break;
                case "a":
                case "ArrowLeft":
                    this.prev();
                    break;
                case "d":
                case "ArrowRight":
                    this.next();
                    break;
                case "+":
                    this.zoomIn();
                    break;
                case "-":
                    this.zoomOut();
                    break;
            }
        }
    }

    /**
     * Resets attributes and classes on all slides.
     * @private
     */
    _resetCurrentAttributes() {
        if (this.current.slide) {
            this.current.wasDragged = false;
            this._updateZoomPosition(0, 0);
        }
        const allSlides = Array.from(this.containers.slidesTrack.children || []);
        const allThumbnails = Array.from(this.containers.thumbnailsTrack?.children || []);

        [...allSlides, ...allThumbnails].forEach((element) => {
            element.isZoomed = false;
            element.classList.remove("is-current", "is-zoomed", "was-current", "is-animating");
            element.setAttribute("aria-selected", "false");
            element.setAttribute("tabindex", "-1");
            const elementIndex = parseInt(element.dataset.index);
            if (this.visibleSlides.has(elementIndex)) {
                element.classList.add("is-visible");
            } else {
                element.classList.remove("is-visible");
            }
        });

        this.containers.container.classList.remove("is-zoomed");
    }

    /**
     * Applies 'is-current' class and ARIA attributes to the active slide(s).
     * @private
     */
    _updateCurrentSlideAttributes() {
        const newPages = [this.current];
        if (this.config.book && this.secondPage.slide) {
            newPages.push(this.secondPage);
        }
        newPages.forEach((page) => {
            [page.slide, page.thumbnailSlide].forEach((element) => {
                element?.classList.add("is-current");
            });
            page.slide.setAttribute("aria-selected", "true");
            page.slide.setAttribute("tabindex", "0");
        });
        if (this.config.book) {
            const animatingPages = [];
            const oldLeftSlide = this.wasCurrent.slide;
            const oldRightSlide = this.wasCurrent.index == 0 ? this.wasCurrent.slide : this.wasSecondPage.slide;
            [oldLeftSlide, oldRightSlide].forEach((element) => {
                if (element) {
                    element.classList.add("was-current");
                }
            });
            if (this.isForward) {
                if (oldRightSlide) {
                    animatingPages.push(oldRightSlide);
                }
                if (this.current.slide) {
                    animatingPages.push(this.current.slide);
                }
            } else {
                if (oldLeftSlide) {
                    animatingPages.push(oldLeftSlide);
                }
                if (this.secondPage.slide) {
                    animatingPages.push(this.secondPage.slide);
                }
            }
            animatingPages.forEach((element) => {
                if (element) {
                    element.classList.add("is-animating");
                }
            });
        }
    }

    /**
     * Binds click event listeners to action buttons in the toolbar.
     * @private
     */
    _setupActionButtons() {
        this._addManagedEventListener(this.containers.actionsContainer, "click", (e) => {
            e.preventDefault();
            if (e.target.classList.contains("ig-lightbox-action")) {
                this._handleActionButtonClick(e.target);
            }
        });
    }

    /**
     * Handles clicks on action buttons.
     * @param {HTMLElement} action - The clicked action button.
     * @private
     */
    _handleActionButtonClick(action) {
        const method = action.dataset.action;
        if (method !== "playToggle") {
            this._resetAutoPlay();
        }
        this[method]();
    }

    /**
     * Stops the autoplay slideshow.
     * @private
     */
    _resetAutoPlay() {
        clearInterval(this.autoPlayInterval);
        this.containers.container.classList.remove("is-playing");
    }

    /**
     * Binds event listeners for zoom functionality (wheel and double-click).
     * @private
     */
    _setupScrollZoom() {
        this._addManagedEventListener(this.containers.container, "wheel", (e) => this._handleWheelZoom(e));
        this._addManagedEventListener(this.containers.slidesTrack, "dblclick", (e) => this._handleDoubleClickZoom(e));
    }

    /**
     * Handles zooming via the mouse wheel.
     * @param {WheelEvent} e - The wheel event.
     * @private
     */
    _handleWheelZoom(e) {
        if (e.target.closest(".ig-lightbox-slides")) {
            e.preventDefault();
            if (e.deltaY < 0) this.zoomIn();
            else this.zoomOut();
        }
    }

    /**
     * Handles zooming via double-clicking.
     * @param {MouseEvent} e - The click event.
     * @private
     */
    _handleDoubleClickZoom(e) {
        e.preventDefault();
        if (!this.isZoomed) {
            const doubleClickZoomScale = 2;
            this.current.wasDragged = true;
            this.current.lastStaticX = parseInt((window.innerWidth / 2 - e.clientX) * doubleClickZoomScale);
            this.current.lastStaticY = parseInt((window.innerHeight / 2 - e.clientY) * doubleClickZoomScale);
            this._updateZoomPosition(0, 0);
            this.zoomIn(doubleClickZoomScale);
        } else {
            this.zoomOut(10);
        }
    }

    /**
     * Creates the main DOM structure for the lightbox container.
     * @returns {HTMLDivElement} The lightbox container element.
     * @private
     */
    _createContainerElement() {
        const hasMultipleImages = this.max > 1;
        const showAutoplay = hasMultipleImages && this.config.autoPlay;
        const showThumbnails = hasMultipleImages && this.config.thumbnails;
        const fullscreen = typeof document.documentElement.requestFullscreen != "undefined" && this.config.fullscreen;
        const isBook = this.config.book;

        const container = document.createElement("div");
        container.tabIndex = -1;
        container.dataset.id = this.config.id;
        container.className = `ig-lightbox-container${showThumbnails ? " active-thumbnails" : ""}${isBook ? " is-book" : ""}`;

        const actionsContainer = document.createElement("div");
        actionsContainer.className = "ig-lightbox-action-container";

        if (hasMultipleImages) {
            actionsContainer.appendChild(this._createActionButton("prev", "Previous image"));
            actionsContainer.appendChild(this._createActionButton("next", "Next image"));
            const spacer = document.createElement("span");
            spacer.className = "ig-lightbox-spacer";
            actionsContainer.appendChild(spacer);
        }
        if (showThumbnails) {
            actionsContainer.appendChild(this._createActionButton("thumbnailsToggle", "Toggle Thumbnails"));
        }
        if (this.config.zoom) {
            actionsContainer.appendChild(this._createActionButton("zoomOut", "Zoom out"));
            actionsContainer.appendChild(this._createActionButton("zoomIn", "Zoom in"));
        }
        if (this.config.download) {
            actionsContainer.appendChild(this._createActionButton("download", "Download"));
        }
        if (showAutoplay) {
            actionsContainer.appendChild(this._createActionButton("playToggle", "Autoplay"));
        }
        if (fullscreen) {
            actionsContainer.appendChild(this._createActionButton("fullscreenToggle", "Toggle Fullscreen"));
        }
        actionsContainer.appendChild(this._createActionButton("close", "Close Lightbox"));

        container.appendChild(actionsContainer);

        const slides = document.createElement("div");
        slides.className = "ig-lightbox-slides";
        const slidesTrack = document.createElement("div");
        slidesTrack.className = "ig-lightbox-track";
        slidesTrack.setAttribute("role", "listbox");
        slidesTrack.setAttribute("aria-live", "polite");
        slides.appendChild(slidesTrack);
        container.appendChild(slides);

        if (showThumbnails) {
            const thumbnails = document.createElement("div");
            thumbnails.className = "ig-lightbox-thumbnails";
            const thumbnailsTrack = document.createElement("div");
            thumbnailsTrack.className = "ig-lightbox-track";
            thumbnailsTrack.setAttribute("role", "listbox");
            thumbnailsTrack.setAttribute("aria-live", "polite");
            thumbnails.appendChild(thumbnailsTrack);
            container.appendChild(thumbnails);
        }
        return container;
    }

    /**
     * Creates an action button element.
     * @param {string} action - The action name.
     * @param {string} ariaLabel - The ARIA label for the button.
     * @returns {HTMLButtonElement} The button element.
     * @private
     */
    _createActionButton(action, ariaLabel) {
        const button = document.createElement("button");
        button.className = "ig-lightbox-action";
        button.dataset.action = action;
        button.setAttribute("aria-label", ariaLabel);
        button.title = ariaLabel;
        return button;
    }

    _createSlideElement(slideContainer, index, thumbs) {
        const dataset = slideContainer.dataset;
        const slide = document.createElement("div");
        this._setSlideAttributes(slide, dataset, index, thumbs);
        const content = this._createSlideContent(dataset, thumbs, slideContainer, index);
        slide.appendChild(content);
        if (!thumbs && !this.config.book) {
            const caption = this._createCaptionElement(dataset, index, slideContainer);
            if (caption) {
                slide.appendChild(caption);
            }
        }
        return slide;
    }

    /**
     * Generates attributes for a slide element.
     * @param {DOMStringMap} dataset - The dataset of the original image link.
     * @param {number} index - The index of the slide.
     * @param {boolean} [thumbs=false] - Whether the slide is a thumbnail.
     * @returns {string} The HTML attributes for the slide.
     * @private
     */
    _setSlideAttributes(element, dataset, index, thumbs = false) {
        if (!thumbs) {
            Object.entries(dataset).forEach(([key, value]) => {
                if (key !== "tag") {
                    element.dataset[key] = value;
                }
            });
            element.setAttribute("aria-labelledby", `caption-title-${index}`);
            element.setAttribute("aria-describedby", `caption-description-${index}`);
            element.setAttribute("aria-selected", index === this.current.index ? "true" : "false");
            element.setAttribute("tabindex", "-1");
        }
        if (thumbs) {
            const page = (() => {
                if (index === 0) {
                    return 1;
                }
                const firstPage = Math.floor((index - 1) / 2) * 2 + 2;
                const secondPage = firstPage + 1;

                if (!this.config.book) {
                    return index + 1;
                }
                if (index === this.max - 1) {
                    return this.max % 2 === 0 ? firstPage : secondPage;
                }
                return `${firstPage} - ${secondPage}`;
            })();
            element.dataset.page = page;
        }
        element.dataset.index = index;
        element.className = `ig-lightbox-slide ${index === this.current.index ? "is-current is-visible" : ""}`;
    }

    /**
     * Creates the inner content element for a slide.
     * @param {DOMStringMap} dataset - The dataset of the original image link.
     * @param {boolean} thumbs - Whether the slide is a thumbnail.
     * @param {HTMLElement} slideContainer - The original image link element.
     * @param {number} index - The index of the slide.
     * @returns {HTMLElement} The inner content element for the slide.
     * @private
     */
    _createSlideContent(dataset, thumbs, slideContainer, index) {
        const customContent = this._getAjaxContent(dataset, slideContainer, index) || this._getCustomContent(dataset);
        if (customContent) {
            if (thumbs) {
                const placeholder = document.createElement("div");
                placeholder.className = "ig-lightbox-slide-placeholder";
                return placeholder;
            }
            const customContentEl = document.createElement("div");
            customContentEl.className = "ig-lightbox-custom-content";
            if (customContent !== true) {
                customContentEl.innerHTML = customContent;
            }
            return customContentEl;
        }

        const img = document.createElement("img");
        img.loading = "lazy";
        img.src = slideContainer.href;
        return img;
    }

    /**
     * Fetches and injects content for an AJAX slide.
     * @param {DOMStringMap} dataset - The dataset of the original image link.
     * @param {HTMLElement} slideContainer - The original image link element.
     * @param {number} index - The index of the slide.
     * @returns {boolean|undefined} True if the content is being fetched.
     * @private
     */
    _getAjaxContent(dataset, slideContainer, index) {
        if (dataset.ajax || slideContainer.rel) {
            this.updateConsents = true;
            const url = slideContainer.getAttribute("rel") || slideContainer.getAttribute("href");
            this._fetchAjax(url, index, dataset?.ajaxJson, dataset?.ajaxAttr);
            return true;
        }
        return false;
    }

    /**
     * Retrieves custom HTML content from a data attribute.
     * @param {DOMStringMap} dataset - The dataset of the original image link.
     * @returns {string|boolean} The sanitized HTML or false.
     * @private
     */
    _getCustomContent(dataset) {
        if (dataset.html) {
            this.updateConsents = true;
            return dataset.html;
        }
        return false;
    }

    /**
     * Creates the caption element for a slide.
     * @param {DOMStringMap} dataset - The dataset of the original image link.
     * @param {number} index - The index of the slide.
     * @param {HTMLElement} slideContainer - The original image link element.
     * @returns {HTMLDivElement|null} The caption element or null if no data.
     * @private
     */
    _createCaptionElement(dataset, index, slideContainer) {
        const title = slideContainer.title || dataset.title || dataset.alt;
        const description = dataset.caption || dataset.description;
        if (!title && !description) return null;

        const caption = document.createElement("div");
        caption.className = "ig-lightbox-caption";

        if (title) {
            const titleEl = document.createElement("div");
            titleEl.id = `caption-title-${index}`;
            titleEl.className = "ig-lightbox-caption-title";
            const strongEl = document.createElement("strong");
            strongEl.textContent = title;
            titleEl.appendChild(strongEl);
            caption.appendChild(titleEl);
        }

        if (description) {
            const descriptionEl = document.createElement("div");
            descriptionEl.id = `caption-description-${index}`;
            descriptionEl.className = "ig-lightbox-caption-description";
            descriptionEl.textContent = description;
            caption.appendChild(descriptionEl);
        }

        return caption;
    }
    _addSlide(index, type, container, isThumb = false) {
        const actualIndex = ((index % this.max) + this.max) % this.max;
        const group = this.cache[type];
        const imgData = this.config.sources[actualIndex];
        if (group[actualIndex] || !imgData) return;
        const slide = this._createSlideElement(imgData, actualIndex, isThumb);
        group[actualIndex] = slide;
        container.appendChild(slide);
    }

    _addSlides(from, to, type, container, isThumb = false) {
        for (let i = from; i <= to; i++) {
            this._addSlide(i, type, container, isThumb);
        }
    }
    _preload(targetIndex) {
        const startIndex = this.wasCurrent.index;
        const getLimit = (offset) => Math.min(this.max, targetIndex + offset);
        this._addSlides(startIndex, getLimit(this.config.preload), "slides", this.containers.slidesTrack);
        if (this.config.thumbnails && this.max > 1) {
            const thumbLimit = getLimit(this.config.preload + this.config.loadingOffset);
            this._addSlides(startIndex, thumbLimit, "thumbnails", this.containers.thumbnailsTrack, true);
        }
    }
    /**
     * Creates and injects the slide and thumbnail elements into the DOM.
     * @private
     */
    _generateSlides() {
        this._preload(this.current.index);
        this.containers.images = document.querySelectorAll(`[data-id="${this.config.id}"] .ig-lightbox-slides .ig-lightbox-track .ig-lightbox-slide`);
        if (this.config.book) {
            this._addManagedEventListener(this.containers.slidesTrack, "transitionend", (e) => {
                const element = e.target;
                if (element.classList.contains("ig-lightbox-slide")) {
                    if (e.propertyName !== "rotate") return;
                    if (element.classList.contains("is-animating")) {
                        element.classList.remove("is-animating");
                    }
                    if (element.classList.contains("was-current")) {
                        element.classList.remove("was-current");
                    }
                }
            });
        }
        if (this.config.thumbnails && this.max > 1) {
            this._setupThumbnailClick();
        }
    }

    /**
     * Binds click event listeners to thumbnail images.
     * @private
     */
    _setupThumbnailClick() {
        this._addManagedEventListener(this.containers.thumbnailsTrack, "click", (e) => {
            const target = e.target;
            if (target.classList.contains("ig-lightbox-slide")) {
                this._setTransitionMultiplier(1);
                this.current.index = parseInt(target.dataset.index, 10);
                this.goTo(this.current.index);
                this._resetAutoPlay();
            }
        });
    }

    /**
     * Opens the lightbox.
     */
    open() {
        if (!this.containers.container) this._init();
        if (!this.containers.images) this._generateSlides();
        this.current = this._getSlideReferences(this.current.index || 0);
        this._updateBoundaryClasses(this.current.index);
        document.documentElement.classList.add("ig-lightbox-open");
        this.containers.container.setAttribute("aria-hidden", "false");
        this.containers.container.classList.add("active");
        this.containers.container.focus();
        this.isOpen = true;
        if (this.updateConsents) {
            window.IgConsentManager?.renderContextualConsentNotices();
        }
        this._triggerDispatchEvent(this.availableEvents.open);
    }

    /**
     * Navigates to the previous slide.
     */
    prev() {
        const step = this.config.book && this.current.index > 1 ? 2 : 1;
        let targetIndex = this.current.index - step;
        const isLooping = targetIndex < 0;
        if (isLooping) {
            targetIndex = (targetIndex + this.max) % this.max;
            this._setTransitionMultiplier(0);
        } else {
            this._setTransitionMultiplier(1);
        }
        this.goTo(targetIndex);
    }

    /**
     * Navigates to the next slide.
     */
    next() {
        const step = this.config.book && this.current.index + 2 < this.max ? 2 : 1;
        let targetIndex = this.current.index + step;
        const isLooping = targetIndex >= this.max;
        if (isLooping) {
            targetIndex = targetIndex % this.max;
            this._setTransitionMultiplier(0);
        } else {
            this._setTransitionMultiplier(1);
        }
        this.goTo(targetIndex);
    }

    /**
     * Toggles the autoplay functionality.
     */
    playToggle() {
        this.isPlaying = !this.isPlaying;
        if (this.isPlaying) {
            this.autoPlayInterval = setInterval(() => this.next(), this.config.autoPlayTime);
            this.containers.container.classList.add("is-playing");
        } else {
            clearInterval(this.autoPlayInterval);
            this.containers.container.classList.remove("is-playing");
        }
    }

    /**
     * Zooms out of the current image.
     * @param {number} [speed=this.config.zoomConfig.speed] - The zoom-out speed/step.
     */
    zoomOut(speed = this.config.zoomConfig.speed) {
        this.config.zoomConfig.scale = Math.max(this.config.zoomConfig.min, this.config.zoomConfig.scale - speed);
        if (this.config.zoomConfig.scale <= this.config.zoomConfig.min) {
            this._resetZoom();
        } else {
            this._doZoom();
        }
    }

    /**
     * Zooms in on the current image.
     * @param {number} [speed=this.config.zoomConfig.speed] - The zoom-in speed/step.
     */
    zoomIn(speed = this.config.zoomConfig.speed) {
        this.config.zoomConfig.scale = Math.min(this.config.zoomConfig.max, this.config.zoomConfig.scale + speed);
        if (this.config.zoomConfig.scale <= this.config.zoomConfig.max) {
            this._applyZoom();
        } else {
            this._doZoom();
        }
    }

    /**
     * Initiates a download of the current image file.
     */
    download() {
        const filePath = this.config.downloadTarget || this.current.slide?.dataset.orig || this.current.slide?.querySelector("img")?.src;

        if (filePath) {
            const fileName = filePath.substring(filePath.lastIndexOf("/") + 1);
            const link = document.createElement("a");
            link.href = filePath;
            link.download = fileName || "image-file";
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
    }

    /**
     * Resets the zoom level and position of the current slide.
     * @private
     */
    _resetZoom() {
        this.isZoomed = false;
        this.current.wasDragged = false;
        this._updateZoomPosition(0, 0);
        this.current.slide.classList.remove("is-zoomed");
        this.containers.container.classList.remove("is-zoomed");
        this.config.zoomConfig.scale = this.config.zoomConfig.min;
        this._doZoom();
    }

    /**
     * Applies the zoom effect to the current slide.
     * @private
     */
    _applyZoom() {
        this.isZoomed = true;
        this.current.slide.classList.add("is-zoomed");
        this.containers.container.classList.add("is-zoomed");
        if (this.current.slide.dataset.orig) {
            const img = this.current.slide.querySelector("img");
            if (img && !img.dataset?.swapped) {
                img.dataset.src = img.src;
                img.src = this.current.slide.dataset.orig;
                img.dataset.swapped = true;
            }
        }
        this._doZoom();
    }

    /**
     * Sets the CSS custom property for the zoom scale.
     * @private
     */
    _doZoom() {
        this._updateProperty(this.containers.container, "--ig-lightbox-scale", parseFloat(this.config.zoomConfig.scale).toFixed(2));
    }

    /**
     * Toggles the visibility of the thumbnail gallery.
     */
    thumbnailsToggle() {
        this.containers.container.classList.toggle("active-thumbnails");
    }

    /**
     * Toggles fullscreen mode for the lightbox.
     */
    fullscreenToggle() {
        this.containers.container.classList.toggle("is-fullscreen");
        if (!document.fullscreenElement) {
            document.documentElement.requestFullscreen();
        } else {
            document.exitFullscreen();
        }
    }

    /**
     * Closes the lightbox.
     */
    close() {
        document.documentElement.classList.remove("ig-lightbox-open");
        this.containers.container.setAttribute("aria-hidden", "true");
        this.containers.container.classList.remove("active");
        this._setTransitionMultiplier(0);
        this.containers.triggerElement?.focus();
        this.isOpen = false;
        this._triggerDispatchEvent(this.availableEvents.close);
    }

    /**
     * Closes the lightbox, removes event listeners, and removes the lightbox from the DOM.
     */
    destroy() {
        this.close();
        this._removeEventListeners();
        this.containers.container.remove();
    }

    _updateBoundaryClasses(index) {
        this.containers.container.classList.toggle("is-first", index === 0);
        this.containers.container.classList.toggle("is-last", index === this.max - 1);
    }

    /**
     * Navigates to a specific slide index.
     * @param {number} index - The index of the slide to navigate to.
     */
    goTo(index) {
        if (index > 0) this._preload(index);
        this._updateBoundaryClasses(index);
        this._prepareSlideTransition(index);
        const { actualIndex, secondIndex } = this._calculateBookIndices(index);
        this._updateSlideReferences(actualIndex, secondIndex);
        this._finalizeSlideTransition();
    }

    /**
     * Prepares the lightbox for a slide transition.
     * @param {number} index - The target slide index.
     * @private
     */
    _prepareSlideTransition(index) {
        this.isZoomed = false;
        this._removeCustomProperties();
        this._updateProperty(this.containers.container, "--ig-lightbox-translateX", index);
        this.wasCurrent = { ...this.current };
        this.wasSecondPage = { ...this.secondPage };
        this.config.zoomConfig.scale = 1;
        this._updateProperty(this.containers.container, "--ig-lightbox-scale", this.config.zoomConfig.scale);
        this._setVisibleSlides(index);
        this._resetCurrentAttributes();
    }

    _setVisibleSlides(index) {
        this.visibleSlides.clear();
        const start = -this.config.visibleSlidesAmount;
        const end = this.config.visibleSlidesAmount;
        for (let i = start; i <= end; i++) {
            const slideIndex = Math.max(0, Math.min(index + i, this.max - 1));
            this.visibleSlides.add(slideIndex);
        }
    }

    /**
     * Calculates slide indices for book mode.
     * @param {number} index - The target slide index.
     * @returns {{actualIndex: number, secondIndex: number|null}}
     * @private
     */
    _calculateBookIndices(index) {
        if (!this.config.book) {
            return { actualIndex: index, secondIndex: null };
        }

        if (index === 0) {
            return { actualIndex: 0, secondIndex: null };
        }

        const actualIndex = index % 2 !== 0 ? index : index - 1;
        const secondIndex = actualIndex + 1 < this.max ? actualIndex + 1 : null;

        return { actualIndex, secondIndex };
    }

    /**
     * Updates references to the current and second page slides.
     * @param {number} actualIndex - The index of the current slide.
     * @param {number|null} secondIndex - The index of the second slide.
     * @private
     */
    _updateSlideReferences(actualIndex, secondIndex) {
        this.drag.index = actualIndex;
        this.current = this._getSlideReferences(actualIndex);
        this.secondPage = secondIndex !== null ? this._getSlideReferences(secondIndex) : { index: null, slide: null, thumbnailSlide: null };
    }

    /**
     * Retrieves references to a slide and its corresponding thumbnail.
     * @param {number} index - The slide index.
     * @returns {{index: number, slide: Element, thumbnailSlide: Element}}
     * @private
     */
    _getSlideReferences(index) {
        const slide = this.containers.slidesTrack.querySelector(`[data-index="${index}"]`);
        const thumbnailSlide = this.containers.thumbnailsTrack?.querySelector(`[data-index="${index}"]`);
        this.isForward = this.current.index > this.wasCurrent.index;
        return { index, slide, thumbnailSlide };
    }

    /**
     * Finalizes a slide transition by updating attributes and firing events.
     * @private
     */
    _finalizeSlideTransition() {
        this._addCustomProperties();
        this._updateCurrentSlideAttributes();
        this._triggerDispatchEvent(this.availableEvents.goTo);
    }

    /**
     * Updates the zoom position of the current slide.
     * @param {number} currentX - The current X position of the pointer.
     * @param {number} currentY - The current Y position of the pointer.
     * @private
     */
    _updateZoomPosition(currentX, currentY) {
        if (this.current.wasDragged) {
            currentX = currentX + this.current.lastStaticX;
            currentY = currentY + this.current.lastStaticY;
        }
        this.current.currentX = currentX;
        this.current.currentY = currentY;
        this._updateProperty(this.containers.slidesTrack, "--ig-lightbox-img-translateX", `${currentX}px`);
        this._updateProperty(this.containers.slidesTrack, "--ig-lightbox-img-translateY", `${currentY}px`);
    }

    /**
     * Updates the horizontal position of the slide track.
     * @param {number} value - The new translateX value.
     * @private
     */
    _updateTrackPosition(value) {
        this._updateProperty(this.containers.container, "--ig-lightbox-translateX", value);
    }

    /**
     * Sets or removes a CSS custom property on an element.
     * @param {HTMLElement} element - The element to modify.
     * @param {string} property - The name of the CSS property.
     * @param {string|number} value - The value to set.
     * @private
     */
    _updateProperty(element, property, value) {
        if (parseInt(value) == 0) {
            element.style.removeProperty(property);
        } else {
            element.style.setProperty(property, value);
        }
    }

    /**
     * Applies custom properties from the current slide's dataset to the container.
     * @private
     */
    _addCustomProperties() {
        if (this.current.slide.dataset.vars) {
            this.current.slide.dataset.vars
                .split(";")
                .filter(Boolean)
                .forEach((cssVar) => {
                    const [property, value] = cssVar.split(":");
                    this._updateProperty(this.containers.container, property.trim(), value.trim());
                });
        }
    }

    /**
     * Removes all custom properties from the container, preserving fixed properties.
     * @private
     */
    _removeCustomProperties() {
        for (const property in this.fixedProperties) {
            this.fixedProperties[property] = this.containers.container.style.getPropertyValue(property);
        }
        this.containers.container.style.cssText = "";
        for (const property in this.fixedProperties) {
            if (this.fixedProperties[property]) {
                this.containers.container.style.setProperty(property, this.fixedProperties[property]);
            }
        }
    }

    /**
     * Sets the transition speed multiplier.
     * @param {number} multiplier - The multiplier value.
     * @private
     */
    _setTransitionMultiplier(multiplier) {
        this._updateProperty(this.containers.container, "--ig-lightbox-track-transition-multiplier", multiplier);
    }

    /**
     * Dispatches a custom event from the lightbox container.
     * @param {string} name - The name of the event to dispatch.
     * @private
     */
    _triggerDispatchEvent(name) {
        this.containers.container.dispatchEvent(
            new CustomEvent(name, {
                detail: {
                    current: this.current,
                    slides: this.containers.images,
                },
            })
        );
    }

    /**
     * Fetches content for a slide via AJAX.
     * @param {string} url - The URL to fetch.
     * @param {number} index - The index of the slide to update.
     * @param {boolean} [json=false] - Whether to parse the response as JSON.
     * @param {string|null} [attr=null] - The attribute to extract from the JSON response.
     * @private
     */
    async _fetchAjax(url, index, json = false, attr = null) {
        try {
            const response = await fetch(url);
            if (!response.ok) throw new Error("Network response was not ok");

            let data = json ? await response.json() : await response.text();
            if (attr) data = data[attr];

            const container = this.containers.slidesTrack.querySelector(`[data-index="${index}"] .ig-lightbox-custom-content`);
            if (container) {
                if (typeof data === "object" && data !== null) {
                    container.innerHTML = JSON.stringify(data, null, 2);
                } else {
                    if (!container.shadowRoot) {
                        container.attachShadow({ mode: "open" });
                    }
                    container.shadowRoot.innerHTML = data;
                }
            }
        } catch (error) {
            console.error("There has been a problem with your fetch operation:", error);
        }
    }
}

document.addEventListener("DOMContentLoaded", () => {
    let igLightboxImages = document.querySelectorAll('[data-fslightbox="gallery"]');
    if (igLightboxImages.length === 0) {
        igLightboxImages = document.querySelectorAll(".lightbox");
    }
    if (igLightboxImages.length > 0) {
        const config = typeof IgLightboxConfig !== "undefined" ? IgLightboxConfig : {};
        window.IgLightbox = new IgLightbox({ ...{ sources: igLightboxImages }, ...config });
    }
});
