class IgLightbox {
    /**
     * Create a lightbox.
     * @param {Object} config - The configuration object for the lightbox.
     * @param {Object} [config.images=null] - The image objects
     * @param {number} [config.dragSpeedMultiplier=0.2] - The multiplier for drag speed.
     * @param {boolean} [config.draggable=true] - Whether the images are draggable.
     * @param {boolean} [config.autoPlay=true] - Whether the lightbox should display auto-play button.
     * @param {number} [config.autoPlayTime=3000] - The interval for auto-play in milliseconds.
     * @param {boolean} [config.zoom=true] - Zoom on images
     * @param {Object} [config.zoomConfig] - The zoom configuration.
     * @param {number} [config.zoomConfig.min=1] - The minimum zoom level.
     * @param {number} [config.zoomConfig.max=5] - The maximum zoom level.
     * @param {number} [config.zoomConfig.scale=1] - The initial zoom scale.
     * @param {number} [config.zoomConfig.speed=0.1] - The zoom speed.
     */
    constructor(config) {
        this.loaded = { containers: false, slides: false, thumbnails: false, consentUpdated: false };
        this.isOpen = false;
        this.defaults = { id: 1, images: null, dragSpeedMultiplier: 0.4, draggable: true, autoPlay: true, autoPlayTime: 3000, fullscreen: true, zoom: true, zoomConfig: { min: 1, max: 5, scale: 1, speed: 0.1 } };
        this.config = { ...this.defaults, ...config }
        this.containers = {};
        this.current = { index: 0, slide: null, thumbnailSlide: null };
        this.drag = { isDragging: false, startX: null, startY: null, currentX: null, currentY: null, mX: null, mY: null, translateX: null }
        this.autoPlayInterval = null;
        this.max = 1;
        this.addClickEvent();
        this.containerHtml = this.getContainerHtml();
        this.fixedProperties = { '--ig-lightbox-translateX': 0, '--ig-lightbox-scale': 0, '--ig-lightbox-track-transition-multiplier': 0 };
        this.updateConsents = false;
        this.availableEvents = {
            open: 'lightbox:open',
            close: 'lightbox:close',
            goTo: 'lightbox:goTo',
        };
    }
    /**
     * Add click event listeners to the images.
     */
    addClickEvent() {
        this.max = this.config.images.length;
        if (this.max) {
            this.config.images.forEach((img, index) => img.addEventListener('click', (e) => this.handleImageClick(e, index, img)));
        }
    }
    /**
     * Setup Image Click:
     * - Open Lightbox
     * - Go to Image Index
     * - add triggerElement so focus can be set for accessibility
     */
    handleImageClick(e, index, el) {
        e.preventDefault();
        this.open();
        this.goTo(index);
        this.containers.triggerElement = el;
    }
    /**
     * Initializes the lightbox by performing several setup steps.
     */
    init() {
        this.addOverlay();
        this.loadContainers();
        this.setupDrag();
        this.setUpKeyBoardEvents();
        this.setupActionButtons();
        if (this.config.zoom) {
            this.setupScrollZoom();
        }
    }
    /**
     * Adds the necessary HTML overlay to the body tag.
     */
    addOverlay() {
        document.body.insertAdjacentHTML('beforeend', this.containerHtml);
    }
    /**
     * Finds and stores the required container elements.
     */
    loadContainers() {
        this.containers = {
            container: document.querySelector('[data-id="' + this.config.id + '"].ig-lightbox-container'),
            actionsContainer: document.querySelector('[data-id="' + this.config.id + '"] .ig-lightbox-action-container'),
            actions: document.querySelectorAll('[data-id="' + this.config.id + '"] .ig-lightbox-action'),
            imagesTrack: document.querySelector('[data-id="' + this.config.id + '"] .ig-lightbox-slides .ig-lightbox-track'),
            thumbnailsTrack: document.querySelector('[data-id="' + this.config.id + '"] .ig-lightbox-thumbnails .ig-lightbox-track'),
        };
        this.loaded.containers = true;
    }
    /**
     * Sets up drag functionality if configured.
     */
    setupDrag() {
        if (this.config.draggable) {
            const dragHandler = (e) => this.initDrag(e);
            document.addEventListener('mousedown', dragHandler);
            document.addEventListener('mousemove', dragHandler);
            document.addEventListener('mouseup', dragHandler);
            document.addEventListener('touchstart', dragHandler);
            document.addEventListener('touchmove', dragHandler);
            document.addEventListener('touchend', dragHandler);
        }
    }
    /**
     * Init Drag:
     * - startDrag (mousedown, touchstart)
     * - duringDrag (mousemove, touchmove)
     * - endDrag (mouseup, touchend)
     * @param {MouseEvent} event - The Mouse event object.
     */
    initDrag(e) {
        const touchEvent = e.type.startsWith('touch');
        let clientX, clientY;
        let isValidTarget = e.target.closest('.ig-lightbox-slides') ? true : false;
        if (!touchEvent) {
            e.preventDefault();
        }
        if (e.type === 'mousedown' || e.type === 'touchstart' || e.type === 'mousemove' || e.type === 'touchmove') {
            clientX = touchEvent ? e.touches[0].clientX : e.clientX;
            clientY = touchEvent ? e.touches[0].clientY : e.clientY;
        }
        if (isValidTarget && (e.type === 'mousedown' || e.type === 'touchstart')) {
            this.startDrag(clientX, clientY);
        } else if (isValidTarget && (e.type === 'mousemove' || e.type === 'touchmove')) {
            this.duringDrag(clientX, clientY);
        } else if (e.type === 'mouseup' || e.type === 'touchend') {
            this.endDrag(this.drag.mX, this.drag.mY);
        }
    }
    /**
     * startDrag (mousedown, touchstart)
     * @param {number} clientX - The Mouse X position.
     * @param {number} clientY - The Mouse Y position.
     */
    startDrag(clientX, clientY) {
        if (this.isOpen) {
            this.drag.isDragging = true;
            this.drag.startX = clientX;
            this.drag.startY = clientY;
            this.drag.mX = clientX;
            this.drag.mY = clientY;
            this.containers.imagesTrack.classList.add('is-dragging');
            this.setTransitionMultiplier(1);
        }
    }
    /**
     * duringDrag (mousemove, touchmove)
     * @param {number} clientX - The Mouse X position.
     * @param {number} clientY - The Mouse Y position.
     */
    duringDrag(clientX, clientY) {
        if (this.isOpen && this.drag.isDragging) {
            this.updateDrag(clientX, clientY);
            this.drag.mX = clientX;
            this.drag.mY = clientY;
            if (this.current.slide.isZoomed) {
                this.updateZoomPosition(this.drag.currentX, this.drag.currentY);
            } else {
                this.updateTrackPosition(Math.max(0, Math.min(this.max - 1, this.drag.translateX)));
            }
        }
    }
    /**
     * update Drag Object
     * @param {number} clientX - The Mouse X position.
     * @param {number} clientY - The Mouse Y position.
     */
    updateDrag(clientX, clientY) {
        if (this.isOpen && this.drag.isDragging) {
            this.drag.currentX = clientX - this.drag.startX;
            this.drag.currentY = clientY - this.drag.startY;
            if (!this.current.slide.isZoomed) {
                this.drag.translateX = (Number(this.current.index) - this.drag.currentX / (this.containers.container.clientWidth * this.config.dragSpeedMultiplier)).toFixed(2);
                this.drag.index = Math.round(parseFloat(this.drag.translateX));
            }
        }
    }
    /**
     * endDrag (mouseup, touchend)
     * @param {number} clientX - The Mouse X position.
     * @param {number} clientY - The Mouse Y position.
     */
    endDrag(clientX, clientY) {
        if (this.isOpen && this.drag.isDragging) {
            // call updateDrag this.duringDrag was never called
            if (!this.drag.index) {
                this.updateDrag(clientX, clientY);
            }
            this.drag.isDragging = false;
            this.containers.imagesTrack.classList.remove('is-dragging');
            if (this.current.slide.isZoomed) {
                this.current.slide.lastStaticX = this.current.slide.currentX;
                this.current.slide.lastStaticY = this.current.slide.currentY;
                this.current.slide.wasDragged = true;
            } else {
                this.goTo(Math.max(0, Math.min(this.max - 1, this.drag.index)))
            }
        }
    }
    /**
     * Configures keyboard event listeners if multiple images are present.
     */
    setUpKeyBoardEvents() {
        if (this.max > 1) {
            document.addEventListener('keydown', (event) => this.handleKeyboardEvents(event));
        }
    }
    /**
     * Handles keyboard events to navigate and control the lightbox.
     * @param {KeyboardEvent} event - The keyboard event object.
     */
    handleKeyboardEvents(event) {
        const key = event.key;
        if (this.isOpen) {
            switch (key) {
                case 'Escape':
                    this.close();
                    break;
                case 'a':
                case 'ArrowLeft':
                    this.prev();
                    break;
                case 'd':
                case 'ArrowRight':
                    this.next();
                    break;
                case '+':
                    this.zoomIn();
                    break;
                case '-':
                    this.zoomOut();
                    break;
                default:
                    // Do nothing for other keys
                    break;
            }
        }
    }
    /**
     * Resets the attributes and states of the current image and thumbnail elements.
     *
     * This method performs the following steps:
     * 1. Iterates over the image and thumbnail container elements.
     * 2. Resets the `isZoomed` property of each element to `false`.
     * 3. Resets the `wasDragged` property of the current image if it exists.
     * 4. Updates the zoom position to the default (0, 0).
     * 5. Removes the 'is-current' and 'is-zoomed' CSS classes from each element.
     * 6. Sets the 'aria-selected' attribute of each element to 'false'.
     * 7. Sets the 'tabindex' attribute of each element to '-1'.
     */
    resetCurrentAttributes() {
        [this.containers.images, this.containers.thumbnailsImages].forEach(elements => {
            elements?.forEach(element => {
                element.isZoomed = false;
                if (this.current.slide) {
                    this.current.slide.wasDragged = false;
                    this.updateZoomPosition(0, 0);
                }
                element.classList.remove('is-current', 'is-zoomed');
                element.setAttribute('aria-selected', 'false');
                element.setAttribute('tabindex', '-1');
            });
        });
        this.containers.container.classList.remove('is-zoomed');
    }
    /**
     * Updates the attributes and states of the current image and thumbnail elements.
     *
     * This method performs the following steps:
     * 1. Iterates over the image and thumbnail container elements.
     * 2. Add the 'is-current' CSS class on each element.
     * 3. Sets the 'aria-selected' attribute of each element to 'true'.
     * 4. Sets the 'tabindex' attribute of each element to '0'.
     */
    updateCurrentSlideAttributes() {
        [this.current.slide, this.current.thumbnailSlide].forEach(element => {
            element?.classList.add('is-current');
        });
        this.current.slide.setAttribute('aria-selected', 'true');
        this.current.slide.setAttribute('tabindex', '0');
    }
    /**
     * Sets up action buttons for navigation and interaction.
     */
    setupActionButtons() {
        this.containers.actions.forEach(action => {
            action.addEventListener('click', () => this.handleActionButtonClick(action));
        });
    }
    /**
     * Handles action button clicks by invoking the corresponding method.
     * @param {HTMLElement} action - The action button element.
     */
    handleActionButtonClick(action) {
        const method = action.dataset.action;
        if (method !== 'playToggle') {
            this.resetAutoPlay();
        }
        this[method]();
    }
    /**
     * Reset Auto Play state
     */
    resetAutoPlay() {
        clearInterval(this.autoPlayInterval);
        this.containers.container.classList.remove('is-playing');
    }
    /**
     * Enables scroll and dblclick zoom functionality.
     */
    setupScrollZoom() {
        this.containers.container.addEventListener('wheel', (e) => this.handleWheelZoom(e));
        this.containers.imagesTrack.addEventListener('dblclick', (e) => this.handleDoubleClickZoom(e));
    }
    /**
     * Handles when to zoom in/out
     * @param {WheelEvent} event - The Wheel Event object.
     */
    handleWheelZoom(e) {
        if (e.target.closest('.ig-lightbox-slides')) {
            e.preventDefault();
            if (e.deltaY < 0) this.zoomIn();
            else this.zoomOut();
        }
    }
    /**
     * Handles dblclick zoom in/out
     * @param {WheelEvent} event - The Wheel Event object.
     */
    handleDoubleClickZoom(e) {
        e.preventDefault();
        const target = e.target;
        if (target.classList.contains('ig-lightbox-slide')) {
            if (!target.classList.contains('is-zoomed')) this.zoomIn(2);
            else this.zoomOut(10);
        }
    }
    /**
     * Create HTML that is added to body
     */
    getContainerHtml() {
        const hasMultipleImages = this.max > 1;
        const showAutoplay = hasMultipleImages && this.config.autoPlay;
        const showThumbnails = hasMultipleImages;

        return `
            <div tabindex="-1" data-id="${this.config.id}" class="ig-lightbox-container ${showThumbnails ? 'active-thumbnails' : ''}">
                <div class="ig-lightbox-action-container">
                    ${hasMultipleImages ? this.getActionButton('prev', 'Previous image') : ''}
                    ${hasMultipleImages ? this.getActionButton('next', 'Next image') : ''}
                    ${hasMultipleImages ? `<span class="ig-lightbox-spacer"></span>` : ''}
                    ${showThumbnails ? this.getActionButton('thumbnailsToggle', 'Toggle Thumbnails') : ''}
                    ${this.config.zoom ? this.getActionButton('zoomOut', 'Zoom out') : ''}
                    ${this.config.zoom ? this.getActionButton('zoomIn', 'Zoom in') : ''}
                    ${showAutoplay ? this.getActionButton('playToggle', 'Autoplay') : ''}
                    ${this.config.fullscreen ? this.getActionButton('fullscreenToggle', 'Toggle Fullscreen') : ''}
                    ${this.getActionButton('close', 'Close Lightbox')}
                </div>
                <div class="ig-lightbox-slides">
                    <div class="ig-lightbox-track" role="listbox" aria-live="polite"></div>
                </div>
                ${showThumbnails ? '<div class="ig-lightbox-thumbnails"><div class="ig-lightbox-track" role="listbox" aria-live="polite"></div></div>' : ''}
            </div>
        `;
    }
    /**
     * Handles dblclick zoom in/out
     * @param {string} action - Data Action
     * @param {string} label - Arial Label
     */
    getActionButton(action, ariaLabel) {
        return `<button class="ig-lightbox-action" data-action="${action}" aria-label="${ariaLabel}"></button>`;
    }
    /**
     * Generate Slide
     * @param {bool} thumbs - is Thumbnail
     */
    getSlideHtml(thumbs = false) {
        return Array.from(this.config.images).map((slideContainer, index) => {
            const dataset = slideContainer.dataset;
            const attributes = this.getSlideAttributes(dataset, index, thumbs);
            const contentHtml = this.getSlideContentHtml(dataset, thumbs, slideContainer, index);
            const captionHtml = !thumbs && this.getCaptionHtml(dataset, index, slideContainer);
            return `
                <div ${attributes}>
                    ${contentHtml}
                    ${captionHtml || ''}
                </div>
            `;
        }).join('');
    }
    /**
     * generate tag attributes
     * @param {array} dataset - slide data attributes
     * @param {number} index - slide index
     * @param {bool} thumbs - is Thumbnail
     */
    getSlideAttributes(dataset, index, thumbs = false) {
        let attribute = [];
        if (!thumbs) {
            attribute = Object.entries(dataset).map(([key, value]) => key !== 'tag' ? `data-${key}="${value}"` : '');
            attribute.push(`aria-labelledby="caption-title-${index}"`);
            attribute.push(`aria-describedby="caption-description-${index}" `);
            attribute.push(`aria-selected="${index === this.current.index ? 'true' : 'false'}"`);
            attribute.push(`tabindex="-1"`);
        }
        return `${attribute.join(' ')} data-index="${index}" class="ig-lightbox-slide ${index === this.current.index ? 'is-current' : ''}"`;
    }
    /**
     * generate content HTML
     * @param {array} dataset - slide data attributes
     * @param {bool} thumbs - is Thumbnail
     * @param {object} slideContainer - Slide Container
     * @param {int} index - Slide index
     */
    getSlideContentHtml(dataset, thumbs, slideContainer, index) {
        const customContent = this.getAjaxContent(dataset, slideContainer, index) || this.getCustomContent(dataset);
        if (customContent) {
            let customContentHtml = customContent === true ? '' : customContent;
            return thumbs ? '<div class="ig-lightbox-slide-placeholder"></div>' : `<div class="ig-lightbox-custom-content">${customContentHtml}</div>`;
        }
        return `<img src="${slideContainer.href}" />`;
    }
    /**
     * get ajax content and load it when it's ready
     * @param {array} dataset - slide data attributes
     * @param {object} slideContainer - Slide Container
     * @param {int} index - Slide index
     */
    getAjaxContent(dataset, slideContainer, index) {
        if (dataset.ajax || slideContainer.rel) {
            this.updateConsents = true;
            const url = slideContainer.getAttribute('rel') || slideContainer.getAttribute('href');
            this.fetchAjax(url, index, dataset?.ajaxJson, dataset?.ajaxAttr);
            return true;
        }
        return false;
    }
    /**
     * return custom tag content
     * @param {array} dataset - slide data attributes
     */
    getCustomContent(dataset) {
        if (dataset.html) {
            this.updateConsents = true;
            return dataset.html;
        }
        return false;
    }
    /**
     * generate caption HTML
     * @param {array} dataset - slide data attributes
     * @param {number} index - slide index
     * @param {object} slideContainer - Slide Container
     */
    getCaptionHtml(dataset, index, slideContainer) {
        const title = slideContainer.title || dataset.title || dataset.alt;
        const description = dataset.caption || dataset.description;
        if (!title && !description) return '';
        return `
            <div class="ig-lightbox-caption">
                ${title ? `<div id="caption-title-${index}" class="ig-lightbox-caption-title"><strong>${title}</strong></div>` : ''}
                ${description ? `<div id="caption-description-${index}" class="ig-lightbox-caption-description">${description}</div>` : ''}
            </div>
        `;
    }
    /**
     * Generate Slides and Thumbnails HTML
     */
    generateSlides() {
        this.containers.imagesTrack.innerHTML = this.getSlideHtml();
        this.loaded.slides = true;
        this.containers.images = document.querySelectorAll('[data-id="' + this.config.id + '"] .ig-lightbox-slides .ig-lightbox-track .ig-lightbox-slide');
        if (this.max > 1) {
            this.containers.thumbnailsTrack.innerHTML = this.getSlideHtml(true);
            this.containers.thumbnailsImages = document.querySelectorAll('[data-id="' + this.config.id + '"] .ig-lightbox-thumbnails .ig-lightbox-track .ig-lightbox-slide')
            this.loaded.thumbnails = true;
            this.setupThumbnailClick();
        }
    }
    /**
     * Thumbnail Click Event
     */
    setupThumbnailClick() {
        const thumbs = this.containers.thumbnailsTrack.querySelectorAll('.ig-lightbox-slide');
        thumbs.forEach(thumb => {
            thumb.addEventListener('click', () => {
                this.setTransitionMultiplier(1);
                this.current.index = parseInt(thumb.dataset.index, 10);
                this.goTo(this.current.index);
                this.resetAutoPlay();
            });
        });
    }
    /**
     * Open Lightbox
     */
    open() {
        if (!this.loaded.containers) this.init();
        if (!this.loaded.slides) this.generateSlides();
        document.documentElement.classList.add('ig-lightbox-open');
        this.containers.container.setAttribute('aria-hidden', 'false');
        this.containers.container.classList.add('active');
        this.containers.container.focus();
        this.isOpen = true;
        if (this.updateConsents && !this.consentUpdated) {
            window.IgConsentManager?.renderContextualConsentNotices();
            this.consentUpdated = true;
        }
        this.triggerDispatchEvent(this.availableEvents.open);
    }
    /**
     * Action: Previous Slide
     */
    prev() {
        this.setTransitionMultiplier(1);
        this.goTo((this.current.index - 1 + this.max) % this.max);
    }
    /**
     * Action: Next Slide
     */
    next() {
        this.setTransitionMultiplier(1);
        this.goTo((this.current.index + 1 + this.max) % this.max);
    }
    /**
     * Action: Autoplay
     */
    playToggle() {
        clearInterval(this.autoPlayInterval);
        if (!this.containers.container.classList.contains('is-playing')) {
            this.autoPlayInterval = setInterval(() => this.next(), this.config.autoPlayTime);
        }
        this.containers.container.classList.toggle('is-playing');
    }
    /**
     * Action: Zoom Out
     */
    zoomOut(speed = this.config.zoomConfig.speed) {
        this.config.zoomConfig.scale = Math.max(this.config.zoomConfig.min, this.config.zoomConfig.scale - speed);
        if (this.config.zoomConfig.scale <= this.config.zoomConfig.min) {
            this.resetZoom();
        }
        this.doZoom();
    }
    /**
     * Action: Zoom In
     */
    zoomIn(speed = this.config.zoomConfig.speed) {
        this.config.zoomConfig.scale = Math.min(this.config.zoomConfig.max, this.config.zoomConfig.scale + speed);
        if (this.config.zoomConfig.scale <= this.config.zoomConfig.max) {
            this.applyZoom();
        }
        this.doZoom();
    }

    /**
     * Reset Zoom
     */
    resetZoom() {
        this.current.slide.wasDragged = false;
        this.updateZoomPosition(0, 0);
        this.current.slide.isZoomed = false;
        this.current.slide.classList.remove('is-zoomed');
        this.containers.container.classList.remove('is-zoomed');
    }
    /**
     * Handle Zoom in. Use Bigger image if data-orig is set
     */
    applyZoom() {
        this.current.slide.classList.add('is-zoomed');
        this.containers.container.classList.add('is-zoomed');
        if (this.current.slide.dataset.orig) {
            const img = this.current.slide.querySelector('img');
            img.dataset.src = img.src;
            img.src = this.current.slide.dataset.orig;
        }
        this.current.slide.isZoomed = true;
    }
    /**
     * Set zoom Attribute
     */
    doZoom() {
        this.updateProperty(this.containers.container, '--ig-lightbox-scale', parseFloat(this.config.zoomConfig.scale).toFixed(2));
    }
    /**
     * Action: Toggle Thumbnails
     */
    thumbnailsToggle() {
        this.containers.container.classList.toggle('active-thumbnails');
    }
    /**
     * Action: Toggle Fullscreen
     */
    fullscreenToggle() {
        this.containers.container.classList.toggle('is-fullscreen');
        if (!document.fullscreenElement) {
            document.documentElement.requestFullscreen();
        } else {
            document.exitFullscreen();
        }
    }
    /**
     * Action: Close
     */
    close() {
        document.documentElement.classList.remove('ig-lightbox-open');
        this.containers.container.setAttribute('aria-hidden', 'true');
        this.containers.container.classList.remove('active');
        this.setTransitionMultiplier(0);
        this.containers.triggerElement.focus();
        this.isOpen = false;
        this.triggerDispatchEvent(this.availableEvents.close);
    }
    getEvents() {
        console.log(this.availableEvents);
    }
    /**
     * Go To slide
     * 1. Update Track translateX
     * 2. Reset zoom
     * 3. Reset Attributes on Current Element
     * 4. find new image and thumbnail
     */
    goTo(index) {
        this.removeCustomProperties();
        this.updateProperty(this.containers.container, '--ig-lightbox-translateX', index);
        this.config.zoomConfig.scale = 1;
        this.updateProperty(this.containers.container, '--ig-lightbox-scale', this.config.zoomConfig.scale);
        this.resetCurrentAttributes();
        this.drag.index = index;
        this.current.index = index;
        this.current.slide = this.containers.imagesTrack.querySelector(`[data-index="${this.current.index}"]`);
        this.current.thumbnailSlide = this.containers.thumbnailsTrack?.querySelector(`[data-index="${this.current.index}"]`);
        this.addCustomProperties();
        this.updateCurrentSlideAttributes();
        this.triggerDispatchEvent(this.availableEvents.goTo);
    }
    /**
     * Update Slide Zoom Position
     */
    updateZoomPosition(currentX, currentY) {
        if (this.current.slide.wasDragged) {
            currentX = currentX + this.current.slide.lastStaticX;
            currentY = currentY + this.current.slide.lastStaticY;
        }
        this.current.slide.currentX = currentX;
        this.current.slide.currentY = currentY;
        this.updateProperty(this.current.slide, '--ig-lightbox-img-translateX', `${currentX}px`);
        this.updateProperty(this.current.slide, '--ig-lightbox-img-translateY', `${currentY}px`);
    }
    /**
     * Update Track Position
     */
    updateTrackPosition(value) {
        this.updateProperty(this.containers.container, '--ig-lightbox-translateX', value);
    }
    /**
     * Update var() in style tag
     */
    updateProperty(element, property, value) {
        if (parseInt(value) == 0) {
            element.style.removeProperty(property, value);
        } else {
            element.style.setProperty(property, value);
        }
    }
    /**
     * Custom Properties from Slide
     */
    addCustomProperties() {
        if (this.current.slide.dataset.vars) {
            this.current.slide.dataset.vars.split(';').filter(Boolean).forEach(cssVar => {
                const [property, value] = cssVar.split(':');
                this.updateProperty(this.containers.container, property.trim(), value.trim());
            });
        }
    }
    /**
     * Removes all custom properties except the specified ones from the container's inline styles.
     */
    removeCustomProperties() {
        // Store the values of the fixedProperties
        for (const property in this.fixedProperties) {
            this.fixedProperties[property] = this.containers.container.style.getPropertyValue(property);
        }
        // Clear all inline styles
        this.containers.container.style.cssText = '';
        // Reapply the fixedProperties with their stored values
        for (const property in this.fixedProperties) {
            if (this.fixedProperties[property]) {
                this.containers.container.style.setProperty(property, this.fixedProperties[property]);
            }
        }
    }
    /**
     * Sets the transition multiplier property on the main container.
     * @param {number} multiplier - The value to set for the transition multiplier.
     */
    setTransitionMultiplier(multiplier) {
        this.updateProperty(this.containers.container, '--ig-lightbox-track-transition-multiplier', multiplier);
    }
    /**
     * Dispatch DOM Event
     * @param { string } name - trigger name
    */
    triggerDispatchEvent(name) {
        document.dispatchEvent(new CustomEvent(name, {
            detail: {
                current: this.current,
                slides: this.containers.images
            }
        }));
    }
    /**
     * Sets the transition multiplier property on the main container.
     * @param {string} url - URL to fetch
     * @param {number} index - Index of element the content should be loaded to
     * @param {boolean} json - handle as json
     * @param {string} attr - return only json[attr]
     */
    async fetchAjax(url, index, json = false, attr = null) {
        fetch(url)
            .then(response => {
                if (!response.ok) throw new Error('Network response was not ok');
                if (json) return response.json();
                return response.text();
            })
            .then(data => {
                const container = document.querySelector('.ig-lightbox-track [data-index="' + index + '"] .ig-lightbox-custom-content');
                container.innerHTML = attr ? data[attr] : data;
            })
            .catch(error => {
                console.error('There has been a problem with your fetch operation:', error);
            });

    }
}
let igLightboxImages = document.querySelectorAll('[data-fslightbox="gallery"]');
if (igLightboxImages.length === 0) {
    igLightboxImages = document.querySelectorAll('.lightbox');
}
// only initialize when needed 
if (igLightboxImages.length > 0 && IgLightboxConfig) {
    window.IgLightbox = new IgLightbox({ ...{ images: igLightboxImages }, ...IgLightboxConfig });
}