<?php

declare(strict_types=1);

namespace Ig\IgMenu\DataProcessing;

use Ig\IgMenu\Domain\Model\MenuItem;
use Ig\IgMenu\Domain\Repository\PageRepository;
use Ig\IgMenu\Source\DataProcessorInterface;
use Ig\IgMenu\Utility\SettingsUtility;
use PDO;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\Restriction\FrontendRestrictionContainer;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;

use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Mvc\Web\Routing\UriBuilder;

class PagesProcessor implements DataProcessorInterface, SingletonInterface
{
    /**
     * PageRepository
     *
     * @var PageRepository
     */
    public $pageRepository = null;


    protected $doktypesInMenu = [1, 4, 3, 7];

    public function __construct()
    {
        $this->conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('pages');
        $this->queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable('pages');
    }

    public function injectPageRepository(PageRepository $pageRepository)
    {
        $this->pageRepository = $pageRepository;
    }

    /**
     * Process first call $menuItem is a page
     *
     * @param MenuItem $menuItem - the current menuItem (data is a page)
     */
    public function process(MenuItem $menuItem)
    {
        return $this->getSubMenuItems($menuItem);
    }

    /**
     * Process subitems - $menuItem is a category
     *
     * @param MenuItem $menuItem - the current menuItem
     */
    public function getChildren(MenuItem $menuItem)
    {
        return $this->getSubMenuItems($menuItem);
    }

    public function addDoktypesInMenu(array $addDoktypesInMenu)
    {
        $this->doktypesInMenu = array_merge($this->doktypesInMenu, $addDoktypesInMenu);
    }

    public function removeDoktypesInMenu(array $removeDoktypesInMenu)
    {
        $this->doktypesInMenu = array_diff($this->doktypesInMenu, $removeDoktypesInMenu);
    }

    public function getDoktypesInMenu(): array
    {
        return $this->doktypesInMenu;
    }

    public function setDoktypesInMenu(array $doktypesInMenu)
    {
        $this->doktypesInMenu = $doktypesInMenu;
    }

    /**
     * convert to menuItems
     */
    public function getSubMenuItems($menuItem)
    {
        $additionalQuery = $this->getAdditionalQuery();
        if ($menuItem->getDoktype() == 7 && $menuItem->getMountPid()) {
            // * or uid,shortcut_mode,doktype (we need uid and checkValidShortcutOfPage in pageRepository needs shortcut_mode, doktype)
            $menu = $this->pageRepository->getIgMenu($menuItem->getMountPid(), '*', 'sorting', $additionalQuery);
            $menuItem->set_MP_PARAM($menuItem->getMountPid() . '-' . $menuItem->getUid());
            //_mount_pid_ol
            if ($menuItem->getMountPidOl() == 1) {
                $page = $this->pageRepository->getPage($menuItem->getMountPid());
                $menuItem->mergeIgMenuAttributes($page);//@todo
            }
        } else {
            $menu = $this->pageRepository->getIgMenu($menuItem->getUid(), '*', 'sorting', $additionalQuery);
        }

        return $this->getMenuItems($menu, $menuItem->get_MP_PARAM());
    }

    /**
     * convert to menuItems
     */
    public function getMenuItems($menu, $mpParam)
    {
        $menuItems = [];
        $totalWidth = 0;
        $anzahlEintraege = 0;
        foreach ($menu as $menuEntry) {
            $menuItem = GeneralUtility::makeInstance(MenuItem::class);
            $menuItem->load($this->pageRepository->getPage($menuEntry['uid']));
            $totalWidth += $menuItem->getIgMenuColWidth();
            ++$anzahlEintraege;
            if ($mpParam) {
                $menuItem->set_MP_PARAM($mpParam);
            }

            $menuItems[] = $menuItem;
        }

        return $menuItems;
    }

    public function getAdditionalQuery()
    {
        $expressionBuilder = $this->conn->getExpressionBuilder();
        $frontendRestriction = GeneralUtility::makeInstance(FrontendRestrictionContainer::class);

        $conditions = [];

        $conditions[] = $frontendRestriction->buildExpression([
            'pages' => 'pages',
        ], $expressionBuilder);

        $menuConf = GeneralUtility::makeInstance(SettingsUtility::class);

        if (!$menuConf->getArgument('includeNotInMenu')) {
            $conditions[] = $expressionBuilder->eq('nav_hide', '0');
        }

        $pagesToExclude = $menuConf->getArgument('pagesToExclude');
        if ($pagesToExclude) {
            $pagesToExcludeArray = array_map('trim', explode(',', (string) $pagesToExclude));
            $conditions[] = $expressionBuilder->notIn('uid', $pagesToExcludeArray);
        }

        $conditions[] = $expressionBuilder->in('doktype', $this->doktypesInMenu);
        $whereExpression = $expressionBuilder->and(...$conditions);
        return $whereExpression->__toString();
        /*
          $this->queryBuilder->andWhere($expressionBuilder->in('doktype', $this->doktypesInMenu))
          ->andWhere($expressionBuilder->eq('nav_hide', '0'));
          return $this->queryBuilder->getQueryPart('where')->__toString();
        */
    }

    /**
     * convert to menuItems
     */
    public function convertCategories($pageUid, $categories)
    {
        $menuitems = [];
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        foreach ($categories as $category) {
            $menuitem = GeneralUtility::makeInstance(MenuItem::class);
            $menuitem->setTitle($category->getName());
            $menuitem->setUid($pageUid);
            $menuitem->setData($category);
            $menuitem->setActiveKey('category-' . $category->getUid());
            $uriBuilder
                ->reset()
                ->setTargetPageUid($pageUid);
            //->setArguments(['category' => $category->getUid()]);
            $menuitem->setLink($uriBuilder->uriFor(
                'list', // actionName
                [
                    'category' => $category->getUid(),
                ], //controllerArguments
                'Product', // controllerName
                'igshop2', // extensionName
                'Feshop' // pluginName
            ));

            $menuitems[] = $menuitem;
        }

        return $menuitems;
    }

    public function getStoragePid()
    {
        $configurationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
        return $configurationManager->getConfiguration(
            ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        )['plugin.']['tx_igshop2.']['settings.']['storagePid'];
    }

    /**
     * Get Active Entries (Parameters with values)
     */
    public function getActiveKeys(int $pageUid, array $queryParams)
    {
        $activeKeys = [];
        if (isset($queryParams['tx_igshop2_feshop']['category'])) {
            $category = (int) $queryParams['tx_igshop2_feshop']['category'];
            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
                'tx_igshop2_domain_model_category'
            );
            $limit = 10;
            while ($category > 0 && $limit > 0) {
                --$limit;
                $activeKeys[] = 'category-' . $category;
                $result = $queryBuilder
                        ->select('parent')
                        ->from('tx_igshop2_domain_model_category')
                        ->where(
                            $queryBuilder->expr()
->eq('uid', $queryBuilder->createNamedParameter($category, PDO::PARAM_INT))
                        )
                        ->executeQuery()
                        ->fetchAssociative();
                $category = (int) $result['parent'];
            }
        }

        return $activeKeys;
    }
}
