<?php

declare(strict_types=1);

namespace Ig\IgNewsletterContainer\DataProcessing;

/*
 * This file is part of TYPO3 CMS-based extension "container" by b13.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 */

use B13\Container\Domain\Factory\Exception;
use B13\Container\Domain\Factory\FrontendContainerFactory;
use B13\Container\Domain\Model\Container;
use Ig\IgResponsiveImages\Domain\Model\ResponsiveImagesConfiguration;
use Ig\IgResponsiveImages\Utility\ResponsiveImagesUtility;
use Psr\Http\Message\ServerRequestInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Frontend\ContentObject\ContentDataProcessor;
use TYPO3\CMS\Frontend\ContentObject\ContentObjectRenderer;
use TYPO3\CMS\Frontend\ContentObject\DataProcessorInterface;

class ContainerProcessor implements DataProcessorInterface
{

    /**
     * @var ContentDataProcessor
     */
    protected $contentDataProcessor;

    protected Context $context;
    protected FrontendContainerFactory $frontendContainerFactory;

    public function __construct(ContentDataProcessor $contentDataProcessor, Context $context, FrontendContainerFactory $frontendContainerFactory)
    {
        $this->contentDataProcessor = $contentDataProcessor;
        $this->context = $context;
        $this->frontendContainerFactory = $frontendContainerFactory;
    }

    public function process(
        ContentObjectRenderer $cObj,
        array $contentObjectConfiguration,
        array $processorConfiguration,
        array $processedData
    ): array {
        if (isset($processorConfiguration['if.']) && !$cObj->checkIf($processorConfiguration['if.'])) {
            return $processedData;
        }
        // BEGIN IG
        $flexFormString = $cObj->data['pi_flexform'];
        $flexForm = GeneralUtility::makeInstance(\TYPO3\CMS\Core\Service\FlexFormService::class)
            ->convertFlexFormContentToArray($flexFormString);
        $imageRenderingClass = $flexForm['image_rendering_class'] ?? null;
        if ($imageRenderingClass && $processorConfiguration['igresponsiveimages.']) {
            $processorConfiguration['igresponsiveimages.']['cols'] = $imageRenderingClass;
        }
        if ($processorConfiguration['igresponsiveimages.'] && $processorConfiguration['igresponsiveimages.']['cols'] == 'X') {
            $colWidthInt = (int)str_replace('ym-g', '', $flexForm['width_column_normal_1']);
            if ($colWidthInt > 0) {
                $colsInt = round(100 / $colWidthInt);

                if ($colsInt > 10) {
                    $colsInt = 10;
                }
                if ($colsInt < 1) {
                    $colsInt = 1;
                }
                $processorConfiguration['igresponsiveimages.']['colWidth'] = (string)$colWidthInt;
                $processorConfiguration['igresponsiveimages.']['cols'] = (string)$colsInt;
            }
        }
        // END IG
        $contentId = null;
        if ($processorConfiguration['contentId.'] ?? false) {
            $contentId = (int)$cObj->stdWrap($processorConfiguration['contentId'], $processorConfiguration['contentId.']);
        } elseif ($processorConfiguration['contentId'] ?? false) {
            $contentId = (int)$processorConfiguration['contentId'];
        }

        try {
            $container = $this->frontendContainerFactory->buildContainer($cObj, $this->context, $contentId);
        } catch (Exception $e) {
            // do nothing
            return $processedData;
        }

        $colPos = (int)$cObj->stdWrapValue('colPos', $processorConfiguration);
        if (empty($colPos)) {
            $allColPos = $container->getChildrenColPos();
            foreach ($allColPos as $colPos) {
                $processedData = $this->processColPos(
                    $cObj,
                    $container,
                    $colPos,
                    'children_' . $colPos,
                    $processedData,
                    $processorConfiguration
                );
            }
        } else {
            $as = $cObj->stdWrapValue('as', $processorConfiguration, 'children');
            $processedData = $this->processColPos(
                $cObj,
                $container,
                $colPos,
                $as,
                $processedData,
                $processorConfiguration
            );
        }
        return $processedData;
    }

    protected function processColPos(
        ContentObjectRenderer $cObj,
        Container $container,
        int $colPos,
        string $as,
        array $processedData,
        array $processorConfiguration
    ): array {
        $children = $container->getChildrenByColPos($colPos);

        $contentRecordRenderer = $cObj->getContentObject('RECORDS');
        if ($contentRecordRenderer === null) {
            throw new ContainerDataProcessingFailedException('RECORDS content object not available.', 1691483526);
        }
        $conf = [
            'tables' => 'tt_content',
        ];
        foreach ($children as &$child) {
            if (!isset($processorConfiguration['skipRenderingChildContent']) || (int)$processorConfiguration['skipRenderingChildContent'] === 0) {
                $conf['source'] = $child['uid'];
        // IG BEGIN
            $responsiveSet = false;
            if (!empty($processorConfiguration['igresponsiveimages.'])) {
                $responsiveImageUtility = GeneralUtility::makeInstance(ResponsiveImagesUtility::class);

                $igresponsiveimages = $processorConfiguration['igresponsiveimages.'];
                $configName = $igresponsiveimages['config'] ?? $igresponsiveimages['cols'] ?? 'default';
                if ((int)$configName > 0) {
                    $configName = $configName . '-spalten-nl';
                    //var_dump($configName);die('a');
                }

                $colWidth = $igresponsiveimages['colWidth'] ?? null;
                // @todo respect all with media query
                 // $child['tx_igcontainer_large_width']
                 // $child['tx_igcontainer_medium_width']
                 // $child['tx_igcontainer_small_width']
                if ($child['tx_igcontainer_large_width'] > 0) {
                    $colWidth = $child['tx_igcontainer_large_width'];
                }
                // IG Newsletter Config
                $setupUtility = GeneralUtility::makeInstance(\Ig\IgNewsletter\Utility\SetupUtility::class);

                $containerRecord = $container->getContainerRecord();
                $cols = isset($igresponsiveimages['cols']) ? intval($igresponsiveimages['cols']) : null;

                $current = $setupUtility->getCurrent();

                // cols="2" layout="{data.layout}" frameClass="{data.frame_class}" responsive="{f:if(condition: '{flexform.mobile_layout_cols} == 1', then: '0',else: '1')}"
                $setupUtility->setTotalNumberOfCols( $igresponsiveimages['cols'] );
                $debugOutput = $setupUtility->startBox('innerBox');
                $debugOutput .= $setupUtility->startCol($setupUtility->getColWidth(), $cols);
                
                if( isset($containerRecord['layout'])) {
                    //echo('layout='.$containerRecord['layout'].'<br />');
                    $debugOutput .= $setupUtility->setLayoutConfig($containerRecord['layout'], 'outerBox');
                }
                if( isset($containerRecord['frame_class'])) {
                    $setupUtility->setFrameClass($containerRecord['frame_class']);
                }
                $debugOutput .= $setupUtility->startTable();

                $totalNumberOfCols = $setupUtility->getTotalNumberOfCols();
                $layout = $setupUtility->getLayout();
                if ($cols==2 && $colPos==10 && $containerRecord['tx_ig_newsletter_column_normal_1']) {
                    $colWidth = $containerRecord['tx_ig_newsletter_column_normal_1'];
                } elseif ($cols==2 && $colPos==11 && $containerRecord['tx_ig_newsletter_column_normal_2']) {
                    $colWidth = $containerRecord['tx_ig_newsletter_column_normal_2'];
                } else {
                    $colWidth = null;
                }
                if ($cols) {
                    $setupUtility->setCurrentCol($cols);
                }
                $col = $setupUtility->getCurrentCol();
                
                $setupUtility->setColWidth($colWidth);
                //$colspan = (int)$igresponsiveimages['colspan'];
                //$content=$setupUtility->getOuterBoxTd( $colspan, $igresponsiveimages['class'] ) . $setupUtility->debug();


                $responsiveSet = false;
                $responsiveImageUtility = GeneralUtility::makeInstance(ResponsiveImagesUtility::class);
                if( $totalNumberOfCols >= 1) {
                    $responsiveImagesConfiguration = GeneralUtility::makeInstance(ResponsiveImagesConfiguration::class);
                    $responsiveImagesConfiguration->setProcessorConfiguration($igresponsiveimages);
                    $responsiveImageUtility->setTagWidth( $setupUtility->getWidth() );
                    //$responsiveImagesConfiguration->setColWidth($colWidth);
                    $responsiveImagesConfiguration->setConfigName($totalNumberOfCols . '-spalten-nl');
                    $responsiveImageUtility->pushContainer($responsiveImagesConfiguration);
                $responsiveSet = true;
                }
                //var_dump('colWidht= ' . $colWidth . ', news1=' . $containerRecord['tx_ig_newsletter_column_normal_1']. ', news2=' . $containerRecord['tx_ig_newsletter_column_normal_2'] . ', cols=' . $igresponsiveimages['cols'] . ', total=' .  $totalNumberOfCols  . ',layout=' .$layout . ', cw='. $colWidth . ', cn='. $configName . ', tw='. $setupUtility->getWidth());
                                               
            }
            // IG END
                $child['renderedContent'] = $cObj->render($contentRecordRenderer, $conf);
            }
            /** @var ContentObjectRenderer $recordContentObjectRenderer */
            $recordContentObjectRenderer = GeneralUtility::makeInstance(ContentObjectRenderer::class);
            $recordContentObjectRenderer->start($child, 'tt_content');
            $child = $this->contentDataProcessor->process($recordContentObjectRenderer, $processorConfiguration, $child);
            // IG BEGIN
            if ($responsiveSet) {
                $responsiveImageUtility->popContainer($responsiveImagesConfiguration);
            }
            $setupUtility->setCurrent( $current );
            // IG END
        }
        $processedData[$as] = $children;
        return $processedData;
    }

    protected function getRequest(): ServerRequestInterface
    {
        return $GLOBALS['TYPO3_REQUEST'];
    }
}
