<?php
namespace Ig\IgPayment\Services;

use Ig\IgPayment\Domain\Model\PaymentStatus;
use InternetGalerie\Igshop2\Domain\Model\Order;
use PostFinanceCheckout\Sdk\ApiClient;
use PostFinanceCheckout\Sdk\Model\AddressCreate;
use PostFinanceCheckout\Sdk\Model\CustomerCreate;
use PostFinanceCheckout\Sdk\Model\Gender;
use PostFinanceCheckout\Sdk\Model\LineItemCreate;
use PostFinanceCheckout\Sdk\Model\LineItemType;
use PostFinanceCheckout\Sdk\Model\TransactionCreate;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Http\RedirectResponse;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Site\Entity\Site;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;

class PostFinanceCheckoutService extends PaymentService
{

    protected string $code = 'PostFinanceCheckout';
    protected array $settings = [];

    /**
     * Constructor
     */
    public function __construct()
    {
        $configurationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
        $typoscript = $configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT);
        $settings = $typoscript['plugin.']['tx_igpayment.']['settings.'];
        $this->settings = ['pfCheckout' => $settings['pfCheckout.']];
    }

    public function getSettings(): array
    {
        return $this->settings;
    }
    

    public function getPaymentPageUrl(
        int $orderId,
        array $billingAddress,
        array $shippingAddress,
        array $lineItems,
        string $currency = 'CHF',
        string $successUrl = '',
        string $failUrl = ''
    ): string
    {
        // Get site language to get the payment page in the correct language
        $site = $GLOBALS['TYPO3_REQUEST']->getAttribute('site');
        $context = GeneralUtility::makeInstance(Context::class);
        $languageId = $context->getPropertyFromAspect('language', 'id');
        $language = $site->getLanguageById($languageId);

        // Configuration
        $spaceId = $this->settings['pfCheckout']['spaceId'];
        $userId = $this->settings['pfCheckout']['userId'];
        $secret = $this->settings['pfCheckout']['secret'];

        // Create transaction
        $transactionPayload = new TransactionCreate();
        $transactionPayload->setCurrency($currency)
                           ->setMetaData(['order_id' => $orderId])
                           ->setLineItems($lineItems)
                           ->setAutoConfirmationEnabled(true)
                           ->setLanguage($language->getLocale()->getLanguageCode());

        // Assign Billing Address
        if (!empty($billingAddress)) {
            $billingAddressObject = new AddressCreate($billingAddress);
            $transactionPayload->setBillingAddress($billingAddressObject);
        }

        // Assign Shipping Address
        if (!empty($shippingAddress)) {
            $shippingAddressObject = new AddressCreate($shippingAddress);
            $transactionPayload->setShippingAddress($shippingAddressObject);
        }

        // Assign URL to redirect to after successful payment
        if($successUrl) {
            $transactionPayload->setSuccessUrl($successUrl);
        }
        // Assign URL to redirect to after failed or aborted payment
        if ($failUrl) {
            $transactionPayload->setFailedUrl($failUrl);
        }

        // Setup API client
        $client = new ApiClient($userId, $secret);

        // Create the effective transaction
        $transaction = $client->getTransactionService()->create($spaceId, $transactionPayload);

        // Create Payment Page URL:
        $redirectionUrl = $client->getTransactionPaymentPageService()->paymentPageUrl($spaceId, $transaction->getId());

        return $redirectionUrl;
    }

    public function invokePayment($orderId, $currency, $amount, &$reference, $redirectUrl, $failUrl = ''): PaymentStatus
    {
        $paymentStatus = GeneralUtility::makeInstance(PaymentStatus::class);

        $lineItems = [];
        foreach ($this->products as $product) {
            $lineItem = new LineItemCreate();
            $lineItem->setName($product['name'])
                ->setUniqueId('Product-' . $product['id'])
                ->setQuantity($product['amount'])
                ->setAmountIncludingTax((float)(string)$product['price'])
                ->setType(LineItemType::PRODUCT);

            $lineItems[] = $lineItem;
        }

        if (!empty($this->shipping)) {
            $lineItem = new LineItemCreate();
            $lineItem->setName($this->shipping['name'])
                ->setUniqueId('Porto-' . $this->shipping['id'])
                ->setQuantity(1)
                ->setAmountIncludingTax((float)(string)$this->shipping['price'])
                ->setType(LineItemType::SHIPPING);

            $lineItems[] = $lineItem;
        }

        $shippingAddress = $this->shippingAddress;
        $billingAddress = $this->billingAddress;

        $shippingAddress['gender'] = $shippingAddress['gender'] == 'm' ? Gender::MALE : Gender::FEMALE;
        $billingAddress['gender'] = $billingAddress['gender'] == 'm' ? Gender::MALE : Gender::FEMALE;

        $paymentPageUrl = $this->getPaymentPageUrl(
            $orderId,
            $shippingAddress,
            $billingAddress,
            $lineItems,
            $currency,
            $redirectUrl,
            $failUrl
        );

        $paymentStatus->setRedirect(new RedirectResponse($paymentPageUrl));

        return $paymentStatus;
    }

    public function checkTransactionSuccess($currency, $amount): PaymentStatus
    {
        $paymentStatus = GeneralUtility::makeInstance(PaymentStatus::class);
        $paymentStatus->setStatus(PaymentStatus::PAID);
        $paymentStatus->setTransactionSuccess(true);
        return $paymentStatus;
    }

    public function updatePayment($orderId, $currency, $amount, $reference, $redirectUrl, $failUrl = '', $paymentMethodCode = ''): PaymentStatus
    {
        $paymentStatus = GeneralUtility::makeInstance(PaymentStatus::class);
        return $paymentStatus;
    }
}
