<?php
namespace Ig\IgPayment\Services;

use Ig\IgPayment\Domain\Model\PaymentStatus;
use SumUp\HttpClients\Response;
use SumUp\Services\Checkouts;
use SumUp\SumUp;
use TYPO3\CMS\Core\Utility\GeneralUtility;

/**
 * This calss provides methods for acces checking and authorization
 */
class SumUpService extends PaymentService
{
    protected string $code = 'SumUp';
    private SumUp $sumup;
    private array $settings;
    private array $sumupEnvironment;
    
    public function __construct()
    {
        $configurationManager = GeneralUtility::makeInstance(\TYPO3\CMS\Extbase\Configuration\ConfigurationManager::class);
        $typoscript= $configurationManager->getConfiguration(\TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT);
        $this->settings=$typoscript['plugin.']['tx_igpayment.']['settings.']['sumup.'];

        $this->sumupEnvironment = $this->settings[$this->settings['environment'] . '.'] ?? $this->settings;

        try {
            $this->sumup = new SumUp([
                'app_id'     => $this->sumupEnvironment['clientId'],
                'app_secret' => $this->sumupEnvironment['clientSecret'],
                'grant_type' => 'client_credentials',
                'scopes'     => ['payments', 'transactions.history', 'user.app-settings', 'user.profile_readonly'],
            ]);
        } catch (\SumUp\Exceptions\SumUpAuthenticationException $e) {
            echo 'Authentication error: ' . $e->getMessage();
        } catch (\SumUp\Exceptions\SumUpResponseException $e) {
            echo 'Response error: ' . $e->getMessage();
        } catch (\SumUp\Exceptions\SumUpSDKException $e) {
            echo 'SumUp SDK error: ' . $e->getMessage();
        }

        // Set external ID if set (Redirect from 3D Secure adds it to the query string)
        $request = $GLOBALS['TYPO3_REQUEST'];
        if (!$this->externalId && isset($request->getQueryParams()['checkout_id']) && $request->getQueryParams()['checkout_id'] !== '') {
            $this->setExternalId($request->getQueryParams()['checkout_id']);
        }
    }

    public function getSumup(): SumUp
    {
        return $this->sumup;
    }

    public function getCheckoutService(): Checkouts
    {
        return $this->sumup->getCheckoutService();
    }

    public function createCheckout($order, $checkoutReference, $redirectUrl = null)
    {
            $checkoutResponse = $this->getCheckoutService()->create(
                $order['amount'],
                $order['currency'],
                $checkoutReference,
                $this->sumupEnvironment['payToEmail'],
                'Bestellung ' . $order['id'],
                null,
                null,
                rawurldecode($redirectUrl)
            );
            return $checkoutResponse->getBody();
    }

    public function getCheckoutResponseBody($order, $redirectUrl = null): mixed
    {
        $checkoutReference = $order['id'];
        $checkout = $this->getCheckoutByReference($checkoutReference);
        if ($checkout === false) {
            $checkout = $this->createCheckout($order, $checkoutReference , $redirectUrl);
        }
        return $checkout;
        /*  
        $checkoutResponse = $this->getCheckoutResponse($order, $redirectUrl);

        if (is_array($checkoutResponse->getBody())) {
            return $checkoutResponse->getBody()[0];
        }
        return $checkoutResponse->getBody();
        */
    }

    public function invokePayment($orderId, $currency, $amount, &$reference, $redirectUrl, $failUrl = ''): PaymentStatus
    {
        $orderArray = [
            'id' => $orderId,
            'currency' => $currency,
            'amount' => $amount,
            'checkout_reference' => $reference,
        ];
        $checkoutResponseBody = $this->getCheckoutResponseBody($orderArray, $redirectUrl);
        $paymentStatus = GeneralUtility::makeInstance(PaymentStatus::class);
        $paymentStatus->setStatus($checkoutResponseBody->status);

        if ($currency != $checkoutResponseBody->currency || '' . $amount != '' . $checkoutResponseBody->amount || $checkoutResponseBody->status === 'EXPIRED') {
            $paymentStatus->setOrderChanged(true);
            $reference = time();
            $orderArray['checkout_reference'] = $reference;
            if ($checkoutResponseBody->status !== 'EXPIRED') {
                $this->getCheckoutService()->delete($checkoutResponseBody->id);
            }
            $checkoutResponseBody = $this->getCheckoutResponseBody($orderArray, $redirectUrl);
        }

        $checkoutId = $checkoutResponseBody->id;

        $this->externalId = $checkoutId;

        $paymentStatus->setStatus($checkoutResponseBody->status);

        $this->setAssigns([
            'orderArray' => $orderArray,
            'checkoutId' => $checkoutId
        ]);

        return $paymentStatus;
    }

    public function checkTransactionSuccess($currency, $amount): PaymentStatus
    {
        $checkoutId = $this->externalId;
        $checkoutService = $this->getCheckoutService();
        $checkoutResponseBody = $checkoutService->findById($checkoutId)->getBody();
        $paymentStatus = GeneralUtility::makeInstance(PaymentStatus::class);
        $paymentStatus->setStatus($checkoutResponseBody->status);

        if ($currency != $checkoutResponseBody->currency || '' . $amount != '' . $checkoutResponseBody->amount) {
            $paymentStatus->setOrderChanged(true);
            if ($checkoutResponseBody->status !== 'EXPIRED') {
                $checkoutService->delete($checkoutId);
            }

            return $paymentStatus;
        }

        $transaction = null;
        foreach ($checkoutResponseBody->transactions as $t) {
            if ($t->id == $checkoutResponseBody->id) {
                $transaction = $t;
                break;
            }
        }

        if($transaction && $transaction->status == 'SUCCESSFUL') {
            $paymentStatus->setTransactionSuccess(true);
        }

        if (!$paymentStatus->getTransactionSuccess()) {
            if ($checkoutResponseBody->status !== 'EXPIRED') {
                $checkoutService->delete($checkoutId);
            }
        }

        return $paymentStatus;
    }

    public function updatePayment($orderId, $currency, $amount, $reference, $redirectUrl, $failurl = '', $paymentMethodCode = ''): PaymentStatus
    {
        $orderArray = [
            'id' => $orderId,
            'currency' => $currency,
            'amount' => $amount,
            'checkout_reference' => $reference,
        ];

        $checkoutResponseBody = $this->getCheckoutResponseBody($orderArray, $redirectUrl);

        $paymentStatus = GeneralUtility::makeInstance(PaymentStatus::class);

        if (($paymentMethodCode && $paymentMethodCode !== 'SumUp') || $currency != $checkoutResponseBody->currency || '' . $amount != '' . $checkoutResponseBody->amount) {
            $paymentStatus->setOrderChanged(true);
            if ($checkoutResponseBody->status !== 'EXPIRED') {
                $this->getCheckoutService()->delete($checkoutResponseBody->id);
            }
        }

        $paymentStatus->setStatus($checkoutResponseBody->status);

        return $paymentStatus;
    }

    public function getCheckoutByReference(string $checkoutReference)
    {
        if (!$checkoutReference) {
            return false;
        }
        $checkoutService = $this->getCheckoutService();
        $checkoutResponse = $checkoutService->findByReferenceId($checkoutReference);
        if(!empty($checkoutResponse->getBody())) {
            return $checkoutResponse->getBody()[0];
        }
        return false;
    }
    
    public function clear(string $checkoutReference): bool
    {
        $checkout = $this->getCheckoutByReference($checkoutReference);
        if ($checkout === false) {
            return false;
        }
        if ($checkout->status !== 'EXPIRED' && $checkout->status !== 'PAID') {
            $this->getCheckoutService()->delete($checkout->id);
            return true;
        }
        return false;
    }
}
