<?php

namespace Internetgalerie\IgRecurrenceDate\Hook;

use DateTime;
use Internetgalerie\IgRecurrenceDate\Domain\Model\Eventgroup;
use Internetgalerie\IgRecurrenceDate\Utility\TcaUtility;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\Restriction\HiddenRestriction;
use TYPO3\CMS\Core\DataHandling\DataHandler;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\StringUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;

class RecurrenceDataHandlerHook
{
    protected string $eventTable = '';
    protected string $eventTableMainDateAttribute = 'date_from';
    protected array $eventTableAdditionalDateAttributes = [];
    protected string $eventTableEventgroupAttribute = 'eventgroup';
    protected string $eventgroupTable = '';
    protected string $registrationTable = '';
    protected string $registrationTableMatchField = 'uid_foreign';
    protected int $pid = 0;
    protected string $temporaryId = '';
    protected static bool $addDeletedToExcludeDate = true; // add deleted dates to exclude dates, so they are not created on recurrence updates
    protected array $translationsToDelete = []; // Track translations to delete
    protected array $translationsMm = []; // Track translations Mm


    /**
     * on create/update also create/update occurences according update_scope and update eventgroup
     */
    public function processDatamap_beforeStart(DataHandler $dataHandler): void
    {
        // Reset translations to delete
        $this->translationsToDelete = [];

        $recurrence = $this->requestGetRecurrence();
        
        foreach ($dataHandler->datamap as $table => $tableDatamap) {
            if ($this->initRecurrenceConfigForTable($table)) {
                foreach ($tableDatamap as $eventUid => $eventData) {

                    $isNew = !is_numeric($eventUid);
                    static::$addDeletedToExcludeDate = false;
                    if ($isNew) {
                        $recurrence['update_scope'] = 'all';
                    }
                    $excludeDateChanged = false;
                    // recurrence is or was set
                    if ($this->isRecurrenceEnable($eventData, $recurrence)) {
                        $eventgroup = $this->getEventgroup($eventData, $recurrence); // on copy eventgroup is null

                        $currentEvent = null;

                        if (!$isNew) {
                            $currentEvent = $this->getEventByUidRaw($eventUid);
                            //$oldEvent = $currentEvent;
                        } else {
                            $this->temporaryId = $eventUid;
                            $currentEvent = $eventData;
                            // always all for new entries
                        }
                        $this->pid = (int)$currentEvent['pid'];

                        // get translation of current event
                        $currentEventTranslations = [];
                        $sysLanguageUid = 0;
                        // default language -> also update translated entries
                        if ($currentEvent['sys_language_uid'] == 0) {
                            if (!$isNew  && $currentEvent['sys_language_uid'] == 0) {
                                $currentEventTranslations = $this->getEventsByL10nParentRaw($eventUid);
                            }
                        } else {
                            // translated entry, only update this langauge
                            $sysLanguageUid = (int)$currentEvent['sys_language_uid'];
                            // do we only need to update event itself
                            if ($eventgroup === null || $eventgroup->isRawEventExcluded($currentEvent)) {
                                return;
                            }
                            $recurrence['update_scope'] = 'all';
                        }

                        // we only update this entry -> exclude from eventGroup
                        if ($recurrence['update_scope'] === '' || $recurrence['update_scope'] === 'this') {
                            $eventgroup->addExcludeDate($eventData[$this->eventTableMainDateAttribute]);
                            $excludeDateChanged = true;
                        } else {
                            $eventgroupUid = $eventgroup ? $eventgroup->getUidString() : null;
                            $dataHandler->datamap[$this->eventTable][$eventUid][$this->eventTableEventgroupAttribute] = $eventgroupUid;
                            $eventData[$this->eventTableEventgroupAttribute] = $eventgroupUid;
                            if ($eventgroup) {
                                if (!$isNew && $eventgroup->getUid() > 0) {
                                    $existingEvents = $this->getEventsByEventgroupGroupByDateFrom(
                                        $eventgroup,
                                        $eventUid,
                                        $recurrence,
                                        $currentEvent[$this->eventTableMainDateAttribute],
                                        $sysLanguageUid
                                    );

                                    $dates = $eventgroup->getDates();

                                    // we are alone -> disable recurrence and clear our eventgroup
                                    if (empty($dates)) {
                                        $eventgroup->setRecurrenceEnable(false);
                                        $eventData[$this->eventTableEventgroupAttribute] = null;
                                        $dataHandler->datamap[$this->eventTable][$eventUid][$this->eventTableEventgroupAttribute] = null;
                                    }
                                    // set current event
                                    $dates[substr((string) $currentEvent[$this->eventTableMainDateAttribute], 0, 10)] = $eventUid;
                                    $deletes = [];
                                    foreach ($existingEvents as $date => $uid) {
                                        if (array_key_exists($date, $dates)) {
                                            $this->createRecurrenceByDate($dataHandler, $uid, $date, $eventUid, $eventData, $currentEventTranslations);
                                            $dates[$date] = $uid;
                                        } else {
                                            $deletes[$date] = $uid;
                                        }
                                    }

                                    // do if eventgroup stuff have changes -> do we want to allow this
                                    foreach ($dates as $date => $uid) {
                                        if ($uid === null || (int)$uid === 0) {
                                            $this->createRecurrenceByDate($dataHandler, $uid, $date, $eventUid, $eventData, $currentEventTranslations);
                                            //DebuggerUtility::var_dump($dataHandler->datamap[$this->eventTable][$uid]);exit(0);
                                        }
                                    }
                                    foreach ($deletes as $date => $uid) {
                                        $count = $this->eventCountReservations($uid);
                                        if ($count === 0) {
                                            $dataHandler->cmdmap[$this->eventTable][$uid]['delete'] = 1;
                                            // Add translations of this event to delete list
                                            $this->addTranslationsToDeleteList($uid);
                                        } else {
                                            $dateFrom = new DateTime($eventData[$this->eventTableMainDateAttribute]);
                                            $this->eventDeleteError(
                                                'Event: ' . $eventData['title'] . ' ' . $dateFrom->format(
                                                    'd.m.Y'
                                                ) . ': ' . $count . ' Reservations'
                                            );
                                        }
                                    }
                                } else {
                                    // is new or had no recurrences -> add all dates
                                    $dates = $eventgroup->getDates(true);
                                    foreach ($dates as $date => $uid) {
                                        $this->createRecurrenceByDate($dataHandler, $uid, $date, $eventUid, $eventData, $currentEventTranslations);
                                    }
                                }
                            }
                        }
                        //DebuggerUtility::var_dump($dataHandler->datamap[$this->eventTable]);die('ID=' . $eventUid . ($isNew ? ' [NEW]' : ' [OLD]') . ' update = ' . $recurrence['update_scope']);
                
                        // add eventgroup
                        if ($eventgroup) {
                            if ($excludeDateChanged || ($eventgroup->getRecurrenceEnable() && $recurrence['update_scope'] == 'all')) {
                                $dataHandler->datamap[$this->eventgroupTable][$eventgroup->getUidString()] = $eventgroup->getData(
                                    $this->pid
                                );
                            } else {
                                if ($recurrence['update_scope'] == 'all') {
                                    $dataHandler->cmdmap[$this->eventgroupTable][$eventgroup->getUid()]['delete'] = 1;
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * on NEW prevent that all forms of all occurence events are shown
     */
    public function processDatamap_afterDatabaseOperations(
        $status,
        $table,
        $id,
        array $fieldArray,
        DataHandler $dataHandler
    ): void {
        // Remove the events from substNEWwithIDs_table so it doesn't open all newly created events at once
        $substNEWwithIDs_table = $dataHandler->substNEWwithIDs_table;
        foreach ($substNEWwithIDs_table as $key => $table) {
            if ($table == $this->eventTable && $key != $this->temporaryId) {
                unset($dataHandler->substNEWwithIDs_table[$key]);
            }
        }
    }


    
    /**
     * called after update, before delete is started
     * delete all occurence of an event, if delete_scope is all
     */
    public function processCmdmap_beforeStart(DataHandler $dataHandler): void
    {
        $recurrence = $this->requestGetRecurrence();
        if ($recurrence['delete_scope'] !== 'all') {
            return;
        }
        foreach ($dataHandler->cmdmap as $table => $tableCmdmap) {
            if ($this->initRecurrenceConfigForTable($table)) {

                // delete all events
                foreach ($tableCmdmap as $eventUid => $eventCmd) {
                    if ($eventCmd['delete'] ?? false) {
                        $currentEvent = $this->getEventByUidRaw($eventUid);
                        $eventgroup = $this->getEventgroupByUid((int)$currentEvent[$this->eventTableEventgroupAttribute]);
                        $events = $this->getEventsByEventgroupGroupByUid($eventgroup, $eventUid, $recurrence);
                        foreach ($events as $uid) {
                            $dataHandler->cmdmap[$table][$uid]['delete'] = 1;
                        }
                        $dataHandler->cmdmap[$this->eventgroupTable][$eventgroup->getUid()]['delete'] = 1;
                    }
                }
            }
        }
    }

    // @todo clean up move to start
    public function processCmdmap_deleteAction(
        string $table,
        int $eventUid,
        array $event,
        bool $recordWasDeleted,
        DataHandler $dataHandler
    ): void {
        if ($table === $this->eventTable && static::$addDeletedToExcludeDate) {
            // Add your custom logic here
            // Example: Log the deletion or trigger custom logic
            $recurrence = $this->requestGetRecurrence();
            if ($this->isRecurrenceEnable($event) && $recordWasDeleted) {
                $eventgroupUid = $event[$this->eventTableEventgroupAttribute];
                $eventgroupArray = $eventgroupUid ? $this->getEventgroupByUidRaw($eventgroupUid) : false;
                if ($eventgroupArray !== false && is_array($eventgroupArray) && !empty($eventgroupArray)) {
                    $dataMapper = GeneralUtility::makeInstance(DataMapper::class);
                    $eventgroups = $dataMapper->map(Eventgroup::class, [$eventgroupArray]);
                    $eventgroup = $eventgroups[0];
                    // delete all occurences
                    if ($recurrence['update_scope'] == 'all') {
                        //die('no');
                        //$events = $this->getEventsByEventgroupGroupByDateFrom($eventgroup, $eventUid, $event[$this->eventTableMainDateAttribute], ...);
                    } else {
                        // only delete this entry, update eventgroup
                        $eventgroup->addExcludeDate($event[$this->eventTableMainDateAttribute]);
                        $this->updateEventgroupSetExcludeDateseventCountReservations(
                            $eventgroup->getUid(),
                            $eventgroup
                        );
                        $dataHandler->datamap[$this->eventgroupTable][$eventgroup->getUidString()] = $eventgroup->getData(
                            $eventgroupArray['pid']
                        );
                    }
                }
                //$GLOBALS['TYPO3_CONF_VARS']['SC_OPTIONS']['logger']->info(                    sprintf('Event with UID %d has been deleted.', $id)                );

                // Optionally perform cleanup, e.g., delete related records
            }
        }
    }

    protected function getAttributesForDatamap(string $tablename, array $excludeFields = []): array
    {
        $attributes = [];
        $excludeFields = array_merge($excludeFields, ['l10n_diffsource', 'l10n_parent', 't3ver_label', 'sorting']);
        foreach ($GLOBALS['TCA'][$tablename]['columns'] as $attribute => $attributeTca) {
            if (!in_array($attribute, $excludeFields)) {
                $attributes[] = $attribute;
            }
        }
        return $attributes;
    }

    protected function getMmByUidForeign(
        string $mmTable,
        string $targetUid,
        string $fieldName,
        string $uidForeign,
        bool $mmUseTablenames,
        string $sorting = 'sorting'
    ) {
        // get filesReferences of target
        if ((int)$targetUid > 0) {
            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($mmTable);
            $queryBuilder
                ->select('*')
                ->from($mmTable)
                ->where(
                    $queryBuilder->expr()
                        ->eq($uidForeign, $queryBuilder->createNamedParameter($targetUid, Connection::PARAM_INT))
                );
            if ($mmUseTablenames) {
                $queryBuilder
                    ->andWhere(
                        $queryBuilder->expr()->eq('tablenames', $queryBuilder->createNamedParameter(
                            $this->eventTable,
                            Connection::PARAM_STR
                        ))
                    )
                    ->andWhere(
                        $queryBuilder->expr()->eq('fieldname', $queryBuilder->createNamedParameter(
                            $fieldName,
                            Connection::PARAM_STR
                        ))
                    );
            }
            return $queryBuilder
                ->executeQuery()
                ->fetchAllAssociative();
        }
        // new target
        return [];
    }

    /**
     * create/update mm relation entries (e.g. fileReferences) on ceate/copy events occurence
     */
    protected function createMmRelations(
        DataHandler $dataHandler,
        string $targetUid,
        string $fieldName,
        string $sourceUid,
        string $sourceValue,
        string $mmTable = 'sys_file_reference',
        string $uidLocal = 'uid_local',
        string $uidForeign = 'uid_foreign',
        string $foreignTable = 'sys_file',
        bool $mmUseTablenames = true,
    ): string {
        $targetFileReferences = $this->getMmByUidForeign(
            $mmTable,
            $targetUid,
            $fieldName,
            $uidForeign,
            $mmUseTablenames
        );
        $oldSourceFileReferences = $this->getMmByUidForeign(
            $mmTable,
            $sourceUid,
            $fieldName,
            $uidForeign,
            $mmUseTablenames
        );
        $attributes = $this->getAttributesForDatamap($mmTable, [$uidForeign]);
        $attributes[] = 'pid';
        $oldSourceEntryUids = array_column($oldSourceFileReferences, 'uid');
        $oldSourceFileReferencesByUid = array_combine($oldSourceEntryUids, $oldSourceFileReferences);
        // get filesReferences of source
        $sourceFileIds = array_filter(GeneralUtility::trimExplode(',', $sourceValue));
        $sourceFileReferences = [];
        foreach ($sourceFileIds as $uid) {
            $sourceFileReferences[] = $dataHandler->datamap[$mmTable][$uid] ?? $this->translationsMm[$mmTable][$uid];
        }

        // compare file references of source and target
        $newTargetFileReferences = [];
        foreach ($sourceFileReferences as $index => $fileReference) {
            if (count($fileReference) == 1 && isset($fileReference['hidden'])) {
                $currentSourceUid = $sourceFileIds[$index];
                $newFileReference = [];
                foreach ($attributes as $attribute) {
                    $v = $oldSourceFileReferencesByUid[$currentSourceUid][$attribute] ?? null;
                    if ($v !== null) {
                        $newFileReference[$attribute] = $v;
                    }
                }
                $newFileReference['hidden'] = $fileReference['hidden'];
                $newUid = StringUtility::getUniqueId('NEW');
                $fileUid = $newFileReference[$uidLocal];
                $newFileReference[$uidLocal] = $foreignTable . '_' . $fileUid;
            } else {
                $newFileReference = $fileReference;
                $fileUid = substr((string) $newFileReference[$uidLocal], strlen($foreignTable . '_'));
            }
            // is file present in target
            $targetFound = false;
            $targetIndex = null;
            foreach ($targetFileReferences as $targetIndex => $targetFileReference) {
                if ((string)$targetFileReference[$uidLocal] === $fileUid) {
                    $targetFound = true;
                    break;
                }
            }
            // file found in target entries
            if ($targetFound) {
                unset($targetFileReferences[$targetIndex]);
                $newUid = $targetFileReference['uid'];
            } else {
                $newUid = StringUtility::getUniqueId('NEW');
                $newFileReference['pid'] = $this->pid;
            }
            $dataHandler->datamap[$mmTable][$newUid] = $newFileReference;
            $newTargetFileReferences[] = $newUid;
        }
        // remove unused file references of target
        foreach ($targetFileReferences as $fileReference) {
            $dataHandler->cmdmap[$mmTable][$fileReference['uid']]['delete'] = 1;
        }
        //var_dump($dataHandler->datamap[$mmTable],$targetFileReferences);        die( implode(',', $newTargetFileReferences));
        return implode(',', $newTargetFileReferences);
    }
    
    protected function getEntriesByUidForeign(
        string $foreignTable,
        string $uidForeign,
        string $uid,
        string $sorting = 'sorting'
    ) {
        // get filesReferences of target
        if ((int)$uid > 0) {
            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($foreignTable);
            // show hidden
            $queryBuilder->getRestrictions()
                ->removeByType(HiddenRestriction::class);

            return $queryBuilder
                ->select('*')
                ->from($foreignTable)
                ->where(
                    $queryBuilder->expr()
                        ->eq($uidForeign, $queryBuilder->createNamedParameter($uid, Connection::PARAM_INT))
                )
                ->orderBy($sorting)
                ->executeQuery()
                ->fetchAllAssociative();
        }
        // empty list
        return [];
    }
    
    /**
     * create/update inline entries on ceate/copy events occurence
     */
    protected function createInline(
        DataHandler $dataHandler,
        string $targetUid,
        string $fieldName,
        string $sourceUid,
        string $sourceValue,
        array $tcaConfig,
    ): string {
        $foreignTable = $tcaConfig['foreign_table'];
        $uidForeign = $tcaConfig['foreign_field'];
        $sorting = $tcaConfig['foreign_sortby'];

        $targetEntries = $this->getEntriesByUidForeign($foreignTable, $uidForeign, $targetUid, $sorting);
        
        // get inline entries of source
        $sourceEntyUids = array_filter(GeneralUtility::trimExplode(',', $sourceValue));
        $sourceEntries = [];
        foreach ($sourceEntyUids as $uid) {
            $sourceEntries[] = $dataHandler->datamap[$foreignTable][$uid] ?? $this->translationsMm[$mmTable][$uid];
        }
        // refetch existing entries
        $oldSourceEntries = $this->getEntriesByUidForeign($foreignTable, $uidForeign, $sourceUid, $sorting);
        $oldSourceEntryUids = array_column($oldSourceEntries, 'uid');
        $oldSourceValue = implode(',', $oldSourceEntryUids);
        $oldSourceEntriesByUid = array_combine($oldSourceEntryUids, $oldSourceEntries);
        // compare entries of source and target
        $newTargetEntries = [];
        
        $attributes = $this->getAttributesForDatamap($foreignTable, [$uidForeign]);
        
        foreach ($sourceEntries as $index => $entry) {
            // check if it is empty e.g. only hidden -> only update hidden
            if (count($entry) == 1 && isset($entry['hidden'])) {
                $currentSourceUid = $sourceEntyUids[$index];
                $newEntry = [];
                foreach ($attributes as $attribute) {
                    $v = $oldSourceEntriesByUid[$currentSourceUid][$attribute] ?? null;
                    if ($v !== null) {
                        $newEntry[$attribute] = $v;
                    }
                }
                $newEntry['hidden'] = $entry['hidden'];
                $newUid = StringUtility::getUniqueId('NEW');
                $newEntry['pid'] = $this->pid;
                //$newEntry['l10n_state'] = '{"additional_form_fields":"custom"}';
                /*
                  if (isset($targetEntries[$index]) && isset($oldSourceEntries[$index])) {
                  // is entry on same position
                  if ($sourceEntyUids[$index] == $oldSourceEntries[$index]['uid']) {
                  // only update hidden
                  $newUid = $targetEntries[$index]['uid'];
                  unset($targetEntries[$index]);
                  } else {
                  // sorting has changed
                  }
                  }
                */
            } else {
                // in the moment we recreate always -> only if not true above
                $newEntry = $entry;
                $newUid = StringUtility::getUniqueId('NEW');
                $newEntry['pid'] = $this->pid;
                //$newEntry['l10n_state'] = '{"additional_form_fields":"custom"}';
            }
            $dataHandler->datamap[$foreignTable][$newUid] = $newEntry;
            $newTargetEntries[] = $newUid;
        }
        // remove unused inline entries of target
        foreach ($targetEntries as $entry) {
            $dataHandler->cmdmap[$foreignTable][$entry['uid']]['delete'] = 1;
        }
        //var_dump($dataHandler->datamap[$foreignTable],$targetEntries);        die( implode(',', $newTargetEntries));
        return implode(',', $newTargetEntries);
    }

    /**
     * get all event by eventgroup expect current event with uid $currentUid
     */
    protected function getEventsByEventgroupGroupByDateFrom(
        EventGroup $eventgroup,
        int $currentUid,
        array $recurrence,
        string $currentDate = '',
        int $sysLanguageUid = 0
    ): array {
        if ($recurrence['update_scope'] === '' || $recurrence['update_scope'] === 'this') {
            return [];
        }
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            $this->eventTable
        );
        $queryBuilder->select('*')
            ->from($this->eventTable)
            ->where(
                $queryBuilder->expr()->eq($this->eventTableEventgroupAttribute, $queryBuilder->createNamedParameter($eventgroup->getUid(), Connection::PARAM_INT)),
                $queryBuilder->expr()->neq('uid', $queryBuilder->createNamedParameter($currentUid, Connection::PARAM_INT)),
                $queryBuilder->expr()->or(
                    $queryBuilder->expr()->eq('sys_language_uid', $queryBuilder->createNamedParameter($sysLanguageUid, Connection::PARAM_INT)),
                    $queryBuilder->expr()->eq('sys_language_uid', $queryBuilder->createNamedParameter(-1, Connection::PARAM_INT))
                )
            );
        $excludeDates = $eventgroup->getExcludeDatesArray();
        if (!empty($excludeDates)) {
            $queryBuilder->andWhere(
                $queryBuilder->expr()->notIn($this->eventTableMainDateAttribute, $queryBuilder->createNamedParameter(
                    $excludeDates,
                    Connection::PARAM_STR_ARRAY
                ))
            );
        }
        if ($recurrence['update_scope'] === 'future') {
            $queryBuilder->andWhere(
                $queryBuilder->expr()->gte($this->eventTableMainDateAttribute, $queryBuilder->createNamedParameter(
                    $currentDate,
                    Connection::PARAM_STR
                ))
            );
        } elseif ($recurrence['update_scope'] !== 'all') {
            die('recurrence.update_scope "' . $recurrence['update_scope'] . '" not found');
        }
        $queryBuilder->orderBy($this->eventTableMainDateAttribute, 'ASC');
        $rows = $queryBuilder->executeQuery()
            ->fetchAllAssociative();

        $existingEvents = [];
        foreach ($rows as $row) {
            $existingEvents[$row[$this->eventTableMainDateAttribute]] = $row['uid'];
        }
        return $existingEvents;
    }

    /**
     * get events by $eventgroup exluide event with uid $currentUid depengind on delete_scope of $recurrence
     */
    protected function getEventsByEventgroupGroupByUid(
        EventGroup $eventgroup,
        int $currentUid,
        array $recurrence
    ): array {
        if ($recurrence['delete_scope'] === '' || $recurrence['delete_scope'] === 'this') {
            return [];
        }

        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($this->eventTable);
        $queryBuilder->select('*')
            ->from($this->eventTable)
            ->where(
                $queryBuilder->expr()
                    ->eq($this->eventTableEventgroupAttribute, $queryBuilder->createNamedParameter($eventgroup->getUid(), Connection::PARAM_INT)),
                $queryBuilder->expr()
                    ->neq('uid', $queryBuilder->createNamedParameter($currentUid, Connection::PARAM_INT))
            );
        if ($this->registrationTable) {
            $queryBuilder->andWhere(
                'NOT EXISTS (SELECT 1 FROM ' . $this->registrationTable . ' WHERE ' . $this->registrationTable . '.' . $this->registrationTableMatchField . '=' . $this->eventTable . '.uid AND deleted=0)'
            );
        }
        $queryBuilder->orderBy($this->eventTableMainDateAttribute, 'ASC');
        $rows = $queryBuilder->executeQuery()
            ->fetchAllAssociative();

        $existingEvents = [];
        foreach ($rows as $row) {
            $existingEvents[$row['uid']] = $row['uid'];
        }
        return $existingEvents;
    }

    /**
     * count reservation of a event
     */
    protected function eventCountReservations(int $eventUid): int
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            $this->registrationTable
        );
        $count = $queryBuilder
            ->count('*')
            ->from($this->registrationTable)
            ->where(
                $queryBuilder->expr()->eq('event', $queryBuilder->createNamedParameter(
                    $eventUid,
                    Connection::PARAM_INT
                ))
            )
            ->executeQuery()
            ->fetchOne();
        return (int) $count;
    }



    /**
     * get eventgroup as object by $uid
     */
    protected function getEventgroupByUid(int $uid): ?Eventgroup
    {
        if (!$uid) {
            return null;
        }
        $eventgroupArray = $this->getEventgroupByUidRaw($uid);
        if ($eventgroupArray !== false && is_array($eventgroupArray) && !empty($eventgroupArray)) {
            $this->pid = $eventgroupArray['pid'];
            $dataMapper = GeneralUtility::makeInstance(DataMapper::class);
            $eventgroups = $dataMapper->map(Eventgroup::class, [$eventgroupArray]);
            return $eventgroups[0];
        }
        return null;
    }

    /**
     * get eventgroup as array by $uid
     */
    protected function getEventgroupByUidRaw(int $uid)
    {
        return $this->getByUid($this->eventgroupTable, $uid);
    }

    /**
     * get event as array by $uid
     */
    protected function getEventByUidRaw(int $uid)
    {
        return $this->getByUid($this->eventTable, $uid);
    }

    protected function getByUid(string $tableName, int $uid)
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tableName);
        return $queryBuilder
            ->select('*')
            ->from($tableName)
            ->where($queryBuilder->expr()->eq('uid', $queryBuilder->createNamedParameter($uid, Connection::PARAM_INT)))
            ->executeQuery()
            ->fetchAssociative();
    }
    protected function getEventByEventgroupAndDatedRaw(int $eventgroupUid, string $targetDate)
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($this->eventTable);
        return $queryBuilder
            ->select('*')
            ->from($this->eventTable)
            ->where(
                $queryBuilder->expr()->eq($this->eventTableEventgroupAttribute, $queryBuilder->createNamedParameter($eventgroupUid, Connection::PARAM_INT)),
                $queryBuilder->expr()->eq($this->eventTableMainDateAttribute, $queryBuilder->createNamedParameter($targetDate))
            )
            ->executeQuery()
            ->fetchAssociative();
    }


    /**
     * diff in dayes of two date strings
     */
    protected function diffInDays(string $oldDateString, string $newDateString): int
    {
        $oldDateString = substr($oldDateString, 0, 10);
        $newDateString = substr($newDateString, 0, 10);
        if ($oldDateString != $newDateString) {
            $oldDateDateTime = new DateTime($oldDateString);
            $newDateDateTime = new DateTime($newDateString);
            $interval = $oldDateDateTime->diff($newDateDateTime);
            return $interval->invert ? -$interval->days : $interval->days;
        }
        return 0;
    }

    /**
     * add $addDays to $date
     */
    protected function addDays(?string $date, int $addDays): ?string
    {
        if ($date === null || $date === '') {
            return null;
        }
        $dateTime = new DateTime($date);
        if ($addDays < 0) {
            $dateTime->modify($addDays . ' days');
        } else {
            $dateTime->modify('+' . $addDays . ' days');
        }
        return $dateTime->format('Y-m-d');
    }

    /**
     * creates an event and set it for update/insert
     * @param string $targetEventUid int for existing events and NEW... for  new ones
     * @param string $targetDate date_from for the new event
     * @param string $sourceEventUid uid of the event to copy (used for references not in form data)
     * @param array $sourceEvent the data of the event to copy
     */
    protected function createRecurrenceByDate(
        DataHandler $dataHandler,
        ?string $targetEventUid,
        string $targetDate,
        string $sourceEventUid,
        array $sourceEvent,
        array $sourceEventTranslations = []
    ): void {
        $targetEvent = $sourceEvent;
        $targetEvent['pid'] = $this->pid; // pid is not in data
        if ($targetEvent['sys_language_uid'] == 0) {
            $oldDate = $targetEvent[$this->eventTableMainDateAttribute];
            $targetEvent[$this->eventTableMainDateAttribute] = $targetDate;
            $addDays = $this->diffInDays($oldDate, $targetDate);
            foreach ($this->eventTableAdditionalDateAttributes as $dateAttribute) {
                $targetEvent[$dateAttribute] = $this->addDays($targetEvent[$dateAttribute], $addDays);
            }
        } else {
            if ($targetEventUid > 0) {
                $parentEvent = $this->getEventByUidRaw($targetEventUid);
                $targetEvent['l10n_parent'] = (int)$parentEvent['l10n_parent'] ?? 0;
            } else {
                $parentEvent = $this->getEventByEventgroupAndDatedRaw($sourceEvent[$this->eventTableEventgroupAttribute], $targetDate);
                $targetEvent['l10n_parent'] = (int)$parentEvent['uid'] ?? 0;                
            }
        }
        if ($targetEventUid === null || $targetEventUid === '') {
            $targetEventUid = StringUtility::getUniqueId('NEW');
        }
        $tcaColumns = $GLOBALS['TCA'][$this->eventTable]['columns'];
        foreach ($tcaColumns as $attribute => $tcaColumn) {
            $tcaConfig = $tcaColumn['config'];
            // copy/create all fileReferences (file, images)
            if ($tcaConfig['type'] == 'file') {
                $targetEvent[$attribute] = $this->createMmRelations(
                    $dataHandler,
                    $targetEventUid,
                    $attribute,
                    $sourceEventUid,
                    $sourceEvent[$attribute]
                );
            }
            // copy/create all inline fields (additional_form_fields)
            if ($tcaConfig['type'] == 'inline') {
                $targetEvent[$attribute] = $this->createInline(
                    $dataHandler,
                    $targetEventUid,
                    $attribute,
                    $sourceEventUid,
                    $sourceEvent[$attribute],
                    $tcaConfig
                );
            }
        }
        $dataHandler->datamap[$this->eventTable][$targetEventUid] = $targetEvent;

        // Handle translations for this event
        if ($targetEvent['sys_language_uid'] == 0) {
            $this->createTranslationsForRecurrence(
                $dataHandler,
                $targetEventUid,
                $targetDate,
                $sourceEventUid,
                $sourceEventTranslations,
                $addDays
            );
        }
    }

    /**
     * Create translations for recurring events
     */
    protected function createTranslationsForRecurrence(
        DataHandler $dataHandler,
        string $targetEventUid,
        string $targetDate,
        string $sourceEventUid,
        array $sourceEventTranslations,
        int $addDays
    ): void {
        if (empty($sourceEventTranslations)) {
            return;
        }

        // Get existing translations for target event (if it exists)
        $existingTargetTranslations = [];
        if (is_numeric($targetEventUid)) {
            $existingTargetTranslations = $this->getEventsByL10nParentRaw($targetEventUid);
        }

        foreach ($sourceEventTranslations as $sourceTranslation) {
            $sysLanguageUid = $sourceTranslation['sys_language_uid'];
            $targetTranslationUid = null;

            // Check if translation already exists for this language
            if (isset($existingTargetTranslations[$sysLanguageUid])) {
                $targetTranslationUid = $existingTargetTranslations[$sysLanguageUid]['uid'];
                // Remove from existing list to track which ones to delete later
                unset($existingTargetTranslations[$sysLanguageUid]);
            } else {
                $targetTranslationUid = StringUtility::getUniqueId('NEW');
            }

            // Prepare translation data
            $targetTranslation = $sourceTranslation;
            $targetTranslation['l10n_parent'] = $targetEventUid;
            $targetTranslation['pid'] = $this->pid;
            $targetTranslation[$this->eventTableMainDateAttribute] = $targetDate;
            foreach ($this->eventTableAdditionalDateAttributes as $dateAttribute) {
                $targetTranslation[$dateAttribute] = $this->addDays($targetTranslation[$dateAttribute], $addDays);
            }

            // Handle MM relations and inline fields for translation
            $this->handleTranslationRelations($dataHandler, $targetTranslationUid, $sourceTranslation, $targetTranslation);

            $dataHandler->datamap[$this->eventTable][$targetTranslationUid] = $targetTranslation;
        }

        // Mark remaining existing translations for deletion
        foreach ($existingTargetTranslations as $translationToDelete) {
            $this->translationsToDelete[] = $translationToDelete['uid'];
        }
    }

    /**
     * Handle MM relations and inline fields for translations
     */
    protected function handleTranslationRelations(
        DataHandler $dataHandler,
        string $targetTranslationUid,
        array $sourceTranslation,
        array &$targetTranslation
    ): void {
        $tcaColumns = $GLOBALS['TCA'][$this->eventTable]['columns'];
        foreach ($tcaColumns as $attribute => $tcaColumn) {
            $tcaConfig = $tcaColumn['config'];
            if ($tcaConfig['type'] == 'file') {
                $targetTranslation[$attribute] = $this->createMmRelations(
                    $dataHandler,
                    $targetTranslationUid,
                    $attribute,
                    $sourceTranslation['uid'],
                    $sourceTranslation[$attribute]
                );
            }
            if ($tcaConfig['type'] == 'inline') {
                $targetTranslation[$attribute] = $this->createInline(
                    $dataHandler,
                    $targetTranslationUid,
                    $attribute,
                    $sourceTranslation['uid'],
                    $sourceTranslation[$attribute],
                    $tcaConfig
                );
            }
        }
    }

    /**
     * Add translations of an event to deletion list
     */
    protected function addTranslationsToDeleteList(int $eventUid): void
    {
        $translations = $this->getEventsByL10nParentRaw($eventUid);
        foreach ($translations as $translation) {
            $this->translationsToDelete[] = $translation['uid'];
        }
    }

    /**
     * Schedule deletion of orphaned translations
     */
    protected function scheduleTranslationDeletions(DataHandler $dataHandler): void
    {
        foreach (array_unique($this->translationsToDelete) as $translationUid) {
            $dataHandler->cmdmap[$this->eventTable][$translationUid]['delete'] = 1;
        }
    }
    
    /**
     * are recurrence enabled or was recurrence enabled (still has eventgroup)
     */
    protected function isRecurrenceEnable(array $event, array $recurrence = null): bool
    {
        // has eventgroup
        $eventgroupUid = (int) $this->getEventgroupFromEvent($event);
        if ($eventgroupUid) {
            return true;
        }
        // recurrence frequency defined (e.g. weekly), update_scope not only 1, event date set
        return $recurrence !== null && ($recurrence['frequency'] ?? '') != '' && ($recurrence['update_scope'] ?? 'this') != 'this' && ($event[$this->eventTableMainDateAttribute] ?? false);
    }

    protected function getEventgroupFromEvent(array $event): ?int
    {
        // is translation
        if ((int)($event['sys_language_uid'] ?? 0) > 0) {
            $parentUid = (int)($event['l10n_parent'] ?? 0);
            if ($parentUid) {
                $parentEvent = $this->getEventByUidRaw($parentUid);
                return $parentEvent[$this->eventTableEventgroupAttribute];
            }
            
        } else {
            return $event[$this->eventTableEventgroupAttribute];
        }
        return null;
    }
    
    /**
     * get existing eventgroup from a event or create a new one, updated/set with values from $recurrence (if update_scope is all)
     * on copy $recurrence is not fully set (detected insetWithFormData) and we are returning null (remove eventgroup on copy)
     */
    protected function getEventgroup(array $event, array $recurrence): ?Eventgroup
    {
        $eventgroupUid = $this->getEventgroupFromEvent($event);
        $eventgroupArray = $eventgroupUid ? $this->getEventgroupByUidRaw($eventgroupUid) : false;
        if ($eventgroupArray !== false && is_array($eventgroupArray) && !empty($eventgroupArray)) {
            $dataMapper = GeneralUtility::makeInstance(DataMapper::class);
            $eventgroups = $dataMapper->map(Eventgroup::class, [$eventgroupArray]);
            $eventgroup = $eventgroups[0];
        } else {
            $eventgroup = GeneralUtility::makeInstance(Eventgroup::class);
            $eventgroup->setRecurrenceStartsOn(new DateTime(substr((string) $event[$this->eventTableMainDateAttribute], 0, 10)));
            $eventgroup->setTablename($this->eventTable);
            //$eventgroup->setLabel(substr($event[$this->eventTableMainDateAttribute], 0 , 10) . ' ' . $recurrence['frequency']);
        }
        // only update e eventgroup if we update all occurences
        if ($recurrence['update_scope'] === 'all') {
            if (!$eventgroup->setWithFormData($recurrence, $event['title'])) {
                return null;
            }
        }
        //DebuggerUtility::var_dump($recurrence);DebuggerUtility::var_dump($eventgroup);DebuggerUtility::var_dump($event);exit(0);
        return $eventgroup;
    }

    /**
     * update exclude_dates of a given eventgroup in database
     */
    protected function updateEventgroupSetExcludeDateseventCountReservations(int $uid, Eventgroup $eventgroup)
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            $this->eventgroupTable
        );
        return $queryBuilder
            ->update($this->eventgroupTable)
            ->where(
                $queryBuilder->expr()
                    ->eq('uid', $uid)
            )->set('recurrence_exclude_dates', $eventgroup->getExcludeDatesJson())
            ->executeStatement();
    }

    /**
     * get recurrence settings from current request
     */
    protected function requestGetRecurrence(): array
    {
        $request = $GLOBALS['TYPO3_REQUEST'];
        $requestData = array_merge($request->getQueryParams() ?? [], $request->getParsedBody() ?? []);
        $recurrence = $requestData['recurrence'] ?? [];
        if (!isset($recurrence['update_scope']) || !$recurrence['update_scope']) {
            $recurrence['update_scope'] = 'this';
        }
        if (!isset($recurrence['delete_scope']) || !$recurrence['delete_scope']) {
            $recurrence['delete_scope'] = 'this';
        }
        return $recurrence;
    }


    protected function getEventsByL10nParentRaw(int $l10nParentUid): array
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($this->eventTable);
        $queryBuilder->getRestrictions()->removeByType(HiddenRestriction::class);
        $res = $queryBuilder
            ->select('*')
            ->from($this->eventTable)
            ->where(
                $queryBuilder->expr()->eq('l10n_parent', $queryBuilder->createNamedParameter($l10nParentUid, Connection::PARAM_INT)),
                $queryBuilder->expr()->eq('deleted', $queryBuilder->createNamedParameter(0, Connection::PARAM_INT))
            )
            ->executeQuery();

        // Enrich translations with MM and inline data
        $rows = [];
        while ($row = $res->fetchAssociative()) {
            $rows[$row['sys_language_uid']] = $this->enrichTranslationWithRelations($row);
        }
        return $rows;
    }

    /**
     * Enrich translation with MM relations and inline data
     */
    protected function enrichTranslationWithRelations(array $translation): array
    {
        $tcaColumns = $GLOBALS['TCA'][$this->eventTable]['columns'];

        foreach ($tcaColumns as $attribute => $tcaColumn) {
            $tcaConfig = $tcaColumn['config'];
            $mmTable = isset($tcaConfig['MM']) ? $tcaConfig['MM'] : 'sys_file_reference';
            if (!isset($this->translationsMm[$mmTable])) {
                $this->translationsMm[$mmTable] = [];
            }

            // Handle file references
            if ($tcaConfig['type'] == 'file') {
                $mms = $this->getMm($translation, $attribute, $tcaConfig);
                $translation[$attribute] = $this->getCsv($mms);
                foreach ($mms as $mm) {
                    $mmUid = isset($mm['uid']) ? $mm['uid'] : $mm['uid_foreign'];
                    $this->translationsMm[$mmTable][$mmUid] = $mm;
                }
            }
            // Handle select fields with MM relations
            if ($tcaConfig['type'] == 'select' && ($tcaConfig['renderType'] ?? '') == 'selectMultipleSideBySide') {
                $mms = $this->getMm($translation, $attribute, $tcaConfig);
                $translation[$attribute] = $this->getCsv($mms);
                foreach ($mms as $mm) {
                    $mmUid = isset($mm['uid']) ? $mm['uid'] : $mm['uid_foreign'];
                    $this->translationsMm[$mmTable][$mmUid] = $mm;
                }
            }
            // Handle inline relations
            if ($tcaConfig['type'] == 'inline') {
                $inlines = $this->getInline($translation, $attribute, $tcaConfig);
                $translation[$attribute] = $this->getCsv($inlines);
                $mmTable = $tcaConfig['foreign_table'];
                foreach($inlines as $inline) {
                    $mmUid = $inline['uid'];
                    $this->translationsMm[$mmTable][$mmUid] =  $inline;
                }
            }
        }
        return $translation;
    }

    protected function getMm(array $translation, string $attribute, array $tcaConfig, $uidForeign = 'uid_foreign')
    {
        $defaultTable = 'sys_file_reference';
        return $this->getMmByUidForeign($tcaConfig['MM'] ?? $defaultTable, $translation['uid'], $attribute, isset($tcaConfig['foreign_field']) ? $tcaConfig['foreign_field'] : $uidForeign, isset($tcaConfig['MM']) ? false : true);
    }

    protected function getInline(array $translation, string $attribute, array $tcaConfig)
    {
        return $this->getEntriesByUidForeign($tcaConfig['foreign_table'], $tcaConfig['foreign_field'], $translation['uid']);
    }

    protected function getCsv(array $rows)
    {
        if (!empty($rows)) {
            return implode(',', array_column($rows, 'uid'));
        }

        return '';
    }

    /**
     * Get recurrence configuration for a specific table
     */
    protected function initRecurrenceConfigForTable(string $tableName): bool
    {

        $config = TcaUtility::getRecurrenceConfigForTable($tableName);
        if (empty($config)) {
            return false;
        }
        $this->eventTable = $config['eventTable'];
        // $config['eventTableLabel'];
        $this->eventTableMainDateAttribute = $config['eventTableMainDateAttribute']; 
        $this->eventTableAdditionalDateAttributes = $config['eventTableAdditionalDateAttributes']; 
        $this->eventTableEventgroupAttribute = $config['eventTableEventgroupAttribute']; 
        $this->eventgroupTable = $config['eventgroupTable']; 
        $this->registrationTable = $config['registrationTable']; 
        $this->registrationTableMatchField = $config['registrationTableMatchField']; 
        return true;
    }
    
}
