<?php

namespace Internetgalerie\IgRender\Utility;

use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\PathUtility;

class PdfUtility
{
    const CONVERTER_WKHTMLTOPDF = 'wkhtmltopdf';
    const CONVERTER_WEASYPRINT = 'weasyprint';

    private $converter = self::CONVERTER_WKHTMLTOPDF;
    private $format = 'pdf';
    private static ?int $weasyprintVersion = null;
    private $filename = null;
    private bool $addHtmlTag = true;

    private $headerHtml = null;
    private $footerHtml = null;
    private $pageFooterHtml = null;
    private $pageHeaderHtml = null;
    private $css = '';
    private $title = '';
    private $pageMarginTop = '10mm';
    private $pageMarginBottom = '10mm';
    private $pageMarginLeft = '10mm';
    private $pageMarginRight = '10mm';
    private $tempFiles= [];

    private $options = [
                        '--print-media-type',
                        '--disable-smart-shrinking',
                        ];
    private $defaultOptions = [
        self::CONVERTER_WKHTMLTOPDF => [
            '--print-media-type',
            '--disable-smart-shrinking',
        ],
        self::CONVERTER_WEASYPRINT => [
            '-q', // Hide logging messages -> prevent stderr to fill buffer
        ],
        ];
    /**
     * get html width Footer/Header (CSS)
     *
     * @param string $html
     * @return string html data
     */
    public function getHtml($html)
    {
        $header = $this->headerHtml===null ? $this->header() : $this->headerHtml;
        $footer = $this->footerHtml===null ? $this->footer() : $this->footerHtml;
        return $header . $html . $footer;
    }
    /**
     * get pdf from html
     *
     * @param string $html
     * @return string PDF data
     */
    public function get(string $html): string
    {
        $pipes = [];
        $descriptorspec = [
                           0 => ['pipe', 'r'],  // STDIN ist eine Pipe, von der das Child liest
                           1 => ['pipe', 'w'],
                           2 => ['pipe', 'w'],  // stderr
        ];
        $pageMargins = '-L '.$this->pageMarginLeft.' -R '.$this->pageMarginRight.' -T '.$this->pageMarginTop.' -B '.$this->pageMarginBottom.'';
        //echo($tmpFooterUrl);
        //unlink($tmpFooterFullPath);
        //die('wkhtmltopdf '.implode(' ', $this->options).' '.$pageMargins.' - -');
            $this->addFooterHtml();
            $this->addHeaderHtml();
        if ($this->converter==self::CONVERTER_WEASYPRINT) {
            $optionsString = '';
            // weasyprint options
            // @todo -f option is dropped in weasyprint version 58?, 54? but mandatory in version 51
            if (self::$weasyprintVersion === null) {
                $versionInfoString = shell_exec('weasyprint --version');
                $versionInfos = explode(' ', $versionInfoString);                
                self::$weasyprintVersion = (int)($versionInfos[2] ?? 51);
            }

            if (self::$weasyprintVersion <= 51) {
                $optionsString .= ' -f pdf';
            }
            if(!empty($this->options)) {
                $optionsString .=' ' . implode(' ', $this->options);
            }
            $process = proc_open('weasyprint ' . $optionsString . ' - -', $descriptorspec, $pipes);
        } else {
            $process = proc_open('wkhtmltopdf '.implode(' ', $this->options).' '.$pageMargins.' - -', $descriptorspec, $pipes);
        }
        if (!is_resource($process)) {
            throw new \RuntimeException('Failed to start WeasyPrint process');
        }

        $header = $this->headerHtml===null ? $this->header() : $this->headerHtml;
        $footer = $this->footerHtml===null ? $this->footer() : $this->footerHtml;
        fwrite($pipes[0], $this->getHtml($html));
        fclose($pipes[0]);
        $pdfOutput = stream_get_contents($pipes[1]);
        $stderr = stream_get_contents($pipes[2]);
        $status = proc_close($process);
        if ($status !== 0) {
            throw new \RuntimeException("WeasyPrint failed with status code $status: " . trim($stderr));
        }

        $this->cleanup();
        return  $pdfOutput;
    }
    /**
     * get html width Footer/Header (CSS)
     *
     * @param string $html
     * @return string html data
     */
    public function debug($html)
    {
        die($this->getPageHeaderHtml() . $this->getHtml($html) . $this->getPageFooterHtml());
    }

    /**
     * print out pdf data
     *
     * @param string $html
     * @return void
     */

    public function out($html)
    {
        $pdf_data = $this->get($html);
        header('Content-type: application/pdf');
        if ($this->filename) {
            header('Content-Disposition: attachment; filename="' . $this->filename .'"');
        }
        echo($pdf_data);
        //exit(0);
    }
    
    public function setConverter($converter)
    {
        if(isset($this->defaultOptions[$converter])) {
            $this->options = $this->defaultOptions[$converter];
        } else {
            die('convert "' . $converter. '" not found');
        }
        $this->converter = $converter;
    }
    public function getConverter()
    {
        return $this->converter;
    }

    public function setFormat($format)
    {
        $this->format = $format;
    }
    public function getFormat()
    {
        return $this->format;
    }

    public function setAddHtmlTag(bool $addHtmlTag): void
    {
        $this->addHtmlTag = $addHtmlTag;
    }
    public function getAddHtmlTag(): bool
    {
        return $this->addHtmlTag;
    }

    public function setTitle($value)
    {
        $this->title = $value;
    }
    public function getTitle()
    {
        return $this->title;
    }

    public function setFilename($value)
    {
        $this->filename=$value;
    }
    public function getFilename()
    {
        return $this->filename;
    }

    public function setHeaderHtml($value)
    {
        $this->headerHtml=$value;
    }
    public function getHeaderHtml()
    {
        return $this->headerHtml;
    }

    public function setCss($value)
    {
        $this->css=$value;
    }
    public function addCss($value)
    {
        $this->css.=$value;
    }
    public function setCssFile($path)
    {
        $this->css=file_get_contents(GeneralUtility::getFileAbsFileName($path));
    }
    public function addCssFile($path)
    {
        $this->css.=file_get_contents(GeneralUtility::getFileAbsFileName($path));
    }
    public function getCss()
    {
        return $this->css;
    }

    public function getCssParsed()
    {
        $replaces = [
            '###SERVER_NAME###' => $_SERVER['SERVER_NAME'],
            '###HTTP_HOST###' => $_SERVER['HTTP_HOST'],
            '###REQUEST_SCHEME###' => $_SERVER['REQUEST_SCHEME'],
            '###BASE_URL###' => $_SERVER['REQUEST_SCHEME'] .'://' . $_SERVER['HTTP_HOST'],
            '###PUBLIC_PATH###' => Environment::getPublicPath(),
        ];
        $css = str_replace(array_keys($replaces), array_values($replaces), $this->css);
        //$site = $GLOBALS['TYPO3_REQUEST']->getAttribute('site');
        //$base = $site->getAttribute('base');
        $base = $GLOBALS['TYPO3_REQUEST']->getAttribute('normalizedParams')->getSiteUrl();
        // callback function to replace EXT:my_ext/Resources/Public/ with the Public Resource Web Path
        $css = preg_replace_callback(
            '/(EXT:[a-z_]+\/Resources\/Public\/)/',
            function ($matches) use ($base) {
                $extPublicPath = $matches[1];
                $publicResourcePath = PathUtility::getPublicResourceWebPath($extPublicPath);
                return rtrim($base, '/') . $publicResourcePath;
            },
            $css,
        );

        return $css;
    }
    
    public function setFooterHtml($value)
    {
        $this->footerHtml=$value;
    }
    public function getFooterHtml()
    {
        return $this->footerHtml;
    }


    public function setPageMargin($value)
    {
        $this->pageMarginTop=$value;
        $this->pageMarginBottom=$value;
        $this->pageMarginLeft=$value;
        $this->pageMarginRight=$value;
    }
    public function setPageMarginLeft($value)
    {
        $this->pageMarginLeft=$value;
    }
    public function getPageMarginLeft()
    {
        return $this->pageMarginLeft;
    }

    public function setPageMarginRight($value)
    {
        $this->pageMarginRight=$value;
    }
    public function getPageMarginRight()
    {
        return $this->pageMarginRight;
    }

    public function setPageMarginTop($value)
    {
        $this->pageMarginTop=$value;
    }
    public function getPageMarginTop()
    {
        return $this->pageMarginTop;
    }

    public function setPageMarginBottom($value)
    {
        $this->pageMarginBottom=$value;
    }
    public function getPageMarginBottom()
    {
        return $this->pageMarginBottom;
    }

    public function setOptions(array $value)
    {
        $this->options=$value;
    }
    public function addOption($value)
    {
        $this->options[]=$value;
    }
    public function getOptions()
    {
        return $this->options;
    }



    /**
     * get the header for the pdf in html
     *
     * @return string HTML data for PDF
     */
    public function header()
    {
        $css = $this->getCssParsed();
        $cssCode = $css ? '<style type="text/css">' . $css . '</style>' : '';
        if ($this->addHtmlTag) {
            setlocale(LC_ALL, 'de_CH.utf8');

            
            return '<html xml:lang="de" lang="de" xmlns="http://www.w3.org/1999/xhtml">
<head>

<meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
' . $cssCode . '
<title>'.$this->title.'</title>
</head>
<body>';
        }
        return $cssCode;
    }


    /**
     * get the footer for the pdf in html
     *
     * @return string HTML data for PDF
     */
    public function footer()
    {
        return $this->addHtmlTag ? '</body></html>' : '';
    }

    public function setPageHeaderHtml($pageHeaderHtml)
    {
        $this->pageHeaderHtml=$pageHeaderHtml;
    }
    
    public function getPageHeaderHtml()
    {
        return $this->pageHeaderHtml;
    }
    public function setPageFooterHtml($pageFooterHtml)
    {
        $this->pageFooterHtml=$pageFooterHtml;
    }
    
    public function getPageFooterHtml()
    {
        return $this->pageFooterHtml;
    }

    public function addHeaderHtml()
    {
        $this->addOptionUrl('--header-html', $this->pageHeaderHtml);
    }
    public function addFooterHtml()
    {
        $this->addOptionUrl('--footer-html', $this->pageFooterHtml);
    }
    /*
    public function addFooterHtmlFile( ) {
        $filename=tempnam( '/tmp', 'pdf-footer-' ) ;

        file_put_contents($filename . '.html', '<!DOCTYPE html><html><head><meta charset="UTF-8" /></head><body>' . $this->pageFooterHtml . '</body></html>');
        $this->addOptionUrl( '--footer-html', "'" . $filename . ".html'");
        $this->tempFiles[]=$filename . '.html';
        $this->tempFiles[]=$filename;

        return;
    }
    */
    public function addOptionUrl($optionName, $content)
    {
        if ($content) {
            $fullPath = tempnam(Environment::getPublicPath() . '/' . '/typo3temp/', 'wkhtmltopdf');
            $filename= basename($fullPath);
            $handle = fopen($fullPath, "w");
            $css = $this->getCssParsed();
            fwrite($handle, '<!DOCTYPE html><html><head><style type="text/css">'.$css.'</style></head><body>' . $content .'</body></html>');
            fclose($handle);
            chmod($fullPath, 0644);
            $tmpUrl = ($_SERVER['HTTPS'] ? 'https://' : 'http://'). $_SERVER['HTTP_HOST'] . '/typo3temp/'.$filename;
            $this->tempFiles[]=$fullPath;

            $this->addOption($optionName . ' ' . $tmpUrl);
        }
    }
    public function cleanup()
    {
        foreach ($this->tempFiles as $file) {
            unlink($file);
        }
    }
}
