<?php
namespace Internetgalerie\IgRender\ViewHelpers;

/***************************************************************
 *  Copyright notice
 *
 *  (c) 2014 Markus Baumgartner <mb@stylesheep.ch>, Internetgalerie AG
 *
 *  All rights reserved
 *
 *  This script is part of the TYPO3 project. The TYPO3 project is
 *  free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  The GNU General Public License can be found at
 *  http://www.gnu.org/copyleft/gpl.html.
 *
 *  This script is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  This copyright notice MUST APPEAR in all copies of the script!
 ***************************************************************/

 /**
  * Renders a GoogleMap
  *
  * EXAMPLE:
  * <ig:map lat="33.3" lng="10" latlng="33.33,10.0" zoom="7" type="satellite" fitToMarkers="TRUE" options="some : map, options : \{ like : this \}">
  *
  *		<ig:map.marker lat="33.3" lng="10" latlng="33.33,10.0" title="Some Title (mouseover)" >
  * 		<strong>HTML Info Window</strong><br />
  * 			etc.
  * 	</ig:map.marker>
  * 	[...]
  * </ig:map>
  *
  *  <ig:map>
  *     <ig:map.layer icon="/fileadmin/icon.png" options="{some: options}"> //set options and icons for ALL markers inside this layer
  *          <ig:map.marker lat="33.3" lng="10" latlng="33.33,10.0" title="Some Title (mouseover)" >
  *            <strong>HTML Info Window</strong><br />
  * 			etc.
  *          </ig:map.marker>
  *			 [...]
  *      </ig:map.layer>
  *
  * 	<ig:map.layer con="/otherIcon.png"  options="draggable:true" >
  *
  *          <ig:map.marker lat="33.3" lng="10" latlng="33.33,10.0" title="Some Title (mouseover)" >
  *            <strong>HTML Info Window</strong><br />
  * 			etc.
  *          </ig:map.marker>
  *
  * 	</ig:map.layer>
  * 	[...]
  *  </ig:map>
  */

use TYPO3\CMS\Core\Utility\GeneralUtility;

class MapViewHelper extends \TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper
{
    protected $tagName = 'div';

    public static $instanceCount = 0;
    protected $elementSuffix;

    public function initializeArguments()
    {
        parent::initializeArguments();
        $this -> registerArgument('lat', 'float', 'Latitude', false);
        $this -> registerArgument('lng', 'float', 'Longitude', false);
        $this -> registerArgument('latlng', 'string', 'Both, as commaseparated string', false, '46.9127509564,8.19580078125');
        $this -> registerArgument('key', 'string', 'Google API Key, see: https://developers.google.com/maps/documentation/javascript/get-api-key', false, '');
        $this -> registerArgument('zoom', 'int', 'Zoom level', false, 7);
        $this -> registerArgument('type', 'string', 'google.maps.MapTypeId.[type] constant', false, 'ROADMAP');
        $this -> registerArgument('fitToMarkers', 'string', 'fit to marker bounds?', false, false);
        $this -> registerArgument('options', 'string', 'A list of options in JSON Format', false);
        $this -> registerUniversalTagAttributes();
    }

    public function render()
    {
        //unique map id
        $this -> elementSuffix = self::$instanceCount++;
        $pageRenderer=GeneralUtility::makeInstance(\TYPO3\CMS\Core\Page\PageRenderer::class);

        //include the API
        //$api = "https://maps.google.com/maps/api/js?v=3.2".($this->arguments['key'] ? '&key='. $this->arguments['key'] : '');
        $api = "https://maps.googleapis.com/maps/api/js?callback=initMap".($this->arguments['key'] ? '&key='. $this->arguments['key'] : '');
        $pageRenderer->addJsFooterFile($api);

        //set vars
        $this -> templateVariableContainer -> add('layers', array());
        $templateVariableContainer = $this->renderingContext->getVariableProvider();
        $templateVariableContainer->add('__googleMap', [
                                 'directionsService' => [],
                                 'markers' => []
                                 ]);
        $this -> renderChildren();
        $markers = $this -> renderMarkers();
        $directionsService = $this -> renderDirectionsService('gMap'.$this->elementSuffix);
        $options = $this -> getMapOptions();

        //set bounds automagically
        $fitToBounds = '';
        $resizeTrigger = <<< INIT
			var center = gMap{$this->elementSuffix}.getCenter();
			google.maps.event.trigger(gMap{$this->elementSuffix}, "resize");
			gMap{$this->elementSuffix}.setCenter(center); 
INIT;

        if ($this -> arguments['fitToMarkers']) {
            $fitToBounds = <<< INIT
			if(gMapMarkers{$this->elementSuffix}.length>0){
				var gMapBounds{$this->elementSuffix} = new google.maps.LatLngBounds();
				gMapMarkers{$this->elementSuffix}.forEach(function(marker){
					gMapBounds{$this->elementSuffix}.extend(marker.getPosition());
				});
				gMap{$this->elementSuffix}.fitBounds(gMapBounds{$this->elementSuffix});
			}
INIT;
            //if fit to bounds, do the same on resize
            $resizeTrigger = $fitToBounds;
        }

        $js = <<< INIT
	var gMap{$this->elementSuffix};
    var gMapMarkers{$this->elementSuffix} = new Array();
    var gMapinfoWindow{$this->elementSuffix} = new google.maps.InfoWindow();
    
    function initGMap{$this->elementSuffix}() {
    	var myOptions = {$options};
    	gMap{$this->elementSuffix} = new google.maps.Map(document.getElementById("gMap{$this->elementSuffix}"), myOptions);
   	 	{$markers}
   	 	{$directionsService}
		{$fitToBounds}
	google.maps.event.addDomListener(window, "smartresize", function() {
		{$resizeTrigger}
	});


    };
    google.maps.event.addDomListener(window, 'load', initGMap{$this->elementSuffix});
INIT;
        $pageRenderer->addJsFooterInlineCode('gMap'.$this->elementSuffix, $js);

        //could be moved to a Helper class, with safe inlcude and stuff
        $GLOBALS['TSFE'] -> additionalHeaderData['ig_render_map'] .= $headerJs;

        // set div tag attributes
        $this -> tag -> addAttribute('id', 'gMap' . $this -> elementSuffix);
        //we use the dp-map class here to make it easier to use in datapool, i.e. there is some css for it
        $this -> tag -> addAttribute('class', 'dp-map ' . $this -> arguments['class']);
        $this -> tag -> forceClosingTag(true);
        return $this -> tag -> render();
    }

    public function renderMarkers()
    {
        $layers = $this -> templateVariableContainer -> get('layers');

        $allMarkers = array();
        foreach ($layers as $layer) {
            foreach ($layer['markers'] as $marker) {
                $markerId = $marker['id'];
                $infoWindowCode = '';
                if ($marker['infoWindow']) {
                    $infoWindow = $marker['infoWindow'];
                    $infoWindow = str_replace(array("\r\n", "\n"), "\\n", $infoWindow);
                    $infoWindow = str_replace('"', '\\"', $infoWindow);
                    $infoWindowCode = <<< INIT
google.maps.event.addListener({$markerId}, 'click', function(event) {
	gMapinfoWindow{$this->elementSuffix}.close();
	//create aa wrapping div, so the info windows will resize automagically
	var wrapper = document.createElement("div");
	wrapper.className = 'dp-map-infowindow';
	wrapper.innerHTML = "$infoWindow";
	gMapinfoWindow{$this->elementSuffix}.setContent(wrapper);
	gMapinfoWindow{$this->elementSuffix}.open(gMap{$this->elementSuffix}, {$markerId});
});
INIT;
                }
                $options = $this -> getMarkerOptions($marker, $layer);

                $str = <<< INIT
var {$markerId} = new google.maps.Marker({$options});
{$markerId}.set('id', '{$markerId}');
gMapMarkers{$this->elementSuffix}.push({$markerId});
{$infoWindowCode}
INIT;

                array_push($allMarkers, $str);
            }
        }

        $this -> templateVariableContainer -> remove('layers');

        return implode("\n", $allMarkers);
    }

    public function renderDirectionsService($mapVariableName)
    {
        $content='';
        $templateVariableContainer = $this->renderingContext->getVariableProvider();
        $__googleMap=$templateVariableContainer->get('__googleMap');
      
        $directionsService = $__googleMap['directionsService'];
        foreach ($directionsService as $ds) {
            $content.="var directionsService = new google.maps.DirectionsService();
	    directionsDisplay = new google.maps.DirectionsRenderer();
	    directionsDisplay.setMap(".$mapVariableName.");
	    directionsService.route({
	      origin: ".json_encode($ds['origin']).",
		  destination: ".json_encode($ds['destination']).",
		  travelMode: ".json_encode($ds['travelMode']).",
		  }, function(result, status) {
		if (status == 'OK') {
		  directionsDisplay.setDirections(result);
		}
	      });\n";
        }
        return $content;
    }
    protected function getMapOptions()
    {
        $latlng = $this -> getLatLng();

        // hard defaults, cannot override
        $lines = array('center' => '_new google.maps.LatLng(' . $latlng['lat'] . ', ' . $latlng['lng'] . ')', 'mapTypeId' => '_google.maps.MapTypeId.' . strtoupper($this -> arguments['type']), 'zoom' => intval($this -> arguments['zoom']));
        //user params
        if ($this -> hasArgument('options')) {
            if (is_array($this -> arguments['options']) || strpos($this -> arguments['options'], 'Array')) {
                throw new \Exception('Bitte Map Options ohne {...} rundum angeben und geschwungene Klammern mit \\ escapen ( ... \\{ style: google.maps.MapTypeControlStyle.HORIZONTAL_BAR\\}, ... )!');
            }
            $lines['userOptions'] = rtrim(str_replace(array('\\{', '\\}'), array('{', '}'), $this -> arguments['options']), ',');
        }

        return $this -> objWrap($lines);
    }

    protected function getMarkerOptions($marker, $layer)
    {
        $lines = array();
        $lines['position'] = '_new google.maps.LatLng(' . $marker['position']['lat'] . ',' . $marker['position']['lng'] . ')';
        $lines['map'] = '_gMap' . $this -> elementSuffix;
        $title = str_replace(array("\r\n", "\n"), "\\n", $marker['title']);
        $title = str_replace('"', '\\"', $title);
        $lines['title'] = $title;

        // first options from layer
        $options = array_diff(array(trim($layer['options']), trim($marker['options'])), array(''));
        $options = implode(',', $options);
        if ($options) {
            $lines['userOptions'] = $options;
        }
        if ($marker['icon']) {
            $lines['icon'] = $marker['icon'];
        } elseif ($layer['icon']) {
            $lines['icon'] = $layer['icon'];
        }
        return $this -> objWrap($lines);
    }

    /**
     * get the position of the map
     * first, it checks latlng argument
     * then, lat and lng arguments
     *
     * @return array array('lat'=>latitude, 'lng'=>longitude)
     */
    protected function getLatLng()
    {
        $res = array();
        if ($this -> hasArgument('lat') && $this -> hasArgument('lng')) {
            // both argumets set
            $res = array('lat' => floatval($this -> arguments['lat']), 'lng' => floatval($this -> arguments['lng']));
        } elseif ($this -> hasArgument('latlng')) {
            $lat = $lng = null;
            //first priority: commaseparated string or array
            if (is_array($this -> arguments['latlng'])) {
                if (isset($this -> arguments['latlng']['lat'])) {
                    $lat = $this -> arguments['latlng']['lat'];
                    $lng = $this -> arguments['latlng']['lng'];
                } else {
                    $lat = $this -> arguments['latlng'][0];
                    $lng = $this -> arguments['latlng'][1];
                }
            } else {
                list($lat, $lng) = \TYPO3\CMS\Core\Utility\GeneralUtility::trimExplode(',', $this -> arguments['latlng'], true);
            }
            $res = array('lat' => floatval($lat), 'lng' => floatval($lng));
        }
        //validate
        /*
         * XXX: theoretically, this makes it impossible to set the center to (0,0)
         * but since there is only sea and we don't have customers from africa, we can simply ignore that.
         */
        if (!(count($res) == 2 && $res['lat'] && $res['lng'] && $res['lat'] >= -90.0 && $res['lat'] <= 90.0 && $res['lng'] >= -180.0 && $res['lng'] <= 180.0)) {
            //wrong value
            $res = array('lat' => 46.9127509564, 'lng' => 8.19580078125);
        }
        return $res;
    }

    /**
     * Creates the Object Wrap {}
     * like json_encode, BUT strings starting with _ will not be wrapped in ""
     * @param array $lines
     * @return string
     */
    protected function objWrap($lines)
    {
        $res = array();
        foreach ($lines as $name => &$value) {
            if ($name == 'userOptions') {
                $res[] = $value;
                continue;
            } elseif (is_numeric($value)) {
                // NOOP
            } elseif (is_string($value)) {
                //identifier, dont wrap in quotes
                if (strpos($value, '_') === 0) {
                    $value = ltrim($value, '_');
                } else {
                    $value = "\"{$value}\"";
                }
            } elseif (is_null($value)) {
                continue;
            } elseif (is_bool($value)) {
                $value = $value ? 'true' : 'false';
            }
            $res[] = $name . ':' . $value;
        }
        return "{" . implode(", ", $res) . "}";
    }
}
