<?php
namespace Internetgalerie\IgRender\ViewHelpers\Svg;

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

use TYPO3\CMS\Core\Imaging\Icon;
use TYPO3\CMS\Core\Imaging\IconFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Documentation\Domain\Model\DocumentTranslation;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractViewHelper;
use TYPO3Fluid\Fluid\Core\Rendering\RenderingContextInterface;
use TYPO3Fluid\Fluid\Core\ViewHelper\Traits\CompileWithRenderStatic;

/**
 * ViewHelper to display a svg file inline
 *
 * Example: <ig:svg.inline src="" />
 *
 * @internal
 */
class InlineViewHelper extends AbstractViewHelper
{
    use CompileWithRenderStatic;

    /**
     * As this ViewHelper renders HTML, the output must not be escaped.
     *
     * @var bool
     */
    protected $escapeOutput = false;

    /**
     * Initializes the arguments
     *
     * @throws \TYPO3Fluid\Fluid\Core\ViewHelper\Exception
     */
    public function initializeArguments()
    {
        parent::initializeArguments();
        $this->registerArgument('src', 'string', 'a path to a file, a combined FAL identifier or an uid (int). If $treatIdAsReference is set, the integer is considered the uid of the sys_file_reference record. If you already got a FAL object, consider using the $image parameter instead', false, '');
        $this->registerArgument('treatIdAsReference', 'bool', 'given src argument is a sys_file_reference record', false, false);
        $this->registerArgument('image', 'object', 'a FAL object');
        $this->registerArgument('class', 'string', 'css class attribute');
        $this->registerArgument('title', 'string', 'title attribute');
        $this->registerArgument('removeStyle', 'boolean', 'remove style tag');
        $this->registerArgument('appendClass', 'string', 'append given string to all inline classes', false, null);
    }

    /**
     * Statically renders all format download links.
     *
     * @param array $arguments
     * @param \Closure $renderChildrenClosure
     * @param RenderingContextInterface $renderingContext
     *
     * @return string
     */
    public static function renderStatic(array $arguments, \Closure $renderChildrenClosure, RenderingContextInterface $renderingContext)
    {

        $src = (string)$arguments['src'];
        if (($src === '' && $arguments['image'] === null) || ($src !== '' && $arguments['image'] !== null)) {
            throw new Exception('You must either specify a string src or a File object.', 1382284106);
        }
        $imageService = GeneralUtility::makeInstance(\TYPO3\CMS\Extbase\Service\ImageService::class);

        try {
            $image = $imageService->getImage($src, $arguments['image'], (bool)$arguments['treatIdAsReference']);
            $contents =  $image->getContents();
            // start with real svg image, no xml header
            $contents =  substr($contents, strpos($contents, '<svg '));
            // remove more or less all comments
            $contents =  trim(preg_replace('/<!--(.|\s)*?-->/', '', $contents));

            if($arguments['removeStyle']){
                $contents = preg_replace('/(<style[\w\W]+style>)/', '', $contents); // is possible to greedy
            }


            if ($arguments['appendClass'] !== null) {
                if ($arguments['appendClass'] === '') {
                    $appendClass = $arguments['appendClass'] === ''  ? md5($src) : $arguments['appendClass'];
                    $appendClass =  $src === '' ? 'image-' . $arguments['image']->getUid() :  md5($src);
                } else {
                    $appendClass = (string) $arguments['appendClass'];
                }


                //$contentsNoStyle =  preg_replace('/<style\b[^>]*>(.*?)<\/style>/isU', '', $contents);
                //$cleanSvg = \Pelago\Emogrifier\CssInliner::fromHtml($contents)->inlineCss()->render();

                $foundStyle =  preg_match_all('/<style\b[^>]*>(.*?)<\/style>/isU', $contents, $matches);
                if ($foundStyle) {
                    $inlineCss = implode("\n", $matches[1]);
                    $foundClassed = preg_match_all('/(^|\s|})\.[^ {,\t\r\n]*/is', $inlineCss, $classes);
                    if (!empty($classes[0])) {
                        $classesToReplace = array_map('trim', $classes[0]);
                        /*
                        usort($classesToReplace, function($a, $b) {
                            return strlen($b) - strlen($a);
                        });
                        */
                        foreach( $classesToReplace as $classSelector) {
                            $classSelector = trim($classSelector);
                            $class = substr($classSelector,1);
                            // replace all classes in style
                            foreach ([' ', ';', '{', "\t", "\n", "\r"] as $char) {
                                $contents = str_replace($classSelector. $char,   $classSelector . '-' . $appendClass . $char, $contents);
                            }
                            //replace all inline class
                            $contents = str_replace('class="' . $class . '"', 'class="' .  $class . '-' . $appendClass . '"', $contents);
                        }
                    }
                }
            }

            // add extra class to svg - maybe we should check if the class attribute is present in svg tag
            if($arguments['class'] || $arguments['title']) {
                $contents =  substr($contents,0,4)
                          . ($arguments['class'] ? ' class="' . $arguments['class']  . '"' : '')
                          . ($arguments['title'] ? ' title="' . $arguments['title']  . '"' : '')
                          . substr($contents,4);
            }
            return $contents;
        } catch (ResourceDoesNotExistException $e) {
            // thrown if file does not exist
            throw new Exception($e->getMessage(), 1509741911, $e);
        } catch (\UnexpectedValueException $e) {
            // thrown if a file has been replaced with a folder
            throw new Exception($e->getMessage(), 1509741912, $e);
        } catch (\RuntimeException $e) {
            // RuntimeException thrown if a file is outside of a storage
            throw new Exception($e->getMessage(), 1509741913, $e);
        } catch (\InvalidArgumentException $e) {
            // thrown if file storage does not exist
            throw new Exception($e->getMessage(), 1509741914, $e);
        }
        return '';
    }
}
