<?php

namespace Ig\IgResponsiveImages\Domain\Model;

use Ig\IgResponsiveImages\Domain\Model\PictureSource;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Imaging\ImageManipulation\Area;
use TYPO3\CMS\Core\Imaging\ImageManipulation\CropVariantCollection;
use TYPO3\CMS\Core\Resource\FileInterface;
use TYPO3\CMS\Core\Utility\ExtensionManagementUtility;
use TYPO3\CMS\Core\Resource\FileReference;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\PathUtility;
use TYPO3\CMS\Extbase\Service\ImageService;
use TYPO3Fluid\Fluid\Core\ViewHelper\TagBuilder;

class Picture
{
    /**
     * Image Service
     *
     * @var \TYPO3\CMS\Extbase\Service\ImageService
     */
    public $imageService;


    protected FileInterface $originalImage;
    protected CropVariantCollection $cropVariantCollection;
    protected FileInterface $fallbackImage;
    protected bool $lazy = false;
    protected string $lazyPlaceholderImage = '';
    protected bool $absoluteUri = false;
    //protected string $imgFileExtension = '';
    protected array $srcset = [];
    protected string $sizesQuery = '';
    protected string $cropVariant = '';
    protected Area $cropArea;
    protected string $imageAspect = '';
    protected array $sources = [];
    protected array $ignoreFileExtensions = ['svg','gif'];
    protected string $fileExtension = '';

    /**
     * id, style, class, title, alt
     **/
    protected array $imgAttributes = [];

    public function __construct()
    {
        $this->imageService = $this->getImageService();
    }
    
    public function setOriginalImage(FileInterface $originalImage):void
    {
        $this->originalImage = $originalImage;
    }
    public function getOriginalImage():FileInterface
    {
        return $this->originalImage;
    }
    public function getCropVariantCollection():CropVariantCollection
    {
        if(!isset($this->cropVariantCollection)) {
            $cropString = $this->originalImage instanceof FileReference ? $this->originalImage->getProperty('crop') : '';
            $this->cropVariantCollection = CropVariantCollection::create((string) $cropString);
        }
        return $this->cropVariantCollection;
    }


    public function setFallbackImage(FileInterface $fallbackImage):void
    {
        $this->fallbackImage = $fallbackImage;
    }
    public function getFallbackImage():FileInterface
    {
        return $this->fallbackImage;
    }
    public function getReferenceWidth():int
    {
        return 100;//$this->fallbackImage->getProperty('width') ? : 100;
    }
    
    public function setLazy(bool $lazy):void
    {
        $this->lazy = $lazy;
    }
    public function getLazy():bool
    {
        return $this->lazy;
    }
    public function getAttributeSrc()
    {
        return $this->lazy ? 'data-src' : 'src';
    }
    public function getAttributeSrcset()
    {
        return $this->lazy ? 'data-srcset' : 'srcset';
    }
    public function getAttributeSizes()
    {
        return $this->lazy ? 'data-sizes' : 'sizes';
    }
    public function setLazyPlaceholderImage(string $lazyPlaceholderImage = null)
    {
        $this->lazyPlaceholderImage = $lazyPlaceholderImage ?? '';
    }
    public function getLazyPlaceholderImage()
    {
        if(!$this->lazyPlaceholderImage) {
            // unstable - https://stackoverflow.com/questions/6018611/smallest-data-uri-image-possible-for-a-transparent-image
            //$this->lazyPlaceholderImage = 'data:image/gif;base64,R0lGODlhAQABAAD/ACwAAAAAAQABAAACADs=';
            $this->lazyPlaceholderImage = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';
            //$this->lazyPlaceholderImage = '/' . PathUtility::stripPathSitePrefix(ExtensionManagementUtility::extPath('ig_responsive_images') .'Resources/Public/Images/load.gif');
        }
        return $this->lazyPlaceholderImage;
    }

    public function setAbsoluteUri(bool $absoluteUri):void
    {
        $this->absoluteUri = $absoluteUri;
    }
    public function getAbsoluteUri():bool
    {
        return $this->absoluteUri;
    }

    
    public function setImageAspect(string $imageAspect):void
    {
        $this->imageAspect = $imageAspect;
    }
    public function getImageAspect():string
    {
        return $this->imageAspect;
    }

    public function setSrcset(array $srcset):void
    {
        $this->srcset = $srcset;
    }
    public function setSrcsetString(string $srcsetString):void
    {
        $this->srcset = GeneralUtility::trimExplode(',', $srcsetString);
    }
    public function getSrcSet():array
    {
        return $this->srcset;
    }
    public function getSrcSetString():array
    {
        return implode(',', $this->srcset);
    }

    
    public function setSizesQuery(string $sizesQuery):void
    {
        $this->sizesQuery = $sizesQuery;
    }
    public function getSizesQuery():string
    {
        return $this->sizesQuery;
    }
    
    public function setSources(array $sources):void
    {
        $this->sources = $sources;
    }
    public function addSource(PictureSource $pictureSource):void
    {
        $this->sources[] = $pictureSource;
    }
    public function getSources():array
    {
        return $this->sources;
    }


    public function setImgAttributes(array $imgAttributes):void
    {
        $this->imgAttributes = $imgAttributes;
    }
    public function getImgAttributes():array
    {
        return $this->imgAttributes;
    }


    public function setCropVariant(string $cropVariant):void
    {
        $this->cropVariant = $cropVariant;
    }
    public function getCropVariant():string
    {
        return $this->cropVariant;
    }

    public function setCropArea(Area $cropArea):void
    {
        $this->cropArea = $cropArea;
    }
    public function getCropArea():Area
    {
        return $this->cropArea;
    }
    public function setFileExtension(string $fileExtension):void
    {
        $this->fileExtension = $fileExtension;
    }
    public function getFileExtension():string
    {
        return $this->fileExtension;
    }
    
    public function render()
    {
        $this->generateImages();
        $tagContent = $this->renderSources();
        $tagContent .= $this->renderImg();
        $pictureTag = new TagBuilder('picture');
        $pictureTag->setContent(
            $tagContent
        );
        return $pictureTag->render();
    }
    public function renderImgTag()
    {
        $this->generateImages();
        return $this->renderImg();
    }
    public function renderSrcset():string
    {
        $this->generateImages();
        $srcsetImages = $this->img['srcsetImages'];
        return $srcsetImages ? $this->generateSrcsetAttribute($srcsetImages) : $this->img['src'];
    }
    public function renderImg()
    {
        $tag = new TagBuilder('img');
        if(!empty($this->img['srcsetImages'])) {
            $srcsetImages = $this->img['srcsetImages'];
            $tag->addAttribute($this->getAttributeSrc(), reset($srcsetImages));
            $tag->addAttribute($this->getAttributeSizes(), sprintf($this->sizesQuery, $this->getReferenceWidth()));
            $tag->addAttribute($this->getAttributeSrcset(), $this->generateSrcsetAttribute($srcsetImages));
        }
        if($this->lazy) {
            //$tag->addAttribute('class', 'lazy'); // is set in viewhelper 
            $tag->addAttribute('src', $this->getLazyPlaceholderImage());
            if(!empty($this->img['src'])) {
                $tag->addAttribute($this->getAttributeSrc(), $this->img['src']);
            }
        } else {
            if(!empty($this->img['src'])) {
                $tag->addAttribute('src', $this->img['src']);
            }
        }
        // Meta Data
        $focusArea=null; // @todo remove?
        $this->addMetadataToImageTag($tag, $this->originalImage, $focusArea);
        if(!empty($this->imgAttributes)) {
            foreach ($this->imgAttributes as $name=>$value) {
                if($value!==null) {
                    $tag->addAttribute($name, $value);
                }
            }
        }
        return $tag->render();

    }
    public function renderSources()
    {
        $tagContent = '';
        $attributeSizes = $this->getAttributeSizes();
        $attributeSrcset = $this->getAttributeSrcset();
        foreach($this->sources as $source) {
            $srcset = $source->getSrcsetImages();
            if(empty($srcset)) {
                continue;
            }
            $tag = new TagBuilder('source');
            // fill images
            $tag->addAttribute($attributeSizes, sprintf($this->sizesQuery, $this->getReferenceWidth()));
            $tag->addAttribute($attributeSrcset, $this->generateSrcsetAttribute($srcset));
            if($this->lazy) {
                $tag->addAttribute('srcset', $this->getLazyPlaceholderImage());
                //$tag->addAttribute('sizes', '');
              }

            if($source->getCropVariant()) {
                $tag->addAttribute('data-crop-variant', $source->getCropVariant());
            }
            if($source->getMedia()) {
                $tag->addAttribute('media', $source->getMedia());
            }
            if($source->getType()) {
                $tag->addAttribute('type', $source->getType());
            }
            $tagContent .= $tag->render();
        }
        return $tagContent;
    }

    public function generateImages( ):void {
        $cropAreaImageAspect =  $this->imageAspect>0 ? $this->getCropAreaWithImageAspect() : null;

        if($cropAreaImageAspect === null) {
            $cropArea = $this->getCropVariantCollection()->getCropArea($this->cropVariant);
            $cropArea = $cropArea ?? Area::createEmpty();
        } else {
            $cropArea = $cropAreaImageAspect;
        }
        //$cropArea = $this->cropArea ?: Area::createEmpty();
        if(in_array($this->originalImage->getExtension(),$this->ignoreFileExtensions)) {
            $this->img= [
                'src' =>  $this->imageService->getImageUri($this->originalImage, $this->absoluteUri)
            ];
            return;
        }
        $this->img = [
            'srcsetImages' => $this->generateSrcsetImages($cropArea, $this->fileExtension),
        ];
        
        if(!empty($this->sources)) {
            foreach($this->sources as $source) {
                if($cropAreaImageAspect===null) {
                    $cropArea = $this->getCropVariantCollection()->getCropArea($source->getCropVariant() ? : $this->cropVariant);
                    $cropArea = $cropArea ?? Area::createEmpty();
                } else {
                    $cropArea = $cropAreaImageAspect;
                }
                $srcsets = $this->generateSrcsetImages($cropArea, $source->getFileExtension(), $source->getQuality());

                foreach($srcsets as $widthDescriptor=>$srcset) {
                    $source->addSrcsetImage($srcset,$widthDescriptor);
                }
            }
        }
    }

    /**
     * Renders different image sizes for use in a srcset attribute
     *
     * Input:
     *   1: $srcset = [200, 400]
     *   2: $srcset = ['200w', '400w']
     *   3: $srcset = ['1x', '2x']
     *   4: $srcset = '200, 400'
     *
     * Output:
     *   1+2+4: ['200w' => 'path/to/image@200w.jpg', '400w' => 'path/to/image@200w.jpg']
     *   3: ['1x' => 'path/to/image@1x.jpg', '2x' => 'path/to/image@2x.jpg']
     * 
     * @param Area $cropArea
     * @param string $fileExt
     *
     * @return array
     */
    public function generateSrcsetImages(Area $cropArea, string $fileExt, int $quality = 0):array {
        $image = $this->originalImage;
        $srcsetImages = [];
        
        foreach ($this->srcset as $widthDescriptor) {
            // Determine image width
            switch (substr($widthDescriptor, -1)) {
            case 'x':
                $candidateWidth = (int) ($this->getReferenceWidth() * (float) substr($widthDescriptor, 0, -1));
                break;

            case 'w':
                $candidateWidth = (int) substr($widthDescriptor, 0, -1);
                break;

            default:
                $candidateWidth = (int) $widthDescriptor;
                $widthDescriptor = $candidateWidth . 'w';
            }

            if($cropArea->isEmpty()) {
                $cropAreaAbsoluteBasedOnFile=null;
            } else {
                // TYPO3 9.5 - exif beachten - ist in Version 11 gefixt
                if (\TYPO3\CMS\Core\Utility\VersionNumberUtility::convertVersionNumberToInteger(\TYPO3\CMS\Core\Utility\VersionNumberUtility::getNumericTypo3Version()) < 11000000 && function_exists('exif_read_data')) {
                    $exif = @exif_read_data(Environment::getPublicPath() . '/' .$image->getPublicUrl());
                    if(isset($exif['Orientation']) && $exif['Orientation'] >= 5 && $exif['Orientation'] <= 8) {
                        $cropArray=$cropArea->asArray();
                        // crop area drehen, falls exif bild drehen wird (bugfix fuer TYPO3 9.5)
                        $cropAreaAbsoluteBasedOnFile = new \TYPO3\CMS\Core\Imaging\ImageManipulation\Area($cropArray['y']*$image->getProperty('height'), $cropArray['x']*$image->getProperty('width'), $cropArray['height']*$image->getProperty('height'), $cropArray['width']*$image->getProperty('width'));
                    } else {
                        $cropAreaAbsoluteBasedOnFile= $cropArea->makeAbsoluteBasedOnFile($image);
                    }
                } else {
                    $cropAreaAbsoluteBasedOnFile= $cropArea->makeAbsoluteBasedOnFile($image);
                }
                // Limit width to maxium with of crop area (bug? crop are not upscaled, even if processor_allowUpscaling is set to true)
                if ($candidateWidth > $cropAreaAbsoluteBasedOnFile->getWidth()) {
                    $candidateWidth = $cropAreaAbsoluteBasedOnFile->getWidth();
                }
            }
            // Generate image
            $processingInstructions = [
                'width' => $candidateWidth,
                'height' => 0,
                'maxWidth' => $candidateWidth,
                'crop' => $cropAreaAbsoluteBasedOnFile,
            ];
            /*
            // @todo rearange functions should always a TYPO3\CMS\Core\Resource\File and no TYPO3\CMS\Core\Resource\FileReference
            // @todo change it some werde upstream (will be forced in TYPO3 11 to have proper method signature)
            if(is_callable([$image, 'getOriginalFile'])) {
            // Get the original file from the file reference
            $image = $image->getOriginalFile();
            }
        
            $processedImage = $image->process(\TYPO3\CMS\Core\Resource\ProcessedFile::CONTEXT_IMAGECROPSCALEMASK, $processingInstructions);
            // we don't need setCompatibilityValues should speed up by 25%
            //$processedImage =$this->imageService->setCompatibilityValues($processedFile);
            */
            // img
            if($fileExt) {
                $processingInstructions['fileExtension'] = $fileExt;
            }

            if ($quality > 0) {
                $backupJpgQuality = $GLOBALS['TYPO3_CONF_VARS']['GFX']['jpg_quality'];
                $backupWebpQuality = $GLOBALS['TYPO3_CONF_VARS']['GFX']['webp_quality'] ?? null;
                $backupAvifQuality = $GLOBALS['TYPO3_CONF_VARS']['GFX']['avif_quality'] ?? null;
                $GLOBALS['TYPO3_CONF_VARS']['GFX']['jpg_quality'] = $quality;
                $processingInstructions['quality'] = $quality;
                $processedImage = $this->imageService->applyProcessingInstructions($image, $processingInstructions);
                $GLOBALS['TYPO3_CONF_VARS']['GFX']['jpg_quality'] = $backupJpgQuality;
                if ($backupWebpQuality !== null) {
                    $GLOBALS['TYPO3_CONF_VARS']['GFX']['webp_quality'] = $backupWebpQuality;
                }
                if ($backupAvifQuality !== null) {
                    $GLOBALS['TYPO3_CONF_VARS']['GFX']['avif_quality'] = $backupAvifQuality;
                }
            } else {
                $processedImage = $this->imageService->applyProcessingInstructions($image, $processingInstructions);
            }
            $srcsetImages[$widthDescriptor] = $this->imageService->getImageUri($processedImage, $this->absoluteUri);
            /*
            var_dump($processedImage->getProperty('checksum'));
            
            //if(($this->originalImage->getUid()==25093 && $fileExt=='webp')) {//
            if ( $processedImage->getProperty('checksum')== 'c87743e91c' ||) {
                    echo(get_class($this->originalImage ));
                    var_dump($cropArea);
                    var_dump($processingInstructions);
                    var_dump($fileExt);
                    var_dump($processedImage);
                    exit(0);
                }
            */
            
            //$this->img['srcsetImages'][$widthDescriptor] = $this->imageService->getImageUri($processedImage, $this->absoluteUri);
            /*  
            if(!empty($this->sources)) {
                foreach($this->sources as $source) {
                    $processingInstructions['fileExtension']=$source->getFileExtension();
                    $processedImage = $this->imageService->applyProcessingInstructions($image, $processingInstructions);
                    $source->addSrcsetImage($this->imageService->getImageUri($processedImage, $this->absoluteUri),$widthDescriptor);
                }
            }
            */
        }
        return $srcsetImages;
    }

    /**
     * Generates the content for a srcset attribute from an array of image urls
     *
     * Input:
     * [
     *   '200w' => 'path/to/image@200w.jpg',
     *   '400w' => 'path/to/image@400w.jpg'
     * ]
     *
     * Output:
     * 'path/to/image@200w.jpg 200w, path/to/image@400w.jpg 400w'
     *
     * @param array $srcsetImages
     *
     * @return string
     */
    public function generateSrcsetAttribute(array $srcsetImages): string
    {
        $srcsetString = [];
        foreach ($srcsetImages as $widthDescriptor => $imageCandidate) {
            $srcsetString[] = $imageCandidate . ' ' . $widthDescriptor;
        }
        return implode(', ', $srcsetString);
    }

    /**
     * Adds metadata to image tag
     *
     * @param TagBuilder    $tag
     * @param FileInterface $originalImage
     * @param FileInterface $fallbackImage
     * @param Area          $focusArea
     *
     * @return void
     */
    public function addMetadataToImageTag(
        TagBuilder $tag,
        FileInterface $originalImage,
        Area $focusArea = null
    ) {
        $focusArea = $focusArea ?: Area::createEmpty();

        // Add focus area to image tag
        if(!$tag->hasAttribute('data-focus-area') && !$focusArea->isEmpty()) {
            $tag->addAttribute('data-focus-area', $focusArea->makeAbsoluteBasedOnFile($originalImage));
        }

        // The alt-attribute is mandatory to have valid html-code, therefore add it even if it is empty
        $alt = $originalImage->getProperty('alternative');
        if(!$tag->getAttribute('alt')) {
            $tag->addAttribute('alt', $alt);
        }
        $title = $originalImage->getProperty('title');
        if(!$tag->getAttribute('title') && $title) {
            $tag->addAttribute('title', $title);
        }
    }






    public function getCropAreaWithImageAspect()
    {
        // Get crop variants
        $cropString = $this->originalImage instanceof FileReference ? $this->originalImage->getProperty('crop') : '';
        //$cropVariantCollection = CropVariantCollection::create((string) $cropString);

        // DA Start
        // TODO: Hässlich !!!!!!!!!!!!!!!!!!!!!!
        // Nur falls nicht in File im Editor etwas eingestellt ist
        if ($cropString=='' || $cropString=='{"default":{"cropArea":{"x":0,"y":0,"width":1,"height":1},"selectedRatio":"NaN","focusArea":null}}') {
            $cropArea = Area::createEmpty();
            $focusArea = Area::createEmpty();
            //echo('cropString='.$cropString ."\n");
            // @todo is this correct? removed FileReference check in next line 7.12.2021
            if ($this->imageAspect>0) {// && $this->originalImage instanceof FileReference
                $origWidth=floatval($this->originalImage->getProperty('width'));
                $origHeight=floatval($this->originalImage->getProperty('height'));
                if ($origWidth>0 &&  $origHeight >0) {

                    // TYPO3 9.5 - exif beachten - ist in Version 11 gefixt
                    $imageOrientationRotate=false;
                    if (\TYPO3\CMS\Core\Utility\VersionNumberUtility::convertVersionNumberToInteger(\TYPO3\CMS\Core\Utility\VersionNumberUtility::getNumericTypo3Version()) < 11000000 && function_exists('exif_read_data')) {
                        $exif = @exif_read_data(Environment::getPublicPath() . '/' .$this->originalImage->getPublicUrl());
                        if (isset($exif['Orientation']) && $exif['Orientation'] >= 5 && $exif['Orientation'] <= 8) {
                            /*
                              $temp=$origWidth;
                              $origWidth=$origHeight;
                              $origHeight=$temp;
                            */
                            $imageOrientationRotate=true;
                        }
                    }
                    if ($imageOrientationRotate) {
                        // crop area aus ratio geht in 9.5 nicht -> ist verdreht -> falls exif bild drehen wird (bugfix fuer TYPO3 9.5) selbst area berechnen (verdreht)
                        // imageAspect drehen
                        $expectedRatio=floatval(1/$this->imageAspect);
                        //$expectedRatio=floatval(1/2);
                        if ($origHeight * $expectedRatio > $origWidth) {
                            //$newHeight=1/$expectedRatio;
                            //$newStart = ($expectedRatio-$newHeight/2)/2;

                            // Logisch ist:
                            $newStart =(1-$origWidth / $origHeight * $this->imageAspect)/2;
                            $newHeight = 1-2*$newStart;

                            //if($this->originalImage->getOriginalFile()->getUid()==709) {var_dump($expectedRatio,$origHeight,$origWidth);die('a='.$this->originalImage->getPublicUrl());}

                            $cropArea= new \TYPO3\CMS\Core\Imaging\ImageManipulation\Area($newStart, 0, 1, $newHeight);
                            //if($this->originalImage->getOriginalFile()->getUid()==709) {echo ($origWidth .'x' . $origHeight .'='.($origWidth / $origHeight).'<br />');var_dump($cropArea,$origHeight,$origWidth);die('a='.$this->originalImage->getPublicUrl());}
                            //$cropArea= new \TYPO3\CMS\Core\Imaging\ImageManipulation\Area(0, $newStart, 0, 1, $newHeight);
                             //$cropArea= new \TYPO3\CMS\Core\Imaging\ImageManipulation\Area($newStart, 0, 0, $newHeight, 1);
                            //$cropArea= new \TYPO3\CMS\Core\Imaging\ImageManipulation\Area(($expectedRatio-$newHeight/2)/2, 0, $newHeight, 1);
                        } else {
                            $newWidth= 1*$expectedRatio;
                            $newStart = (1/$expectedRatio-$newWidth/2)/2;
                            //die('R='.$expectedRatio .',' . $origHeight . ',' . $origWidth .', newWidth='.$newWidth . ', Area=0, '. ((1/$expectedRatio-$newWidth/2)/2) . ', 1, ' .   $newWidth);
                            $cropArea= new \TYPO3\CMS\Core\Imaging\ImageManipulation\Area(0, $newStart, 1, $newWidth);
                        }

                        //\TYPO3\CMS\Extbase\Utility\DebuggerUtility::var_dump( $cropArea );
                    } else {

                        // Ratio muss relativ zu Bildgrösse angegeben werden
                        $ratio = new \TYPO3\CMS\Core\Imaging\ImageManipulation\Ratio('tempIdIG', 'imageAspectElementTitle', $origHeight/$origWidth / $this->imageAspect);

                        /*
                        //var_dump($this->originalImage->getProperties());
                        // var_dump($this->originalImage->getReferenceProperties());
                        echo( 'width='.$width .', height='. $height . "\n");
                        echo( 'width='.$origWidth .', height='. $origHeight . "\n");
                        echo('imageAspect='.$this->arguments['imageAspect'] ." ergibt=".(1/$this->arguments['imageAspect'])."\n");
                        echo('ratio='.$ratio->getRatioValue() .'[free='.$ratio->isFree().' ]');
                        */
        
                        //$cropArea = new Area(0,0.5,1,0.206);

                        // Relativ
                        $cropArea = $cropArea->applyRatioRestriction($ratio);
                        //\TYPO3\CMS\Extbase\Utility\DebuggerUtility::var_dump( $cropArea );
                    }
                }
            }
        } else {
            // Crop von Element (d.h. Editor)
            $cropVariant = $this->cropVariant ?: 'default';
            $cropArea = $this->getCropVariantCollection()->getCropArea($cropVariant);
            //  $focusArea = $this->getCropVariantCollection()->getFocusArea($cropVariant);
        }

        return $cropArea;
        // DA End
        // Generate fallback image
        //$fallbackImage = $this->generateFallbackImage($this->originalImage, $width, $cropArea);


    }






    /**
     * Return an instance of ImageService
     *
     * @return ImageService
     */
    protected function getImageService()
    {
        return GeneralUtility::makeInstance(ImageService::class);
    }


    
}
