<?php

namespace Ig\IgResponsiveImages\ViewHelpers;

use Ig\IgResponsiveImages\Utility\ResponsiveImagesUtility;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Imaging\ImageManipulation\Area;
use TYPO3\CMS\Core\Imaging\ImageManipulation\CropVariantCollection;
use TYPO3\CMS\Core\Resource\Exception\ResourceDoesNotExistException;
use TYPO3\CMS\Core\Resource\FileInterface;
use TYPO3\CMS\Core\Resource\FileReference;
use TYPO3\CMS\Core\Resource\Rendering\RendererRegistry;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Service\ImageService;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper;
use TYPO3Fluid\Fluid\Core\ViewHelper\Exception;


class ImageViewHelper extends AbstractTagBasedViewHelper
{
    /**
     * @var string
     */
    protected $tagName = 'img';

    protected ImageService $imageService;

    public function __construct()
    {
        parent::__construct();
        $this->imageService = GeneralUtility::makeInstance(ImageService::class);
    }

    /**
     * Initialize arguments.
     */
    public function initializeArguments(): void
    {
        parent::initializeArguments();
        $this->registerUniversalTagAttributes();
        $this->registerTagAttribute('alt', 'string', 'Specifies an alternate text for an image', false);
        $this->registerTagAttribute('ismap', 'string', 'Specifies an image as a server-side image-map. Rarely used. Look at usemap instead', false);
        $this->registerTagAttribute('longdesc', 'string', 'Specifies the URL to a document that contains a long description of an image', false);
        $this->registerTagAttribute('usemap', 'string', 'Specifies an image as a client-side image-map', false);
        $this->registerTagAttribute('loading', 'string', 'Native lazy-loading for images property. Can be "lazy", "eager" or "auto"', false);
        $this->registerTagAttribute('decoding', 'string', 'Provides an image decoding hint to the browser. Can be "sync", "async" or "auto"', false);

        $this->registerArgument('src', 'string', 'a path to a file, a combined FAL identifier or an uid (int). If $treatIdAsReference is set, the integer is considered the uid of the sys_file_reference record. If you already got a FAL object, consider using the $image parameter instead', false, '');
        $this->registerArgument('treatIdAsReference', 'bool', 'given src argument is a sys_file_reference record', false, false);
        $this->registerArgument('image', 'object', 'a FAL object (\\TYPO3\\CMS\\Core\\Resource\\File or \\TYPO3\\CMS\\Core\\Resource\\FileReference)');
        $this->registerArgument('crop', 'string|bool', 'overrule cropping of image (setting to FALSE disables the cropping set in FileReference)');
        $this->registerArgument('cropVariant', 'string', 'select a cropping variant, in case multiple croppings have been specified or stored in FileReference', false, 'default');
        $this->registerArgument('fileExtension', 'string', 'Custom file extension to use');

        $this->registerArgument('width', 'string', 'width of the image. This can be a numeric value representing the fixed width of the image in pixels. But you can also perform simple calculations by adding "m" or "c" to the value. See imgResource.width for possible options.');
        $this->registerArgument('height', 'string', 'height of the image. This can be a numeric value representing the fixed height of the image in pixels. But you can also perform simple calculations by adding "m" or "c" to the value. See imgResource.width for possible options.');
        $this->registerArgument('minWidth', 'int', 'minimum width of the image');
        $this->registerArgument('minHeight', 'int', 'minimum height of the image');
        $this->registerArgument('maxWidth', 'int', 'maximum width of the image');
        $this->registerArgument('maxHeight', 'int', 'maximum height of the image');
        $this->registerArgument('absolute', 'bool', 'Force absolute URL', false, false);


        $this->registerArgument('srcset', 'mixed', 'Image sizes that should be rendered.', true);
        $this->registerArgument('sizes', 'string', 'Sizes query for responsive image.', false, '(min-width: %1$dpx) %1$dpx, 100vw');
        $this->registerArgument('imageAspect', 'string', 'IG: Aspect Ratio', false, null);
        $this->registerArgument('lazy', 'boolean', 'Lazy Loading Images', false, false);
        $this->registerArgument('webp', 'boolean', 'Create WEBP images', false, null);
        $this->registerArgument('sourcesName', 'string', 'Name of plugin.tx_igresponsiveimages.settings.sources.NAME', false, 'default');
        $this->registerArgument('sources', 'array', 'source definitions for picture element', false);
        $this->registerArgument('lazyClass', 'string', 'class if image is Lazy Loading', false, 'lazy');
        // @todo addd ignoreFileExtensions add pass to Picture
        // @todo placeholder image in lazyloading with color
        // die
        $this->registerArgument('breakpoints', 'array', 'Image breakpoints from responsive design.', false);
        $this->registerArgument('breakpointsName', 'string', 'Name of plugin.tx_igresponsiveimages.settings.breakpoints.NAME', false, null);
    }

    public function initialize()
    {
        if ($this->arguments['lazy']) {
            $this->arguments['class'].=' ' . $this->arguments['lazyClass'];
        }
        parent::initialize();
    }

    /**
     * Resizes a given image (if required) and renders the respective img tag
     *
     * @see https://docs.typo3.org/typo3cms/TyposcriptReference/ContentObjects/Image/
     *
     * @throws Exception
     * @return string Rendered tag
     */
    public function render(): string
    {
        $src = (string)$this->arguments['src'];
        if (($src === '' && $this->arguments['image'] === null) || ($src !== '' && $this->arguments['image'] !== null)) {
            throw new Exception('You must either specify a string src or a File object.', 1382284106);
        }

        if ((string)$this->arguments['fileExtension'] && !GeneralUtility::inList($GLOBALS['TYPO3_CONF_VARS']['GFX']['imagefile_ext'], (string)$this->arguments['fileExtension'])) {
            throw new Exception(
                'The extension ' . $this->arguments['fileExtension'] . ' is not specified in $GLOBALS[\'TYPO3_CONF_VARS\'][\'GFX\'][\'imagefile_ext\']'
                . ' as a valid image file extension and can not be processed.',
                1618989190
            );
        }

        // A URL was given as src, this is kept as is, and we can only scale
       try {
            $image = $this->imageService->getImage($src, $this->arguments['image'], (bool)$this->arguments['treatIdAsReference']);
        } catch (ResourceDoesNotExistException $e) {
            // thrown if file does not exist
            throw new Exception($e->getMessage(), 1509741911, $e);
        } catch (\UnexpectedValueException $e) {
            // thrown if a file has been replaced with a folder
            throw new Exception($e->getMessage(), 1509741912, $e);
            } catch (\RuntimeException $e) {
            // RuntimeException thrown if a file is outside of a storage
            throw new Exception($e->getMessage(), 1509741913, $e);
        } catch (\InvalidArgumentException $e) {
            // thrown if file storage does not exist
            throw new Exception($e->getMessage(), 1509741914, $e);
        }
       $additionalConfig = (array)($this->arguments['additionalConfig'] ?? []);
        $width = $this->arguments['width'];
        $height = $this->arguments['height'];

        // get Resource Object (non ExtBase version)
        if (is_callable([$image, 'getOriginalResource'])) {
            // We have a domain model, so we need to fetch the FAL resource object from there
            $image = $image->getOriginalResource();
        }

        if (!$image instanceof FileInterface) {
            throw new \UnexpectedValueException('Supplied file object type ' . get_class($image) . ' must be FileInterface.', 1454252193);
        }

        return $this->renderImageSrcset($image, $width, $height);
        /*
        $imageRenderer = RendererRegistry::getInstance()->getRenderer($image);

        // Fallback to image when no renderer is found
        if ($imageRenderer === null) {
            return $this->renderImageSrcset($image, $width, $height);
        }
        $additionalConfig = array_merge_recursive($this->arguments, $additionalConfig);
        return $imageRenderer->render($image, $width, $height, $additionalConfig);
        */
    }
  

    /**
     * Render img tag with srcset/sizes attributes
     *
     * @param FileInterface $image
     * @param string        $width
     * @param string        $height
     *
     * @return string                 Rendered img tag
     */
    protected function renderImageSrcset(FileInterface $image, $width, $height)
    {
        $responsiveImageUtility=GeneralUtility::makeInstance(ResponsiveImagesUtility::class);
        /*
        if ($this->arguments['imageAspect']>0) {
            $imageAspect = $this->arguments['imageAspect'];
        }
        */
        //        echo('a='.$imageAspect .'='. $image->hasProperty('image_aspect'));
        // Generate image tag
        
        // @todo ($this->tag als img tag verwenden
        //var_dump($this->tag);exit(0);

        return $this->getResponsiveImagesUtility()->createImageHtmlWithSrcset(
            $image,
            $this->arguments['srcset'],
            $this->arguments['sizes'],
            $this->arguments['imageAspect'],
            $this->arguments['crop'],
            false,
            $this->arguments['lazy'],
            [
                'class' => $this->arguments['class'],
                'style' => $this->arguments['style'],
                'title' => $this->arguments['title'],
                'alt' => $this->arguments['alt'],
                'lang' => $this->arguments['lang'],
                'id' => $this->arguments['id'],
                'data' => $this->arguments['data'],
            ],
            $this->arguments['webp'],
            $this->arguments['sourcesName']
        );

        //      return $this->tag->render();
    }

    
    /**
     * Generates a fallback image for picture and srcset markup
     *
     * @param FileInterface $image
     * @param string        $width
     * @param Area          $cropArea
     *
     * @return FileInterface
     */
    protected function generateFallbackImage(FileInterface $image, $width, Area $cropArea)
    {
        if ($this->arguments['width'] ||!$cropArea->isEmpty()) {
            $processingInstructions = [
                'width' => $width,
                'height' => 0,
                'crop' => $cropArea->isEmpty() ? null : $cropArea->makeAbsoluteBasedOnFile($image),
            ];
            $imageService = $this->getImageService();
            return $imageService->applyProcessingInstructions($image, $processingInstructions);
        } else {
            return $image;
        }
    }

    /**
     * Returns an instance of the responsive images utility
     * This fixes an issue with DI after clearing the cache
     *
     * @return ResponsiveImagesUtility
     */
    protected function getResponsiveImagesUtility()
    {
        return GeneralUtility::makeInstance(ResponsiveImagesUtility::class);
    }

    /**
     * Return an instance of ImageService
     *
     * @return ImageService
     */
    protected function getImageService()
    {
        return GeneralUtility::makeInstance(ImageService::class);
    }
}
