<?php

namespace Ig\IgResponsiveImages\ViewHelpers;

use Ig\IgResponsiveImages\Utility\ResponsiveImagesUtility;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Imaging\ImageManipulation\CropVariantCollection;
use TYPO3\CMS\Core\Imaging\ImageManipulation\Area;
use TYPO3\CMS\Core\Resource\FileInterface;
use TYPO3\CMS\Core\Resource\FileReference;
use TYPO3\CMS\Core\Resource\Rendering\RendererRegistry;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Service\ImageService;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper;
use TYPO3Fluid\Fluid\Core\ViewHelper\Exception;

class MediaViewHelper extends AbstractTagBasedViewHelper
{
    /**
     * @var string
     */
    protected $tagName = 'img';
    
    /**
     * Initialize arguments.
     */
    public function initializeArguments(): void
    {
        parent::initializeArguments();
        $this->registerUniversalTagAttributes();
        $this->registerTagAttribute('alt', 'string', 'Specifies an alternate text for an image', false);
        $this->registerArgument('file', 'object', 'File', true);
        $this->registerArgument('additionalConfig', 'array', 'This array can hold additional configuration that is passed though to the Renderer object', false, []);
        $this->registerArgument('width', 'string', 'This can be a numeric value representing the fixed width of in pixels. But you can also perform simple calculations by adding "m" or "c" to the value. See imgResource.width for possible options.');
        $this->registerArgument('height', 'string', 'This can be a numeric value representing the fixed height in pixels. But you can also perform simple calculations by adding "m" or "c" to the value. See imgResource.width for possible options.');
        $this->registerArgument('cropVariant', 'string', 'select a cropping variant, in case multiple croppings have been specified or stored in FileReference', false, 'default');
        $this->registerArgument('fileExtension', 'string', 'Custom file extension to use for images');
        $this->registerArgument('loading', 'string', 'Native lazy-loading for images property. Can be "lazy", "eager" or "auto". Used on image files only.');
        $this->registerArgument('decoding', 'string', 'Provides an image decoding hint to the browser. Can be "sync", "async" or "auto"', false);
        $this->registerArgument('srcset', 'mixed', 'Image sizes that should be rendered.', false);
        $this->registerArgument('sizes', 'string', 'Sizes query for responsive image.', false, '(min-width: %1$dpx) %1$dpx, 100vw');
        $this->registerArgument('imageAspect', 'string', 'IG: Aspect Ratio', false, null);
        $this->registerArgument('lazy', 'boolean', 'Lazy Loading Images', false, false);
        $this->registerArgument('webp', 'boolean', 'Create WEBP images', false, null);
        $this->registerArgument('sourcesName', 'string', 'Name of plugin.tx_igresponsiveimages.settings.sources.NAME', false, 'default');
        $this->registerArgument('sources', 'array', 'source definitions for picture element', false);
        $this->registerArgument('lazyClass', 'string', 'class if image is Lazy Loading', false, 'lazy');
        // @todo addd ignoreFileExtensions add pass to Picture
        // @todo placeholder image in lazyloading with color
        // die
        $this->registerArgument('breakpoints', 'array', 'Image breakpoints from responsive design.', false);
        $this->registerArgument('breakpointsName', 'string', 'Name of plugin.tx_igresponsiveimages.settings.breakpoints.NAME', false, null);
    }

    public function initialize()
    {
        if ($this->arguments['lazy']) {
            $this->arguments['class'].=' ' . $this->arguments['lazyClass'];
        }
        parent::initialize();
    }
  

    /**
     * Render a given media file. (no changes)
     *
     * @throws \UnexpectedValueException
     * @throws Exception
     */
    public function render(): string
    {
        $file = $this->arguments['file'];
        $additionalConfig = (array)$this->arguments['additionalConfig'];
        $width = $this->arguments['width'];
        $height = $this->arguments['height'];

        // get Resource Object (non ExtBase version)
        if (is_callable([$file, 'getOriginalResource'])) {
            // We have a domain model, so we need to fetch the FAL resource object from there
            $file = $file->getOriginalResource();
        }

        if (!$file instanceof FileInterface) {
            throw new \UnexpectedValueException('Supplied file object type ' . get_class($file) . ' must be FileInterface.', 1454252193);
        }

        if ((string)$this->arguments['fileExtension'] && !GeneralUtility::inList($GLOBALS['TYPO3_CONF_VARS']['GFX']['imagefile_ext'], (string)$this->arguments['fileExtension'])) {
            throw new Exception(
                'The extension ' . $this->arguments['fileExtension'] . ' is not specified in $GLOBALS[\'TYPO3_CONF_VARS\'][\'GFX\'][\'imagefile_ext\']'
                . ' as a valid image file extension and can not be processed.',
                1619030957
            );
        }

        $fileRenderer = GeneralUtility::makeInstance(RendererRegistry::class)->getRenderer($file);

        // Fallback to image when no renderer is found
        if ($fileRenderer === null) {
            return $this->renderImage($file, $width, $height, $this->arguments['fileExtension'] ?? null);
        }
        $additionalConfig = array_merge_recursive($this->arguments, $additionalConfig);
        return $fileRenderer->render($file, $width, $height, $additionalConfig);
    }


    /**
     * Render img tag
     *
     * @param  FileInterface $image
     * @param  string        $width
     * @param  string        $height
     * @param  string|null   $fileExtension
     * @return string Rendered img tag
     */
    protected function renderImage(FileInterface $image, $width, $height, ?string $fileExtension): string
    {
        /*        if ($this->arguments['sources'] || $this->arguments['sourceName']) {
            return $this->renderPicture($image, $width, $height);
        } else
        */
        if ($this->arguments['srcset']) {
            return $this->renderImageSrcset($image, $width, $height);
        } else {
            return $this->renderImageSrc($image, $width, $height, $fileExtension);
        }
    }

    /**
     * Render img tag with srcset/sizes attributes
     *
     * @param FileInterface $image
     * @param string        $width
     * @param string        $height
     *
     * @return string                 Rendered img tag
     */
    protected function renderImageSrcset(FileInterface $image, $width, $height): string
    {
        /*
        if ($this->arguments['imageAspect']>0) {
            $imageAspect = $this->arguments['imageAspect'];
        }
        */
        //        echo('a='.$imageAspect .'='. $image->hasProperty('image_aspect'));
        // Generate image tag
        
        // @todo ($this->tag als img tag verwenden
        //var_dump($this->tag);exit(0);
        
        return $this->getResponsiveImagesUtility()->createImageHtmlWithSrcset(
            $image,
            $this->arguments['srcset'],
            $this->arguments['sizes'] ?? null,
            $this->arguments['imageAspect'] ?? null,
            $this->arguments['crop'] ?? null,
            false,
            $this->arguments['lazy'] ?? false,
            [
                'class' => $this->arguments['class'],
                'style' => $this->arguments['style'],
                'title' => $this->arguments['title'],
                'alt' => $this->arguments['alt'],
                'lang' => $this->arguments['lang'],
                'id' => $this->arguments['id'],
                'data' => $this->arguments['data'],
            ],
            $this->arguments['webp'],
            $this->arguments['sourcesName']
        );
    }











    
    /**
     * Generates a fallback image for picture and srcset markup
     *
     * @param FileInterface $image
     * @param string        $width
     * @param Area          $cropArea
     *
     * @return FileInterface
     */
    protected function generateFallbackImage(FileInterface $image, $width, Area $cropArea)
    {
        if ($this->arguments['width'] ||!$cropArea->isEmpty()) {
            $processingInstructions = [
                'width' => $width,
                'height' => 0,
                'crop' => $cropArea->isEmpty() ? null : $cropArea->makeAbsoluteBasedOnFile($image),
            ];
            $imageService = $this->getImageService();
            return $imageService->applyProcessingInstructions($image, $processingInstructions);
        } else {
            return $image;
        }
    }


    /**
     * Render img tag (no changes from MediaViewHelper.php:renderImage)
     *
     * @param string $width
     * @param string $height
     * @return string Rendered img tag
     */
    protected function renderImageSrc(FileInterface $image, $width, $height, ?string $fileExtension): string
    {
        $cropVariant = $this->arguments['cropVariant'] ?: 'default';
        $cropString = $image instanceof FileReference ? $image->getProperty('crop') : '';
        $cropVariantCollection = CropVariantCollection::create((string)$cropString);
        $cropArea = $cropVariantCollection->getCropArea($cropVariant);
        $processingInstructions = [
            'width' => $width,
            'height' => $height,
            'crop' => $cropArea->isEmpty() ? null : $cropArea->makeAbsoluteBasedOnFile($image),
        ];
        if (!empty($fileExtension)) {
            $processingInstructions['fileExtension'] = $fileExtension;
        }
        $imageService = $this->getImageService();
        $processedImage = $imageService->applyProcessingInstructions($image, $processingInstructions);
        $imageUri = $imageService->getImageUri($processedImage);

        if (!$this->tag->hasAttribute('data-focus-area')) {
            $focusArea = $cropVariantCollection->getFocusArea($cropVariant);
            if (!$focusArea->isEmpty()) {
                $this->tag->addAttribute('data-focus-area', $focusArea->makeAbsoluteBasedOnFile($image));
            }
        }
        $this->tag->addAttribute('src', $imageUri);
        $this->tag->addAttribute('width', $processedImage->getProperty('width'));
        $this->tag->addAttribute('height', $processedImage->getProperty('height'));
        if (in_array($this->arguments['loading'] ?? '', ['lazy', 'eager', 'auto'], true)) {
            $this->tag->addAttribute('loading', $this->arguments['loading']);
        }
        if (in_array($this->arguments['decoding'] ?? '', ['sync', 'async', 'auto'], true)) {
            $this->tag->addAttribute('decoding', $this->arguments['decoding']);
        }

        $alt = $image->getProperty('alternative');
        $title = $image->getProperty('title');

        // The alt-attribute is mandatory to have valid html-code, therefore add it even if it is empty
        if (empty($this->arguments['alt'])) {
            $this->tag->addAttribute('alt', $alt);
        }
        if (empty($this->arguments['title']) && $title) {
            $this->tag->addAttribute('title', $title);
        }

        return $this->tag->render();
    }

    /**
     * Returns an instance of the responsive images utility
     * This fixes an issue with DI after clearing the cache
     *
     * @return ResponsiveImagesUtility
     */
    protected function getResponsiveImagesUtility()
    {
        return GeneralUtility::makeInstance(ResponsiveImagesUtility::class);
    }


    protected function getImageService(): ImageService
    {
        return GeneralUtility::makeInstance(ImageService::class);
    }
}
