<?php

namespace Ig\IgResponsiveImages\Utility;

use Ig\IgResponsiveImages\Domain\Model\ResponsiveImagesConfiguration;
use Ig\IgResponsiveImages\Domain\Model\Picture;
use Ig\IgResponsiveImages\Domain\Model\PictureSource;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Imaging\ImageManipulation\CropVariantCollection;
use TYPO3\CMS\Core\Imaging\ImageManipulation\Area;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Resource\FileInterface;
use TYPO3\CMS\Core\Utility\ExtensionManagementUtility;
use TYPO3\CMS\Core\Utility\PathUtility;
use TYPO3Fluid\Fluid\Core\ViewHelper\TagBuilder;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Cache\CacheManager;
use TYPO3\CMS\Core\Resource\FileReference;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Service\ImageService;

class ResponsiveImagesUtility implements SingletonInterface
{
    /**
     * Image Service
     *
     * @var \TYPO3\CMS\Extbase\Service\ImageService
     */
    protected $imageService;

    /**
     * cache
     *
     * @var \TYPO3\CMS\Core\Cache\Frontend\FrontendInterface
     */
    protected $cache;

    /**
     * cache
     *
     * @var array
     */
    protected $settings;
    
    protected $currentImageConfig = [
        'class' => null,
        'configName' => 'default',
        'srcset' => null,
        'sizes' => [
            0 => [
                'media' => '',
                'width' => '100vw'
            ]
        ],
        'config' => []
    ];

    protected $currentImageConfigStack = [];

    protected $arguments = [];

    public function __construct()
    {
        $this->cache = GeneralUtility::makeInstance(CacheManager::class)->getCache('ig_responsive_images');        
        $this->imageService = $this->getImageService();
        $configurationManager = GeneralUtility::makeInstance(\TYPO3\CMS\Extbase\Configuration\ConfigurationManager::class);
        //$configurationManager = GeneralUtility::makeInstance(ConfigurationManagerInterface::class);
        $settings = $configurationManager->getConfiguration(\TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface::CONFIGURATION_TYPE_SETTINGS, 'igResponsiveImages');
        $this->settings = $settings;
        if($settings['default']) {
            $this->currentImageConfig['srcset']=$settings['default']['srcset'];
            $this->currentImageConfig['sizes']=$settings['default']['sizes'];
        }        
    }

    /**
     * Interpretes media query and returns the info in an array
     *
     * @param string $media
     *
     * @return array
     */
    public function parseMedia($media)
    {
        $media= trim(trim($media), '()');
        $parts=explode(':', $media);
        if(count($parts)==2) {
            return [
                'feature' => trim($parts[0]),
                'value' => trim($parts[1])
            ];
        } else {
            return false;
        }
    }

    /**
     * Creates an image tag with the provided srcset candidates frueher return TagBuilder
     *
     * @param FileInterface $originalImage
     * @param array|string  $srcset
     * @param string        $sizesQuery
     * @param string        $imageAspect 
     * @param string        $cropVariant
     * @param bool          $absoluteUri
     * @param bool          $lazy              use lazy load images (data attributes)
     * @param array         $imgAttributes     attributes of img tag
     * @param bool          $webp              generate webp images
     * @param string        $sourcesName
     *
     * @return string
     */
    public function createImageHtmlWithSrcset(
        FileInterface $originalImage,
        $srcset,
        string $sizesQuery = null,
        string $imageAspect = null,
        string $cropVariant = null,
        bool $absoluteUri = false,
        bool $lazy = false,
        array $imgAttributes = [],
        bool $webp = null,
        string $sourcesName = 'default'
    ): string {

        /*
        FileInterface $fallbackImage,
        Area $cropArea = null,
        Area $focusArea = null,
        */
        $cropString = $originalImage instanceof FileReference ? $originalImage->getProperty('crop') : '';
        //$cropVariantCollection = CropVariantCollection::create((string) $cropString);
        if ($cropString=='' || $cropString=='{"default":{"cropArea":{"x":0,"y":0,"width":1,"height":1},"selectedRatio":"NaN","focusArea":null}}') {
            //$cropArea = Area::createEmpty();
            //$focusArea = Area::createEmpty();
            // @tood implempent imageAspect
            $cropVariant = 'default';
        } else {
            // Crop von Element (d.h. Editor)
            $cropVariant = $cropVariant ?: 'default';
            //$cropArea = $cropVariantCollection->getCropArea($cropVariant);
            //$focusArea = $cropVariantCollection->getFocusArea($cropVariant);
        }
        ///$fallbackImage = $this->generateFallbackImage($originalImage, $width, $cropArea);

        // ($fallbackImage ? $fallbackImage->getUid() : '').','. 
        $sources = $this->settings['sources'][$sourcesName];
        $cacheString = $originalImage->getIdentifier();
        $cacheString .=  'srcset' . $srcset . $cropString . $sizesQuery;
        $cacheString .= $imageAspect;
        $cacheString .= ($absoluteUri ? 'absoluteUri' : '');
        $cacheString .= ($lazy ? 'lazy' : '');
        $cacheString .= json_encode($imgAttributes);
        $cacheString .= json_encode($sources);
        $cacheString .= ($webp ? 'webp' : '');
        $cacheIdentifier= sha1($cacheString);
        //335 = erstes Bild


        /*
        if($originalImage->getUid()==25093) {
            var_dump($webp, $cacheString);
            echo(get_class($originalImage ));
            var_dump($cropString);
            exit(0);
        }
        */
        // try to find the cached content
        if(($content = $this->cache->get($cacheIdentifier)) === false) {

            // no cache content
            $referenceWidth = 100;//$fallbackImage->getProperty('width') ? : 100;
            $picture = new Picture();
            $picture->setLazyPlaceholderImage($this->settings['lazyPlaceholderImage']);
            $picture->setOriginalImage($originalImage);
            //$picture->setFallbackImage($fallbackImage);
            $picture->setAbsoluteUri($absoluteUri);
            $picture->setLazy($lazy);
            $picture->setImgAttributes($imgAttributes);

            $picture->setSrcSetString($srcset);
            $picture->setSizesQuery($sizesQuery);
            $picture->setCropVariant($cropVariant);


            if(!empty($sources)) {
                foreach($sources as $source) {
                    // no webp if it is disable local
                    if ($webp === false && isset($source['type']) && $source['type'] == 'image/webp') {
                        continue;
                    }
                    $pictureSource = new PictureSource();
                    if(isset($source['type'])) {
                        $pictureSource->setType($source['type']);
                    }
                    if(isset($source['fileExt'])) {
                        $pictureSource->setFileExtension($source['fileExt']);
                    }
                    if(isset($source['media'])) {
                        $pictureSource->setMedia($source['media']);
                    }
                    if(isset($source['cropVariant'])) {
                        $pictureSource->setCropVariant($source['cropVariant'] ?? $cropVariant);
                    }
                    if ($source['quality'] ?? false) {
                        $pictureSource->setQuality((int)($source['quality'] ?? 0));
                    }
                    /*
                    if(isset($source['sizes'])) {
                        $pictureSource->setSizes($source['sizes']);
                    }
                    if(isset($source['srcset'])) {
                        $pictureSource->setSrcset($source['srcset']);
                    }
                     */
                    $picture->addSource($pictureSource);
                }
                //if($sourcesName=='slider') {                die('done');}
                if($imageAspect>0) {
                    $picture->setImageAspect($imageAspect);
                }
                $content = $picture->render();
            } elseif($webp===true || ($webp===null && $this->settings['webp'])) {
                // Compatibility for webp
                $pictureSource = new PictureSource();
                $pictureSource->setType('image/webp');
                $pictureSource->setFileExtension('webp');
                $picture->addSource($pictureSource);
                //$pictureSource->setMedia('(prefers-color-scheme: dark)');
                //$pictureSource->setMedia('(orientation: landscape)');
                if($imageAspect>0) {
                    $picture->setImageAspect($imageAspect);
                }
                $content = $picture->render();
            } else {
                // Render withou picture
                $content = $picture->renderImgTag();
                // Render with picture
                //$content = $picture->renderImg();
            }
            $this->cache->set($cacheIdentifier, $content);
        }
        return $content;
    }

    


    
    /**
     * Creates an image tag with the provided srcset candidates frueher return TagBuilder
     *
     * @param FileInterface $originalImage
     * @param array|string  $srcset
     * @param string        $imageAspect 
     * @param string        $cropVariant
     * @param bool          $absoluteUri
     *
     * @return string
     */
    public function getSrcsetAttribute(
        FileInterface $originalImage,
        $srcset,
        string $imageAspect = null,
        string $cropVariant = null,
        bool $absoluteUri = false
    ): string {

        /*
        FileInterface $fallbackImage,
        Area $cropArea = null,
        Area $focusArea = null,
        */
        $cropString = $originalImage instanceof FileReference ? $originalImage->getProperty('crop') : '';
        //$cropVariantCollection = CropVariantCollection::create((string) $cropString);
        if ($cropString=='' || $cropString=='{"default":{"cropArea":{"x":0,"y":0,"width":1,"height":1},"selectedRatio":"NaN","focusArea":null}}') {
            $cropVariant = 'default';
        } else {
            $cropVariant = $cropVariant ?: 'default';
        }
        ///$fallbackImage = $this->generateFallbackImage($originalImage, $width, $cropArea);

        // ($fallbackImage ? $fallbackImage->getUid() : '').','. 
        $cacheString = 'lightbox-' . $originalImage->getIdentifier();
        $cacheString .=  'srcset' . $srcset . $cropString ;
        $cacheString .= $imageAspect;
        $cacheString .= ($absoluteUri ? 'absoluteUri' : '');
        $cacheIdentifier= sha1($cacheString);
        // try to find the cached content
        if(($content = $this->cache->get($cacheIdentifier)) === false) {
            // no cache content
            $referenceWidth = 100;//$fallbackImage->getProperty('width') ? : 100;
            $picture = new Picture();
            $picture->setLazyPlaceholderImage($this->settings['lazyPlaceholderImage']);
            $picture->setOriginalImage($originalImage);
            //$picture->setFallbackImage($fallbackImage);
            $picture->setAbsoluteUri($absoluteUri);
            //$picture->setLazy($lazy);
            //$picture->setImgAttributes($imgAttributes);

            $picture->setSrcSetString($srcset);
            //$picture->setSizesQuery($sizesQuery);
            $picture->setCropVariant($cropVariant);
            //$picture->setFileExtension('webp');
            $content = $picture->renderSrcset();
            $this->cache->set($cacheIdentifier, $content);
        }
        return $content;
    }

    


    
    /**
     * Adds metadata to image tag
     *
     * @param TagBuilder    $tag
     * @param FileInterface $originalImage
     * @param FileInterface $fallbackImage
     * @param Area          $focusArea
     *
     * @return void
     */
    public function addMetadataToImageTag(
        TagBuilder $tag,
        FileInterface $originalImage,
        FileInterface $fallbackImage,
        Area $focusArea = null
    ) {
        $focusArea = $focusArea ?: Area::createEmpty();

        // Add focus area to image tag
        if(!$tag->hasAttribute('data-focus-area') && !$focusArea->isEmpty()) {
            $tag->addAttribute('data-focus-area', $focusArea->makeAbsoluteBasedOnFile($fallbackImage));
        }

        // The alt-attribute is mandatory to have valid html-code, therefore add it even if it is empty
        $alt = $originalImage->getProperty('alternative');
        if(!$tag->getAttribute('alt')) {
            $tag->addAttribute('alt', $alt);
        }
        $title = $originalImage->getProperty('title');
        if(!$tag->getAttribute('title') && $title) {
            $tag->addAttribute('title', $title);
        }
    }

    /**
     * Renders different image sizes for use in a srcset attribute
     *
     * Input:
     *   1: $srcset = [200, 400]
     *   2: $srcset = ['200w', '400w']
     *   3: $srcset = ['1x', '2x']
     *   4: $srcset = '200, 400'
     *
     * Output:
     *   1+2+4: ['200w' => 'path/to/image@200w.jpg', '400w' => 'path/to/image@200w.jpg']
     *   3: ['1x' => 'path/to/image@1x.jpg', '2x' => 'path/to/image@2x.jpg']
     *
     * @param FileInterface $image
     * @param int           $defaultWidth
     * @param array|string  $srcset
     * @param Area          $cropArea
     * @param bool          $absoluteUri
     * @param string        $fileExtension
     *
     * @return array
     */
    public function generateSrcsetImages(
        FileInterface $image,
        int $defaultWidth,
        $srcset,
        Area $cropArea = null,
        string $imageAspect = null,
        bool $absoluteUri = false,
        array $fileExtensions = ['default']
    ): array {
        $cropArea = $cropArea ?: Area::createEmpty();

        // Convert srcset input to array
        if(!is_array($srcset)) {
            $srcset = GeneralUtility::trimExplode(',', $srcset);
        }

        $images = [];
        foreach( $fileExtensions as $fileExtension) {
            $images[$fileExtension] = [];
        }
        foreach ($srcset as $widthDescriptor) {
            // Determine image width
            switch (substr($widthDescriptor, -1)) {
            case 'x':
                $candidateWidth = (int) ($defaultWidth * (float) substr($widthDescriptor, 0, -1));
                break;

            case 'w':
                $candidateWidth = (int) substr($widthDescriptor, 0, -1);
                break;

            default:
                $candidateWidth = (int) $widthDescriptor;
                $widthDescriptor = $candidateWidth . 'w';
            }

            if($cropArea->isEmpty()) {
                $cropAreaAbsoluteBasedOnFile=null;
            } else {
                // TYPO3 9.5 - exif beachten
                if(function_exists('exif_read_data')) {
                    $exif = @exif_read_data(Environment::getPublicPath() . '/' .$image->getPublicUrl());
                    if(isset($exif['Orientation']) && $exif['Orientation'] >= 5 && $exif['Orientation'] <= 8) {
                        $cropArray=$cropArea->asArray();
                        // crop area drehen, falls exif bild drehen wird (bugfix fuer TYPO3 9.5)
                        $cropAreaAbsoluteBasedOnFile = new \TYPO3\CMS\Core\Imaging\ImageManipulation\Area($cropArray['y']*$image->getProperty('height'), $cropArray['x']*$image->getProperty('width'), $cropArray['height']*$image->getProperty('height'), $cropArray['width']*$image->getProperty('width'));
                    } else {
                        $cropAreaAbsoluteBasedOnFile= $cropArea->makeAbsoluteBasedOnFile($image);
                    }
                } else {
                    $cropAreaAbsoluteBasedOnFile= $cropArea->makeAbsoluteBasedOnFile($image);
                }
            }
      
            // Generate image
            $processingInstructions = [
                'width' => $candidateWidth,
                'height' => 0,
                'crop' => $cropAreaAbsoluteBasedOnFile,
            ];
            /*
            // @todo rearange functions should always a TYPO3\CMS\Core\Resource\File and no TYPO3\CMS\Core\Resource\FileReference
            // @todo change it some werde upstream (will be forced in TYPO3 11 to have proper method signature)
            if(is_callable([$image, 'getOriginalFile'])) {
                // Get the original file from the file reference
                $image = $image->getOriginalFile();
            }
        
            $processedImage = $image->process(\TYPO3\CMS\Core\Resource\ProcessedFile::CONTEXT_IMAGECROPSCALEMASK, $processingInstructions);
            // we don't need setCompatibilityValues should speed up by 25%
            //$processedImage =$this->imageService->setCompatibilityValues($processedFile);
            */
            foreach( $fileExtensions as $fileExtension) {
                if($fileExtension=='default') {
                    if(isset($processingInstructions['fileExtension'])) {
                        unset($processingInstructions['fileExtension']);
                    }
                }else {
                    $processingInstructions['fileExtension']=$fileExtension;
                }
                
                $processedImage = $this->imageService->applyProcessingInstructions($image, $processingInstructions);
                $images[$fileExtension][$widthDescriptor] = $this->imageService->getImageUri($processedImage, $absoluteUri);
            }


        }

        return $images;
    }

    /**
     * Generates the content for a srcset attribute from an array of image urls
     *
     * Input:
     * [
     *   '200w' => 'path/to/image@200w.jpg',
     *   '400w' => 'path/to/image@400w.jpg'
     * ]
     *
     * Output:
     * 'path/to/image@200w.jpg 200w, path/to/image@400w.jpg 400w'
     *
     * @param array $srcsetImages
     *
     * @return string
     */
    public function generateSrcsetAttribute(array $srcsetImages): string
    {
        $srcsetString = [];
        foreach ($srcsetImages as $widthDescriptor => $imageCandidate) {
            $srcsetString[] = $imageCandidate . ' ' . $widthDescriptor;
        }
        return implode(', ', $srcsetString);
    }






    /**
     * Aktuelle Sizes fuer img Tag
     *
     * @return string Aktuelle sizes
     */
    public function getSizes()
    {
        $ret='';
        foreach ($this->currentImageConfig['sizes'] as $currentSize) {
            if(strlen($ret)>4000) {
                var_dump($this->currentImageConfig['sizes']);
                die('image size Error - Stack too big');
            }
            if($ret) {
                $ret .= ', ';
            }
            $ret .= ($currentSize['media'] ? $currentSize['media'] .' ' : '').$currentSize['width'];
        }
        return $ret ;
    }

    /**
     * Aktuelle Srcset fuer img Tag
     *
     * @return string Aktuelle srcset
     */
    public function getSrcset($maxWidth=0)
    {
        if(intval($maxWidth)<=0) {
            return $this->currentImageConfig['srcset'];
        } else {
            $srcset= explode(',', $this->currentImageConfig['srcset']);
            sort($srcset, SORT_NUMERIC);
            $ret=[];
            $maxWidthAndNotClose=$maxWidth*0.9; // Werte nur nehmen wenn nicht in naehe
            foreach ($srcset as $width) {
                if($width>$maxWidthAndNotClose) {
                    break;
                }
                $ret[]=trim($width);
            }
            $ret[]=$maxWidth;
            return implode(',', $ret);
        }
    }
    public function setTagWidth($tagWidth)
    {
        $this->currentImageConfig['config']['tagWidth']=$tagWidth;
    }

    public function getTagWidth()
    {
        return $this->currentImageConfig['config']['tagWidth'] ?? 0;
        $tagWidth=$this->currentImageConfig['config']['tagWidth'];
        if($tagWidth == 0) {
            $tagWidth=590;
        }
        if(isset($this->currentImageConfig['parent']['width'])) {
            $colWidth=$this->currentImageConfig['parent']['width'];
            $tagWidth=round($tagWidth*floatval($colWidth)/100);
        } else {
            $colWidth=$this->currentImageConfig['config']['colDefaultWidth'] ? : 100;
            $tagWidth=round($tagWidth*floatval($colWidth)/100);
        }
        if(isset($this->currentImageConfig['config']['tagWidthMargin'])) {
            $tagWidth-=intval($this->currentImageConfig['config']['tagWidthMargin']);
        }
        return $tagWidth;
        //return $this->currentImageConfig['config']['tagWidth'];
    }
    public function getImageWidth()
    {
        return ($this->currentImageConfig['config']['imageWidth'] ?? null && $this->currentImageConfig['config']['imageWidth']>0) ? $this->currentImageConfig['config']['imageWidth'] : 590;
    }
    public function getConfig($name)
    {
        return $name && isset($this->currentImageConfig['config'][$name]) ? $this->currentImageConfig['config'][$name] : '';
    }
  
    /**
     * Klasse auf Stack schieben und aktuelle Bildwerte entsprechend setzen
     *
     * @param string $class bzw. Typoscript Konfigurationsnamen
     *
     * @return void
     */
    public function calculateSize($currentWidth, $newWidth, $width=null)
    {
        /*
           Unterstuetzte Endungen:
           %: -> Wert berechnen
           sonst: Wert setzen (fuer z.B. vw, px)
        */
        $newWidth=trim($newWidth);
      
        $last=substr($newWidth, -1);
        if($last!='%') {
            return $newWidth;
        }
        preg_match_all('/^(\d+|\d*\.\d+)(\w+)$/', $currentWidth, $matches);      // Oder stur: $currentUnit=substr($currentWidth,-2);
        $currentValue=$matches[1][0];
        $currentUnit=$matches[2][0];
        $newValue=trim(substr($newWidth, 0, -1));
        if($width!==null && $width>0) {
            $newValue=$width;
        }
        return round($currentValue*$newValue/100, 4) . $currentUnit;
    }
    /**
     * Klasse auf Stack schieben und aktuelle Bildwerte entsprechend setzen
     *
     * @param string $class      bzw. Typoscript Konfigurationsnamen
     * @param float  $width      Breite in % 0 - 100
     * @param string $configName bzw. Typoscript Konfigurationsnamen fuer Debug bei grid wenn class=col ist, d.h. $class keine Config hat
     *
     * @return void
     */
    public function pushClass($class, $width = null, $configName = null)
    {
        array_push($this->currentImageConfigStack, $this->currentImageConfig);
        $newSizes = [];

        //var_dump($cols, $width,$classes, $this->settings['classes']['3'], $this->currentImageConfig['sizes']);die('a');
        
        foreach ($this->currentImageConfig['sizes'] as $currentSize) {
            $newSizesConfig = $this->settings['classes'][$class]['sizes'] ?? [];
            $newConfig = $this->settings['classes'][$class]['config'] ?? [];



            if(is_array($newSizesConfig) && count($newSizesConfig)) {
                foreach ($newSizesConfig as $newSize) {
                    //echo( $class .':'. $currentSize['width'].'*'.$newSize['width'] .'='.($currentSize['width']*$newSize['width']/100) .'::');
                    if(($newSize['media'] ?? false) && ($currentSize['media'] ?? false)) {
                        $newMedia = $this->parseMedia($newSize['media']);
                        $currentMedia = $this->parseMedia($currentSize['media']);
                        // only combine identical media queries
                        /**
                         * @todo media Queries interpretieren und richtig zusammenfuehren
                         * Minimumziel:
                         * 1) Funktion: min/max erkennen + Pixel
                         * 2) Beide Array sortieren und zusammenführen: media Query identisch 1 Eintrag sonst beide?
                         **/
                        if($newMedia && $currentMedia && $newMedia['feature'] != $currentMedia['feature']) {
                            continue;
                        }
                    }
                    // should the value be overridden with flexform width
                    $overrideWidth = isset($newSize['override']) && $newSize['override'] == 1 ? $width : null;
                    $newElement = [
                        'configName' => $configName,
                        'parent' => [
                            'class' => $class,
                            'width' => $width  ? : ($this->currentImageConfig['parent']['width'] ?? 0),
                        ],
                        'media' => isset($newSize['media']) && $newSize['media'] ? $newSize['media'] : $currentSize['media'],
                        'width' => $this->calculateSize($currentSize['width'], $newSize['width'], $overrideWidth),
                        'config' => array_merge($this->currentImageConfig['config'], $newConfig)
                    ];
                    //$newElement['config']['tagWidth']=$this->currentImageConfig['config']['tagWidth']/;
                    $this->currentImageConfig['config']=$newElement['config'];
                    $this->currentImageConfig['configName']=$newElement['configName'];
                    $this->currentImageConfig['parent']= $newElement['parent'];
                    if(!in_array($newElement, $newSizes)) {
                        //file_put_contents('/tmp/ri.log', "push media=".$newElement['media'].", width=".$newElement['width']."\n", FILE_APPEND);
                        $newSizes[] = $newElement;
                    }
                }
            } else {
            }
        }
        // Was neues?
        if(count($newSizes)>0) {
            if(isset($this->settings['classes'][$class]['srcset'])) {
                $this->currentImageConfig['srcset']=$this->settings['classes'][$class]['srcset'];
            } else {
                // Alter Wert belassen
            }
            $this->currentImageConfig['sizes'] = $newSizes;
        }
        //      echo('test['.$class.'] (stack:'.count($this->currentImageConfigStack).'):' . print_r($newSize['media'],1). ', old='.print_r($this->currentImageConfig['sizes'],1) .'<br />'."\n");
    }


      
    /**
     * Aktuelle Klasse vom Stack entfernen
     *
     * @return void
     */
    public function popClass()
    {
        /// DADADA
        $currentImageConfig = array_pop($this->currentImageConfigStack);
        if ($currentImageConfig !== null) {
            $this->currentImageConfig = $currentImageConfig;
        }
    }


    /**
     * @TODO Breite der Spalte beruecksichtigen
     * Klasse auf Stack schieben und aktuelle Bildwerte entsprechend setzen
     *
     * @param integer $cols    Anzahl Spalten des Grids
     * @param string  $width   Aktuelle Breite der Spalte in Prozent %
     * @param array   $classes alternative Konfigurationsname (class)
     *
     * @return boolean class is pushed
     */
    public function pushGridelements($cols, $width, $classes=[])
    {

        // Existiert eine Konfiguration zu einer der gewünschten Klassen
        foreach ($classes as $class) {
            if($class!=null && $class!='1' && isset($this->settings['classes'][$class])) {
                $this->pushClass($class, $width, $class);
                return true;
            }
        }
        if($cols>1) {
            $this->pushClass($cols, $width, $class);
            return true;
        } else {
            if($width>0) {
                $cols=round(100/$width);
                $this->pushClass($cols, null, $class);
                return true;
            }
            return false;
        }
    }

    /**
     * Aktuelle Klasse vom Stack entfernen fuer Gridelements
     *
     * @return void
     */
    public function popGridelements()
    {
        $this->popClass();
    }

    /**
     * @TODO Breite der Spalte beruecksichtigen
     * Klasse auf Stack schieben und aktuelle Bildwerte entsprechend setzen
     *
     * @param ResponsiveImagesConfiguration $responsiveImagesConfiguration
     */
    public function pushContainer(ResponsiveImagesConfiguration $responsiveImagesConfiguration)
    {
        //$processorConfiguration = $responsiveImagesConfiguration->getProcessorConfiguration();
        $colWidth = $responsiveImagesConfiguration->getColWidth();
        $configName = $responsiveImagesConfiguration->getConfigName();        
        $cols = $responsiveImagesConfiguration->getCols();
        $tagWidth = $responsiveImagesConfiguration->getTagWidth();
        
        if (($configName && $configName!='1') || $colWidth!==null) {
            if ($colWidth!==null || isset($cols)) {
                if ($this->pushGridelements($cols, $colWidth, [$configName])) {
                    $responsiveImagesConfiguration->push();
                }
            } else {
                $this->pushCols($configName);
                $responsiveImagesConfiguration->push();
            }
            if (isset($tagWidth)) {
                $this->setTagWidth(intval($tagWidth));
            }
        } else {
            $responsiveImagesConfiguration->push();
            $responsiveImagesConfiguration->setOuterTagWidth($this->getTagWidth());
            $this->setTagWidth(intval($tagWidth));
        }
    }

    /**
     * Aktuelle Klasse vom Stack entfernen fuer Gridelements
     *
     *@param ResponsiveImagesConfiguration $responsiveImagesConfiguration
     * @return void
     */
    public function popContainer(ResponsiveImagesConfiguration $responsiveImagesConfiguration)
    {
        if ($responsiveImagesConfiguration->isPushed()) {
            $outerTagWidth = $responsiveImagesConfiguration->getOuterTagWidth();
            if ($outerTagWidth !== null) {
                $this->setTagWidth($outerTagWidth);
            } else {
                $this->popWidth();
            }
        }
    }

    
    public function getSrcsetByClass($class)
    {
        return $this->settings['classes'][$class]['srcset'] ?? null;
    }
    public function getSizesByClass($class)
    {
        return $this->settings['classes'][$class]['sizes'] ?? null;
    }
    

    // Veraltete Funktionsnamen
    /**
     * Debug Funktion
     *
     * @return void
     */
    private function getColSizes($cols, $width)
    {
        return $this->settings['classes'][$cols];
    }


    // Breite automaitsch berechnen
    public function pushWidth($width)
    {
        array_push($this->currentImageConfigStack, $this->currentImageConfig);
        foreach ($this->currentImageConfig['sizes'] as $key=>$size) {
            $this->currentImageConfig[$key]['width']*=$width/100;
        }
    }

    public function pushCols($cols)
    {
        $this->pushClass($cols);
    }
    public function popWidth()
    {
        $this->popClass();
    }

    /**
     * Obly for debug - return current stack
     *
     * @return array current stack
     */
    public function debugStack()
    {
        return $this->currentImageConfigStack;
    }
  
    /**
     * Obly for debug - return current settings
     *
     * @return array current image config
     */
    public function debugCurrent()
    {
        return $this->currentImageConfig;
    }

    public function setWidth($width)
    {
        echo('function setWidth');
        exit(0);
        $newSize = [
            'media' => '',
            'width' => $width
        ];
    
        $this->currentImageConfig = $newSize;
    }
    public function getWidth()
    {
        echo('function getWidth');
        exit(0);
      
        return $this->currentImageConfig['width'];
    }

    /**
     * Generates a fallback image for picture and srcset markup
     *
     * @param FileInterface $image
     * @param string        $width
     * @param Area          $cropArea
     *
     * @return FileInterface
     */
    protected function generateFallbackImage(FileInterface $image, $width, Area $cropArea)
    {
        if (($this->arguments['width'] ?? null) ||!$cropArea->isEmpty()) {
            $processingInstructions = [
                'width' => $width,
                'height' => 0,
                'crop' => $cropArea->isEmpty() ? null : $cropArea->makeAbsoluteBasedOnFile($image),
            ];
            $imageService = $this->getImageService();
            return $imageService->applyProcessingInstructions($image, $processingInstructions);
        } else {
            return $image;
        }
    }



    /**
     * Return an instance of ImageService
     *
     * @return ImageService
     */
    protected function getImageService()
    {
        return GeneralUtility::makeInstance(ImageService::class);
    }

    
}
