<?php

namespace Ig\IgResponsiveImages\ViewHelpers;

use Ig\IgResponsiveImages\Utility\ResponsiveImagesUtility;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Imaging\ImageManipulation\CropVariantCollection;
use TYPO3\CMS\Core\Imaging\ImageManipulation\Area;
use TYPO3\CMS\Core\Resource\FileInterface;
use TYPO3\CMS\Core\Resource\FileReference;
use TYPO3\CMS\Core\Resource\Rendering\RendererRegistry;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Domain\Model\AbstractFileFolder;
use TYPO3\CMS\Extbase\Service\ImageService;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper;

class LightboxViewHelper extends AbstractTagBasedViewHelper
{
    /**
     * @var string
     */
    protected $tagName = 'a';

    // handleAdditionalArguments() von AbstractTagBasedViewHelper müsste dies eigentlich machen
    protected $lightboxAttributes= ['data-caption', 'data-width', 'data-height', 'data-type', 'data-src', 'data-ratio', 'data-options'];
    
    /**
     * Initialize arguments.
     */
    public function initializeArguments()
    {
        parent::initializeArguments();
        $this->registerUniversalTagAttributes();
        $this->registerArgument('file', 'object', 'File', true);
        $this->registerArgument('data-fancybox', 'string', 'fancybox Group', false, null);
        foreach ($this->lightboxAttributes as $name) {
            $this->registerArgument($name, 'string', 'fancybox: ' . $name, false, null);
        }
        $this->registerArgument('setupClass', 'string', 'setupclass', false, 'lightbox');
        $this->registerArgument('additionalConfig', 'array', 'This array can hold additional configuration that is passed though to the Renderer object', false, []);
        $this->registerArgument('width', 'string', 'This can be a numeric value representing the fixed width of in pixels. But you can also perform simple calculations by adding "m" or "c" to the value. See imgResource.width for possible options.');
        $this->registerArgument('height', 'string', 'This can be a numeric value representing the fixed height in pixels. But you can also perform simple calculations by adding "m" or "c" to the value. See imgResource.width for possible options.');
        $this->registerArgument('cropVariant', 'string', 'select a cropping variant, in case multiple croppings have been specified or stored in FileReference', false, 'default');
        $this->registerArgument('srcset', 'mixed', 'Image sizes that should be rendered.', false, null);
        $this->registerArgument('imageAspect', 'string', 'IG: Aspect Ratio', false, null);
    }

    /**
     * Render a given media file
     *
     * @return string Rendered tag
     * @throws \UnexpectedValueException
     */
    public function render()
    {
        $file = $this->arguments['file'];
        $additionalConfig = $this->arguments['additionalConfig'];
        $width = $this->arguments['width'];
        $height = $this->arguments['height'];

        // get Resource Object (non ExtBase version)
        if (is_callable([$file, 'getOriginalResource'])) {
            // We have a domain model, so we need to fetch the FAL resource object from there
            $file = $file->getOriginalResource();
        }

        if (!($file instanceof FileInterface || $file instanceof AbstractFileFolder)) {
            throw new \UnexpectedValueException('Supplied file object type ' . get_class($file) . ' must be FileInterface or AbstractFileFolder.', 1454252193);
        }

        $fileRenderer = RendererRegistry::getInstance()->getRenderer($file);
        // Fallback to image when no renderer is found
        if ($fileRenderer === null) {
            $fancybox = $this->arguments['data-fancybox'];
            $srcset = $this->renderSrcset($file, $width, $height);
            $this->tag->addAttribute('data-fancybox', $fancybox);
            $this->tag->addAttribute('data-srcset', $srcset);
            $this->tag->addAttribute('href', $file->getPublicUrl());
            foreach ($this->lightboxAttributes as $name) {
                if ($this->arguments[$name]) {
                    $this->tag->addAttribute($name, $this->arguments[$name]);
                }
            }
            $this->tag->setContent($this->renderChildren());
            $this->tag->forceClosingTag(true);
            return  $this->tag->render();
        }
        $additionalConfig = array_merge_recursive($this->arguments, $additionalConfig);
        return $fileRenderer->render($file, $width, $height, $additionalConfig);
    }

    /**
     * Render srcset attributes
     *
     * @param FileInterface $image
     * @param string        $width
     * @param string        $height
     *
     * @return string                 srcset sttribute
     */
    protected function renderSrcset(FileInterface $image, $width, $height)
    {
        $responsiveImageUtility=$this->getResponsiveImagesUtility();
        if ($this->arguments['srcset']) {
            $srcset = $this->arguments['srcset'];
        } else {
            $srcset = $responsiveImageUtility->getSrcsetByClass($this->arguments['setupClass']);
        }
        return $this->getResponsiveImagesUtility()->getSrcsetAttribute(
            $image,
            $srcset,
            $this->arguments['imageAspect'],
            $this->arguments['crop'],
            false
        );
    }

    /**
     * Returns an instance of the responsive images utility
     * This fixes an issue with DI after clearing the cache
     *
     * @return ResponsiveImagesUtility
     */
    protected function getResponsiveImagesUtility()
    {
        return GeneralUtility::makeInstance(ResponsiveImagesUtility::class);
    }
    /**
     * Return an instance of ImageService
     *
     * @return ImageService
     */
    protected function getImageService()
    {
        return GeneralUtility::makeInstance(ImageService::class);
    }
}
