<?php

namespace Ig\IgRuckzuckevent\Controller\Backend;

use DateTime;
use Ig\IgRuckzuckevent\Controller\TYPO3\CMS\Extbase\Annotation\IgnoreValidation;
use Ig\IgRuckzuckevent\Domain\Model\Event;
use Ig\IgRuckzuckevent\Domain\Model\Registration;
use Ig\IgRuckzuckevent\Domain\Model\WaitlistPromotionStatus;
use Ig\IgRuckzuckevent\Domain\Repository\EventRepository;
use Ig\IgRuckzuckevent\Domain\Repository\RegistrationRepository;
use Ig\IgRuckzuckevent\Service\WaitlistService;
use Internetgalerie\IgBackendHelpers\Controller\AbstractBackendTableListController;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Shared\Date as ExcelDate;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Backend\Attribute\AsController;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Backend\Utility\BackendUtility;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\Restriction\DeletedRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\EndTimeRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\HiddenRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\StartTimeRestriction;
use TYPO3\CMS\Core\Imaging\Icon;
use TYPO3\CMS\Core\Imaging\IconFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Annotation\IgnoreValidation as AnnotationIgnoreValidation;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3\CMS\Backend\Template\Components\ButtonBar;
use TYPO3\CMS\Backend\Template\Components\Menu\Menu;
use TYPO3\CMS\Backend\Template\Components\Menu\MenuItem;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\TypoScript\TypoScriptService;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;

#[AsController]
class EventBackendController extends AbstractBackendTableListController
{
    protected string $tableName = 'tx_igruckzuckevent_domain_model_event';
    protected string $tableEventgroup = 'tx_igrecurrencedate_domain_model_eventgroup';
    protected string $entryIdSearchAttribute = 'categories';

    protected ?EventRepository $eventRepository = null;
    protected ?RegistrationRepository $registrationRepository = null;
    protected ?PageRenderer $pageRenderer = null;

    /**
     * waitlistService
     *
     * @var WaitlistService
     */
    protected $waitlistService = null;

    public function injectEventRepository(EventRepository $eventRepository): void
    {
        $this->eventRepository = $eventRepository;
    }

    public function injectRegistrationRepository(RegistrationRepository $registrationRepository): void
    {
        $this->registrationRepository = $registrationRepository;
    }

    public function injectPageRenderer(PageRenderer $pageRenderer): void
    {
        $this->pageRenderer = $pageRenderer;
    }

    public function injectWaitlistService(WaitlistService $waitlistService): void
    {
        $this->waitlistService = $waitlistService;
    }

    public function initializeAction(): void
    {
        parent::initializeAction();

        // Get plugin settings from TypoScript to use same mail configuration as frontend
        $fullTypoScript = $this->configurationManager->getConfiguration(
            ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT,
            'IgRuckzuckevent',
            'Igruckzuckevent'
        );
        
        $settings = $fullTypoScript['plugin.']['tx_igruckzuckevent.']['settings.'] ?? null;
        $typoScriptService = GeneralUtility::makeInstance(TypoScriptService::class);
        $typoScriptSettings = $typoScriptService->convertTypoScriptArrayToPlainArray($settings);

        if ($settings) {
            $this->waitlistService->setSettings($typoScriptSettings);
        }
    }

    /**
     * Action index
     */
    public function indexAction(): ResponseInterface
    {
        $view = $this->moduleTemplateFactory->create($this->request);
        $filter = $this->search;
        $categoryUid = $this->entryId;
        $storagePid = $this->pageUid;

        if ($storagePid !== null) {
            $tableName = 'tx_igruckzuckevent_domain_model_event';
            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tableName);
            $queryBuilder->getRestrictions()
                         ->removeByType(HiddenRestriction::class);
            $queryBuilder->getRestrictions()
                         ->removeByType(StartTimeRestriction::class);
            $queryBuilder->getRestrictions()
                         ->removeByType(EndTimeRestriction::class);
            $queryBuilder->getRestrictions()
                         ->add(GeneralUtility::makeInstance(DeletedRestriction::class));

            $now = new \DateTime();
            $nowString = $now->format('Y-m-d H:i:s');

            $queryBuilder->selectLiteral(
                '*, ' .
                // Count only confirmed registrations (waitlist=0) for display
                '(SELECT COUNT(*) FROM tx_igruckzuckevent_domain_model_registration WHERE event = tx_igruckzuckevent_domain_model_event.uid AND deleted = 0 AND hidden = 0 AND waitlist = 0) AS registration_count, ' .
                // Count actual waitlist (excludes registrations with active promotions)
                '(SELECT COUNT(*) FROM tx_igruckzuckevent_domain_model_registration WHERE event = tx_igruckzuckevent_domain_model_event.uid AND deleted = 0 AND hidden = 0 AND waitlist = 1 AND waitlist_promotion_status <> 2 AND waitlist_promotion_status <> 3 AND (' .
                '  waitlist_promotion_token = \'\' OR ' .
                '  waitlist_promotion_deadline < \'' . $nowString . '\'' .
                ')) AS waitlist_count'
            )
                         ->from($tableName)
                         ->where($queryBuilder->expr()->in(
                             $tableName . '.pid',
                             [$storagePid]
                         ))
                         ->andWhere($queryBuilder->expr()->in(
                             $tableName . '.sys_language_uid',
                             [
                                 $queryBuilder->createNamedParameter(-1, Connection::PARAM_INT),
                                 $queryBuilder->createNamedParameter(0, Connection::PARAM_INT),
                             ]
                         ))
                         ->orderBy('date_from');

            if ($filter['searchword'] ?? false) {
                $queryBuilder->andWhere(
                    $queryBuilder->expr()
                                 ->or(
                                     $queryBuilder->expr()->eq('uid', $queryBuilder->createNamedParameter($filter['searchword'], Connection::PARAM_INT)),
                                     $queryBuilder->expr()->like(
                                         'title',
                                         $queryBuilder->createNamedParameter('%' . $queryBuilder->escapeLikeWildcards($filter['searchword']) . '%')
                                     )
                                 )
                );
            }

            if ($categoryUid > 0) {
                $queryBuilder
                    ->join(
                        'tx_igruckzuckevent_domain_model_event',
                        'tx_igruckzuckevent_event_category_mm',
                        'catmm',
                        'catmm.uid_local = tx_igruckzuckevent_domain_model_event.uid'
                    )
                    ->andWhere('catmm.uid_foreign = ' . $categoryUid);
            }

            if (!isset($filter['mode']) || !$filter['mode']) {
                $filter['mode'] = 1;
            }

            $today = new DateTime();
            $today->setTime(2, 0, 0);

            if (isset($filter['mode']) && $filter['mode'] == 1) {
                $queryBuilder->andWhere(
                    $queryBuilder->expr()
                                 ->gte('date_from', $queryBuilder->createNamedParameter($today->format('Y-m-d')))
                );
            } elseif (isset($filter['mode']) && $filter['mode'] == 2) {
                $queryBuilder->andWhere(
                    $queryBuilder->expr()
                                 ->lte('date_from', $queryBuilder->createNamedParameter($today->format('Y-m-d')))
                );
            }

            $events = $queryBuilder->executeQuery()
                                   ->fetchAllAssociative();

            foreach ($events as &$event) {
                $iconFactory = GeneralUtility::makeInstance(IconFactory::class);
                $iconHtml = $iconFactory->getIconForRecord($tableName, $event, Icon::SIZE_SMALL)->render();
                $event['iconWithLink'] = BackendUtility::wrapClickMenuOnIcon($iconHtml, $tableName, $event['uid']);
            }

            $view->assign('events', $events);
            $view->assign('tableName', $tableName);
            $view->assign('filter', $filter);
        }
        $view->assign('entryId', $this->entryId);
        $view->assign('searchArgumentName', $this->searchArgumentName);
        $view->assign('entryIdArgumentName', $this->entryIdArgumentName);
        $view->assign('entryIdSearchAttribute', $this->entryIdSearchAttribute);
        $view->assign('pageUid', $this->pageUid);
        $view->assign('storagePid', $storagePid);
        $view->assign('search', $this->search);
        $returnUrlParameters = [
            $this->searchArgumentName => $this->search,
        ];
        if ($this->entryId > 0) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->entryId;
        }
        $view->assign('returnUrlParameters', $returnUrlParameters);
        $this->pageRenderer->loadJavaScriptModule('@typo3/backend/context-menu.js');

        return $view->renderResponse('EventBackend/Index');
    }

    /**
     * list participants for an event
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'event',
    ])]
    public function listParticipantsAction(Event $event, array $filter = []): ResponseInterface
    {
        $view = $this->moduleTemplateFactory->create($this->request);
        $storagePid = $event->getPid();
        $tableName = 'tx_igruckzuckevent_domain_model_registration';

        // Only get regular registrations (not waitlist)
        $filter['waitlist'] = 0;
        $registrations = $this->registrationRepository->findByEventAndFilter($event, $filter);

        // Get waitlist statistics for event statistics partial
        $waitlistRegistrations = $this->registrationRepository->findWaitlistByEvent($event);
        $declinedWaitlist = $this->registrationRepository->findDeclinedWaitlistByEvent($event);
        $cancelledWaitlist = $this->registrationRepository->findCancelledWaitlistByEvent($event);
        $waitlistCount = $waitlistRegistrations->count();

        // Calculate total spaces (not just registration count)
        $waitlistSpaces = $this->calculateTotalSpaces($waitlistRegistrations);
        $declinedWaitlistSpaces = $this->calculateTotalSpaces($declinedWaitlist);
        $cancelledWaitlistSpaces = $this->calculateTotalSpaces($cancelledWaitlist);

        $view->assign('storagePid', $storagePid);
        $view->assign('event', $event);
        $view->assign('registrations', $registrations);
        $view->assign('waitlistRegistrations', $waitlistRegistrations);
        $view->assign('declinedWaitlist', $declinedWaitlist);
        $view->assign('cancelledWaitlist', $cancelledWaitlist);
        $view->assign('waitlistCount', $waitlistCount);
        $view->assign('waitlistSpaces', $waitlistSpaces);
        $view->assign('declinedWaitlistSpaces', $declinedWaitlistSpaces);
        $view->assign('cancelledWaitlistSpaces', $cancelledWaitlistSpaces);
        $view->assign('tableName', $tableName);
        $view->assign('defaultValues', [
            $tableName => [
                'event' => $event->getUid(),
            ],
        ]);
        $view->assign('filter', $filter);
        $view->assign('search', $this->search);
        $returnUrlParameters = [
            $this->searchArgumentName => $this->search,
        ];
        if ($this->entryId > 0) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->entryId;
        } elseif ($this->search[$this->entryIdArgumentName] ?? false) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->search[$this->entryIdArgumentName];
        }
        $view->assign('returnUrlParameters', $returnUrlParameters);

        return $view->renderResponse('EventBackend/ListParticipants');
    }

    /**
     * list waitlist registrations for an event
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'event',
    ])]
    public function waitlistAction(Event $event, array $filter = []): ResponseInterface
    {
        $view = $this->moduleTemplateFactory->create($this->request);
        $storagePid = $event->getPid();
        $tableName = 'tx_igruckzuckevent_domain_model_registration';

        // Get waitlist registrations with optional search filter for display
        $waitlistRegistrationsWithDeclined = $this->registrationRepository->findWaitlistByEvent($event, true, $filter);

        // Get unfiltered counts for statistics (always show total counts, not filtered)
        $waitlistRegistrations = $this->registrationRepository->findWaitlistByEvent($event);
        $declinedWaitlist = $this->registrationRepository->findDeclinedWaitlistByEvent($event);
        $cancelledWaitlist = $this->registrationRepository->findCancelledWaitlistByEvent($event);

        // Calculate total spaces (not just registration count)
        $waitlistSpaces = $this->calculateTotalSpaces($waitlistRegistrations);
        $declinedWaitlistSpaces = $this->calculateTotalSpaces($declinedWaitlist);
        $cancelledWaitlistSpaces = $this->calculateTotalSpaces($cancelledWaitlist);

        $view->assign('storagePid', $storagePid);
        $view->assign('event', $event);
        $view->assign('waitlistRegistrations', $waitlistRegistrations);
        $view->assign('waitlistRegistrationsWithDeclined', $waitlistRegistrationsWithDeclined);
        $view->assign('declinedWaitlist', $declinedWaitlist);
        $view->assign('cancelledWaitlist', $cancelledWaitlist);
        $view->assign('waitlistSpaces', $waitlistSpaces);
        $view->assign('declinedWaitlistSpaces', $declinedWaitlistSpaces);
        $view->assign('cancelledWaitlistSpaces', $cancelledWaitlistSpaces);
        $view->assign('tableName', $tableName);
        $view->assign('filter', $filter);
        $view->assign('search', $this->search);
        $returnUrlParameters = [
            $this->searchArgumentName => $this->search,
        ];
        if ($this->entryId > 0) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->entryId;
        } elseif ($this->search[$this->entryIdArgumentName] ?? false) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->search[$this->entryIdArgumentName];
        }
        $view->assign('returnUrlParameters', $returnUrlParameters);

        return $view->renderResponse('EventBackend/Waitlist');
    }

    /**
     * list participants for an events
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'event',
    ])]
    public function calendarAction(Event $event, array $filter = [], ?int $startYear = null): ResponseInterface
    {
        $view = $this->moduleTemplateFactory->create($this->request);
        $storagePid = $event->getPid();

        $eventgroup = $event->getEventgroup();
        $this->eventRepository->setDefaultQuerySettings(
            $this->eventRepository->createQuery()->getQuerySettings()->setRespectStoragePage(false)
        );
        $events = $this->eventRepository->findBy([
            'eventgroup' => $eventgroup,
        ]);
        // TODO: CALENDAR
        $dataSource = [];
        $i = 0;
        $now = new DateTime();
        $startYear ??= $now->format('Y');

        /** @var Event $ev */
        foreach ($events as $ev) {
            //echo($ev->getDateFrom()->format('Y-m-d').'<br />');
            $editLinkParams = [
                'edit' => [
                    'tx_igruckzuckevent_domain_model_event' => [
                        $ev->getUid() => 'edit',
                    ],
                ],
                'returnUrl' => $this->request->getAttribute('normalizedParams')
                                             ->getRequestUri(),
                'recurrence' => [
                    'update_scope' => 'this',
                ],
            ];
            $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
            $editUri = (string)$uriBuilder->buildUriFromRoute('record_edit', $editLinkParams);
            if ($i == 0) {
                $startYear = $ev->getDateTimeFrom()
                                ->format('Y');
            }
            $registrations = $this->registrationRepository->findBy([
                'event' => $ev->getUid(),
            ]);
            $registrationsCount = $registrations->count();
            if ($registrationsCount > 0) {
                $color = $eventgroup->isEventExcluded($ev) ? '#c88' : '#f88';
                $title = ($eventgroup->isEventExcluded(
                    $ev
                ) ? LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.calendar.special_date_with_registrations') : LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.calendar.date_with_registrations')) . ' (' . $registrationsCount . ')';
            } elseif ($eventgroup->isEventExcluded($ev)) {
                $color = '#fc8';
                $title = LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.calendar.special_date_no_registrations');
            } else {
                $color = '#8f8';
                $title = LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.calendar.date_in_series');
            }

            $dataSource[] = [
                'name' => $ev->getDateText(),
                'startDate' => $ev->getDateTimeFrom()
                                  ->format('Y-m-d H:i:s'),
                'endDate' => $ev->getDateTimeTo() ? $ev->getDateTimeTo()
                                                       ->format('Y-m-d H:i:s') : $ev->getDateTimeFrom()
                                                                                    ->format('Y-m-d H:i:s'),
                'color' => $color,
                'title' => $title,
                'eventId' => $event->getUid(),
                'editUri' => $editUri,
                'deleteUri' => $registrations->count() == 0 ? $this->uriBuilder->reset()
                                                                               ->uriFor('deleteEvent', [
                                                                                   'event' => $ev->getUid(),
                                                                               ]) : '',
                'deleteMessage' => $registrations->count() == 0 ? LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:delete-event-confirm', 'ig_ruckzuckevent') . ' (' . $ev->getDateText() . ')' : '',
                'deleteTitle' => $registrations->count() == 0 ? LocalizationUtility::translate('LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:delete', 'core') : '',
                'buttonCloseText' => $registrations->count() == 0 ? LocalizationUtility::translate('LLL:EXT:core/Resources/Private/Language/locallang_common.xlf:cancel', 'core') : '',
                'buttonOkText' => $registrations->count() == 0 ? LocalizationUtility::translate('LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:delete', 'core') : '',
                'id' => $ev->getUid(),
                //'name' => $event->getTitle()
            ];
            $eventgroup->removeExcludeDate($ev->getDateFrom()->format('Y-m-d'));
            $i++;
        }

        // add deleted entries present in excludeDates
        $excludeDates = $eventgroup->getExcludeDatesArray();
        foreach ($excludeDates as $date) {
            $dateTime = (new DateTime($date))->format('Y-m-d');
            $dataSource[] = [
                'name' => 'deleted',
                'title' => LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.calendar.deleted_date'),
                'startDate' => $dateTime,
                'endDate' => $dateTime,
                'color' => '#eee',
                //'eventId' => $event->getUid(),
                //'editUri' => $editUri,
                //'deleteUri' => $registrations->count() == 0 ? $this->uriBuilder->reset()->uriFor('deleteEvent', ['event' => $ev->getUid()]) : '',
                //'deleteMessage' => $registrations->count() == 0 ? 'Termin vom ' . $ev->getDateText() . ' wirklich löschen?' : '',
                //'id' => $ev->getUid(),
                //'name' => $event->getTitle()
            ];
        }

        
        $view->assign('storagePid', $storagePid);
        $view->assign('startYear', $startYear);
        $view->assign('event', $event);
        $view->assign('eventgroup', $eventgroup);
        $view->assign('tableEventgroup', $this->tableEventgroup);
        $view->assign('events', $events);
        $view->assign('filter', $filter);
        $view->assign('dataSource', $dataSource);
        $view->assign('dataSourceJson', json_encode($dataSource));
        $view->assign('search', $this->search);
        if ($this->request->hasArgument('return')) {
            $returnUrlParameters = $this->request->getArgument('return');
        } else {
            $returnUrlParameters = [
                $this->searchArgumentName => $this->search,
            ];
        }
        if ($this->entryId > 0) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->entryId;
        }
        $view->assign('returnUrlParameters', $returnUrlParameters);

        return $view->renderResponse('EventBackend/Calendar');
    }

    /**
     * Helper method to write registrations to a worksheet
     */
    private function writeRegistrationsToSheet(Worksheet $worksheet, Event $event, array $registrations, string $sheetTitle, bool $includeWaitlistStatus = false): void
    {
        $worksheet->getCell('A1')->setValue($sheetTitle);
        $worksheet->getCell('B1')->setValue(
            LocalizationUtility::translate('tx_igruckzuckevent_domain_model_event.num_participants', 'IgRuckzuckevent') . ':'
        );
        $worksheet->getCell('C1')->setValue(count($registrations));
        $worksheet->getStyle('A1:C1')->applyFromArray(['font' => ['bold' => true]]);

        // Headers
        $columnIndex = 0;
        $row = 2;
        $headerKeys = [
            'tx_igruckzuckevent_domain_model_registration.gender',
            'tx_igruckzuckevent_domain_model_registration.first_name',
            'tx_igruckzuckevent_domain_model_registration.name',
            'tx_igruckzuckevent_domain_model_registration.street',
            'tx_igruckzuckevent_domain_model_registration.zip',
            'tx_igruckzuckevent_domain_model_registration.city',
            'tx_igruckzuckevent_domain_model_registration.phone',
            'tx_igruckzuckevent_domain_model_registration.phone_work',
            'tx_igruckzuckevent_domain_model_registration.email',
            'tx_igruckzuckevent_domain_model_registration.comment',
            'tx_igruckzuckevent_domain_model_registration.company',
            'tx_igruckzuckevent_domain_model_registration.company_street',
            'tx_igruckzuckevent_domain_model_registration.company_zip',
            'tx_igruckzuckevent_domain_model_registration.company_city',
            'tx_igruckzuckevent_domain_model_registration.bill',
            'date',
            'number-of-registrations',
            'additional_person',
        ];

        if ($includeWaitlistStatus) {
            $headerKeys[] = 'waitlist-status';
        }

        $headerKeys[] = 'hidden';

        foreach ($headerKeys as $key) {
            $columnLetter = Coordinate::stringFromColumnIndex(++$columnIndex);
            $worksheet->SetCellValue($columnLetter . $row, LocalizationUtility::translate($key, 'IgRuckzuckevent'));
        }

        if ($event->getAdditionalFormFields()->count()) {
            foreach ($event->getAdditionalFormFields() as $formField) {
                if ($formField->getType() != 'Title') {
                    $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . '2', $formField->getTitle());
                }
            }
        }

        $headerRowLastColumnIndex = $columnIndex;
        $headerRowLastColumnLetter = Coordinate::stringFromColumnIndex($headerRowLastColumnIndex);
        $worksheet->getStyle('A' . $row . ':' . $headerRowLastColumnLetter . '2')->getFont()->setBold(true);

        // Data rows
        $row++;
        foreach ($registrations as $registration) {
            $columnIndex = 0;
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, LocalizationUtility::translate(
                'tx_igruckzuckevent_domain_model_registration.gender.' . $registration->getGender(),
                'IgRuckzuckevent'
            ));

            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getFirstName());
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getName());
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getStreet());
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getZip());
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getCity());
            $worksheet->SetCellValueExplicit(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getPhone(), DataType::TYPE_STRING);
            $worksheet->SetCellValueExplicit(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getPhoneWork(), DataType::TYPE_STRING);
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getEmail());
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, preg_replace('#\R+#', "\n", $registration->getComment()));
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getCompany());
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getCompanyStreet());
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, preg_replace('#\R+#', ' ', $registration->getCompanyZip()));
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getCompanyCity());
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, LocalizationUtility::translate(
                'tx_igruckzuckevent_domain_model_registration.bill.' . ($registration->getBill() ? '1' : '0'),
                'IgRuckzuckevent'
            ));

            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, ExcelDate::PHPToExcel($registration->getCrdate()));
            $worksheet->getStyle(Coordinate::stringFromColumnIndex($columnIndex) . $row)
                      ->getNumberFormat()
                      ->setFormatCode('dd.mm.yyyy');

            $personCount = $registration->getHidden() ? 0 : $registration->getRegistrationCount();
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $personCount);
            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, implode("\n", $registration->getAdditionalPersonsNames()));

            if ($includeWaitlistStatus) {
                $statusText = 'Warteliste';
                // Check if there's a promotion status (PENDING=0, ACCEPTED=1, DECLINED=2, or CANCELLED=3)
                $promotionStatus = $registration->getWaitlistPromotionStatus();
                if ($promotionStatus !== null && $promotionStatus !== '') {
                    // Has a promotion status set, show the label
                    $statusText = $registration->getWaitlistPromotionStatusEnum()->getLabel();
                }
                $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $statusText);
            }

            $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $registration->getHidden() ? '1' : '');

            if ($registration->getAdditionalFields()) {
                foreach ($event->getAdditionalFormFields() as $formField) {
                    foreach ($registration->getAdditionalFields() as $fieldname => $additionalField) {
                        if ('additional-' . $formField->getUid() == $fieldname) {
                            if ($formField->getType() == 'Checkbox') {
                                $index = 0;
                                $v = '';
                                foreach ($additionalField as $value) {
                                    if ($index != 0) {
                                        $v .= ', ';
                                    }
                                    $v .= $formField->getOptionsArray()[$value];
                                    $index++;
                                }
                                $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $v);
                            } elseif ($formField->getType() == 'Radio' || $formField->getType() == 'Dropdown') {
                                $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, $formField->getOptionsArray()[$additionalField] ?? '');
                            } elseif ($formField->getType() != 'Title') {
                                $worksheet->SetCellValue(Coordinate::stringFromColumnIndex(++$columnIndex) . $row, preg_replace('#\R+#', "\n", (string) $additionalField));
                            }
                        }
                    }
                }
            }

            $row++;
        }

        // Set column width automatically
        for ($i = 1; $i <= $headerRowLastColumnIndex; $i++) {
            $columnLetter = Coordinate::stringFromColumnIndex($i);
            $worksheet->getColumnDimension($columnLetter)->setAutoSize(true);
        }
    }

    /**
     * list participants for an events
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'event',
    ])]
    public function exportRegistrationsAction(Event $event): ResponseInterface
    {
        $phpExcel = new Spreadsheet();
        $phpExcel->removeSheetByIndex(0);

        // Separate regular and waitlist registrations
        $regularRegistrations = [];
        $waitlistRegistrations = [];
        foreach ($event->getRegistrations() as $registration) {
            if ($registration->getWaitlist()) {
                $waitlistRegistrations[] = $registration;
            } else {
                $regularRegistrations[] = $registration;
            }
        }

        // Sheet 1: Regular participants
        $worksheet = new Worksheet($phpExcel, 'Teilnehmer');
        $phpExcel->addSheet($worksheet, 0);
        $this->writeRegistrationsToSheet($worksheet, $event, $regularRegistrations, $event->getTitle(), false);

        // Sheet 2: Waitlist (if there are any)
        // Use repository method to get waitlist with same ordering as backend view
        $waitlistRegistrationsOrdered = $this->registrationRepository->findWaitlistByEvent($event, true);
        if ($waitlistRegistrationsOrdered->count() > 0) {
            $waitlistSheet = new Worksheet($phpExcel, 'Warteliste');
            $phpExcel->addSheet($waitlistSheet, 1);
            // Convert query result to array for writeRegistrationsToSheet
            $waitlistArray = $waitlistRegistrationsOrdered->toArray();
            $this->writeRegistrationsToSheet($waitlistSheet, $event, $waitlistArray, $event->getTitle() . ' - Warteliste', true);
        }

        $fileName = 'kurs-' . $event->getUid() . '-' . time() . '.xlsx'; // Temporary filename

        header('Content-Type: application/force-download');
        header('Content-Description: File Transfer');

        header('Content-Type: application/octet-stream');
        header('Content-Type: application/download');
        header('Content-Disposition: attachment; filename=' . $fileName);
        header('Content-Transfer-Encoding: binary');
        header('Connection: Keep-Alive');
        header('Expires: 0');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Pragma: public');
        //header('Content-Length: 30000000000000000');//".filesize($filename));

        // Set the first sheet as active before saving
        $phpExcel->setActiveSheetIndex(0);

        $objWriter = new Xlsx($phpExcel);
        $tmpname = tempnam('', 'tempfile');
        $objWriter->save($tmpname);
        header('Content-Length: ' . filesize($tmpname));
        unlink($tmpname);
        $objWriter->save('php://output');
        exit(0);

        return $this->htmlResponse();
    }


    /**
     * check participants for an events
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'event',
    ])]
    public function checkRegistrationsAction(Event $event): ResponseInterface
    {
        $results = [];
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_igruckzuckevent_domain_model_registration');

        $checks = [
            [
                'sql' => 'SELECT DISTINCT first_name AS firstName,name AS lastName,r.company,r.uid FROM tx_igruckzuckevent_domain_model_registration  AS r WHERE r.event=6 AND r.deleted=0 AND EXISTS (SELECT 1 FROM tx_igruckzuckevent_domain_model_registration AS jr WHERE jr.name=r.name AND  r.first_name=jr.first_name AND jr.deleted=0 AND jr.event=6 AND jr.uid<r.uid AND jr.hidden=0) AND r.hidden=0 ORDER BY first_name,name;',
                'title' => 'Doppelte Anmeldung',
            ],
            [
                'sql' => 'SELECT p.first_name AS firstName,p.last_name AS lastName,r.first_name,r.name,r.company,r.uid  FROM tx_igruckzuckevent_domain_model_person AS p JOIN tx_igruckzuckevent_domain_model_registration  AS r ON r.uid=registration AND r.event=6 AND r.hidden=0 AND r.deleted=0 AND p.deleted=0  WHERE EXISTS (SELECT 1 FROM tx_igruckzuckevent_domain_model_registration AS jr WHERE jr.name=p.last_name AND  p.first_name=jr.first_name AND jr.deleted=0 AND jr.event=6 AND jr.hidden=0);',
                'title' => 'Zusätzliche Person existiert als Anmeldung',
            ],
            [
                'sql' => 'SELECT p.first_name AS firstName,p.last_name AS lastName,r.first_name,r.name,r.company,r.uid  FROM tx_igruckzuckevent_domain_model_person AS p JOIN tx_igruckzuckevent_domain_model_registration  AS r ON r.uid=registration AND r.event=6 AND r.deleted=0 AND p.deleted=0  WHERE EXISTS (SELECT 1 FROM tx_igruckzuckevent_domain_model_person AS jp JOIN tx_igruckzuckevent_domain_model_registration  AS jr ON jr.uid=jp.registration AND jr.event=6 AND jr.hidden=0 AND jr.deleted=0 AND jp.deleted=0 AND jp.last_name=p.last_name AND  jp.first_name=p.first_name AND jp.uid<p.uid);',
                'title' => 'Zusätzliche Person existiert doppelt',
            ],
        ];
        foreach ($checks as $check) {
            $stmt = $conn->executeQuery($check['sql']);
            $entries = $stmt->fetchAllAssociative();
            if (!empty($entries)) {
                $results[] = [
                    'title' => $check['title'],
                    'entries' => $entries,
                ];
            }
        }
        $view = $this->moduleTemplateFactory->create($this->request);
        $tableName = 'tx_igruckzuckevent_domain_model_registration';

        $view->assignMultiple([
            'results' => $results,
            'event' => $event,
            'tableName' => $tableName,
            'currentAction' => $this->request->getControllerActionName(),
        ]);
        return $view->renderResponse('EventBackend/CheckRegistrations');
    }

    /**
     * Delete a registration
     *
     * @param Registration $registration
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'registration',
    ])]
    public function deleteRegistrationAction(Registration $registration, string $returnAction = 'listParticipants'): ResponseInterface
    {
        $event = $registration->getEvent();
        $this->registrationRepository->remove($registration);
        return $this->redirect($returnAction, null, null, [
            'event' => $event->getUid(),
        ]);
    }

    /**
     * Delete the registration of an aevent
     *
     * @param Event $event
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'event',
    ])]
    public function deleteRegistrationsAction(Event $event): ResponseInterface
    {
        foreach ($event->getRegistrations() as $registration) {
            $this->registrationRepository->remove($registration);
        }
        return $this->redirect('listParticipants', null, null, [
            'event' => $event->getUid(),
        ]);
    }

    /**
     * Delete the registration of an aevent
     *
     * @param Event $event
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'event',
    ])]
    public function deleteEventAction(Event $event): ResponseInterface
    {
        if (!$event->getRegistrationsCount()) {
            $this->eventRepository->remove($event);
        }
        return $this->redirect('index');
    }


    protected function viewAddMenu($view)
    {
        // Create a menu in the module docheader
        $menu = $view->getDocHeaderComponent()->getMenuRegistry()->makeMenu();
        $menu->setIdentifier('eventModuleMenu');

        // Define your actions
        $actions = [
            'index' => ['label' => 'Zurück', 'icon' => 'actions-view-go-back'],
            'listParticipants' => ['label' => 'Teilnehmer Liste', 'icon' => 'actions-list'],
            'exportRegistrations' => ['label' => 'Teilnehmer exportieren (.xlsx)', 'icon' => 'actions-database-export'],
            'checkRegistrations' => ['label' => 'Teilnehmer überprüfen', 'icon' => 'actions-exclamation-circle'],
        ];

        $currentAction = $this->request->getControllerActionName();

        // Add each action as a menu item
        foreach ($actions as $actionName => $data) {
            $menuItem = $menu->makeMenuItem()
                             ->setTitle($data['label'])
                             ->setHref($this->uriBuilder->reset()->uriFor($actionName/*, ['event' => $event->getUid()]*/)) // TODO: add event param if needed
                             ->setActive($currentAction === $actionName);

            $menu->addMenuItem($menuItem);
        }

        // Register the menu
        $view->getDocHeaderComponent()->getMenuRegistry()->addMenu($menu);
    }

    /**
     * Promote a waitlist registration to a regular registration
     *
     * @param Registration $registration
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'registration',
    ])]
    public function promoteWaitlistAction(Registration $registration): ResponseInterface
    {
        $event = $registration->getEvent();

        if ($this->waitlistService->promoteRegistration($registration)) {
            $this->addFlashMessage(
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_promotion_success'),
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_updated'),
                \TYPO3\CMS\Core\Type\ContextualFeedbackSeverity::OK
            );
        } else {
            $this->addFlashMessage(
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_promotion_error'),
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.error'),
                \TYPO3\CMS\Core\Type\ContextualFeedbackSeverity::ERROR
            );
        }

        return $this->redirect('waitlist', null, null, [
            'event' => $event->getUid(),
        ]);
    }

    /**
     * Automatically promote waitlist registrations when space becomes available
     *
     * @param Event $event
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'event',
    ])]
    public function propagateWaitlistAction(Event $event): ResponseInterface
    {
        $promoted = $this->waitlistService->propagateWaitlist($event);

        if ($promoted > 0) {
            $this->addFlashMessage(
                sprintf(LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_propagate_success'), $promoted),
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_updated'),
                \TYPO3\CMS\Core\Type\ContextualFeedbackSeverity::OK
            );
        } else {
            $this->addFlashMessage(
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_propagate_none'),
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.no_changes'),
                \TYPO3\CMS\Core\Type\ContextualFeedbackSeverity::INFO
            );
        }

        return $this->redirect('waitlist', null, null, [
            'event' => $event->getUid(),
        ]);
    }

    /**
     * Cancel a waitlist promotion (admin action)
     *
     * @param Registration $registration
     */
    #[AnnotationIgnoreValidation([
        'argumentName' => 'registration',
    ])]
    public function cancelWaitlistPromotionAction(Registration $registration): ResponseInterface
    {
        $event = $registration->getEvent();

        if ($this->waitlistService->cancelPromotion($registration)) {
            $this->addFlashMessage(
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_promotion_cancelled'),
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_updated'),
                \TYPO3\CMS\Core\Type\ContextualFeedbackSeverity::OK
            );
        } else {
            $this->addFlashMessage(
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.waitlist_promotion_cancel_error'),
                LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:backend.flash.error'),
                \TYPO3\CMS\Core\Type\ContextualFeedbackSeverity::ERROR
            );
        }

        return $this->redirect('waitlist', null, null, [
            'event' => $event->getUid(),
        ]);
    }

    /**
     * Calculate total spaces for a collection of registrations
     * Takes into account used_spaces and additional persons
     *
     * @param \TYPO3\CMS\Extbase\Persistence\QueryResultInterface|array $registrations
     * @return int
     */
    protected function calculateTotalSpaces($registrations): int
    {
        $totalSpaces = 0;
        foreach ($registrations as $registration) {
            $totalSpaces += $registration->getRegistrationCount();
        }
        return $totalSpaces;
    }

}
