<?php

namespace Ig\IgRuckzuckevent\Controller;

use GeorgRinger\News\Pagination\QueryResultPaginator;
use GeorgRinger\NumberedPagination\NumberedPagination;
use Ig\IgRuckzuckevent\Controller\TYPO3\CMS\Extbase\Annotation\Validate;
use Ig\IgRuckzuckevent\Domain\Model\FeUser;
use Ig\IgRuckzuckevent\Domain\Model\Subscription;
use Ig\IgRuckzuckevent\Domain\Repository\FeUserRepository;
use Ig\IgRuckzuckevent\Domain\Repository\SubscriptionOfferRepository;
use Ig\IgRuckzuckevent\Domain\Repository\SubscriptionRepository;
use Ig\IgRuckzuckevent\Domain\Validator\PasswordChangeValidator;
use Ig\IgRuckzuckevent\Domain\Validator\PasswordConfirmationValidator;
use Ig\IgRuckzuckevent\Domain\Validator\UserExistsValidator;
use Ig\IgRuckzuckevent\Domain\Validator\UserRegistrationFormValidator;
use Ig\IgRuckzuckevent\Utility\UserUtility;
use Ig\IgRuckzuckevent\Service\MailService;
use Internetgalerie\IgDynval\Controller\DynamicValidationActionController;
use Internetgalerie\IgDynval\Validation\Validator\DynamicValidator;
use Internetgalerie\IgFrontendUser\Domain\Repository\FrontendUserGroupRepository;
use Internetgalerie\IgFrontendUser\Domain\Repository\FrontendUserRepository;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Crypto\PasswordHashing\PasswordHashFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Annotation\Validate as AnnotationValidate;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Extbase\Property\TypeConverter\DateTimeConverter;
use TYPO3\CMS\Fluid\View\StandaloneView;

/**
 * The EventController handles all actions directly related to Events.
 */
class UserController extends ActionController
{
    use DynamicValidationActionController;

    /**
     * feUserRepository
     *
     * @var FeUserRepository
     */
    protected $feUserRepository = null;

    /**
     * feUserGroupRepository
     *
     * @var FrontendUserGroupRepository
     */
    protected $feUserGroupRepository = null;

    /**
     * productRepository
     *
     * @var SubscriptionOfferRepository
     */
    protected $subscriptionOfferRepository = null;

    /**
     * subscriptionRepository
     *
     * @var SubscriptionRepository
     */
    protected $subscriptionRepository = null;

    /**
     * mailService
     *
     * @var MailService
     */
    protected $mailService = null;

    public function injectFeUserRepository(FeUserRepository $feUserRepository): void
    {
        $this->feUserRepository = $feUserRepository;
    }

    /**
     * @param FrontendUserRepository $feUserGroupRepository
     */
    public function injectFeUserGroupRepository(FrontendUserGroupRepository $feUserGroupRepository): void
    {
        $this->feUserGroupRepository = $feUserGroupRepository;
    }

    public function injectSubscriptionOfferRepository(SubscriptionOfferRepository $subscriptionOfferRepository): void
    {
        $this->subscriptionOfferRepository = $subscriptionOfferRepository;
    }

    public function injectSubscriptionRepository(SubscriptionRepository $subscriptionRepository): void
    {
        $this->subscriptionRepository = $subscriptionRepository;
    }

    public function injectMailService(MailService $mailService): void
    {
        $this->mailService = $mailService;
    }

    public function initializeAction(): void
    {
        // Setup mail service
        $this->mailService->setSettings($this->settings);
        $this->mailService->setRequest($this->request);

        if (isset($this->arguments['subscription']) || isset($this->arguments['newSubscription'])) {
            if (isset($this->arguments['subscription'])) {
                $subscription = $this->arguments['subscription'];
            } else {
                $subscription = $this->arguments['newSubscription'];
            }

            // Set date format of date fields
            foreach (['startDate'] as $dateField) {
                $subscription->getPropertyMappingConfiguration()
                ->forProperty($dateField)
                ->setTypeConverterOption(
                    DateTimeConverter::class,
                    DateTimeConverter::CONFIGURATION_DATE_FORMAT,
                    'd.m.Y'
                );
            }
        }
    }

    public function listAction(): ResponseInterface
    {
        if (!UserUtility::isAdminLoggedIn()) {
            exit(0);
        }

        if ($this->request->hasArgument('search')) {
            $search = $this->request->getArgument('search');
        } else {
            $search = [];
        }
        if ($this->request->hasArgument('user')) {
            $user = $this->request->getArgument('user');
        } else {
            $user = null;
        }
        $this->view->assign('search', $search);

        //$users = $this->feUserRepository->findByUsergroup(4);
        $users = $this->feUserRepository->findWithSearch(4, $search, $user);

        $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument(
            'currentPage'
        ) : 1;
        $paginator = new QueryResultPaginator($users, $currentPage, 20);
        $pagination = new NumberedPagination($paginator, 10); // zweiter Argument: maximal Anzahl Links

        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);

        $this->view->assign('users', $users);

        return $this->htmlResponse();
    }

    /**
     * Shows edit user form
     */
    public function showAction(?FeUser $user = null): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        if (!$user && $userId) {
            $user = $this->feUserRepository->findByUid($userId);
        }

        if ($isLoggedIn && (UserUtility::isAdminLoggedIn() || $user->getUid() == $userId)) {
            $this->view->assign('user', $user);
        }

        return $this->htmlResponse();
    }

    /**
     * Shows the registration form
     *
     * @param string $referer
     */
    public function newAction($referer = ''): ResponseInterface
    {
        $this->view->assign('referer', $referer);

        return $this->htmlResponse();
    }

    /**
     * Inserts a new user
     *
     * @param string $passwordConfirmation
     * @param string $referer
     */
    #[AnnotationValidate([
        'validator' => DynamicValidator::class,
        'param' => 'newUser',
    ])]
    #[AnnotationValidate([
        'validator' => UserExistsValidator::class,
        'param' => 'newUser',
    ])]
    #[AnnotationValidate([
        'validator' => UserRegistrationFormValidator::class,
        'param' => 'newUser',
    ])]
    #[AnnotationValidate([
        'validator' => 'NotEmpty',
        'param' => 'password',
    ])]
    #[AnnotationValidate([
        'validator' => PasswordConfirmationValidator::class,
        'param' => 'password',
    ])]
    public function createAction(
        FeUser $newUser,
        string $password,
        $passwordConfirmation,
        $referer = ''
    ): ResponseInterface {
        $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
        $newUser->setPid(5);
        $newUser->setUsername($newUser->getEmail());
        $user = $this->feUserRepository->findHiddenByUsername($newUser->getUsername());

        if ($user && $user->getDisable()) {
            $this->feUserRepository->remove($user);
        }

        $hashFactory = GeneralUtility::makeInstance(PasswordHashFactory::class);
        $hashInstance = $hashFactory->getDefaultHashInstance('FE');
        $hashedPassword = $hashInstance->getHashedPassword($password);
        $newUser->setPassword($hashedPassword);

        $newUser->setHash(md5(uniqid()));

        if (UserUtility::isAdminLoggedIn()) {
            $newUser->setDisable(false);
            $newUser->addUsergroup($this->feUserGroupRepository->findByUid(4));
        }

        $this->feUserRepository->add($newUser);
        $persistenceManager->persistAll();

        if (!UserUtility::isAdminLoggedIn()) {
            $this->mailService->sendMail(
                'Physio-Team Naderi: Bestätigung der E-Mail-Adresse',
                [$this->settings['debugMail'] ?: $this->settings['mailFrom']],
                [$this->settings['debugMail'] ?: $newUser->getEmail()],
                'IgRuckzuckevent/User/New/User',
                [
                    'user' => $newUser,
                    'referer' => $referer,
                ]
            );

            return $this->redirect('reminder', 'User', 'IgRuckzuckevent', [
                'user' => $newUser->getUid(),
            ]);
        }
        return $this->redirect('list');
    }

    /**
     * Reminds the user to check his email
     *
     * @param integer $user
     */
    public function reminderAction($user): ResponseInterface
    {
        $this->view->assign('user', $this->feUserRepository->findHiddenByUid($user));

        return $this->htmlResponse();
    }

    /**
     * Inserts a new user and sends mail to admin
     *
     * @param integer $user
     * @param string $hash
     * @param string $referer
     */
    public function confirmationAction($user, $hash, $referer = ''): ResponseInterface
    {
        $user = $this->feUserRepository->findHiddenByUid($user);

        if ($user->getDisable() && $user->getHash() == $hash) {
            $usergroup = $this->feUserGroupRepository->findByUid(4);

            $user->setDisable(false);
            $user->addUsergroup($usergroup);

            $this->feUserRepository->update($user);

            $this->mailService->sendMail(
                'Physio-Team Naderi: Neuer Benutzer hat sich angemeldet',
                [$this->settings['debugMail'] ?: $this->settings['mailFrom']],
                [$this->settings['debugMail'] ?: $this->settings['mailFrom']],
                'IgRuckzuckevent/User/Create/Admin',
                ['user' => $user]
            );

            $this->mailService->sendMail(
                'Physio-Team Naderi: Vielen Dank für Ihre Anmeldung',
                [$this->settings['debugMail'] ?: $this->settings['mailFrom']],
                [$this->settings['debugMail'] ?: $user->getEmail()],
                'IgRuckzuckevent/User/Create/User',
                ['user' => $user]
            );

            $this->view->assign('user', $user);
            $this->view->assign('referer', $referer);
        }

        return $this->htmlResponse();
    }

    /**
     * Shows edit user form
     */
    public function editAction(?FeUser $user = null): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        if (!$user && $userId) {
            $user = $this->feUserRepository->findByUid($userId);
        }

        if ($isLoggedIn && (UserUtility::isAdminLoggedIn() || $user->getUid() == $userId)) {
            $this->view->assign('user', $user);
        }

        return $this->htmlResponse();
    }

    /**
     * Updates a user
     */
    #[AnnotationValidate([
        'validator' => DynamicValidator::class,
        'param' => 'user',
    ])]
    #[AnnotationValidate([
        'validator' => UserRegistrationFormValidator::class,
        'param' => 'user',
    ])]
    public function updateAction(FeUser $user): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        if ($isLoggedIn && (UserUtility::isAdminLoggedIn() || $user->getUid() == $userId)) {
            $user->setUsername($user->getEmail());
            $this->feUserRepository->update($user);
        }
        if (!UserUtility::isAdminLoggedIn()) {
            return $this->redirect('show', 'User');
        }
        return $this->redirect('list', 'User');
    }

    /**
     * Shows password change form for a user
     */
    public function passwordChangeAction(?FeUser $user = null): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        if (!$user && $userId) {
            $user = $this->feUserRepository->findByUid($userId);
        }

        if ($isLoggedIn && (UserUtility::isAdminLoggedIn() || $user->getUid() == $userId)) {
            $this->view->assign('user', $user);
        }

        return $this->htmlResponse();
    }

    /**
     * Updates the password of a user
     */
    #[AnnotationValidate([
        'validator' => DynamicValidator::class,
        'param' => 'user',
    ])]
    #[AnnotationValidate([
        'validator' => PasswordChangeValidator::class,
        'param' => 'user',
    ])]
    #[AnnotationValidate([
        'validator' => 'NotEmpty',
        'param' => 'oldPassword',
    ])]
    #[AnnotationValidate([
        'validator' => 'NotEmpty',
        'param' => 'newPassword',
    ])]
    #[AnnotationValidate([
        'validator' => PasswordConfirmationValidator::class,
        'param' => 'newPassword',
    ])]
    public function updatePasswordAction(
        FeUser $user,
        string $oldPassword,
        string $newPassword,
        string $passwordConfirmation
    ): ResponseInterface {
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        if ($isLoggedIn && (UserUtility::isAdminLoggedIn() || $user->getUid() == $userId)) {
            $hashFactory = GeneralUtility::makeInstance(PasswordHashFactory::class);
            $hashInstance = $hashFactory->getDefaultHashInstance('FE');
            $hashedPassword = $hashInstance->getHashedPassword($newPassword);
            $user->setPassword($hashedPassword);
            $this->feUserRepository->update($user);
        }
        if (!UserUtility::isAdminLoggedIn()) {
            return $this->redirect('show', 'User');
        }
        return $this->redirect('list', 'User');
    }

    public function deleteAction(FeUser $user): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        if ($isLoggedIn && (UserUtility::isAdminLoggedIn() || $user->getUid() == $userId)) {
            if ($user->getActiveSubscriptions()->count() == 0) {
                $this->feUserRepository->remove($user);
            }
        }
        return $this->redirect('list');
    }

    public function newSubscriptionAction(FeUser $user): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        if ($isLoggedIn && (UserUtility::isAdminLoggedIn() || $user->getUid() == $userId)) {
            $this->view->assign('user', $user);
            $this->view->assign('subscriptionOffers', $this->subscriptionOfferRepository->findAll());
        }

        return $this->htmlResponse();
    }

    public function createSubscriptionAction(Subscription $subscription): ResponseInterface
    {
        /*
        if( !$this->request->hasArgument('agb') || $this->request->getArgument('agb')==0)
    $this->forward('newSubscription');
        */
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        if ($isLoggedIn && (UserUtility::isAdminLoggedIn() || $subscription->getFeUser()->getUid() == $userId)) {
            $inThreeMonths = clone $subscription->getStartDate();
            $inThreeMonths->modify('+' . $subscription->getSubscriptionOffer()->getDuration() . ' month');

            $subscription->setExpirationDate($inThreeMonths);
            $subscription->setDaysLeft($subscription->getSubscriptionOffer()->getDays());

            $subscription->getFeUser()
->addSubscription($subscription);

            $this->subscriptionRepository->add($subscription);
            $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
            $persistenceManager->persistAll();
        }
        if (UserUtility::isAdminLoggedIn()) {
            $uriBuilder = $this->uriBuilder;
            $uri = $uriBuilder->setTargetPageUid(168)
->setArguments([
    'tx_igruckzuckevent_managementuser' => [
        'user' => $subscription->getFeUser(),
    ],
])->build();
            return $this->redirectToURI($uri, $delay = 0, $statusCode = 303);
        }
        return $this->redirect('list');
    }

    public function deleteSubscriptionAction(Subscription $subscription): ResponseInterface
    {
        if (UserUtility::isAdminLoggedIn()) {
            if ($subscription->getDaysLeft() == $subscription->getSubscriptionOffer()->getDays() || $subscription->getDaysLeft() == 0) {
                $this->subscriptionRepository->remove($subscription);
                $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
                $persistenceManager->persistAll();
            }
        }
        return $this->redirect('newSubscription', 'User', 'IgRuckzuckevent', [
            'user' => $subscription->getFeUser(),
        ]);
    }

    public function exportXlsAction(FeUser $user): ResponseInterface
    {
        $registrations = $user->getPassedRegistrations()
->toArray();

        //$phpExcelService = GeneralUtility::makeInstanceService('phpexcel');
        $phpExcel = new Spreadsheet();
        $phpExcel->setActiveSheetIndex(0);
        $row = 1;

        $phpExcel->getActiveSheet()
->SetCellValue('A' . $row, 'Name');
        $phpExcel->getActiveSheet()
->SetCellValue('B' . $row, 'Vorname');
        $phpExcel->getActiveSheet()
->SetCellValue('C' . $row, 'E-Mail-Adresse');
        $phpExcel->getActiveSheet()
->SetCellValue('D' . $row, 'Telefon');

        $phpExcel->getActiveSheet()
->getStyle('A' . $row . ':D' . $row)->getFont()->setBold(true);

        $row++;

        $phpExcel->getActiveSheet()
->SetCellValue('A' . $row, $user->getLastName());
        $phpExcel->getActiveSheet()
->SetCellValue('B' . $row, $user->getFirstName());
        $phpExcel->getActiveSheet()
->SetCellValue('C' . $row, $user->getEmail());
        $phpExcel->getActiveSheet()
->SetCellValueExplicit('D' . $row, $user->getTelephone(), DataType::TYPE_STRING);

        $row++;
        $row++;

        $phpExcel->getActiveSheet()
->SetCellValue('A' . $row, 'Kurs');
        $phpExcel->getActiveSheet()
->SetCellValue('B' . $row, 'Datum');
        $phpExcel->getActiveSheet()
->SetCellValue('C' . $row, 'Zeit von');
        $phpExcel->getActiveSheet()
->SetCellValue('D' . $row, 'Zeit bis');
        $phpExcel->getActiveSheet()
->SetCellValue('E' . $row, 'Bezahlmethode/Abo');
        $phpExcel->getActiveSheet()
->getStyle('A' . $row . ':E' . $row)->getFont()->setBold(true);

        $row++;

        usort($registrations, fn ($a, $b) => $a->getEvent()->getDateTimeFrom() > $b->getEvent()->getDateTimeFrom());

        foreach ($registrations as $registration) {
            // First row: Output event title and registration count
            $phpExcel->getActiveSheet()
->SetCellValue('A' . $row, $registration->getEvent()->getTitle());

            // Output registration count
            $phpExcel->getActiveSheet()
->SetCellValue('B' . $row, html_entity_decode($registration->getEvent()->getDateFrom()->format('d.m.Y')));
            $phpExcel->getActiveSheet()
->SetCellValue('C' . $row, html_entity_decode($registration->getEvent()->getDateTimeFrom()->format('H:i')));
            $phpExcel->getActiveSheet()
->SetCellValue('D' . $row, html_entity_decode($registration->getEvent()->getDateTimeTo()->format('H:i')));

            // Output registration count
            if ($registration->getPaymentMethod() == 'office') {
                $phpExcel->getActiveSheet()
->SetCellValue('E' . $row, 'Sekretariat');
            } else {
                $phpExcel->getActiveSheet()
->SetCellValue('E' . $row, html_entity_decode(
    $registration->getSubscription()
->getSubscriptionOffer()
->getName() . ', '
                    . $registration->getSubscription()->getStartDate()->format('d.m.Y')
                    . ' bis '
                    . $registration->getSubscription()->getExpirationDate()->format('d.m.Y')
));
            }
            $row++;
        }

        // Set column width automatically according to their content
        foreach (range('A', 'E') as $column) {
            $phpExcel->getActiveSheet()
->getColumnDimension($column)
->setAutoSize(true);
        }

        // Temporary fielname
        $fileName = 'typo3temp/events-user-' . $user->getUid() . '-' . time() . '.xlsx';

        // Create an Excel Writer
        $excelWriter = new Xlsx($phpExcel); // Create an Excel Writer
        // Save to temporary file
        $excelWriter->save($fileName);

        // Set headers
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . str_replace('typo3temp/', '', $fileName) . '"');
        header('Cache-Control: max-age=0');
        // If you're serving to IE 9, then the following may be needed
        header('Cache-Control: max-age=1');
        // If you're serving to IE over SSL, then the following may be needed
        header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
        // Date in the past
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT');
        // always modified
        header('Cache-Control: cache, must-revalidate');
        // HTTP/1.1
        header('Pragma: public');
        // HTTP/1.0
        header('Content-Length: ' . filesize($fileName));
        //header('Pragma: no-cache');

        echo file_get_contents($fileName);
        // Output file

        unlink($fileName);
        // Delete temporary file

        exit();

        return $this->htmlResponse();
    }
}
