<?php

declare(strict_types=1);

namespace Ig\IgRuckzuckevent\Domain\Model;

/**
 * Waitlist Promotion Status Enum
 *
 * Represents the different states of a waitlist promotion
 */
enum WaitlistPromotionStatus: int
{
    /**
     * Promotion sent, waiting for user response
     */
    case PENDING = 0;

    /**
     * User accepted the promotion
     */
    case ACCEPTED = 1;

    /**
     * User declined the promotion
     */
    case DECLINED = 2;

    /**
     * Admin cancelled the promotion
     */
    case CANCELLED = 3;

    /**
     * Get human-readable label for the status
     */
    public function getLabel(): string
    {
        return match($this) {
            self::PENDING => 'Promotion ausstehend',
            self::ACCEPTED => 'Promotion akzeptiert',
            self::DECLINED => 'Promotion abgelehnt',
            self::CANCELLED => 'Promotion abgebrochen',
        };
    }

    /**
     * Check if the promotion is still active (can be accepted/declined)
     */
    public function isActive(): bool
    {
        return $this === self::PENDING;
    }

    /**
     * Check if the promotion was completed (either accepted or declined)
     */
    public function isCompleted(): bool
    {
        return $this === self::ACCEPTED || $this === self::DECLINED;
    }

    /**
     * Check if the promotion can be cancelled
     */
    public function isCancellable(): bool
    {
        return $this === self::PENDING;
    }
}
