<?php

namespace Ig\IgRuckzuckevent\Domain\Repository;

use DateTime;
use Ig\IgRuckzuckevent\Domain\Model\Event;
use Ig\IgRuckzuckevent\Domain\Model\Person;
use Ig\IgRuckzuckevent\Domain\Model\WaitlistPromotionStatus;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use TYPO3\CMS\Extbase\Persistence\Repository;

/**
 * The repository for Registrations
 */
class RegistrationRepository extends Repository
{
    /**
     * @var array
     */
    protected $defaultOrderings = [
        'crdate' => QueryInterface::ORDER_DESCENDING,
    ];

    public function initializeObject(): void
    {
        $querySettings = $this->createQuery()
->getQuerySettings();

        //$querySettings->setLanguageUid(0);
        $querySettings->setRespectSysLanguage(false);
        $querySettings->setRespectStoragePage(false);

        $this->setDefaultQuerySettings($querySettings);
    }

    public function getTableName(): string
    {
        $dataMapper = GeneralUtility::makeInstance(DataMapper::class);
        return $dataMapper->getDataMap($this->objectType)
->getTableName();
    }
    public function getTableNamePerson(): string
    {
        $dataMapper = GeneralUtility::makeInstance(DataMapper::class);
        return $dataMapper->getDataMap(Person::class)->getTableName();
    }
    
    public function setStoragePid($pid): void
    {
        $querySettings = $this->createQuery()
->getQuerySettings();

        $querySettings->setStoragePageIds([$pid]);

        $this->setDefaultQuerySettings($querySettings);
    }

    public function findByEventAndFilter($event, array $filter = [])
    {
        $query = $this->createQuery();

        $constraints = [];
        $constraints[] = $query->equals('event', $event);

        if (isset($filter['waitlist'])) {
            $constraints[] = $query->equals('waitlist', $filter['waitlist']);
        }

        if ($filter['searchword'] ?? false) {
            $keywordsArray = GeneralUtility::trimExplode(' ', $filter['searchword'], true);
            foreach ($keywordsArray as $keyword) {
                $constraints[] = $query->logicalOr(
                    $query->like('name', '%' . $keyword . '%'),
                    $query->like('firstName', '%' . $keyword . '%'),
                    $query->like('street', '%' . $keyword . '%'),
                    $query->like('zip', '%' . $keyword . '%'),
                    $query->like('city', '%' . $keyword . '%'),
                    $query->like('email', '%' . $keyword . '%'),
                    $query->like('additionalPersons.lastName', '%' . $keyword . '%'),
                    $query->like('additionalPersons.firstName', '%' . $keyword . '%'),
                );
            }
        }

        $query->matching($query->logicalAnd(...$constraints));

        return $query->execute();
    }

    /**
     * Count registrations by event
     *
     * @param int $eventId
     * @param bool $includePendingPromotions If true, includes ALL active waitlist registrations (for calculating free places, to reserve spots). If false, only counts confirmed registrations (for participant display)
     * @return int
     */
    public function countRegistrationsByEvent($eventId, bool $includePendingPromotions = false)
    {
        $tablenameRegistration = $this->getTableName();
        $tablenamePerson = $this->getTableNamePerson();
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)
                      ->getQueryBuilderForTable($tablenameRegistration);

        $subQuery = $queryBuilder->getConnection()
->createQueryBuilder();
        $subQuery
            ->addSelectLiteral('COUNT(1)')
            ->from($tablenamePerson)
            ->where(
                $subQuery->expr()
->eq('registration', 'r.uid'),
                //$subQuery->expr()->eq('deleted', 0)
            );

        $queryBuilder
            //->select('r.event')
            ->addSelectLiteral('SUM((1 + (' . $subQuery->getSQL() . ')) * used_spaces) AS total')
            ->from($tablenameRegistration, 'r')
            ->where(
                $queryBuilder->expr()
->eq('r.event', $queryBuilder->createNamedParameter($eventId, Connection::PARAM_INT))
            );

        if ($includePendingPromotions) {
            // Include regular registrations AND ALL active waitlist entries (to reserve spots for them)
            // Waitlist people have "dibs" on spots, so they should count against free places
            // This prevents new registrations from taking spots that should go to waitlist
            $now = new \DateTime();
            $queryBuilder->andWhere(
                $queryBuilder->expr()->or(
                    // Confirmed registrations
                    $queryBuilder->expr()->eq('r.waitlist', $queryBuilder->createNamedParameter(0, Connection::PARAM_INT)),
                    // Waitlist with no promotion yet (regular waitlist, have dibs)
                    $queryBuilder->expr()->and(
                        $queryBuilder->expr()->eq('r.waitlist', $queryBuilder->createNamedParameter(1, Connection::PARAM_INT)),
                        $queryBuilder->expr()->or(
                            $queryBuilder->expr()->eq('r.waitlist_promotion_token', $queryBuilder->createNamedParameter('', Connection::PARAM_STR)),
                            $queryBuilder->expr()->isNull('r.waitlist_promotion_token')
                        ),
                        $queryBuilder->expr()->notIn('r.waitlist_promotion_status', [
                            $queryBuilder->createNamedParameter(WaitlistPromotionStatus::DECLINED->value, Connection::PARAM_INT), // DECLINED
                            $queryBuilder->createNamedParameter(WaitlistPromotionStatus::CANCELLED->value, Connection::PARAM_INT)  // CANCELLED
                        ])
                    ),
                    // Waitlist with ACTIVE promotion (has token, deadline not expired, not declined/cancelled)
                    $queryBuilder->expr()->and(
                        $queryBuilder->expr()->eq('r.waitlist', $queryBuilder->createNamedParameter(1, Connection::PARAM_INT)),
                        $queryBuilder->expr()->neq('r.waitlist_promotion_token', $queryBuilder->createNamedParameter('', Connection::PARAM_STR)),
                        $queryBuilder->expr()->isNotNull('r.waitlist_promotion_token'),
                        $queryBuilder->expr()->gte('r.waitlist_promotion_deadline', $queryBuilder->createNamedParameter($now->format('Y-m-d H:i:s'), Connection::PARAM_STR)),
                        $queryBuilder->expr()->notIn('r.waitlist_promotion_status', [
                            $queryBuilder->createNamedParameter(WaitlistPromotionStatus::DECLINED->value, Connection::PARAM_INT), // DECLINED
                            $queryBuilder->createNamedParameter(WaitlistPromotionStatus::CANCELLED->value, Connection::PARAM_INT)  // CANCELLED
                        ])
                    )
                )
            );
        } else {
            // Only count confirmed registrations (waitlist=0)
            $queryBuilder->andWhere(
                $queryBuilder->expr()->eq('r.waitlist', $queryBuilder->createNamedParameter(0, Connection::PARAM_INT))
            );
        }

        $queryBuilder->groupBy('r.event');

        $result = $queryBuilder->executeQuery()
->fetchAssociative();

        return (int)($result['total'] ?? 0);
    }

    /*
     * Funktionen fuer Physio Team
     */

    public function findNextRegistrations($user)
    {
        $query = $this->createQuery();

        $today = new DateTime();
        $constraints = [];

        $constraints[] = $query->equals('feUser', $user);
        $constraints[] = $query->greaterThanOrEqual('event.dateFrom', $today->format('Y-m-d'));

        $query->matching($query->logicalAnd(...$constraints));
        $query->setLimit(3);
        $query->setOrderings([
            'event.dateFrom' => 'ASC',
            'event.timeFrom' => 'ASC',
        ]);

        return $query->execute();
    }


    public function findPassedRegistrations($user)
    {
        $query = $this->createQuery();

        $today = new DateTime();
        $aYearAgo = clone $today;
        $aYearAgo->modify('-1 year');

        $constraints = [];

        $constraints[] = $query->equals('feUser', $user);
        $constraints[] = $query->greaterThanOrEqual('event.dateFrom', $aYearAgo->format('Y-m-d'));
        $constraints[] = $query->lessThanOrEqual('event.dateFrom', $today->format('Y-m-d'));

        $query->matching($query->logicalAnd(...$constraints));
        $query->setLimit(3);
        $query->setOrderings([
            'event.dateFrom' => 'ASC',
            'event.timeFrom' => 'ASC',
        ]);

        return $query->execute();
    }

    /**
     * Find all waitlist registrations for a specific event, ordered by creation date (FIFO)
     * Excludes declined (status 2) and cancelled (status 3) registrations by default
     * Excludes registrations with active promotion offers (those count as regular registrations)
     *
     * @param Event $event
     * @param bool $includeDeclined Include declined/cancelled registrations
     * @param array $filter Optional filter array with 'searchword' key
     * @return \TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     */
    public function findWaitlistByEvent(Event $event, bool $includeDeclined = false, array $filter = [])
    {
        $query = $this->createQuery();

        $constraints = [];
        $constraints[] = $query->equals('event', $event);
        $constraints[] = $query->equals('waitlist', true);

        if(!$includeDeclined) {
             // Exclude declined (status 2) and cancelled (status 3) registrations
            $constraints[] = $query->logicalNot($query->equals('waitlistPromotionStatus', 2));
            $constraints[] = $query->logicalNot($query->equals('waitlistPromotionStatus', 3));
        }

        // Add search filter
        if ($filter['searchword'] ?? false) {
            $keywordsArray = GeneralUtility::trimExplode(' ', $filter['searchword'], true);
            foreach ($keywordsArray as $keyword) {
                $constraints[] = $query->logicalOr(
                    $query->like('name', '%' . $keyword . '%'),
                    $query->like('firstName', '%' . $keyword . '%'),
                    $query->like('street', '%' . $keyword . '%'),
                    $query->like('zip', '%' . $keyword . '%'),
                    $query->like('city', '%' . $keyword . '%'),
                    $query->like('email', '%' . $keyword . '%'),
                    $query->like('additionalPersons.lastName', '%' . $keyword . '%'),
                    $query->like('additionalPersons.firstName', '%' . $keyword . '%'),
                );
            }
        }

        // Exclude registrations with active promotion offers
        // These are still waitlist=1 but have a valid promotion token and non-expired deadline
        // They should count toward regular registrations, not waitlist

        $query->matching($query->logicalAnd(...$constraints));
        $query->setOrderings([
            'waitlistPromotionStatus' => QueryInterface::ORDER_ASCENDING,
            'crdate' => QueryInterface::ORDER_ASCENDING,
            'uid' => QueryInterface::ORDER_ASCENDING,
        ]);

        return $query->execute();
    }

    public function findDeclinedWaitlistByEvent(Event $event)
    {
        $query = $this->createQuery();

        $constraints = [];
        $constraints[] = $query->equals('event', $event);
        $constraints[] = $query->equals('waitlist', true);
        $constraints[] = $query->equals('waitlistPromotionStatus', 2);

        $query->matching($query->logicalAnd(...$constraints));
        $query->setOrderings([
            'crdate' => QueryInterface::ORDER_ASCENDING,
        ]);

        return $query->execute();
    }

    public function findCancelledWaitlistByEvent(Event $event)
    {
        $query = $this->createQuery();

        $constraints = [];
        $constraints[] = $query->equals('event', $event);
        $constraints[] = $query->equals('waitlist', true);
        $constraints[] = $query->equals('waitlistPromotionStatus', 3);

        $query->matching($query->logicalAnd(...$constraints));
        $query->setOrderings([
            'crdate' => QueryInterface::ORDER_ASCENDING,
        ]);

        return $query->execute();
    }

    public function findPendingPromotionsByEvent(Event $event)
    {
        $query = $this->createQuery();

        $now = new \DateTime();

        $constraints = [];
        $constraints[] = $query->equals('event', $event);
        $constraints[] = $query->equals('waitlist', true);
        $constraints[] = $query->logicalNot($query->equals('waitlistPromotionStatus', 2));
        $constraints[] = $query->logicalAnd(
            $query->logicalNot($query->equals('waitlistPromotionToken', '')),
            $query->greaterThanOrEqual('waitlistPromotionDeadline', $now)
        );

        $query->matching($query->logicalAnd(...$constraints));
        $query->setOrderings([
            'crdate' => QueryInterface::ORDER_ASCENDING,
        ]);

        return $query->execute();
    }

}
