<?php

declare(strict_types=1);

namespace Ig\IgRuckzuckevent\Hook;

use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\DataHandling\DataHandler;
use TYPO3\CMS\Core\Localization\LanguageService;
use TYPO3\CMS\Core\Messaging\FlashMessage;
use TYPO3\CMS\Core\Messaging\FlashMessageService;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Core\Utility\GeneralUtility;

/**
 * Validates that date_deadline is not after date_from for events
 */
class ValidateEventDeadlineDataHandlerHook
{
    protected string $eventTable = 'tx_igruckzuckevent_domain_model_event';

    /**
     * Hook method called before a record is saved
     * Validates date_deadline vs date_from
     */
    public function processDatamap_preProcessFieldArray(
        array &$incomingFieldArray,
        string $table,
        string|int $id,
        DataHandler $dataHandler
    ): void {
        // Only process our event table
        if ($table !== $this->eventTable) {
            return;
        }

        // Get date_deadline and date_from from incoming data or existing record
        $dateDeadline = $incomingFieldArray['date_deadline'] ?? null;
        $dateFrom = $incomingFieldArray['date_from'] ?? null;

        // If dates are not in the incoming array and this is an update, fetch from database
        if (($dateDeadline === null || $dateFrom === null) && is_numeric($id)) {
            $existingRecord = $this->getExistingRecord((int)$id);
            $dateDeadline = $dateDeadline ?? $existingRecord['date_deadline'] ?? null;
            $dateFrom = $dateFrom ?? $existingRecord['date_from'] ?? null;
        }

        // Only validate if both dates are set and not empty
        if (empty($dateDeadline) || empty($dateFrom)) {
            return;
        }

        // Convert to timestamps for comparison
        $deadlineTimestamp = is_numeric($dateDeadline) ? (int)$dateDeadline : strtotime($dateDeadline);
        $fromTimestamp = is_numeric($dateFrom) ? (int)$dateFrom : strtotime($dateFrom);

        // Validate: deadline must not be after event start
        if ($deadlineTimestamp > $fromTimestamp) {
            // Format dates for display
            $deadlineFormatted = date('d.m.Y', $deadlineTimestamp);
            $fromFormatted = date('d.m.Y', $fromTimestamp);

            // Get localized error message
            $languageService = $this->getLanguageService();
            $errorMessage = \sprintf(
                $languageService->sL('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:validation.event.deadline_after_start'),
                $deadlineFormatted,
                $fromFormatted
            );
            $errorTitle = $languageService->sL('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:validation.event.deadline_after_start.title');

            // Add error to DataHandler
            $dataHandler->log(
                $this->eventTable,
                $id,
                2, // Action: update
                0,
                1, // Error
                $errorMessage,
                -1,
                ['date_deadline' => $dateDeadline, 'date_from' => $dateFrom]
            );
            // Prevent the field from being updated
            unset($incomingFieldArray['date_deadline']);
        }
    }

    /**
     * Get existing record from database
     */
    protected function getExistingRecord(int $uid): array
    {
        $queryBuilder = GeneralUtility::makeInstance(\TYPO3\CMS\Core\Database\ConnectionPool::class)
            ->getQueryBuilderForTable($this->eventTable);

        $record = $queryBuilder
            ->select('date_deadline', 'date_from')
            ->from($this->eventTable)
            ->where(
                $queryBuilder->expr()->eq('uid', $queryBuilder->createNamedParameter($uid, Connection::PARAM_INT))
            )
            ->executeQuery()
            ->fetchAssociative();

        return $record ?: [];
    }

    /**
     * Add a flash message
     */
    protected function addFlashMessage(string $message, string $title, ContextualFeedbackSeverity $severity): void
    {
        $flashMessage = GeneralUtility::makeInstance(
            FlashMessage::class,
            $message,
            $title,
            $severity,
            true
        );

        $flashMessageService = GeneralUtility::makeInstance(FlashMessageService::class);
        $flashMessageService
            ->getMessageQueueByIdentifier()
            ->enqueue($flashMessage);
    }

    /**
     * Get the language service
     */
    protected function getLanguageService(): LanguageService
    {
        return $GLOBALS['LANG'];
    }
}
