<?php

declare(strict_types=1);

namespace Ig\IgRuckzuckevent\Upgrades;

use Doctrine\DBAL\ArrayParameterType;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Install\Attribute\UpgradeWizard;
use TYPO3\CMS\Install\Updates\DatabaseUpdatedPrerequisite;
use TYPO3\CMS\Install\Updates\UpgradeWizardInterface;

#[UpgradeWizard('igRuckzuckeventPluginToCType')]
final class PluginToCTypeUpgradeWizard implements UpgradeWizardInterface
{
    /**
     * Mapping of old list_type values to new CType values
     */
    private const PLUGIN_MAPPING = [
        'igruckzuckevent_eventmanagement' => 'igruckzuckevent_eventmanagement',
        'igruckzuckevent_eventsinprocessing' => 'igruckzuckevent_eventsinprocessing',
        'igruckzuckevent_eventsarchive' => 'igruckzuckevent_eventsarchive',
        'igruckzuckevent_categorymanagement' => 'igruckzuckevent_categorymanagement',
        'igruckzuckevent_sponsormanagement' => 'igruckzuckevent_sponsormanagement',
        'igruckzuckevent_feedbackmanagement' => 'igruckzuckevent_feedbackmanagement',
        'igruckzuckevent_organizermanagement' => 'igruckzuckevent_organizermanagement',
        'igruckzuckevent_events' => 'igruckzuckevent_events',
        'igruckzuckevent_eventlistfromjson' => 'igruckzuckevent_eventlistfromjson',
        'igruckzuckevent_eventdetailfromjson' => 'igruckzuckevent_eventdetailfromjson',
        'igruckzuckevent_currentevents' => 'igruckzuckevent_currentevents',
        'igruckzuckevent_registration' => 'igruckzuckevent_registration',
        'igruckzuckevent_thankyou' => 'igruckzuckevent_thankyou',
        'igruckzuckevent_feedback' => 'igruckzuckevent_feedback',
        'igruckzuckevent_searchform' => 'igruckzuckevent_searchform',
        'igruckzuckevent_create' => 'igruckzuckevent_create',
        'igruckzuckevent_managementstatistik' => 'igruckzuckevent_managementstatistik',
        'igruckzuckevent_managementuser' => 'igruckzuckevent_managementuser',
        'igruckzuckevent_user' => 'igruckzuckevent_user',
        'igruckzuckevent_eventscalendar' => 'igruckzuckevent_eventscalendar',
        'igruckzuckevent_management' => 'igruckzuckevent_management',
        'igruckzuckevent_payment' => 'igruckzuckevent_payment',
        'igruckzuckevent_eventdetail' => 'igruckzuckevent_eventdetail',
        'igruckzuckevent_resteventlist' => 'igruckzuckevent_resteventlist',
        'igruckzuckevent_resteventshow' => 'igruckzuckevent_resteventshow',
        'igruckzuckevent_htmltitle' => 'igruckzuckevent_htmltitle',
        'igruckzuckevent_slugsuggest' => 'igruckzuckevent_slugsuggest',
    ];

    public function getIdentifier(): string
    {
        return 'igRuckzuckeventPluginToCType';
    }

    public function getTitle(): string
    {
        return 'Migrate ig_ruckzuckevent plugins from list_type to CType';
    }

    public function getDescription(): string
    {
        return 'Migrates all ig_ruckzuckevent content elements from using list_type to the new TYPO3 13 CType approach.';
    }

    public function getPrerequisites(): array
    {
        return [
            DatabaseUpdatedPrerequisite::class,
        ];
    }

    public function updateNecessary(): bool
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getQueryBuilderForTable('tt_content');

        $count = $queryBuilder
            ->count('uid')
            ->from('tt_content')
            ->where(
                $queryBuilder->expr()->eq(
                    'CType',
                    $queryBuilder->createNamedParameter('list')
                ),
                $queryBuilder->expr()->in(
                    'list_type',
                    $queryBuilder->createNamedParameter(
                        array_keys(self::PLUGIN_MAPPING),
                        ArrayParameterType::STRING
                    )
                )
            )
            ->executeQuery()
            ->fetchOne();

        return $count > 0;
    }

    public function executeUpdate(): bool
    {
        $connection = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getConnectionForTable('tt_content');

        foreach (self::PLUGIN_MAPPING as $oldListType => $newCType) {
            $queryBuilder = $connection->createQueryBuilder();
            $queryBuilder
                ->update('tt_content')
                ->set('CType', $newCType)
                ->set('list_type', '')
                ->where(
                    $queryBuilder->expr()->eq(
                        'CType',
                        $queryBuilder->createNamedParameter('list')
                    ),
                    $queryBuilder->expr()->eq(
                        'list_type',
                        $queryBuilder->createNamedParameter($oldListType)
                    )
                )
                ->executeStatement();
        }

        return true;
    }
}
