<?php
namespace Ig\IgRuckzuckevent\Controller;

use DateTime;
use Doctrine\DBAL\ParameterType;
use Ig\IgRuckzuckevent\Domain\Model\Event;
use Internetgalerie\IgBackendHelpers\Controller\AbstractBackendTableListController;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\Restriction\DeletedRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\EndTimeRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\HiddenRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\StartTimeRestriction;
use TYPO3\CMS\Core\Imaging\Icon;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

class EventBackendController extends AbstractBackendTableListController
{
    protected string $tableName = 'tx_igruckzuckevent_domain_model_event';
    protected string $tableEventgroup = 'tx_igruckzuckevent_domain_model_eventgroup';
    protected string $entryIdSearchAttribute = 'categories';

    /**
     * The repository for events
     *
     * @var \Ig\IgRuckzuckevent\Domain\Repository\EventRepository
     */
    protected $eventRepository = null;

    /**
     * The repository for registrations
     *
     * @var \Ig\IgRuckzuckevent\Domain\Repository\RegistrationRepository
     */
    protected $registrationRepository = null;

    /**
     * @param  \Ig\IgRuckzuckevent\Domain\Repository\EventRepository $eventRepository
     * @return void
     */
    public function injectEventRepository(\Ig\IgRuckzuckevent\Domain\Repository\EventRepository $eventRepository): void
    {
        $this->eventRepository = $eventRepository;
    }

    /**
     * @param  Ig\IgRuckzuckevent\Domain\Repository\RegistrationRepository $registrationRepository
     * @return void
     */
    public function injectRegistrationRepository(\Ig\IgRuckzuckevent\Domain\Repository\RegistrationRepository $registrationRepository): void
    {
        $this->registrationRepository = $registrationRepository;
    }

    /**
     * Action index
     *
     * @return void
     */
    public function indexAction(): ResponseInterface
    {
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $filter = $this->search;
        $categoryUid = $this->entryId;
        $storagePid = $this->pageUid;

        if (!is_null($storagePid)) {
            $tableName = 'tx_igruckzuckevent_domain_model_event';
            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tableName);
            $queryBuilder->getRestrictions()->removeByType(HiddenRestriction::class);
            $queryBuilder->getRestrictions()->removeByType(StartTimeRestriction::class);
            $queryBuilder->getRestrictions()->removeByType(EndTimeRestriction::class);
            $queryBuilder->getRestrictions()->add(GeneralUtility::makeInstance(DeletedRestriction::class));

            $queryBuilder->selectLiteral('*, (SELECT COUNT(*) FROM tx_igruckzuckevent_domain_model_registration WHERE event = tx_igruckzuckevent_domain_model_event.uid AND deleted = 0 AND hidden = 0) AS registration_count')
            ->from($tableName)
                ->where($queryBuilder->expr()->in(
                    $tableName . '.pid',
                    [$storagePid]
                ))
                ->andWhere($queryBuilder->expr()->in(
                    $tableName . '.sys_language_uid',
                    [
                        $queryBuilder->createNamedParameter(-1, ParameterType::INTEGER),
                        $queryBuilder->createNamedParameter(0, ParameterType::INTEGER),
                    ]
                ))
                ->orderBy('date_from');

            if ($filter['searchword'] ?? false) {
                $queryBuilder->andWhere(
                    $queryBuilder->expr()->or(
                        $queryBuilder->expr()->eq('uid', $queryBuilder->createNamedParameter($filter['searchword'], ParameterType::INTEGER)),
                        $queryBuilder->expr()->like(
                            'title',
                            $queryBuilder->createNamedParameter('%' . $queryBuilder->escapeLikeWildcards($filter['searchword']) . '%')
                        )
                    )
                );
            }

            if($categoryUid > 0) {
                $queryBuilder
                    ->join(
                        'tx_igruckzuckevent_domain_model_event',
                        'tx_igruckzuckevent_event_category_mm',
                        'catmm',
                        'catmm.uid_local = tx_igruckzuckevent_domain_model_event.uid'
                    )
                    ->andWhere('catmm.uid_foreign = ' . $categoryUid);
            }

            if (!isset($filter['mode']) || !$filter['mode']) {
                $filter['mode'] = 1;
            }

            $today = new DateTime();
            $today->setTime(2, 0, 0);

            if (isset($filter['mode']) && $filter['mode'] == 1) {
                $queryBuilder->andWhere(
                    $queryBuilder->expr()->gte('date_from', $queryBuilder->createNamedParameter($today->format('Y-m-d')))
                );
            } else if (isset($filter['mode']) && $filter['mode'] == 2) {
                $queryBuilder->andWhere(
                    $queryBuilder->expr()->lte('date_from', $queryBuilder->createNamedParameter($today->format('Y-m-d')))
                );
            }

            $events = $queryBuilder->executeQuery()->fetchAllAssociative();

            foreach ($events as &$event) {
                $iconFactory = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance(\TYPO3\CMS\Core\Imaging\IconFactory::class);
                $iconHtml = $iconFactory->getIconForRecord($tableName, $event, Icon::SIZE_SMALL)->render();
                $event['iconWithLink'] = \TYPO3\CMS\Backend\Utility\BackendUtility::wrapClickMenuOnIcon($iconHtml, $tableName, $event['uid']);
            }

            $moduleTemplate->assign('events', $events);
            $moduleTemplate->assign('tableName', $tableName);
            $moduleTemplate->assign('filter', $filter);
        }
        $moduleTemplate->assign('entryId', $this->entryId);
        $moduleTemplate->assign('searchArgumentName', $this->searchArgumentName);
        $moduleTemplate->assign('entryIdArgumentName', $this->entryIdArgumentName);
        $moduleTemplate->assign('entryIdSearchAttribute', $this->entryIdSearchAttribute);
        $moduleTemplate->assign('pageUid', $this->pageUid);
        $moduleTemplate->assign('storagePid', $storagePid);
        $moduleTemplate->assign('search', $this->search);
        $returnUrlParameters = [
            $this->searchArgumentName => $this->search,
        ];
        if($this->entryId > 0) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->entryId;
        }
        $moduleTemplate->assign('returnUrlParameters', $returnUrlParameters);

        return $moduleTemplate->renderResponse('EventBackend/Index');
    }

    /**
     * list participants for an events
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("event")
     * @param array $filter
     */
    public function listParticipantsAction($event, array $filter = []): ResponseInterface
    {
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $storagePid = $this->pageUid;
        $tableName = 'tx_igruckzuckevent_domain_model_registration';
        $registrations = $this->registrationRepository->findByEventAndFilter($event, $filter);

        $moduleTemplate->assign('storagePid', $storagePid);
        $moduleTemplate->assign('event', $event);
        $moduleTemplate->assign('registrations', $registrations);
        $moduleTemplate->assign('tableName', $tableName);
        $moduleTemplate->assign('defaultValues', [$tableName => ['event' => $event->getUid()]]);
        $moduleTemplate->assign('filter', $filter);
        $moduleTemplate->assign('search', $this->search);

        $returnUrlParameters = [
            $this->searchArgumentName => $this->search,
        ];
        if ($this->entryId > 0) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->entryId;
        } elseif ($this->search[$this->entryIdArgumentName] ?? false) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->search[$this->entryIdArgumentName];
        }
        $moduleTemplate->assign('returnUrlParameters', $returnUrlParameters);

        return $moduleTemplate->renderResponse('EventBackend/ListParticipants');
    }

    /**
     * list participants for an events
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param ?int $startYear
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("event")
     * @param array $filter
     */
    public function calendarAction($event, array $filter = [], ?int $startYear = null): ResponseInterface
    {
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $storagePid = $this->pageUid;

        $eventgroup =  $event->getEventgroup();
        $events = $this->eventRepository->findBy(['eventgroup' => $eventgroup]);
        // TODO: CALENDAR
        $dataSource = [];
        $i = 0;
        $now = new DateTime();
        $startYear = $startYear ?? $now->format('Y');
        foreach ($events as $ev) {
            //echo($ev->getDateFrom()->format('Y-m-d').'<br />');
            $editLinkParams = [
                'edit' => ['tx_igruckzuckevent_domain_model_event' => [$ev->getUid() => 'edit']],
                'returnUrl' => $this->request->getAttribute('normalizedParams')->getRequestUri(),
                'recurrence' => ['update_scope' => 'this'],
            ];
            $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
            $editUri = (string)$uriBuilder->buildUriFromRoute('record_edit', $editLinkParams);
            if ($i == 0) {
                $startYear = $ev->getDateTimeFrom()->format('Y');
            }
            $registrations = $this->registrationRepository->findBy(['event' => $ev->getUid()]);
            $registrationsCount = $registrations->count();
            if ($registrationsCount > 0) {
                $color = $eventgroup->isEventExcluded($ev) ? '#c88' : '#f88';
                $title = ($eventgroup->isEventExcluded($ev) ? 'spezieller Termin - nicht mehr in Serie mit Reservationen' : 'Termin mit Reservationen') . ' (' . $registrationsCount . ')';
            } elseif ($eventgroup->isEventExcluded($ev)) {
                $color = '#fc8';
                $title = 'spezieller Termin - nicht mehr in Serie (ohne Reservationen)';
            } else {
                $color = '#8f8';
                $title = 'Termin in Serie (ohne Reservationen)';
            }

            $dataSource[] = [
                'name' => $ev->getDateText(),
                'startDate' => $ev->getDateTimeFrom()->format('Y-m-d H:i:s'),
                'endDate' => $ev->getDateTimeTo() ? $ev->getDateTimeTo()->format('Y-m-d H:i:s') : $ev->getDateTimeFrom()->format('Y-m-d H:i:s'),
                'color' => $color,
                'title' => $title,
                'eventId' => $event->getUid(),
                'editUri' => $editUri,
                'deleteUri' => $registrations->count() == 0 ? $this->uriBuilder->reset()->uriFor('deleteEvent', ['event' => $ev->getUid()]) : '',
                'deleteMessage' => $registrations->count() == 0 ? 'Termin vom ' . $ev->getDateText() . ' wirklich löschen?' : '',
                'id' => $ev->getUid(),
                //'name' => $event->getTitle()
            ];
            $eventgroup->removeExcludeDate($ev->getDateFrom()->format('Y-m-d'));
            $i++;
        }

        // add deleted entries present in excludeDates
        $excludeDates = $eventgroup->getExcludeDatesArray();
        foreach ($excludeDates as $date) {
            $dateTime = (new DateTime($date))->format('Y-m-d');
            $dataSource[] = [
                'name' => 'deleted',
                'title' => 'manuell gelöschter Termin',
                'startDate' => $dateTime,
                'endDate' => $dateTime,
                'color' => '#eee',
                //'eventId' => $event->getUid(),
                //'editUri' => $editUri,
                //'deleteUri' => $registrations->count() == 0 ? $this->uriBuilder->reset()->uriFor('deleteEvent', ['event' => $ev->getUid()]) : '',
                //'deleteMessage' => $registrations->count() == 0 ? 'Termin vom ' . $ev->getDateText() . ' wirklich löschen?' : '',
                //'id' => $ev->getUid(),
                //'name' => $event->getTitle()
            ];
        }

        
        $moduleTemplate->assign('storagePid', $storagePid);
        $moduleTemplate->assign('startYear', $startYear);
        $moduleTemplate->assign('event', $event);
        $moduleTemplate->assign('eventgroup', $eventgroup);
        $moduleTemplate->assign('tableEventgroup', $this->tableEventgroup);
        $moduleTemplate->assign('events', $events);
        $moduleTemplate->assign('filter', $filter);
        $moduleTemplate->assign('dataSource', $dataSource);
        $moduleTemplate->assign('dataSourceJson', json_encode($dataSource));
        $moduleTemplate->assign('search', $this->search);
        if ($this->request->hasArgument('return')) {
            $returnUrlParameters = $this->request->getArgument('return');
        } else {
            $returnUrlParameters = [
                $this->searchArgumentName => $this->search,
            ];
        }
        if($this->entryId > 0) {
            $returnUrlParameters[$this->entryIdArgumentName] = $this->entryId;
        }
        $moduleTemplate->assign('returnUrlParameters', $returnUrlParameters);

        return $moduleTemplate->renderResponse('EventBackend/Calendar');
    }

    /**
     * list participants for an events
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("event")
     */
    public function exportRegistrationsAction(Event $event): ResponseInterface
    { //->getFont()->setBold(true); // Set first row bold

        $phpExcel = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $phpExcel->removeSheetByIndex(0);
        $worksheet = new \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet($phpExcel, 'members');
        $phpExcel->addSheet($worksheet, 0);
        $phpExcel->setActiveSheetIndex(0);

        $worksheet->getCell('A1')->setValue($event->getTitle());
        $worksheet->getCell('B1')->setValue(LocalizationUtility::translate('tx_igruckzuckevent_domain_model_event.num_participants', 'IgRuckzuckevent') . ':');
        $worksheet->getCell('C1')->setValue($event->getRegistrations()->count());
        $worksheet->getStyle('A1:C1')->applyFromArray(
            [
                'font' => [
                    'bold' => true,
                ],
            ]
        );

        foreach (range('A', 'O') as $column) {
            $worksheet->getColumnDimension($column)->setAutoSize(true);
        }

        $phpExcel->getActiveSheet()->SetCellValue('A1', $event->getTitle()); // First row: Output event title and registration count

        // Output registration count
        $phpExcel->getActiveSheet()->SetCellValue('B1', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_event.num_participants', 'IgRuckzuckevent') . ':');
        $phpExcel->getActiveSheet()->SetCellValue('C1', $event->getRegistrations()->count());
        $phpExcel->getActiveSheet()->getStyle('A1:C1')->getFont()->setBold(true); // Set first row bold

        // Second row: Output attribute titles
        $phpExcel->getActiveSheet()->SetCellValue('A2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.gender', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('B2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.first_name', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('C2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.name', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('D2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.street', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('E2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.zip', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('F2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.city', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('G2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.phone', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('H2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.phone_work', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('I2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.email', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('J2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.comment', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('K2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.company', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('L2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.company_street', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('M2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.company_zip', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('N2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.company_city', 'IgRuckzuckevent'));
        $phpExcel->getActiveSheet()->SetCellValue('O2', LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.bill', 'IgRuckzuckevent'));

        $lastHeaderRow = 'P';
        if ($event->getAdditionalFormFields()->count()) {
            $start = ord('P');
            $end = $start + (int)$event->getAdditionalFormFields()->count();

            $char = $start;
            foreach ($event->getAdditionalFormFields() as $formField) {
                $phpExcel->getActiveSheet()->SetCellValue(chr($char) . '2', $formField->getTitle());
                $char++;
            }

            $lastHeaderRow = chr($char);
        }

        $phpExcel->getActiveSheet()->getStyle('A2:' . $lastHeaderRow . '2')->getFont()->setBold(true); // Set second row bold

        // For each registration of the event set the values of the attributes to the according column
        $i = 3; // We are beginning at the 3rd row (Excel rows count from 1)
        foreach ($event->getRegistrations() as $registration) {
            // Concatenate all statusses separated by a ","
            $statusses = '';
            $isFirst = true;
            foreach ($registration->getStatus() as $status) {
                if (!$isFirst) {
                    $statusses .= ', ';
                }
                $statusses .= $status->getStatus();

                if ($isFirst) {
                    $isFirst = false;
                }
            }

            $phpExcel->getActiveSheet()->SetCellValue('A' . $i, LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.gender.' . $registration->getGender(), 'IgRuckzuckevent'));
            $phpExcel->getActiveSheet()->SetCellValue('B' . $i, $registration->getFirstName());
            $phpExcel->getActiveSheet()->SetCellValue('C' . $i, $registration->getName());
            $phpExcel->getActiveSheet()->SetCellValue('D' . $i, $registration->getStreet());
            $phpExcel->getActiveSheet()->SetCellValue('E' . $i, $registration->getZip());
            $phpExcel->getActiveSheet()->SetCellValue('F' . $i, $registration->getCity());
            $phpExcel->getActiveSheet()->SetCellValueExplicit('G' . $i, $registration->getPhone(), \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING);
            $phpExcel->getActiveSheet()->SetCellValueExplicit('H' . $i, $registration->getPhoneWork(), \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING);
            $phpExcel->getActiveSheet()->SetCellValue('I' . $i, $registration->getEmail());
            $phpExcel->getActiveSheet()->SetCellValue('J' . $i, preg_replace('#\R+#', ' ', $registration->getComment()));
            $phpExcel->getActiveSheet()->SetCellValue('K' . $i, $registration->getCompany());
            $phpExcel->getActiveSheet()->SetCellValue('L' . $i, $registration->getCompanyStreet());
            $phpExcel->getActiveSheet()->SetCellValue('M' . $i, preg_replace('#\R+#', ' ', $registration->getCompanyZip()));
            $phpExcel->getActiveSheet()->SetCellValue('N' . $i, $registration->getCompanyCity());
            $phpExcel->getActiveSheet()->SetCellValue('O' . $i, LocalizationUtility::translate('tx_igruckzuckevent_domain_model_registration.bill.' . ($registration->getBill() ? '1' : '0'), 'IgRuckzuckevent'));

            if ($registration->getAdditionalFields()) {
                $start = ord('P');
                $end = $start + (int)$event->getAdditionalFormFields()->count();

                $char = $start;
                foreach ($event->getAdditionalFormFields() as $formField) {
                    foreach ($registration->getAdditionalFields() as $fieldname => $additionalField) {
                        if ('additional-' . $formField->getUid() == $fieldname) {
                            if ($formField->getType() == 'Checkbox') {
                                $index = 0;
                                $v = '';
                                foreach ($additionalField as $value) {
                                    if ($index != 0) {
                                        $v .= ', ';
                                    }
                                    $v .= $formField->getOptionsArray()[$value];
                                    $index++;
                                }
                                $phpExcel->getActiveSheet()->SetCellValue(chr($char) . $i, $v);
                            } elseif ($formField->getType() == 'Radio' || $formField->getType() == 'Dropdown') {
                                $phpExcel->getActiveSheet()->SetCellValue(chr($char) . $i, $formField->getOptionsArray()[$additionalField]);
                            } elseif ($formField->getType() != 'Title') {
                                $phpExcel->getActiveSheet()->SetCellValue(chr($char) . $i, preg_replace('#\R+#', ' ', $additionalField));
                            }
                            $char++;
                        }
                    }
                }
            }
            $i++;
        }

        // Set column width automatically according to their content
        foreach (range('A', $lastHeaderRow) as $column) {
            $phpExcel->getActiveSheet()->getColumnDimension($column)->setAutoSize(true);
        }

        $fileName = 'kurs-' . $event->getUid() . '-' . time() . '.xlsx'; // Temporary filename

        header('Content-Type: application/force-download');
        header('Content-Description: File Transfer');

        header('Content-Type: application/octet-stream');
        header('Content-Type: application/download');
        header('Content-Disposition: attachment; filename=' . $fileName);
        header('Content-Transfer-Encoding: binary');
        header('Connection: Keep-Alive');
        header('Expires: 0');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Pragma: public');
        //header('Content-Length: 30000000000000000');//".filesize($filename));
        $objWriter = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($phpExcel);
        $tmpname = tempnam("", "tempfile");
        $objWriter->save($tmpname);
        header("Content-Length: " . filesize($tmpname));
        unlink($tmpname);
        $objWriter->save('php://output');
        exit(0);

        return $this->htmlResponse();
    }

    /**
     * Delete a registration
     *
     * @param  \Ig\IgRuckzuckevent\Domain\Model\Registration $registration
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("registration")
     */
    public function deleteRegistrationAction($registration): ResponseInterface
    {
        $event = $registration->getEvent();
        $this->registrationRepository->remove($registration);
        return $this->redirect('listParticipants', null, null, ['event' => $event->getUid()]);
    }

    /**
     * Delete the registration of an aevent
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("event")
     */
    public function deleteRegistrationsAction($event): ResponseInterface
    {
        foreach ($event->getRegistrations() as $registration) {
            $this->registrationRepository->remove($registration);
        }
        return $this->redirect('listParticipants', null, null, ['event' => $event->getUid()]);
    }

    /**
     * Delete the registration of an aevent
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("event")
     */
    public function deleteEventAction($event): ResponseInterface
    {
        if(!$event->getRegistrationsCount()) {
            $this->eventRepository->remove($event);
        }
        return $this->redirect('index');
    }
}
