<?php
namespace Ig\IgRuckzuckevent\Controller;

use GeorgRinger\NumberedPagination\NumberedPagination;
use Ig\IgRuckzuckevent\Domain\Model\Event;
use Ig\IgRuckzuckevent\Domain\Model\FileReference;
use Ig\IgRuckzuckevent\Domain\Repository\FeUserRepository;
use Ig\IgRuckzuckevent\Event\BeforeRenderDetailViewEvent;
use Ig\IgRuckzuckevent\PageTitle\EventPageTitleProvider;
use Ig\IgRuckzuckevent\Utility\UserUtility;
use Internetgalerie\IgDynval\Controller\DynamicValidationActionController;
use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Http\Message\RequestFactoryInterface;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Pagination\ArrayPaginator;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Pagination\QueryResultPaginator;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;

use TYPO3\CMS\Frontend\Controller\ErrorController;

/**
 * The EventController handles all actions directly related to Events.
 */
class EventController extends ActionController
{
    use DynamicValidationActionController;

    protected FeUserRepository $feUserRepository;

    /**
     * The repository for events
     *
     * @var \Ig\IgRuckzuckevent\Domain\Repository\EventRepository
     */
    protected $eventRepository = null;

    /**
     * The repository for eventgroups
     *
     * @var \Ig\IgRuckzuckevent\Domain\Repository\EventgroupRepository
     */
    protected $eventgroupRepository = null;

    /**
     * The repository for categories
     *
     * @var \Ig\IgRuckzuckevent\Domain\Repository\CategoryRepository
     */
    protected $categoryRepository = null;

    /**
     * The repository for sponsors
     *
     * @var \Ig\IgRuckzuckevent\Domain\Repository\SponsorRepository
     */
    protected $sponsorRepository = null;

    /**
     * The repository for organizers
     *
     * @var \Ig\IgRuckzuckevent\Domain\Repository\OrganizerRepository
     */
    protected $organizerRepository = null;

    /**
     * The repository for qualifications
     *
     * @var \Ig\IgRuckzuckevent\Domain\Repository\QualificationRepository
     */
    protected $qualificationRepository = null;

    /** @var RequestFactoryInterface */
    private $requestFactory;

    /**
     * Upload file service for handling uploaded files
     *
     */
    protected $uploadFileService = null;

    public function injectFeUserRepository(FeUserRepository $feUserRepository)
    {
        $this->feUserRepository = $feUserRepository;
    }

    /**
     * @param \Ig\IgRuckzuckevent\Domain\Repository\EventRepository $eventRepository
     */
    public function injectEventRepository(\Ig\IgRuckzuckevent\Domain\Repository\EventRepository $eventRepository): void
    {
        $this->eventRepository = $eventRepository;
    }

    /**
     * @param \Ig\IgRuckzuckevent\Domain\Repository\EventgroupRepository $eventgroupRepository
     */
    public function injectEventgroupRepository(\Ig\IgRuckzuckevent\Domain\Repository\EventgroupRepository $eventgroupRepository): void
    {
        $this->eventgroupRepository = $eventgroupRepository;
    }

    /**
     * @param \Ig\IgRuckzuckevent\Domain\Repository\CategoryRepository $categoryRepository
     */
    public function injectCategoryRepository(\Ig\IgRuckzuckevent\Domain\Repository\CategoryRepository $categoryRepository): void
    {
        $this->categoryRepository = $categoryRepository;
    }

    /**
     * @param \Ig\IgRuckzuckevent\Domain\Repository\SponsorRepository $sponsorRepository
     */
    public function injectSponsorRepository(\Ig\IgRuckzuckevent\Domain\Repository\SponsorRepository $sponsorRepository): void
    {
        $this->sponsorRepository = $sponsorRepository;
    }

    /**
     * @param \Ig\IgRuckzuckevent\Domain\Repository\OrganizerRepository $organizerRepository
     */
    public function injectOrganizerRepository(\Ig\IgRuckzuckevent\Domain\Repository\OrganizerRepository $organizerRepository): void
    {
        $this->organizerRepository = $organizerRepository;
    }

    /**
     * @param \Ig\IgRuckzuckevent\Domain\Repository\QualificationRepository $qualificationRepository
     */
    public function injectQualificationRepository(\Ig\IgRuckzuckevent\Domain\Repository\QualificationRepository $qualificationRepository): void
    {
        $this->qualificationRepository = $qualificationRepository;
    }

    /**
     * @param \Psr\Http\Message\RequestFactoryInterface $requestFactory
     */
    public function injectRequestFactory(\Psr\Http\Message\RequestFactoryInterface $requestFactory): void
    {
        $this->requestFactory = $requestFactory;
    }

    public function initializeView($view): void
    {
        $this->view->assign('controller', 'Event');
        $this->view->assign('pageUid', $GLOBALS['TSFE']->id);
    }

    /**
     * Initialize the current action
     *
     * @return void
     */
    public function initializeAction(): void
    {
        parent::initializeAction();

        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_igruckzuckevent_domain_model_event');
        $germans = $conn->executeQuery('select uid,date_from,time_from,date_to,time_to,date_deadline,time_deadline FROM tx_igruckzuckevent_domain_model_event WHERE sys_language_uid=0 AND deleted=0 AND exists (SELECT 1 FROM tx_igruckzuckevent_domain_model_event AS e WHERE e.l10n_parent=tx_igruckzuckevent_domain_model_event.uid AND e.tstamp>tx_igruckzuckevent_domain_model_event.tstamp )')->fetchAllAssociative();
        foreach ($germans as $german) {
            // tstamp='".$german['tstamp'] ."'
            $sets = '';
            foreach (['date_from', 'time_from', 'date_to', 'time_to', 'date_deadline', 'time_deadline'] as $attribute) {
                if ($sets) {
                    $sets .= ', ';
                }
                $sets .= $attribute . "='" . $german[$attribute] . "'";
            }
            $sql = "UPDATE tx_igruckzuckevent_domain_model_event SET " . $sets . " WHERE l10n_parent=" . $german['uid'];
            $germans = $conn->executeQuery($sql);
        }

        if(isset($this->arguments['search'])) {
            $search = $this->arguments['search'];
            $propertyMappingConfiguration = $search->getPropertyMappingConfiguration();
            $propertyMappingConfiguration->forProperty('dateFrom')->setTypeConverterOption('TYPO3\\CMS\\Extbase\\Property\\TypeConverter\\DateTimeConverter', \TYPO3\CMS\Extbase\Property\TypeConverter\DateTimeConverter::CONFIGURATION_DATE_FORMAT, 'd.m.Y');
            $propertyMappingConfiguration->forProperty('dateTo')->setTypeConverterOption('TYPO3\\CMS\\Extbase\\Property\\TypeConverter\\DateTimeConverter', \TYPO3\CMS\Extbase\Property\TypeConverter\DateTimeConverter::CONFIGURATION_DATE_FORMAT, 'd.m.Y');
        }


        if (isset($this->arguments['event']) || isset($this->arguments['newEvent'])) {
            if (isset($this->arguments['event'])) {
                $event = $this->arguments['event'];
            } else {
                $event = $this->arguments['newEvent'];
            }
            $propertyMappingConfiguration = $event->getPropertyMappingConfiguration();
            // Set date format of date fields
            foreach (['dateFrom', 'dateTo', 'dateDeadline'] as $dateField) {
                $propertyMappingConfiguration->forProperty($dateField)->setTypeConverterOption('TYPO3\\CMS\\Extbase\\Property\\TypeConverter\\DateTimeConverter', \TYPO3\CMS\Extbase\Property\TypeConverter\DateTimeConverter::CONFIGURATION_DATE_FORMAT, 'd.m.Y');
            }
            // Set time format of time fields
            foreach (['timeFrom', 'timeTo', 'timeDeadline'] as $timeField) {
                $propertyMappingConfiguration->forProperty($timeField)->setTypeConverterOption('TYPO3\\CMS\\Extbase\\Property\\TypeConverter\\DateTimeConverter', \TYPO3\CMS\Extbase\Property\TypeConverter\DateTimeConverter::CONFIGURATION_DATE_FORMAT, 'H:i');
            }
            foreach (['categories', 'sponsors', 'organizers', 'qualifications'] as $f) {
                $propertyMappingConfiguration->allowProperties($f);
                $propertyMappingConfiguration->forProperty($f . '.*')->allowAllProperties();
                $propertyMappingConfiguration->allowCreationForSubProperty($f . '.*');
                $propertyMappingConfiguration->allowModificationForSubProperty($f . '.*');
            }
        }
    }

    /**
     * Liast all events with possibility to sort and filter them
     *
     * @return void
     */
    public function listAction(): ResponseInterface
    {
        $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument('currentPage') : 1;

        $events = $this->eventRepository->findBySearchAndSorting($this->getAssignSearch(), $this->getSorting());
        $categories = $this->categoryRepository->findAll();
        $qualifications = $this->qualificationRepository->findAll();

        $paginator = new QueryResultPaginator($events, $currentPage, 18); // TODO: FLexForm Config
        $pagination = new NumberedPagination($paginator, 10); // zweiter Argument: maximal Anzahl Links

        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);

        $this->view->assign('events', $events);
        $this->view->assign('categories', $categories);
        $this->view->assign('qualifications', $qualifications);
        if ($this->request->hasArgument('managementUser') && UserUtility::isAdminLoggedIn()) {
            $this->view->assign('managementUser', $this->request->getArgument('managementUser'));
        } else {
            $this->view->assign('managementUser', '');
        }

        return $this->htmlResponse();
    }

    public function listJsonAction(): ResponseInterface
    {
        $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument('currentPage') : 1;
        $jsonPath = $this->settings['eventListJson'];
        $search = $this->getAssignSearch();
        $sorting = $this->getSorting();
        $queryArr = [];

        if(!$search) {
            $search = [];
        }
        if(!$sorting) {
            $sorting = [];
        }

        $queryArr = [];
        $queryStr = '';

        if(!empty($search) || !empty($sorting)) {
            if(!empty($search)) {
                $queryArr['tx_igruckzuckevent_resteventlist']['search'] = $search;
            }
            /*if(!empty($sorting)) {
                $queryArr['tx_igruckzuckevent_resteventlist']['sorting'] = $sorting;
            }*/
            $queryStr = '?' . http_build_query($queryArr);
        }
        $additionalOptions = [
            // Additional headers for this specific request
            'headers' => ['Cache-Control' => 'no-cache'],
            // Additional options, see http://docs.guzzlephp.org/en/latest/request-options.html
            'allow_redirects' => false,
        ];

        $url = $jsonPath . $queryStr;

        $response = $this->requestFactory->request($url, 'GET', $additionalOptions);

        $json = '';
        if ($response->getStatusCode() === 200) {
            $json = json_decode($response->getBody()->getContents(), true);
        }

        $paginator = new ArrayPaginator($json, $currentPage, 20);
        $pagination = new NumberedPagination($paginator, 10); // zweiter Argument: maximal Anzahl Links

        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);
        $this->view->assign('events', $json);

        return $this->htmlResponse();
    }

    /**
     * List all events with possibility to sort and filter them.
     * This action is for organizers, so all the edit, create and delete links are visible here.
     *
     * @return void
     */
    public function organizerListAction(): ResponseInterface
    {
        $this->view->assign('action', 'organizerList');
        $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument('currentPage') : 1;

        $this->eventRepository->setShowOnlyActiveEntries(false);
        $events = $this->eventRepository->findBySearchAndSorting($this->getAssignSearch(), $this->getSorting());
        $categories = $this->categoryRepository->findAll();
        $qualifications = $this->qualificationRepository->findAll();

        $paginator = new QueryResultPaginator($events, $currentPage, 20);
        $pagination = new NumberedPagination($paginator, 10); // zweiter Argument: maximal Anzahl Links

        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');

        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);

        $this->view->assign('events', $events);
        $categories = $this->categoryRepository->findAll();
        $this->view->assign('events', $events);
        $this->view->assign('organizer', $this->feUserRepository->findByUid($userId));
        $this->view->assign('categories', $categories);
        $this->view->assign('qualifications', $qualifications);
        $this->view->assign('pageUid', $GLOBALS['TSFE']->id);

        return $this->htmlResponse();
    }

    /**
     * Shows the detail view of an event
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param string $returnUrl
     * @return void
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("event")
     */
    public function showAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event = null, $returnUrl = ''): ResponseInterface
    {
        $showRecurrence = (bool)($this->settings['showRecurrence'] ?? false);
        $eventUid = (int)($this->settings['eventUid'] ?? 0);
        if ($eventUid > 0) {
            $event = $this->eventRepository->findByUid($eventUid);
        }
        if (!$event instanceof Event) {
            // raise 404
            $errorController = GeneralUtility::makeInstance(ErrorController::class);
            $response = $errorController->pageNotFoundAction(
                $this->request,
                    'Page not found',
            );
            throw new \TYPO3\CMS\Core\Http\PropagateResponseException($response);
        }
        // add tag to the cache
        //$this->request->getAttribute('frontend.page.information');
        $GLOBALS['TSFE']->addCacheTags(['tx_igruckzuckevent_show_' . $event->getUid()]);

        $titleProvider = GeneralUtility::makeInstance(EventPageTitleProvider::class);
        $titleProvider->setTitle(($event->getCoursno() ? $event->getCoursno() . ' - ' : '') . $event->getTitle());

        if (!$event) {
            $this->redirectToUri($this->uriBuilder->reset()->setTargetPageUid($this->settings['listPageUid'])->build());
        }
        $this->view->assign('pageUid', $GLOBALS['TSFE']->id);
        $this->view->assign('event', $event);
        $this->view->assign('returnUrl', $returnUrl);


        $this->view = GeneralUtility::makeInstance(EventDispatcherInterface::class)->dispatch(
            new BeforeRenderDetailViewEvent($this->view, $event)
        )->getView();

        return $this->htmlResponse();
    }

    /**
     * Shows the detail view of an event
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param string $returnUrl
     * @return void
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("event")
     */
    public function showJsonAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event, $returnUrl = ''): ResponseInterface
    {
        $titleProvider = GeneralUtility::makeInstance(EventPageTitleProvider::class);
        $titleProvider->setTitle(($event->getCoursno() ? $event->getCoursno() . ' - ' : '') . $event->getTitle());
        $jsonPath = $this->settings['eventShowJson'];

        if (!$event) {
            $this->redirectToUri($this->uriBuilder->reset()->setTargetPageUid($this->settings['listPageUid'])->build());
        }

        $queryArr = [
            'tx_igruckzuckevent_resteventshow' => [
                'event' => $event->getUid()
            ]
        ];
        $queryStr = '?' . http_build_query($queryArr);

        $additionalOptions = [
            // Additional headers for this specific request
            'headers' => ['Cache-Control' => 'no-cache'],
            // Additional options, see http://docs.guzzlephp.org/en/latest/request-options.html
            'allow_redirects' => false,
        ];

        $url = $jsonPath . $queryStr;

        $response = $this->requestFactory->request($url, 'GET', $additionalOptions);

        $json = '';
        if ($response->getStatusCode() === 200) {
            $json = json_decode($response->getBody()->getContents(), true);
        }

        $this->view->assign('pageUid', $GLOBALS['TSFE']->id);
        $this->view->assign('event', $json);
        $this->view->assign('returnUrl', $returnUrl);

        return $this->htmlResponse();
    }

    /**
     * Liast current events
     *
     * @return void
     */
    public function listCurrentAction(): ResponseInterface
    {
        $this->view->assign('action', 'listCurrent');
        $eventsCount = (int)($this->settings['flexform']['currentEventsCount'] ?? $this->settings['currentEventsCount']);
        $events = $this->eventRepository->findBySearchAndSorting($this->getAssignSearch(), $this->getSorting(), $eventsCount);
        $this->view->assign('events', $events);

        return $this->htmlResponse();
    }

    /**
     * Shows form to create a new event
     *
     * @param string $returnUrl
     * @return void
     */
    public function newAction($returnUrl = ''): ResponseInterface
    {

        $this->getAssignSearch();
        $this->assignEditForm();
        $this->view->assign('returnUrl', $returnUrl);

        return $this->htmlResponse();
    }

    /**
     * Action called when the Form in the new-Action is submitted.
     * We validate the event with our EventFormValidator, handle the file uploads and finally persist the event in the database.
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param string $returnUrl
     * @return void
     * @TYPO3\CMS\Extbase\Annotation\Validate("Internetgalerie\IgDynval\Validation\Validator\DynamicValidator", param="event")
     * @TYPO3\CMS\Extbase\Annotation\Validate("\Ig\IgRuckzuckevent\Domain\Validator\EventFormValidator", param="event")
     */
    public function createAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event, $returnUrl = ''): ResponseInterface
    {
        if(!$event->getDateFrom() && $event->getDateTo()) {
            $event->setDateFrom($event->getDateTo());
        }
        if ($event->getTimeFrom() == '') {
            $event->setTimeFrom(null);
        }
        if ($event->getTimeTo() == '') {
            $event->setTimeTo(null);
        }
        if ($event->getTimeDeadline() == '') {
            $event->setTimeDeadline(null);
        }
        $this->eventRepository->add($event);
        $search = $this->getAssignSearch();
        return $this->redirect('organizerList', null, null, ['search' => $search]);
    }

    /**
     * Shows form to create a new event
     *
     * @return void
     */
    public function publicNewAction(\Ig\IgRuckzuckevent\Domain\Model\CaptchaEvent $event = null): ResponseInterface
    {
        $categories = $this->categoryRepository->findAll();
        $this->view->assign('categories', $categories);
        $this->view->assign('event', $event);

        return $this->htmlResponse();
    }

    /**
     * Action called when the Form in the new-Action is submitted.
     * We validate the event with our EventFormValidator, handle the file uploads and finally persist the event in the database.
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\CaptchaEvent $event
     * @param string $captcha
     * @TYPO3\CMS\Extbase\Annotation\Validate("Internetgalerie\IgDynval\Validation\Validator\DynamicValidator", param="event")
     * @TYPO3\CMS\Extbase\Annotation\Validate("\Ig\IgRuckzuckevent\Domain\Validator\EventFormValidator", param="event")
     * @return void
     */
    public function publicCreateAction(\Ig\IgRuckzuckevent\Domain\Model\CaptchaEvent $event): ResponseInterface
    {
        if ($event->getTimeFrom() == '') {
            $event->setTimeFrom(null);
        }
        if ($event->getTimeTo() == '') {
            $event->setTimeTo(null);
        }
        if ($event->getTimeDeadline() == '') {
            $event->setTimeDeadline(null);
        }
        $event->setActive(0);
        $event->setIsNew(1);
        $this->eventRepository->add($event);
        GeneralUtility::makeInstance(\TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager::class)->persistAll();

        $variables = [
            'event' => $event,
            'settings' => $this->settings,
            'hash' => sha1($event->getUid() . 'ruckzuck-create'),
            'isAdmin' => true,
        ];

        $this->sendHtmlMail(
            LocalizationUtility::translate('email.newEvent.subject', 'IgRuckzuckevent'),
            [$this->settings['debugMail'] ?: $this->settings['mailFrom']],
            [$this->settings['debugMail'] ?: $this->settings['mailTo']],
            'IgRuckzuckevent/Event/NewEvent',
            $variables
        );

        $variables['isAdmin'] = false;

        $this->sendHtmlMail(
            LocalizationUtility::translate('email.newEvent.user.subject', 'IgRuckzuckevent'),
            [$this->settings['debugMail'] ?: $this->settings['mailTo']],
            [$this->settings['debugMail'] ?: $event->getEmail()],
            'IgRuckzuckevent/Event/NewEvent',
            $variables
        );

        return $this->redirectToUri($this->uriBuilder->reset()->setTargetPageUid((int)$this->settings['publicCreateThankYouPageUid'])->build());
    }

    /**
     * Action accept
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param string $hash
     */
    public function acceptAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event, $hash): ResponseInterface
    {
        $compareHash = sha1($event->getUid() . 'ruckzuck-create');

        if (!$event->getActive() && $hash == $compareHash) {
            $event->setActive(1);
            $this->eventRepository->update($event);
        }

        $this->view->assign('event', $event);

        return $this->htmlResponse();
    }

    /**
     * Shows the edit form of an event
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param string $returnUrl
     * @return void
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("event")
     */
    public function editAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event, $returnUrl = ''): ResponseInterface
    {

        $this->getAssignSearch();
        $this->assignEditForm();
            
        $this->view->assign('event', $event);
        $this->view->assign('returnUrl', $returnUrl);

        return $this->htmlResponse();
    }

    /**
     * Calculate slug and return as json
     * @param  Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @return string Slug JSON
     */
    public function slugSuggestAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event): ResponseInterface
    {
        $event->makeSlug();
        return $this->jsonResponse(json_encode(['slug' => $event->getPathSegment()]));
    }

    /**
     * This action is called when the edit form of an event is submitted.
     * It persists the changes made to the event to the database.
     * The event is validated by our EventFormValidator.
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param string $returnUrl
     * @return void
     * @TYPO3\CMS\Extbase\Annotation\Validate("Internetgalerie\IgDynval\Validation\Validator\DynamicValidator", param="event")
     * @TYPO3\CMS\Extbase\Annotation\Validate("\Ig\IgRuckzuckevent\Domain\Validator\EventFormValidator", param="event")
     */
    public function updateAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event, $returnUrl = ''): ResponseInterface
    {
        if(!$event->getDateFrom() && $event->getDateTo()) {
            $event->setDateFrom($event->getDateTo());
        }
        if ($event->getTimeFrom() == '') {
            $event->setTimeFrom(null);
        }
        if($event->getTimeTo() == '') {
            $event->setTimeTo(null);
        }
        if ($event->getTimeDeadline() == '') {
            $event->setTimeDeadline(null);
        }
        /*        $arguments = $this->request->getArguments();
        // Remove images to delete
        if (isset($arguments['imagesToDelete']) && !empty($arguments['imagesToDelete'])) {
        $imagesToDelete = [];
        foreach ($event->getImages() as $image) {
        if (in_array($image->getUid(), $arguments['imagesToDelete'])) {
        $imagesToDelete[] = $image;
        }
        }
        foreach ($imagesToDelete as $imageToDelete) {
        $event->removeImage($imageToDelete);
        }
        }
         */
        $event->makeSlug($event->getPathSegment());
        $this->eventRepository->update($event);

        // Delete realurl cache
        /*$connectionPool = GeneralUtility::makeInstance(ConnectionPool::class);
        $qbUrlCache = $connectionPool->getQueryBuilderForTable('tx_realurl_urldata');
        $qbCacheMap = $connectionPool->getQueryBuilderForTable('tx_realurl_uniqalias_cache_map');
        $qbCacheMapDelete = $connectionPool->getQueryBuilderForTable('tx_realurl_uniqalias_cache_map');
        $qbAlias = $connectionPool->getQueryBuilderForTable('tx_realurl_uniqalias');

        $urlCacheResult = $qbUrlCache->select('uid')
        ->from('tx_realurl_urldata')
        ->where("request_variables like '%\"tx_igruckzuckevent_eventdetail[event]\":\"" . intval($event->getUid()) . "\"%'")
        ->execute();

        foreach ($urlCacheResult->fetchAll() as $urlCacheRow) {
        $cacheMapResult = $qbCacheMap->select('*')
        ->from('tx_realurl_uniqalias_cache_map')
        ->where($qbCacheMap->expr()->eq('url_cache_id', $qbCacheMap->createNamedParameter($urlCacheRow['uid'])))
        ->execute();

        foreach ($cacheMapResult->fetchAll() as $cacheMapRow) {
        $aliasDeleted = $qbAlias->delete('tx_realurl_uniqalias')
        ->where($qbAlias->expr()->eq('uid', $qbAlias->createNamedParameter($cacheMapRow['alias_uid'])))
        ->execute();

        $cacheMapDeleted = $qbCacheMapDelete->delete('tx_realurl_uniqalias_cache_map')
        ->where($qbCacheMapDelete->expr()->eq('uid', $qbCacheMapDelete->createNamedParameter($cacheMapRow['uid'])))
        ->execute();
        }
        }

        $urlCacheDeleted = $qbUrlCache->delete('tx_realurl_urldata')
        ->where("request_variables like '%\"tx_igruckzuckevent_eventdetail[event]\":\"" . intval($event->getUid()) . "\"%'")
        ->execute();*/
        
        if ($returnUrl) {
            return $this->redirectToUri($returnUrl);
        } else {
            $search = $this->getAssignSearch();
            return $this->redirect('organizerList', null, null, ['search' => $search]);
        }
    }

    /**
     * Deletes an event.
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param string $returnUrl
     * @return void
     */
    public function deleteAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event, $returnUrl = ''): ResponseInterface
    {
        $this->eventRepository->remove($event);

        if ($returnUrl) {
            return $this->redirectToUri($returnUrl);
        } else {
            $search = $this->getAssignSearch();
            return $this->redirect('organizerList', null, null, ['search' => $search]);
        }
    }

    /**
     * Duplicates an event without its registrations
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @param string $returnUrl
     * @return void
     */
    public function duplicateAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event, $returnUrl = ''): ResponseInterface
    {
        $now = new \DateTime();

        if ($event->getDateTimeFrom() && $event->getDateTimeFrom() < $now) {
            $event->setPathSegment($event->getPathSegment() . '-' . $event->getDateTimeFrom()->format('Y'));
            $this->eventRepository->update($event);
        }

        $newEvent = GeneralUtility::makeInstance(\Ig\IgRuckzuckevent\Domain\Model\Event::class);
        $properties = $event->_getProperties();

        unset($properties['uid']);
        unset($properties['dateFrom']);
        unset($properties['registrations']);
        unset($properties['categories']);
        unset($properties['additionalFormFields']);
        unset($properties['sponsors']);
        unset($properties['pathSegment']);

        foreach ($properties as $key => $value) {
            $newEvent->_setProperty($key, $value);
        }

        foreach ($event->getCategories() as $category) {
            $newEvent->addCategory($category);
        }

        foreach ($event->getSponsors() as $sponsor) {
            $newEvent->addSponsor($sponsor);
        }

        foreach ($event->getQualifications() as $qualification) {
            $newEvent->addQualification($qualification);
        }

        foreach ($event->getAdditionalFormFields() as $additionalFormField) {
            $newFormField = GeneralUtility::makeInstance(\Ig\IgRuckzuckevent\Domain\Model\FormField::class);
            $formFieldProperties = $additionalFormField->_getProperties();
            unset($formFieldProperties['uid']);
            foreach ($formFieldProperties as $key => $value) {
                $newFormField->_setProperty($key, $value);
            }
            $newEvent->addAdditionalFormField($newFormField);
        }
        $newEvent->setTitle($newEvent->getTitle() . ' (Kopie)');

        $this->eventRepository->add($newEvent);

        if ($returnUrl) {
            return $this->redirectToUri($returnUrl);
        } else {
            $search = $this->getAssignSearch();
            return $this->redirect('organizerList', null, null, ['search' => $search]);
        }
    }

    public function eventsInProcessingAction(): ResponseInterface
    {
        $this->view->assign('action', 'eventsInProcessing');
        $this->eventRepository->setShowOnlyActiveEntries(false);
        $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument('currentPage') : 1;
        $search = $this->getAssignSearch(['ignoreToday' => 1, 'withoutDate' => 1]);
        $sorting = $this->request->hasArgument('sorting') ? $this->getSorting() : [];
        if (empty($sorting)) {
            $sorting = [
                'uid' => 'DESC',
                'title' => 'ASC',
            ];
        }
        $events = $this->eventRepository->findBySearchAndSorting($search, $sorting);
        $categories = $this->categoryRepository->findAll();
        $qualifications = $this->qualificationRepository->findAll();

        $paginator = new QueryResultPaginator($events, $currentPage, 20);
        $pagination = new NumberedPagination($paginator, 10); // zweiter Argument: maximal Anzahl Links

        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');

        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);

        $this->view->assign('events', $events);
        $this->view->assign('organizer', $this->feUserRepository->findByUid($userId));
        $this->view->assign('categories', $categories);
        $this->view->assign('qualifications', $qualifications);

        return $this->htmlResponse();
    }

    public function eventsArchiveAction(): ResponseInterface
    {
        $this->view->assign('action', 'eventsArchive');
        $this->eventRepository->setShowOnlyActiveEntries(false);
        $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument('currentPage') : 1;

        $search = $this->getAssignSearch(['ignoreToday' => 1]);
        $events = $this->eventRepository->findBySearchAndSorting($search, $this->getSorting());
        $categories = $this->categoryRepository->findAll();
        $qualifications = $this->qualificationRepository->findAll();

        $paginator = new QueryResultPaginator($events, $currentPage, 20);
        $pagination = new NumberedPagination($paginator, 10); // zweiter Argument: maximal Anzahl Links

        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');

        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);

        $this->view->assign('events', $events);
        $this->view->assign('organizer', $this->feUserRepository->findByUid($userId));
        $this->view->assign('categories', $categories);
        $this->view->assign('qualifications', $qualifications);

        return $this->htmlResponse();
    }

    public function htmlTitleAction(): ResponseInterface
    {
        $queryParams = $this->request->getQueryParams();
        $eventDetailGet = $queryParams['tx_igruckzuckevent_eventdetail'] ?? null;
        $useRegistration = false;
        if (!$eventDetailGet) {
            $eventDetailGet = $queryParams['tx_igruckzuckevent_registration'] ?? null;
            $useRegistration = true;
        }
        $event = $this->eventRepository->findByUid($eventDetailGet['event']);
        if ($useRegistration) {
            $title = 'Anmeldung ' . $title;
        } else  {
            $title = 'Kurs: ' . $event->getTitle();
        }
        
        if (strlen($title) > 70) {
            $title = substr($title, 0, 67) . '...';
        }
        $title = '<title>' . $title . '</title>';

        $metaDescription = strip_tags('Zielgruppe: ' . $event->getTargetGroup() . ' | Daten und Zeit: ' . $event->getDatetimelabel() . ' | Ort: ' . $event->getPlace() . ' ' . $event->getDescription());
        $metaDescription = '<meta name="description" content="' . $metaDescription . '" />';

        return $this->htmlResponse($title . "\n" . $metaDescription);
    }

    /**
     * Exports the reservations of an event to an XLSX-File openable in Microsoft Excel
     *
     * @param \Ig\IgRuckzuckevent\Domain\Model\Event $event
     * @return void
     */
    public function exportAction(\Ig\IgRuckzuckevent\Domain\Model\Event $event): ResponseInterface
    {
        //die('not installed');
        //$phpExcelService = GeneralUtility::makeInstanceService('phpexcel');
        $phpExcel = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $phpExcel->setActiveSheetIndex(0);
        $phpExcel->getActiveSheet()->SetCellValue('A1', $event->getTitle()); // First row: Output event title and registration count

        // Output registration count
        $phpExcel->getActiveSheet()->SetCellValue('B1', $this->langService->ll('tx_igruckzuckevent_domain_model_event.num_participants') . ':');
        $phpExcel->getActiveSheet()->SetCellValue('C1', $event->getRegistrations()->count());
        $phpExcel->getActiveSheet()->getStyle('A1:C1')->getFont()->setBold(true); // Set first row bold

        // Second row: Output attribute titles
        $phpExcel->getActiveSheet()->SetCellValue('A2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.gender'));
        $phpExcel->getActiveSheet()->SetCellValue('B2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.first_name'));
        $phpExcel->getActiveSheet()->SetCellValue('C2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.name'));
        $phpExcel->getActiveSheet()->SetCellValue('D2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.street'));
        $phpExcel->getActiveSheet()->SetCellValue('E2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.zip'));
        $phpExcel->getActiveSheet()->SetCellValue('F2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.city'));
        $phpExcel->getActiveSheet()->SetCellValue('G2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.phone'));
        $phpExcel->getActiveSheet()->SetCellValue('H2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.phone_work'));
        $phpExcel->getActiveSheet()->SetCellValue('I2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.email'));
        $phpExcel->getActiveSheet()->SetCellValue('J2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.comment'));
        $phpExcel->getActiveSheet()->SetCellValue('K2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.company'));
        $phpExcel->getActiveSheet()->SetCellValue('L2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.company_street'));
        $phpExcel->getActiveSheet()->SetCellValue('M2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.company_zip'));
        $phpExcel->getActiveSheet()->SetCellValue('N2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.company_city'));
        $phpExcel->getActiveSheet()->SetCellValue('O2', $this->langService->ll('tx_igruckzuckevent_domain_model_registration.bill'));

        $lastHeaderRow = 'P';
        if ($event->getAdditionalFormFields()->count()) {
            $start = ord('P');
            $end = $start + (int)$event->getAdditionalFormFields()->count();

            $char = $start;
            foreach ($event->getAdditionalFormFields() as $formField) {
                $phpExcel->getActiveSheet()->SetCellValue(chr($char) . '2', $formField->getTitle());
                $char++;
            }

            $lastHeaderRow = chr($char);
        }

        $phpExcel->getActiveSheet()->getStyle('A2:' . $lastHeaderRow . '2')->getFont()->setBold(true); // Set second row bold

        // For each registration of the event set the values of the attributes to the according column
        $i = 3; // We are beginning at the 3rd row (Excel rows count from 1)
        foreach ($event->getRegistrations() as $registration) {
            // Concatenate all statusses separated by a ","
            $statusses = '';
            $isFirst = true;
            foreach ($registration->getStatus() as $status) {
                if (!$isFirst) {
                    $statusses .= ', ';
                }
                $statusses .= $status->getStatus();

                if ($isFirst) {
                    $isFirst = false;
                }
            }

            $phpExcel->getActiveSheet()->SetCellValue('A' . $i, $this->langService->ll('tx_igruckzuckevent_domain_model_registration.gender.' . $registration->getGender()));
            $phpExcel->getActiveSheet()->SetCellValue('B' . $i, $registration->getFirstName());
            $phpExcel->getActiveSheet()->SetCellValue('C' . $i, $registration->getName());
            $phpExcel->getActiveSheet()->SetCellValue('D' . $i, $registration->getStreet());
            $phpExcel->getActiveSheet()->SetCellValue('E' . $i, $registration->getZip());
            $phpExcel->getActiveSheet()->SetCellValue('F' . $i, $registration->getCity());
            $phpExcel->getActiveSheet()->SetCellValueExplicit('G' . $i, $registration->getPhone(), \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING);
            $phpExcel->getActiveSheet()->SetCellValueExplicit('H' . $i, $registration->getPhoneWork(), \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING);
            $phpExcel->getActiveSheet()->SetCellValue('I' . $i, $registration->getEmail());
            $phpExcel->getActiveSheet()->SetCellValue('J' . $i, preg_replace('#\R+#', ' ', $registration->getComment()));
            $phpExcel->getActiveSheet()->SetCellValue('K' . $i, $registration->getCompany());
            $phpExcel->getActiveSheet()->SetCellValue('L' . $i, $registration->getCompanyStreet());
            $phpExcel->getActiveSheet()->SetCellValue('M' . $i, preg_replace('#\R+#', ' ', $registration->getCompanyZip()));
            $phpExcel->getActiveSheet()->SetCellValue('N' . $i, $registration->getCompanyCity());
            $phpExcel->getActiveSheet()->SetCellValue('O' . $i, $this->langService->ll('tx_igruckzuckevent_domain_model_registration.bill.' . ($registration->getBill() ? '1' : '0')));

            if ($registration->getAdditionalFields()) {
                $start = ord('P');
                $end = $start + (int)$event->getAdditionalFormFields()->count();

                $char = $start;
                foreach ($event->getAdditionalFormFields() as $formField) {
                    foreach ($registration->getAdditionalFields() as $fieldname => $additionalField) {
                        if ('additional-' . $formField->getUid() == $fieldname) {
                            if ($formField->getType() == 'Checkbox') {
                                $index = 0;
                                $v = '';
                                foreach ($additionalField as $value) {
                                    if ($index != 0) {
                                        $v .= ', ';
                                    }
                                    $v .= $formField->getOptionsArray()[$value];
                                    $index++;
                                }
                                $phpExcel->getActiveSheet()->SetCellValue(chr($char) . $i, $v);
                            } elseif ($formField->getType() == 'Radio' || $formField->getType() == 'Dropdown') {
                                $phpExcel->getActiveSheet()->SetCellValue(chr($char) . $i, $formField->getOptionsArray()[$additionalField]);
                            } elseif ($formField->getType() != 'Title') {
                                $phpExcel->getActiveSheet()->SetCellValue(chr($char) . $i, preg_replace('#\R+#', ' ', $additionalField));
                            }
                            $char++;
                        }
                    }
                }
            }
            $i++;
        }

        // Set column width automatically according to their content
        foreach (range('A', $lastHeaderRow) as $column) {
            $phpExcel->getActiveSheet()->getColumnDimension($column)->setAutoSize(true);
        }

        // get base name from domain (subdomain if not www, else domain part (e.x. www.aaa.ch -> aaa,  aaa.domain.ch -> aaa, aaa.ch -> aaa)
        $site = $this->request->getAttribute('site', null);
        $host = $site->getBase()->getHost();
        // @todo do we need this for äöü (e.g. xn--kleinwiederkuer-clb.ch)
        $host = idn_to_utf8($host);
        list($servername, $domainName) = explode('.', $host);
        $baseName = $servername == 'www' ? $domainName : $servername;

        $fileName = $baseName . '-kurs-' . $event->getUid() . '-' . time() . '.xlsx'; // Temporary filename

        // Set headers
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . str_replace('typo3temp/', '', $fileName) . '"');
        header('Cache-Control: max-age=0'); // If you're serving to IE 9, then the following may be needed
        header('Cache-Control: max-age=1'); // If you're serving to IE over SSL, then the following may be needed
        header('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); // Date in the past
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT'); // always modified
        header('Cache-Control: cache, must-revalidate'); // HTTP/1.1
        header('Pragma: public'); // HTTP/1.0

        $excelWriter = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($phpExcel); // Create an Excel Writer
        $tempFilename = tempnam('', 'tempfile');
        $excelWriter->save($tempFilename);
        header('Content-Length: ' . filesize($tempFilename));
        unlink($tempFilename);
        $excelWriter->save('php://output');
        exit(0);
        return $this->htmlResponse();
    }


    /**
     * Get the search array
     *
     * @return array
     */
    private function getAssignSearch($extraSearch = [])
    {
        $search = [];
        if ($this->request) {
            $search = $this->request->hasArgument('search') ? $this->request->getArgument('search') : [];
        } elseif (GeneralUtility::_GP('tx_igruckzuckevent_events') && GeneralUtility::_GP('tx_igruckzuckevent_events')['search']) {
            $search = GeneralUtility::_GP('tx_igruckzuckevent_events')['search'];
        } elseif (GeneralUtility::_GP('tx_igruckzuckevent_eventsarchive') && GeneralUtility::_GP('tx_igruckzuckevent_eventsarchive')['search']) {
            //$arguments = [];
        }
        $search = array_merge($search, $extraSearch);

        $this->view->assign('search', $search);
        return $search;
    }

    /**
     * Get the sorting array for the repository
     *
     * @return array
     */
    private function getSorting()
    {
        $defaultSorting = [
            'field' => 'dateFrom',
            'direction' =>  [
                'dateFrom' => 'ASC',
                'title' => 'ASC',
            ],
        ];
        
        if ($this->request) {
            $search = $this->request->hasArgument('search') ? $this->request->getArgument('search') : [];
            if ($search['sorting'] ?? false) {
                list($sortingField, $sortingDirection) =  GeneralUtility::trimExplode(' ', $search['sorting'], true) + ['dateFrom', 'ASC'];
                $sorting = [
                    'field' => $sortingField,
                    'direction' =>  [
                        $sortingField => $sortingDirection,
                    ]
                ];
            } else {
                $sorting = $this->request->hasArgument('sorting') ? $this->request->getArgument('sorting') : $defaultSorting;
            }
        } else {
            $sorting = $defaultSorting;
        }
        // This is the sorting array given to the repository
        $sortingArr = [$sorting['field'] => $sorting['direction'][$sorting['field']]];
        if ($sorting['field'] == 'dateFrom') {
            $sortingArr['timeFrom'] = $sorting['direction'][$sorting['field']];
        }
        // If the current sorting direction of the current sorting field is ASC, switch it  to DESC, but all others stay at ASC
        // Else switch all sorting directions to ASC
        $currentSorting = $sorting;
        if ($sorting['direction'][$sorting['field']] == 'ASC') {
            foreach ($sorting['direction'] as &$sortingDirection) {
                $sortingDirection = 'ASC';
            }
            $sorting['direction'][$sorting['field']] = 'DESC';
        } else {
            foreach ($sorting['direction'] as &$sortingDirection) {
                $sortingDirection = 'ASC';
            }
        }
        $this->view->assign('currentSorting', $currentSorting);
        $this->view->assign('sorting', $sorting);
        return $sortingArr;
    }
    

    public function getAssignGlobalSearch()
    {
        if (!empty(GeneralUtility::_GP('tx_igruckzuckevent_events'))) {
            if (isset(GeneralUtility::_GP('tx_igruckzuckevent_events')['search'])) {
                $search = GeneralUtility::_GP('tx_igruckzuckevent_events')['search'];
            }
            if (isset(GeneralUtility::_GP('tx_igruckzuckevent_events')['sorting'])) {
                $sorting = GeneralUtility::_GP('tx_igruckzuckevent_events')['sorting'];
            }
        } elseif (!empty(GeneralUtility::_GP('tx_igruckzuckevent_eventmanagement'))) {
            if (isset(GeneralUtility::_GP('tx_igruckzuckevent_eventmanagement')['search'])) {
                $search = GeneralUtility::_GP('tx_igruckzuckevent_eventmanagement')['search'];
            }
            if (isset(GeneralUtility::_GP('tx_igruckzuckevent_eventmanagement')['sorting'])) {
                $sorting = GeneralUtility::_GP('tx_igruckzuckevent_eventmanagement')['sorting'];
            }
        }
        $this->view->assign('search', $search);
        $this->view->assign('sorting', $sorting);
    }

    public function searchFormAction(): ResponseInterface
    {
        $categories = $this->categoryRepository->findAll();
        $this->view->assign('categories', $categories);
        $this->view->assign('pageUid', $GLOBALS['TSFE']->id);
        $this->getAssignGlobalSearch();

        return $this->htmlResponse();
    }
    public function assignEditForm()
    {
        $categories = $this->categoryRepository->findAll();
        $sponsors = $this->sponsorRepository->findAll();
        $organizers = $this->organizerRepository->findAll();
        $eventgroups = $this->eventgroupRepository->findAll();
        $qualifications = $this->qualificationRepository->findAll();
        
        $context = GeneralUtility::makeInstance(Context::class);
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');

        $this->view->assign('categories', $categories);
        $this->view->assign('sponsors', $sponsors);
        $this->view->assign('organizers', $organizers);
        $this->view->assign('eventgroups', $eventgroups);
        $this->view->assign('qualifications', $qualifications);
        $this->view->assign('organizer', $this->feUserRepository->findByUid($userId));
    }


    /**
     * Handle image upload
     * We need our own FileReference-class because with the one provided
     * by TYPO3 there is no possibility to set the actual file it is referencing.
     *
     * @param \Internetgalerie\IgUploadtest\Domain\Model\Test $test
     * @param string $field
     * @return void
     */
    /*
    private function handleUpload(\Ig\IgRuckzuckevent\Domain\Model\Event $event)
    {
        $storage = \TYPO3\CMS\Core\Resource\ResourceFactory::getInstance()->getStorageObject(1);
        // Get standard storage with uid 1
        $uploadedFiles = $this->uploadFileService->getUploadedFiles('images');
        foreach ($uploadedFiles as $uploadedFile) {
            // Add file to storage
            $file = $storage->addFile($uploadedFile->getTemporaryFileNameAndPath(), $storage->getRootLevelFolder(), $uploadedFile->getFileName(), 'changeName');
            // Create file reference and set the file
            $fileReference = GeneralUtility::makeInstance(FileReference::class);
            $fileReference->setFile($file);
            // Add reference to Model
            $event->addImage($fileReference);
        }
    }
    */
}
