<?php
namespace Ig\IgRuckzuckevent\Domain\Model;

use DateTime;
use IntlDateFormatter;
use Ig\IgRuckzuckevent\Domain\Repository\EventRepository;
use Ig\IgRuckzuckevent\Utility\JsonUtility;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\StringUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

/**
 * The Eventgroup model class is representing a group of events
 */
class Eventgroup extends \TYPO3\CMS\Extbase\DomainObject\AbstractEntity
{
    protected string $label = '';
    protected bool $recurrenceEnable = false;
    protected string $recurrenceFrequency = '';
    protected int $recurrenceInterval = 1;
    protected ?DateTime $recurrenceStartsOn = null;
    protected ?DateTime $recurrenceEndsOn = null;
    protected ?int $recurrenceEndsAfter = null;
    protected string $recurrenceRule = '';
    protected string $recurrenceExcludeDates = '';
    protected ?array $recurrenceExcludeDatesArray = null;
    protected string $recurrenceAdditionalDates = '';
    protected ?array $recurrenceAdditionalDatesArray = null;
    protected ?string $uidString = null;
    protected ?DateTime $maxRecurrenceDate = null;

    
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->firstEvent = null;
    }

    public function getMaxRecurrenceDate(): DateTime
    {
        if ($this->maxRecurrenceDate === null) {
            $this->maxRecurrenceDate = new DateTime((date('Y') + 2) . '-12-31');
            // @todo set this from constants? and do something like
            // $maxRecurrenceDateModifier = 'last day of December +2 years';
            // $currentDate = new DateTime();
            // $this->maxRecurrenceDate = $currentDate->modify($maxRecurrenceDateModifier);
        }
        return $this->maxRecurrenceDate;
    }

    public function getUidString()
    {
        if ($this->uidString === null) {
            $this->uidString = $this->getUid() ?? StringUtility::getUniqueId('NEW');
        }
        return $this->uidString;
    }
    
    public function setWithFormData(array $recurrence, string $title = ''): bool
    {
        if (!isset($recurrence['frequency'])) {
            return false;
        }
        $this->recurrenceFrequency = $recurrence['frequency'];
        $this->recurrenceEnable = $this->recurrenceFrequency != '';
        $this->recurrenceInterval = 1;
        if ($recurrence['ends_on'] ?? false) {
            $this->recurrenceEndsOn = new DateTime($recurrence['ends_on']);
        } else {
            $this->recurrenceEndsOn = null;
        }
        // 0 is for ever
        if ($recurrence['ends_after'] ?? false) {
            $this->recurrenceEndsAfter = (int)$recurrence['ends_after'];
        } else {
            $this->recurrenceEndsAfter = null;
        }
        $labels = [];
        if ($title) {
            $labels[] = $title;
        }
        $labels[] = '(' . $this->recurrenceFrequency . ($this->recurrenceStartsOn ? ' from ' . $this->recurrenceStartsOn->format('Y-m-d') : '') . ')';
        $this->setLabel(implode(' ' , $labels));
        return true;
    }
    
    protected function initrecurrenceExcludeDatesArray(): void
    {
        if ($this->recurrenceExcludeDatesArray === null) {
            $this->recurrenceExcludeDatesArray = json_decode($this->recurrenceExcludeDates, true) ?? [];
        }
    }

    protected function initrecurrenceAdditionalDatesArray(): void
    {
        if ($this->recurrenceAdditionalDatesArray === null) {
            $this->recurrenceAdditionalDatesArray = json_decode($this->recurrenceAdditionalDates, true) ?? [];
        }
    }

    public function getData(int $pid): array
    {
        return [
            'uid' => $this->getUidString(),
            'pid' => $pid,
            'label' => $this->label,
            'recurrence_enable' => $this->recurrenceEnable ? 1 : 0,
            'recurrence_frequency' => $this->recurrenceFrequency,
            'recurrence_interval' => $this->recurrenceInterval,
            'recurrence_starts_on' => $this->recurrenceStartsOn ? $this->recurrenceStartsOn->format('Y-m-d') : null,
            'recurrence_ends_on' => $this->recurrenceEndsOn ? $this->recurrenceEndsOn->format('Y-m-d') : null,
            'recurrence_ends_after' => $this->recurrenceEndsAfter,
            'recurrence_exclude_dates' => $this->getExcludeDatesJson(),
            'recurrence_additional_dates' => $this->getAdditionalDatesJson(),
            'recurrence_rule' => $this->recurrenceRule,
        ];
    }

    protected function dateformat(string $date): ?string
    {
        if ($date === null || $date === '') {
            return null;
        }
        return substr($date, 0, 10);
    }


    // Getter and Setter for recurrenceAdditionalDates
    public function getRecurrenceExcludeDates(): string
    {
        return $this->recurrenceExcludeDates;
    }

    public function setRecurrenceExcludeDates(string $recurrenceExcludeDates): void
    {
        $this->recurrenceExcludeDates = $recurrenceExcludeDates;
    }
    public function addExcludeDate(string $excludeDate): void
    {
        $excludeDate = $this->dateformat($excludeDate);
        $this->initrecurrenceExcludeDatesArray();
        if (!in_array($excludeDate, $this->recurrenceExcludeDatesArray, true)) {
            $this->recurrenceExcludeDatesArray[] = $excludeDate;
        }

    }
    public function removeExcludeDate(string $excludeDate): void
    {
        $excludeDate = $this->dateformat($excludeDate);
        $this->initrecurrenceExcludeDatesArray();
        $this->recurrenceExcludeDatesArray = array_filter(
            $this->recurrenceExcludeDatesArray,
            fn($date) => $date !== $excludeDate
        );
    }
    public function getExcludeDatesArray(): array
    {
        $this->initrecurrenceExcludeDatesArray();
        return $this->recurrenceExcludeDatesArray;
    }
    
    public function getExcludeDatesJson(): string
    {
        return json_encode($this->getExcludeDatesArray());
    }


    public function isExcludeDate(string $date): bool
    {
        $date = $this->dateformat($date);
        return in_array($date, $this->getExcludeDatesArray());
    }

    public function isEventExcluded(Event $event): bool
    {
        $date = $event->getDateFrom()->format('Y-m-d');
        return $this->isExcludeDate($date);
    }


    // Getter and Setter for recurrenceAdditionalDates
    public function getRecurrenceAdditionalDates(): string
    {
        return $this->recurrenceAdditionalDates;
    }

    public function setRecurrenceAdditionalDates(string $recurrenceAdditionalDates): void
    {
        $this->recurrenceAdditionalDates = $recurrenceAdditionalDates;
    }
    public function addAdditionalDate(string $additionalDate): void
    {
        $additionalDate = $this->dateformat($additionalDate);
        $this->initrecurrenceAdditionalDatesArray();
        if (!in_array($additionalDate, $this->recurrenceAdditionalDatesArray, true)) {
            $this->recurrenceAdditionalDatesArray[] = $this->dateformat($additionalDate);
        }
    }
    public function removeAdditionalDate(string $additionalDate): void
    {
        $additionalDate = $this->dateformat($additionalDate);
        $this->initrecurrenceAdditionalDatesArray();
        $this->recurrenceAdditionalDatesArray = array_filter(
            $this->recurrenceAdditionalDatesArray,
            fn($date) => $date !== $additionalDate
        );
    }
    public function getAdditionalDatesArray(): array
    {
        $this->initrecurrenceAdditionalDatesArray();
        return $this->recurrenceAdditionalDatesArray;
    }
    
    public function getAdditionalDatesJson(): string
    {
        return json_encode($this->getAdditionalDatesArray());
    }


    public function isAdditionalDate(string $date): bool
    {
        $date = $this->dateformat($date);
        return in_array($date, $this->getAdditionalDatesArray());
    }

    public function getLabel(): string
    {
        return $this->label;
    }

    public function setLabel(string $label): void
    {
        $this->label = $label;
    }

    public function getText(): string
    {
        $language = $GLOBALS['TYPO3_REQUEST']->getAttribute('language');
        $locale = $language ? $language->getLocale() : 'de_DE';
        $dateFormatter = new IntlDateFormatter(
            $locale,
            IntlDateFormatter::SHORT,
            IntlDateFormatter::NONE
        );
        $startsOn = $dateFormatter->format($this->getRecurrenceStartsOn());

        $endsOn = $this->getRecurrenceEndsOn() ? $dateFormatter->format($this->getRecurrenceEndsOn()) : '';
        $endsAfter =  $this->getRecurrenceEndsAfter();
        return $this->getRecurrenceFrequencyText() . ' ' . $startsOn . ($endsOn ? ' - ' . $endsOn : '') . ($endsAfter ? ' x ' . $endsAfter : '');
    }
    
    public function getRecurrenceEnable(): bool
    {
        return $this->recurrenceEnable;
    }

    public function setRecurrenceEnable(bool $recurrenceEnable): void
    {
        $this->recurrenceEnable = $recurrenceEnable;
    }

    // Getter and Setter for recurrenceFrequency
    public function getRecurrenceFrequency(): string
    {
        return $this->recurrenceFrequency;
    }
    public function getRecurrenceFrequencyKey(): string
    {
        return $this->recurrenceFrequency ? : 'none';
    }
    public function getRecurrenceFrequencyText(): string
    {
        return LocalizationUtility::translate('LLL:EXT:ig_ruckzuckevent/Resources/Private/Language/locallang_db.xlf:tx_igruckzuckevent_domain_model_eventgroup.recurrence_frequency.' . $this->getRecurrenceFrequencyKey()) ?? '';
    }

    public function setRecurrenceFrequency(string $recurrenceFrequency): void
    {
        $this->recurrenceFrequency = $recurrenceFrequency;
    }

    public function getRecurrenceFrequencyModifier(): array
    {
        $modifyInIntervalString = '';
        $modifyString = null;
        if ($this->recurrenceEnable) {
            // recurrenceStartsOn is used to get current weekday
            if ($this->recurrenceFrequency == 'weekly') {
                $modifyString = '+1 week';
                $modifyInIntervalString = '';
            }if ($this->recurrenceFrequency == 'weekly') {
                $modifyString = '+1 week';
            } elseif ($this->recurrenceFrequency == 'monthly') {
                $modifyString = '+1 month';
            } elseif ($this->recurrenceFrequency == 'firstinmonth') {
                $modifyString = 'first day of next month';
                $modifyWeekday = strtolower($this->getRecurrenceStartsOn()->format('l')); // Get the current weekday name (e.g., 'monday', 'tuesday')
                $modifyInIntervalString = 'first ' . $modifyWeekday . ' of this month';
            } elseif ($this->recurrenceFrequency == 'lastinmonth') {
                $modifyString = 'first day of next month';
                $modifyWeekday = strtolower($this->getRecurrenceStartsOn()->format('l')); // Get the current weekday name (e.g., 'monday', 'tuesday')
                $modifyInIntervalString = 'last ' . $modifyWeekday . ' of this month';
            } elseif ($this->recurrenceFrequency == 'yearly') {
                $modifyString = '+1 year';
            } elseif ($this->recurrenceFrequency == 'twoweeks') {
                $modifyString = '+2 weeks';
            } else {
                die('error recurrenceFrequency "' . $this->recurrenceFrequency);
            }
        }
        return [
            'modify' => $modifyString,
            'modifyInInterval' => $modifyInIntervalString,
        ];
    }


    
    // Getter and Setter for recurrenceInterval
    public function getRecurrenceInterval(): int
    {
        return $this->recurrenceInterval;
    }

    public function setRecurrenceInterval(int $recurrenceInterval): void
    {
        $this->recurrenceInterval = $recurrenceInterval;
    }

    // Getter and Setter for recurrenceStartsOn
    public function getRecurrenceStartsOn(): ?DateTime
    {
        return $this->recurrenceStartsOn;
    }

    public function setRecurrenceStartsOn(?DateTime $recurrenceStartsOn): void
    {
        $this->recurrenceStartsOn = $recurrenceStartsOn;
    }

    // Getter and Setter for recurrenceEndsOn
    public function getRecurrenceEndsOn(): ?DateTime
    {
        return $this->recurrenceEndsOn;
    }

    public function setRecurrenceEndsOn(?DateTime $recurrenceEndsOn): void
    {
        $this->recurrenceEndsOn = $recurrenceEndsOn;
    }

    // Getter and Setter for recurrenceEndsAfter
    public function getRecurrenceEndsAfter(): ?int
    {
        return $this->recurrenceEndsAfter;
    }

    public function setRecurrenceEndsAfter(?int $recurrenceEndsAfter): void
    {
        $this->recurrenceEndsAfter = $recurrenceEndsAfter;
    }

    // Getter and Setter for recurrenceRule
    public function getRecurrenceRule(): string
    {
        return $this->recurrenceRule;
    }

    public function setRecurrenceRule(string $recurrenceRule): void
    {
        $this->recurrenceRule = $recurrenceRule;
    }


    /**
     * get all afftected dates of the current $eventgroup if $skipFirst is true the first one is not included (only usefull for new events)
     */
    public function getDates(bool $skipFirst = false): array
    {
        $dates = [];
        $modifier = $this->getRecurrenceFrequencyModifier();
        if ($modifier['modify'] === null) {
            return [];
        }

        $endsAfter = $this->getRecurrenceEndsAfter();
        $endsOn = $this->getRecurrenceEndsOn();
        $date = clone ($this->getRecurrenceStartsOn());
        while (($endsAfter === null || $endsAfter > 0) && ($endsOn  === null || $date <= $endsOn) && $date <= $this->getMaxRecurrenceDate()) {
            $newData = $date->format('Y-m-d');
            if ($skipFirst) {
                $skipFirst = false;
            } else {
                if (!$this->isExcludeDate($newData)) {
                    $dates[$newData] = null;
                }
            }
            if ($endsAfter !== null) {
                $endsAfter--;
            }
            $date->modify($modifier['modify']);
            if ($modifier['modifyInInterval']) {
                $date->modify($modifier['modifyInInterval']);
            }
        }
        $additionalDates = $this->getAdditionalDatesArray();
        foreach ($additionalDates as $additionalDate) {
            $dates[$additionalDate] = null;
        }
        return $dates;
    }








    



    public function setRepeated(bool $repeated): void
    {
        $this->recurrenceEnable = $repeated;
    }
    public function getRepeated(): bool
    {
        return $this->recurrenceEnable;
    }
    public function isRepeated(): bool
    {
        return $this->getRepeated();
    }


    
    /**
     * Returns the events
     *
     * @return \TYPO3\CMS\Extbase\Persistence\ObjectStorage<\Ig\IgRuckzuckevent\Domain\Model\Event> $events
     */
    public function getEvents()
    {
        $eventRepository = GeneralUtility::makeInstance(EventRepository::class);
        return $eventRepository->findByEventgroup($this);
    }

    public function getCurrentEvent()
    {
        $eventRepository = GeneralUtility::makeInstance(EventRepository::class);
        return $eventRepository->findCurrentEvent($this);
    }



    /**
     * Returns the firstDate
     *
     * @return \DateTime $firstDate
     */
    public function getTitle()
    {
        $eventRepository = GeneralUtility::makeInstance(EventRepository::class);
        $event = $eventRepository->findByEventgroup($this)->getFirst();

        if (!$event) {
            return null;
        }

        return $event->getTitle();
    }



    /**
     * Returns the fromDate of next Event
     *
     * @return \DateTime $fromDate
     */
    public function getEventDateFrom()
    {
        $eventRepository = GeneralUtility::makeInstance(EventRepository::class);
        if ($this->firstEvent == null) {
            $this->firstEvent = $eventRepository->findEventInNextYear($this)->getFirst();
        }

        if (!$this->firstEvent) {
            return null;
        }

        return $this->firstEvent->getDateFrom();
    }

    /**
     * Returns the title of next Event
     *
     * @return \DateTime $fromDate
     */
    public function getEventTitle()
    {
        $eventRepository = GeneralUtility::makeInstance(EventRepository::class);
        if ($this->firstEvent == null) {
            $this->firstEvent = $eventRepository->findEventInNextYear($this)->getFirst();
        }

        if (!$this->firstEvent) {
            return null;
        }

        return $this->firstEvent->getTitle();
    }

    // get numbers of events of this group in next year
    public function getEventsCount()
    {
        $eventRepository = GeneralUtility::makeInstance(EventRepository::class);
        return $eventRepository->countEventInNextYear($this);
    }

    public function __toString()
    {
        return $this->getTitle() . ': '. strftime('%a', $this->getEventDateFrom()->getTimestamp()) .' ' . $this->getEventDateFrom()->format('d.m.Y'). ' - '.($this->getRepeatTill ? $this->getRepeatTill()->format('d.m.Y') : '').' (' .$this->getEventsCount() . ')';
    }

    public function getJson()
    {
        return JsonUtility::getJsonForProperties($this->_getProperties());
    }

}
