<?php

declare(strict_types=1);

namespace Ig\IgRuckzuckevent\Service;

use Psr\Http\Message\ServerRequestInterface;
use TYPO3\CMS\Core\Mail\FluidEmail;
use TYPO3\CMS\Core\Mail\Mailer;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Fluid\View\TemplatePaths;

/**
 * Service for sending emails using FluidEmail
 */
class MailService
{
    /**
     * @var array
     */
    protected $settings;

    /**
     * @var ServerRequestInterface|null
     */
    protected $request;

    /**
     * Send an email using FluidEmail
     *
     * @param string $subject Email subject
     * @param array $from Sender address(es) - format: ['email@example.com' => 'Name'] or ['email@example.com']
     * @param array $to Recipient address(es) - format: ['email@example.com' => 'Name'] or ['email@example.com']
     * @param string $template Template path (e.g., 'IgRuckzuckevent/Registration/Create/User')
     * @param array $variables Variables to assign to the template
     * @return void
     */
    public function sendMail(
        string $subject,
        array $from,
        array $to,
        string $template,
        array $variables = []
    ): void {
        $email = $this->buildFluidEmail($subject, $from, $to, $template, $variables);
        $this->send($email);
    }

    /**
     * Build a FluidEmail instance without sending it
     * Useful when you need to dispatch events before sending
     *
     * @param string $subject
     * @param array $from
     * @param array $to
     * @param string $template
     * @param array $variables
     * @return FluidEmail
     */
    public function buildFluidEmail(
        string $subject,
        array $from,
        array $to,
        string $template,
        array $variables = []
    ): FluidEmail {
        $mailTemplatePaths = $this->getMailTemplatePaths();

        $email = GeneralUtility::makeInstance(FluidEmail::class, $mailTemplatePaths ?? [])
            ->subject($subject)
            ->from(...$from)
            ->to(...$to)
            ->setTemplate($template)
            ->format('html')
            ->assignMultiple($variables);

        // Add settings if available
        if ($this->settings) {
            $email->assign('settings', $this->settings);
        }

        // Set request context if available
        if ($this->request instanceof ServerRequestInterface) {
            $email->setRequest($this->request);
        }

        return $email;
    }

    /**
     * Send a FluidEmail instance
     * Can be used after building an email and potentially modifying it
     *
     * @param FluidEmail $email
     * @return void
     */
    public function send(FluidEmail $email): void
    {
        GeneralUtility::makeInstance(Mailer::class)->send($email);
    }

    /**
     * Returns an instance of TemplatePaths with paths configured in TypoScript and
     * paths configured in $GLOBALS['TYPO3_CONF_VARS']['MAIL']
     *
     * @return TemplatePaths
     */
    protected function getMailTemplatePaths(): TemplatePaths
    {
        $pathArray = [
            'layoutRootPaths'   => $GLOBALS['TYPO3_CONF_VARS']['MAIL']['layoutRootPaths'] ?? [],
            'templateRootPaths' => $GLOBALS['TYPO3_CONF_VARS']['MAIL']['templateRootPaths'] ?? [],
            'partialRootPaths'  => $GLOBALS['TYPO3_CONF_VARS']['MAIL']['partialRootPaths'] ?? [],
        ];

        // Merge with settings if available
        if ($this->settings && isset($this->settings['mail'])) {
            $pathArray = array_replace_recursive(
                $pathArray,
                [
                    'layoutRootPaths'   => $this->settings['mail']['layoutRootPaths'] ?? [],
                    'templateRootPaths' => $this->settings['mail']['templateRootPaths'] ?? [],
                    'partialRootPaths'  => $this->settings['mail']['partialRootPaths'] ?? [],
                ]
            );
        }

        return new TemplatePaths($pathArray);
    }

    /**
     * Set TypoScript settings
     *
     * @param array $settings
     * @return void
     */
    public function setSettings(array $settings): void
    {
        $this->settings = $settings;
    }

    /**
     * Set request context
     *
     * @param ServerRequestInterface $request
     * @return void
     */
    public function setRequest(ServerRequestInterface $request): void
    {
        $this->request = $request;
    }
}
