<?php

namespace Ig\IgRuckzuckevent\Service;

use Ig\IgRuckzuckevent\Domain\Model\Event;
use Ig\IgRuckzuckevent\Domain\Model\Registration;
use Ig\IgRuckzuckevent\Domain\Model\WaitlistPromotionStatus;
use Ig\IgRuckzuckevent\Domain\Repository\RegistrationRepository;
use TYPO3\CMS\Extbase\Mvc\ExtbaseRequestParameters;
use TYPO3\CMS\Extbase\Mvc\Request;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Crypto\Random;
use Psr\Http\Message\ServerRequestInterface;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;
use TYPO3\CMS\Core\Site\SiteFinder;
use TYPO3\CMS\Core\Http\ServerRequestFactory;
use TYPO3\CMS\Core\Core\SystemEnvironmentBuilder;
use TYPO3\CMS\Core\Site\Entity\Site;
use TYPO3\CMS\Core\Site\Entity\SiteLanguage;
use TYPO3\CMS\Frontend\Controller\TypoScriptFrontendController;
use TYPO3\CMS\Extbase\Mvc\Web\Routing\UriBuilder;
use TYPO3\CMS\Frontend\ContentObject\ContentObjectRenderer;

/**
 * Service for managing event waitlists
 *
 * This service encapsulates all waitlist-related functionality including:
 * - Checking if registrations should go to waitlist
 * - Promoting waitlist registrations to regular registrations
 * - Automatic propagation of waitlist (FIFO)
 */
class WaitlistService
{
    protected RegistrationRepository $registrationRepository;
    protected PersistenceManager $persistenceManager;
    protected ?MailService $mailService = null;
    protected array $settings = [];
    protected ?ServerRequestInterface $request = null;

    public function __construct(
        RegistrationRepository $registrationRepository,
        PersistenceManager $persistenceManager
    ) {
        $this->registrationRepository = $registrationRepository;
        $this->persistenceManager = $persistenceManager;
    }

    /**
     * Inject MailService for sending promotion emails
     */
    public function injectMailService(MailService $mailService): void
    {
        $this->mailService = $mailService;
    }

    /**
     * Set settings (e.g., from controller)
     */
    public function setSettings(array $settings): void
    {
        $this->settings = $settings;
    }

    /**
     * Set request context for email rendering
     */
    public function setRequest(?ServerRequestInterface $request): void
    {
        $this->request = $request;
    }

    /**
     * Create a frontend request for email rendering from backend context
     *
     * @param int $pageId The page ID to use for site resolution (defaults to 20)
     * @return ServerRequestInterface|null
     */
    protected function createFrontendRequest(Site $site, ?SiteLanguage $language = null): ?ServerRequestInterface
    {
        try {
            // Create ContentObjectRenderer for URL building
            $cObj = GeneralUtility::makeInstance(ContentObjectRenderer::class);

            $extbaseParameters = new ExtbaseRequestParameters('Registration');
            $extbaseParameters->setControllerExtensionName('IgRuckzuckevent');
            $extbaseParameters->setControllerName('Events');

            // Create a basic frontend request with site information
            $request = GeneralUtility::makeInstance(ServerRequestFactory::class)
                ->createServerRequest('GET', $site->getBase())
                ->withAttribute('site', $site)
                ->withAttribute('language', $language ?? $site->getDefaultLanguage())
                ->withAttribute('applicationType', SystemEnvironmentBuilder::REQUESTTYPE_FE)
                ->withAttribute('currentContentObject', $cObj)
                ->withAttribute('extbase', $extbaseParameters);

            $extbaseRequest = new Request($request);

            // Initialize a minimal TSFE to make ContentObjectRenderer work
            /*$tsfe = GeneralUtility::makeInstance(
                TypoScriptFrontendController::class,
                GeneralUtility::makeInstance(Context::class),
                $site,
                $language ?? $site->getDefaultLanguage(),
                null,
                0
            );

            $tsfe->config = ['config' => []];*/

            // Initialize TSFE for TypoScript rendering (needed for f:format.html ViewHelper)
            $GLOBALS['TYPO3_REQUEST'] = $extbaseRequest;
            //$GLOBALS['TSFE'] = $tsfe;

            return $extbaseRequest;
        } catch (\Exception $e) {
            // If we can't create a proper frontend request, return null
            // Emails will still be sent but may not have full site context
            return null;
        }
    }

    /**
     * Check if an event is full for a given registration
     *
     * @param Event $event The event to check
     * @param int $registrationCount Number of registrations to check
     * @return bool True if the event is full
     */
    protected function isEventFull(Event $event, int $registrationCount): bool
    {
        return ($event->getMinRegistrations() > 0 || $event->getMaxRegistrations() > 0)
            && $event->getFreePlaces() < $registrationCount;
    }

    /**
     * Check if a new registration should be placed on the waitlist
     *
     * @param Event $event The event to check
     * @param Registration $registration The registration being created
     * @return bool True if the registration should be waitlisted
     */
    public function shouldBeWaitlisted(Event $event, Registration $registration): bool
    {
        // If waitlist is not enabled, never waitlist
        if (!$event->getEnableWaitlist()) {
            return false;
        }

        return $this->isEventFull($event, $registration->getRegistrationCount());
    }

    /**
     * Check if a registration can be accepted (either regular or waitlist)
     *
     * @param Event $event The event to check
     * @param Registration $registration The registration being created
     * @return bool True if registration can be accepted (either regular or waitlist)
     */
    public function canAcceptRegistration(Event $event, Registration $registration): bool
    {
        // If registrations are not possible, reject
        if (!$event->getRegistrationsPossible()) {
            return false;
        }

        $isEventFull = $this->isEventFull($event, $registration->getRegistrationCount());

        // If full and no waitlist enabled, reject
        if ($isEventFull && !$event->getEnableWaitlist()) {
            return false;
        }

        // Otherwise accept (will be waitlisted if full)
        return true;
    }

    /**
     * Process a new registration and set waitlist status appropriately
     *
     * @param Event $event The event
     * @param Registration $registration The registration to process
     * @return void
     */
    public function processNewRegistration(Event $event, Registration $registration): void
    {
        if ($this->shouldBeWaitlisted($event, $registration)) {
            $registration->setWaitlist(true);
        } else {
            $registration->setWaitlist(false);
        }
    }

    /**
     * Initiate promotion process for a waitlist registration
     * Generates token and sends email with accept/decline links
     *
     * @param Registration $registration The waitlist registration to promote
     * @param bool $sendEmail Whether to send promotion email
     * @param int $deadlineHours Hours until deadline (default 48)
     * @return bool True if promotion was initiated, false if no space available
     */
    public function promoteRegistration(Registration $registration, bool $sendEmail = true, int $deadlineHours = 48): bool
    {
        $event = $registration->getEvent();
        // Check if there's space available for promotion
        // Use getFreePlacesForPromotion() which doesn't count unpromoted waitlist
        if ($event->getFreePlacesForPromotion() >= $registration->getRegistrationCount()) {
            // Generate cryptographically secure unique token
            $token = $this->generateSecureToken($registration);
            $registration->setWaitlistPromotionToken($token);
            $registration->setWaitlistPromotionStatus(WaitlistPromotionStatus::PENDING);
            $registration->setWaitlistPromotionSentAt(new \DateTime());

            // Set deadline
            $deadline = new \DateTime();
            $deadline->modify('+' . $deadlineHours . ' hours');
            $registration->setWaitlistPromotionDeadline($deadline);

            $this->registrationRepository->update($registration);

            // Send promotion email with accept/decline links
            if ($sendEmail && $registration->getEmail() && $this->mailService) {
                $this->sendPromotionEmail($registration);
            }

            return true;
        }

        return false;
    }

    /**
     * Accept a waitlist promotion (user confirmed)
     *
     * @param Registration $registration
     * @return bool
     */
    public function acceptPromotion(Registration $registration): bool
    {
        // Verify it's still pending and not expired
        if (!$registration->isWaitlistPromotionPending()) {
            return false;
        }

        if ($registration->isWaitlistPromotionExpired()) {
            return false;
        }

        // Check if the event's registration deadline has passed
        $event = $registration->getEvent();
        $registrationDeadline = $event->getDateTimeDeadline();
        if ($registrationDeadline) {
            $now = new \DateTime();
            if ($registrationDeadline < $now) {
                // Registration deadline has passed, cannot accept promotion
                return false;
            }
        }

        // Check if the event has already occurred
        $eventDateTime = $event->getDateTimeTo() ?? $event->getDateTimeFrom();
        if ($eventDateTime) {
            $now = new \DateTime();
            if ($eventDateTime < $now) {
                // Event has already passed, cannot accept promotion
                return false;
            }
        }

        // Check if space is still available
        if ($event->getPendingPromotionsCount() < $registration->getRegistrationCount()) {
            return false;
        }

        // Confirm the promotion
        $registration->setWaitlist(false);
        $registration->setWaitlistPromotionStatus(WaitlistPromotionStatus::ACCEPTED);
        $this->registrationRepository->update($registration);

        return true;
    }

    /**
     * Decline a waitlist promotion (user declined)
     * Optionally propagates to next person on waitlist based on settings
     *
     * @param Registration $registration
     * @return bool
     */
    public function declinePromotion(Registration $registration): bool
    {
        // Verify it's pending
        if (!$registration->isWaitlistPromotionPending()) {
            return false;
        }

        // Check if the event's registration deadline has passed
        $event = $registration->getEvent();
        $registrationDeadline = $event->getDateTimeDeadline();
        if ($registrationDeadline) {
            $now = new \DateTime();
            if ($registrationDeadline < $now) {
                // Registration deadline has passed, cannot decline promotion
                return false;
            }
        }

        // Check if the event has already occurred
        $eventDateTime = $event->getDateTimeTo() ?? $event->getDateTimeFrom();
        if ($eventDateTime) {
            $now = new \DateTime();
            if ($eventDateTime < $now) {
                // Event has already passed, cannot decline promotion (no point)
                return false;
            }
        }

        // Mark as declined
        $registration->setWaitlistPromotionStatus(WaitlistPromotionStatus::DECLINED);
        $registration->setWaitlistPromotionToken(''); // Clear token
        $this->registrationRepository->update($registration);
        $this->persistenceManager->persistAll();

        // Check setting before propagating
        $shouldPropagate = !isset($this->settings['waitlist']['propagateAfterDecline'])
            || (bool)$this->settings['waitlist']['propagateAfterDecline'];

        if ($shouldPropagate) {
            // Automatically propagate to next person
            $this->propagateWaitlist($event, true);
        }

        return true;
    }

    /**
     * Cancel a waitlist promotion (admin cancelled)
     * Marks promotion as cancelled (status 3) and puts registration back on waitlist
     * Optionally propagates to next person on waitlist based on settings
     *
     * @param Registration $registration
     * @param bool $sendEmail Whether to send cancellation email
     * @return bool
     */
    public function cancelPromotion(Registration $registration, bool $sendEmail = true): bool
    {
        // Verify it has a promotion (pending or accepted)
        if (empty($registration->getWaitlistPromotionToken())) {
            return false;
        }

        // Check if already declined or cancelled - cannot cancel those
        if ($registration->getWaitlistPromotionStatus() === WaitlistPromotionStatus::DECLINED->value ||
            $registration->getWaitlistPromotionStatus() === WaitlistPromotionStatus::CANCELLED->value) {
            return false;
        }

        $event = $registration->getEvent();

        // Mark as cancelled
        $registration->setWaitlistPromotionStatus(WaitlistPromotionStatus::CANCELLED);
        $registration->setWaitlistPromotionToken(''); // Clear token
        $registration->setWaitlistPromotionSentAt(null);
        $registration->setWaitlistPromotionDeadline(null);

        // Ensure waitlist flag is set to true (in case it was accepted)
        $registration->setWaitlist(true);

        $this->registrationRepository->update($registration);
        $this->persistenceManager->persistAll();

        // Send cancellation email to user
        if ($sendEmail && $registration->getEmail() && $this->mailService) {
            $this->sendCancellationEmail($registration);
        }

        // Check setting before propagating
        $shouldPropagate = !isset($this->settings['waitlist']['propagateAfterCancel'])
            || (bool)$this->settings['waitlist']['propagateAfterCancel'];

        if ($shouldPropagate) {
            // Automatically propagate to next person on waitlist
            // Cancelled registrations (status 3) will be excluded by findWaitlistByEvent
            $this->propagateWaitlist($event, true);
        }

        return true;
    }

    /**
     * Automatically promote waitlist registrations when space is available (FIFO)
     * Initiates promotion process (sends email with accept/decline links)
     *
     * @param Event $event The event to process
     * @param bool $sendEmails Whether to send promotion emails
     * @return int Number of promotion offers sent
     */
    public function propagateWaitlist(Event $event, bool $sendEmails = true): int
    {
        $promoted = 0;

        // Get all waitlist registrations for this event, ordered by creation date (FIFO)
        $waitlistRegistrations = $this->registrationRepository->findWaitlistByEvent($event);

        /** @var Registration $registration */
        foreach ($waitlistRegistrations as $registration) {
            // Skip registrations that already have a pending promotion
            if ($registration->isWaitlistPromotionPending()) {
                continue;
            }

            // Recalculate free places before each check (important after each promotion)
            // Use getFreePlacesForPromotion() which doesn't count unpromoted waitlist
            $currentFreePlaces = $event->getFreePlacesForPromotion();

            // Check if there's space available for this registration
            if ($currentFreePlaces >= $registration->getRegistrationCount()) {
                // Initiate promotion process (generate token, send email)
                if ($this->promoteRegistration($registration, $sendEmails)) {
                    // Persist immediately so the next iteration sees updated free places
                    $this->persistenceManager->persistAll();
                    $promoted++;
                }
            } else {
                // No more space available, stop processing
                break;
            }
        }

        return $promoted;
    }

    /**
     * Get the number of people on the waitlist for an event
     *
     * @param Event $event The event
     * @return int Number of waitlist registrations
     */
    public function getWaitlistCount(Event $event): int
    {
        $waitlistRegistrations = $this->registrationRepository->findWaitlistByEvent($event);
        return $waitlistRegistrations->count();
    }

    /**
     * Get the total number of spaces occupied by waitlist registrations
     *
     * @param Event $event The event
     * @return int Total spaces on waitlist
     */
    public function getWaitlistSpaces(Event $event): int
    {
        $waitlistRegistrations = $this->registrationRepository->findWaitlistByEvent($event);
        $totalSpaces = 0;

        /** @var Registration $registration */
        foreach ($waitlistRegistrations as $registration) {
            $totalSpaces += $registration->getRegistrationCount();
        }

        return $totalSpaces;
    }

    /**
     * Check if an event has any waitlist registrations
     *
     * @param Event $event The event
     * @return bool True if there are waitlist registrations
     */
    public function hasWaitlist(Event $event): bool
    {
        return $this->getWaitlistCount($event) > 0;
    }

    /**
     * Check if an event is accepting any new registrations (regular or waitlist)
     *
     * @param Event $event The event to check
     * @return bool True if event is accepting registrations
     */
    public function isAcceptingRegistrations(Event $event): bool
    {
        // If registrations are not possible, reject
        if (!$event->getRegistrationsPossible()) {
            return false;
        }

        // Check if event is full (checking for minimum 1 registration)
        $isEventFull = $this->isEventFull($event, 1);

        // If full and no waitlist enabled, reject
        if ($isEventFull && !$event->getEnableWaitlist()) {
            return false;
        }

        // Otherwise accept (will be waitlisted if full)
        return true;
    }

    /**
     * Generate a cryptographically secure unique token for waitlist promotion
     * Ensures no collisions by checking against existing tokens
     *
     * @param Registration $registration The registration to generate token for
     * @return string A unique, secure token
     */
    protected function generateSecureToken(Registration $registration): string
    {
        $random = GeneralUtility::makeInstance(Random::class);
        $maxAttempts = 10;
        $attempt = 0;

        do {
            $token = $random->generateRandomHexString(64);
            $existing = $this->registrationRepository->findOneBy(['waitlistPromotionToken' => $token]);
            $attempt++;
        } while ($existing && $attempt < $maxAttempts);

        if ($existing) {
            // Extremely unlikely, but add timestamp to ensure uniqueness
            $token .= bin2hex(random_bytes(16)) . microtime(true);
        }

        return $token;
    }

    /**
     * Send promotion email to a user who has been moved from waitlist
     *
     * @param Registration $registration The promoted registration
     * @return void
     */
    protected function sendPromotionEmail(Registration $registration): void
    {
        if (!$this->mailService || !$registration->getEmail()) {
            return;
        }

        $event = $registration->getEvent();
        // Use mailFrom from settings, or fallback to global mail configuration, or default
        if (!empty($this->settings['mailFrom'])) {
            $mailFrom = $this->settings['mailFrom'];
        } else {
            $mailFrom = $GLOBALS['TYPO3_CONF_VARS']['MAIL']['defaultMailFromAddress'] ?? 'noreply@example.com';
        }

        // Get the list page UID from settings
        $pageUid = (int)($this->settings['listPageUid'] ?? $event->getPid());
        
        $sysLanguageUid = $registration->getLanguage();

        // Try to find the site for this page
        $siteFinder = GeneralUtility::makeInstance(SiteFinder::class);
        $site = $siteFinder->getSiteByPageId($pageUid);
        $language = null;

        try {
            $language = $site->getLanguageById($sysLanguageUid);
        } catch (\Exception $e) {
            // If language not found, fallback to default
            $language = $site->getDefaultLanguage();
        }
        // Set request context - create frontend request if not already set
        if (!$this->request) {
            $this->request = $this->createFrontendRequest($site, $language);
        }

        if ($this->request) {
            $this->mailService->setRequest($this->request);
        }

        // Set settings for mail service
        if ($this->settings) {
            $this->mailService->setSettings($this->settings);
        }

        // Build accept/decline URLs using UriBuilder
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        $uriBuilder->setRequest($this->request)
            ->setTargetPageUid($pageUid)
            ->setCreateAbsoluteUri(true)
            ->setLanguage($language->getLanguageId());

        $acceptUrl = $uriBuilder->uriFor(
            'acceptWaitlistPromotion',
            ['token' => $registration->getWaitlistPromotionToken()],
            'Registration',
            'IgRuckzuckevent',
            'Events'
        );

        $declineUrl = $uriBuilder->uriFor(
            'declineWaitlistPromotion',
            ['token' => $registration->getWaitlistPromotionToken()],
            'Registration',
            'IgRuckzuckevent',
            'Events'
        );

        $variables = [
            'registration' => $registration,
            'event' => $event,
            'language' => $language,
            'acceptUrl' => $acceptUrl,
            'declineUrl' => $declineUrl,
            'deadline' => $registration->getWaitlistPromotionDeadline(),
        ];

        $this->mailService->sendMail(
            LocalizationUtility::translate('email.user.promotion.subject', 'IgRuckzuckevent', null, $language->getLocale()->getLanguageCode()),
            [$mailFrom],
            [$registration->getEmail()],
            'IgRuckzuckevent/Registration/Promotion/User',
            $variables
        );
    }

    /**
     * Send cancellation email to a user whose promotion was cancelled by admin
     *
     * @param Registration $registration The registration whose promotion was cancelled
     * @return void
     */
    protected function sendCancellationEmail(Registration $registration): void
    {
        if (!$this->mailService || !$registration->getEmail()) {
            return;
        }

        $event = $registration->getEvent();
        // Use mailFrom from settings, or fallback to global mail configuration, or default
        if (!empty($this->settings['mailFrom'])) {
            $mailFrom = $this->settings['mailFrom'];
        } else {
            $mailFrom = $GLOBALS['TYPO3_CONF_VARS']['MAIL']['defaultMailFromAddress'] ?? 'noreply@example.com';
        }

        // Get the list page UID from settings
        $pageUid = (int)($this->settings['listPageUid'] ?? $event->getPid());

        $sysLanguageUid = $registration->getLanguage();

        // Try to find the site for this page
        $siteFinder = GeneralUtility::makeInstance(SiteFinder::class);
        $site = $siteFinder->getSiteByPageId($pageUid);
        $language = null;

        try {
            $language = $site->getLanguageById($sysLanguageUid);
        } catch (\Exception $e) {
            // If language not found, fallback to default
            $language = $site->getDefaultLanguage();
        }

        // Set request context - create frontend request if not already set
        if (!$this->request) {
            $this->request = $this->createFrontendRequest($site, $language);
        }
        if ($this->request) {
            $this->mailService->setRequest($this->request);
        }

        // Set settings for mail service
        if ($this->settings) {
            $this->mailService->setSettings($this->settings);
        }

        $variables = [
            'registration' => $registration,
            'event' => $event,
            'language' => $language,
        ];

        $this->mailService->sendMail(
            LocalizationUtility::translate('email.user.cancellation.subject', 'IgRuckzuckevent', null, $language->getLocale()->getLanguageCode()),
            [$mailFrom],
            [$registration->getEmail()],
            'IgRuckzuckevent/Registration/Promotion/Cancelled',
            $variables
        );
    }
}
