/**
 * Additional Form Fields Dependencies Handler
 * Manages conditional visibility of form fields based on dependencies
 */
(function() {
    'use strict';

    /**
     * Get the current value of a form field by its name
     * @param {string} fieldName - The data-name attribute value
     * @returns {string|Array} The current value(s) of the field
     */
    function getFieldValue(fieldName) {
        // Find all fields with this name
        const fields = document.querySelectorAll('[data-name="' + fieldName + '"]');

        if (!fields.length) {
            return '';
        }

        const firstField = fields[0];

        // Handle radio buttons
        if (firstField.type === 'radio') {
            const checked = document.querySelector('[data-name="' + fieldName + '"]:checked');
            return checked ? checked.value : '';
        }

        // Handle checkboxes
        if (firstField.type === 'checkbox') {
            const checkedBoxes = document.querySelectorAll('[data-name="' + fieldName + '"]:checked');
            const values = [];
            checkedBoxes.forEach(function(cb) {
                values.push(cb.value);
            });
            return values;
        }

        // Handle select, input, textarea
        return firstField.value || '';
    }

    /**
     * Check if a field value matches the dependency requirement
     * @param {string|Array} currentValue - Current field value(s)
     * @param {string} requiredValue - Required value from data-depends-value (can be comma-separated)
     * @param {string} operator - Comparison operator: 'equal' (default) or 'not-equal'
     * @returns {boolean} Whether the dependency is satisfied
     */
    function valueMatches(currentValue, requiredValue, operator) {
        // Default to 'equal' if no operator specified
        operator = operator || 'equal';

        // Check if current value is empty
        var hasValue = false;
        if (Array.isArray(currentValue)) {
            hasValue = currentValue.length > 0;
        } else {
            hasValue = currentValue !== '';
        }

        // Empty dependency value means check if field has any value
        if (!requiredValue) {
            // With 'equal': show when field has any value
            // With 'not-equal': show when field has NO value (hide when anything is selected)
            if (operator === 'not-equal' || operator === 'notEqual' || operator === '!=') {
                return !hasValue;
            }
            return hasValue;
        }

        // Split comma-separated values and trim whitespace
        const requiredValues = requiredValue.split(',').map(function(val) {
            return val.trim();
        });

        var matches = false;

        // Handle array values (checkboxes)
        if (Array.isArray(currentValue)) {
            // Check if any of the current values match any of the required values
            matches = requiredValues.some(function(reqVal) {
                return currentValue.indexOf(reqVal) !== -1;
            });
        } else {
            // Direct comparison - check if current value matches any of the required values
            matches = requiredValues.indexOf(currentValue) !== -1;
        }

        // Apply operator logic
        if (operator === 'not-equal' || operator === 'notEqual' || operator === '!=') {
            return !matches;
        }

        return matches;
    }

    /**
     * Update visibility of all dependent fields
     */
    function updateDependentFields() {
        // Find all elements with dependencies
        const dependentElements = document.querySelectorAll('[data-depends-on]');

        dependentElements.forEach(function(element) {
            const dependsOn = element.getAttribute('data-depends-on');
            const dependsValue = element.getAttribute('data-depends-value');
            const dependsOperator = element.getAttribute('data-depends-operator');

            if (!dependsOn) {
                return;
            }

            const currentValue = getFieldValue(dependsOn);
            const shouldShow = valueMatches(currentValue, dependsValue, dependsOperator);

            if (shouldShow) {
                element.style.display = '';
                element.classList.remove('ig-hidden-dependency');

                // Restore validation classes for required fields
                restoreValidationClasses(element);
            } else {
                element.style.display = 'none';
                element.classList.add('ig-hidden-dependency');

                // Store and remove validation classes before hiding
                storeAndRemoveValidationClasses(element);

                // Clear values of hidden fields to prevent submission
                clearFieldValues(element);
            }
        });
    }

    /**
     * Store validation classes and required attribute, then remove them from form fields
     * @param {HTMLElement} container - The container element
     */
    function storeAndRemoveValidationClasses(container) {
        // List of validation classes used in the system
        const validationClasses = [
            'registration-validate',
            'registration-validate-inline',
            'registration-validate-checkbox',
            'registration-email'
        ];

        // Find all form fields and their parent containers
        const inputs = container.querySelectorAll('input, textarea, select');

        inputs.forEach(function(field) {
            // Store and remove the required attribute
            if (field.hasAttribute('required')) {
                field.setAttribute('data-stored-required', 'true');
                field.removeAttribute('required');
            }

            // Store validation classes from the field itself
            const fieldClasses = [];
            validationClasses.forEach(function(validationClass) {
                if (field.classList.contains(validationClass)) {
                    fieldClasses.push(validationClass);
                    field.classList.remove(validationClass);
                }
            });

            // Store in data attribute if any classes were found
            if (fieldClasses.length > 0) {
                field.setAttribute('data-stored-validation-classes', fieldClasses.join(' '));
            }

            // Also check parent elements for validation classes (e.g., registration-validate-checkbox on wrapper)
            let parent = field.parentElement;
            while (parent && parent !== container) {
                const parentClasses = [];
                validationClasses.forEach(function(validationClass) {
                    if (parent.classList.contains(validationClass)) {
                        parentClasses.push(validationClass);
                        parent.classList.remove(validationClass);
                    }
                });

                if (parentClasses.length > 0) {
                    parent.setAttribute('data-stored-validation-classes', parentClasses.join(' '));
                }

                parent = parent.parentElement;
            }
        });
    }

    /**
     * Restore validation classes and required attribute to form fields
     * @param {HTMLElement} container - The container element
     */
    function restoreValidationClasses(container) {
        // Restore required attributes on form fields
        const fieldsWithStoredRequired = container.querySelectorAll('[data-stored-required]');
        fieldsWithStoredRequired.forEach(function(field) {
            field.setAttribute('required', 'required');
            field.removeAttribute('data-stored-required');
        });

        // Find all elements with stored validation classes
        const elementsWithStoredClasses = container.querySelectorAll('[data-stored-validation-classes]');

        elementsWithStoredClasses.forEach(function(element) {
            const storedClasses = element.getAttribute('data-stored-validation-classes');
            if (storedClasses) {
                const classArray = storedClasses.split(' ');
                classArray.forEach(function(className) {
                    element.classList.add(className);
                });
                // Remove the data attribute after restoring
                element.removeAttribute('data-stored-validation-classes');
            }
        });
    }

    /**
     * Clear all form field values within an element
     * @param {HTMLElement} container - The container element
     */
    function clearFieldValues(container) {
        // Find all form fields within this container
        const inputs = container.querySelectorAll('input, textarea, select');

        inputs.forEach(function(field) {
            if (field.type === 'checkbox' || field.type === 'radio') {
                field.checked = false;
            } else {
                field.value = '';
            }
        });
    }

    /**
     * Initialize event listeners for all fields with data-name
     */
    function initializeEventListeners() {
        const fields = document.querySelectorAll('[data-name]');

        fields.forEach(function(field) {
            const eventType = (field.type === 'checkbox' || field.type === 'radio') ? 'change' : 'input';

            field.addEventListener(eventType, function() {
                updateDependentFields();
            });
        });
    }

    /**
     * Initialize the dependency handler
     */
    function init() {
        // Set initial visibility state
        updateDependentFields();

        // Setup event listeners
        initializeEventListeners();
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Expose update function globally for manual triggering if needed
    window.updateFormFieldDependencies = updateDependentFields;

})();
