<?php

namespace Ig\IgRuckzuckevent\Controller;

use Braintree\Transaction;
use DateTime;
use Ig\IgPayment\Services\BraintreeService;
use Ig\IgRuckzuckevent\Domain\Model\Subscription;
use Ig\IgRuckzuckevent\Domain\Model\SubscriptionOffer;
use Ig\IgRuckzuckevent\Domain\Repository\FeUserRepository;
use Ig\IgRuckzuckevent\Domain\Repository\SubscriptionRepository;
use Ig\IgRuckzuckevent\Utility\UserUtility;
use Ig\IgRuckzuckevent\Service\MailService;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Fluid\View\StandaloneView;

/**
 * The EventController handles all actions directly related to Events.
 */
class SubscriptionController extends ActionController
{
    /**
     * subscriptionRepository
     *
     * @var SubscriptionRepository
     */
    protected $subscriptionRepository = null;

    /**
     * feUserRepository
     *
     * @var FeUserRepository
     */
    protected $feUserRepository = null;

    protected $braintreeService = null;

    /**
     * mailService
     *
     * @var MailService
     */
    protected $mailService = null;

    public function injectSubscriptionRepository(SubscriptionRepository $subscriptionRepository): void
    {
        $this->subscriptionRepository = $subscriptionRepository;
    }

    public function injectFeUserRepository(FeUserRepository $feUserRepository): void
    {
        $this->feUserRepository = $feUserRepository;
    }

    public function injectMailService(MailService $mailService): void
    {
        $this->mailService = $mailService;
    }


    /**
     * Initialize Braintree API
     */
    public function initializeAction(): void
    {
        $this->braintreeService = GeneralUtility::makeInstance(BraintreeService::class);

        // Setup mail service
        $this->mailService->setSettings($this->settings);
        $this->mailService->setRequest($this->request);
    }

    public function listAction(): ResponseInterface
    {
        if (!UserUtility::isAdminLoggedIn()) {
            exit(0);
        }
        /*
        if ( $this->request->hasArgument('search')) {
    $search = $this->request->getArgument('search');
        } else {
    $search = [];
        }
        if ( $this->request->hasArgument('user')) {
    $user = $this->request->getArgument('user');
        } else {
    $user = null;
        }
        $this->view->assign('search', $search);
        */
        //$users = $this->feUserRepository->findByUsergroup(4);
        $subscription = $this->subscriptionRepository->findAll();
        $this->view->assign('subscriptions', $subscription);

        return $this->htmlResponse();
    }



    /**
     * Generates client token and shows payment form for product
     *
     * @param string $referer
     * @param string $startDate
     */
    public function paymentAction(SubscriptionOffer $product, $referer, $startDate): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        // Paypal payment
        if ($isLoggedIn) {
            $this->view->assign('clientToken', $this->braintreeService->generate());
            $this->view->assign('product', $product);
            $this->view->assign('referer', $referer);
            $this->view->assign('startDate', $startDate);
        }

        return $this->htmlResponse();
    }

    /**
     * Performs the transaction if the amount coming per post is the same as the price of the product
     *
     * @param string $startDate
     */
    public function checkoutAction(SubscriptionOffer $product, $referer, $startDate): ResponseInterface
    {
        $amount = $this->request->getParsedBody()['amount'] ?? null;
        $nonce = $this->request->getParsedBody()['payment_method_nonce'] ?? null;

        $context = GeneralUtility::makeInstance(Context::class);
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');
        $userId = $context->getPropertyFromAspect('frontend.user', 'id');

        if ($isLoggedIn && $amount == $product->getPrice()) {
            //if($this->braintreeService->sale($product->getPrice(), $amount, $nonce) === true) {
            $result = $this->braintreeService->sale([
                'amount' => $amount,
                'paymentMethodNonce' => $nonce,
                'options' => [
                    'submitForSettlement' => true,
                ],
            ]);
            $user = $this->feUserRepository->findByUid($userId);
            if ($this->braintreeService->isResultSuccess($result)) {
                $transaction = $result->transaction;
                $subscription = GeneralUtility::makeInstance(Subscription::class);
                //      var_dump($result);exit(0);
                $subscription->setSubscriptionOffer($product);
                $subscription->setTransactionId($transaction->id);
                $subscription->setTransactionAmount($transaction->amount);
                $subscription->setTransactionCurrency($transaction->currencyIsoCode);
                $subscription->setPaymentMethod($transaction->paymentInstrumentType);
                $subscription->setFeUser($user);

                $startDateObject = new DateTime();
                $d = explode('.', $startDate);
                $startDateObject->setDate($d[2], $d[1], $d[0]);

                $inThreeMonths = clone $startDateObject;
                $inThreeMonths->modify('+' . $product->getDuration() . ' month');

                $subscription->setStartDate($startDateObject);
                $subscription->setExpirationDate($inThreeMonths);
                $subscription->setDaysLeft($product->getDays());

                $user->addSubscription($subscription);

                $this->subscriptionRepository->add($subscription);
                $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
                $persistenceManager->persistAll();

                $this->mailService->sendMail(
                    'Physio-Team Naderi: Ihr neues Abonnement',
                    [$this->settings['debugMail'] ?: $this->settings['mailFrom']],
                    [$this->settings['debugMail'] ?: $user->getEmail()],
                    'IgRuckzuckevent/Subscription/User',
                    ['subscription' => $subscription]
                );

                // Same for the admin mail
                if ($this->settings['debugMail']) {
                    $adminMailTo = $this->settings['debugMail'];
                } else {
                    $adminMailTo = $this->settings['mailTo'] ?: $this->settings['mailFrom'];
                }

                $this->mailService->sendMail(
                    'Physio-Team Naderi: Neues Abonnement wurde gelöst',
                    [$this->settings['debugMail'] ?: $this->settings['mailFrom']],
                    [$adminMailTo],
                    'IgRuckzuckevent/Subscription/Admin',
                    ['subscription' => $subscription]
                );
            }
            return $this->redirect(
                'transaction',
                'Subscription',
                'IgRuckzuckevent',
                [
                    'product' => $product,
                    'referer' => $referer,
                    'transactionId' => $transaction->id,
                    'startDate' => $startDate,
                ]
            );
            exit();
        }
        $errorString = '';

        foreach ($result->errors->deepAll() as $error) {
            $errorString .= 'Error: ' . $error->code . ': ' . $error->message . "\n";
        }

        $_SESSION['errors'] = $errorString;
        
        return $this->redirect('payment', 'Subscription', 'IgRuckzuckevent', [
            'product' => $product,
            'referer' => $referer,
            'startDate' => $startDate,
        ]);
    }

    /**
     * Transaction action shows the user if the transaction was succesful
     *
     * @param string $referer
     * @param string $transactionId
     * @param string $startDate
     *
     * @return string
     */
    public function transactionAction(
        SubscriptionOffer $product,
        $referer,
        $transactionId,
        $startDate
    ): ResponseInterface {
        $context = GeneralUtility::makeInstance(Context::class);
        $isLoggedIn = $context->getPropertyFromAspect('frontend.user', 'isLoggedIn');

        if ($isLoggedIn && $transactionId !== '') {
            $transaction = $this->braintreeService->find($transactionId);

            if ($this->isTransactionSuccessful($transaction)) {
                $transactionState = 'success';
            } else {
                $transactionState = 'error';
            }

            $this->view->assign('product', $product);
            $this->view->assign('referer', urldecode($referer));
            $this->view->assign('transaction', $transaction);
            $this->view->assign('transactionState', $transactionState);
        }

        return $this->htmlResponse();
    }
    public function editAction(Subscription $subscription): ResponseInterface
    {
        if (!UserUtility::isAdminLoggedIn()) {
            die('permission denied');
        }
        $this->view->assign('subscription', $subscription);

        return $this->htmlResponse();
    }
    public function updateAction(Subscription $subscription): ResponseInterface
    {
        if (!UserUtility::isAdminLoggedIn()) {
            die('permission denied');
        }
        $this->subscriptionRepository->update($subscription);
        return $this->redirect('newSubscription', 'User', 'IgRuckzuckevent', [
            'user' => $subscription->getFeUser()->getUid(),
        ]);
    }

    protected function isTransactionSuccessful($transaction)
    {
        $transactionSuccessStatuses = [
            Transaction::AUTHORIZED,
            Transaction::AUTHORIZING,
            Transaction::SETTLED,
            Transaction::SETTLING,
            Transaction::SETTLEMENT_CONFIRMED,
            Transaction::SETTLEMENT_PENDING,
            Transaction::SUBMITTED_FOR_SETTLEMENT,
        ];

        if (in_array($transaction->status, $transactionSuccessStatuses)) {
            return true;
        }

        return false;
    }
}
