<?php

declare(strict_types=1);

namespace Ig\IgSecure\Typo3\Service;

use Symfony\Component\Mime\Address;
use TYPO3\CMS\Core\Authentication\AbstractAuthenticationService;
use TYPO3\CMS\Core\Configuration\ExtensionConfiguration;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Mail\MailMessage;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;
use GeoIp2\Database\Reader;
/**
 * Class Authentication
 *
 * @package Ig\IgSecure\Typo3\Service
 */
class Authentication extends AbstractAuthenticationService
{

    private $geoLite2CityFile = '/var/lib/GeoIP/GeoLite2-City.mmdb';
    private $geoLite2CountryFile = '/var/lib/GeoIP/GeoLite2-Country.mmdb';

    /**
     * Erlaubte Länder. Sollte als komma-getrennter String angegeben werden z.B.
     * 'DE,CH,AT'. Wird von der EXT_CONF-Konfiguration überschrieben.
     *
     * @var array $allowedCountries
     */
    private $allowedCountries = [ 'CH' ];

    /**
     * Erlaubte IP-Adressen. Sollte als komma-getrennter String angegeben
     * werden z.B. '212.243.26.187,212.103.70.204'. Wird von der
     * EXT_CONF-Konfiguration überschrieben.
     *
     * @var array $allowedIps
     */
    private $allowedIps = [];

    /**
     * Getätigte Tests
     */
    private $logTest = [];

    /**
     * <0 Login fehlgeschlagen
     * 0-199 Login wird weiterverarbeitet d.h. User+Pwd Test
     * >= 200 Login i.o. - ohne Passwort test
     *
     * @param array $user Benutzerdaten
     *
     * @return int status
     */
    public function authUser(array $user): int
    {
        $config = GeneralUtility::makeInstance(ExtensionConfiguration::class)->get('ig_secure');
        $this->allowedCountries = explode(',', trim($config['countries']));
        $this->allowedIps = $this->getIpsFromString($config['ip_addresses']);
        $remoteIp = $_SERVER['REMOTE_ADDR'];
        //$remoteIp = '90.238.128.212'; //SE

        // IP Testen
        if ($this->testIp($remoteIp)) {
            return 100;
        }        
        // Land nach geoip library
        if ($this->testCountryGeoip($remoteIp)) {
            return 100;
        }
        // Land nach reverse DNS
        if ($this->testCountryReverseDns($remoteIp)) {
            return 100;
        }
        // Land nach RIPE
        if ($this->testCountryRipe($remoteIp)) {
            return 100;
        }
        
        if ($config['email']) {
            $mail = GeneralUtility::makeInstance(MailMessage::class);
            $mail->subject('Unauthorized Person tried to login to the backend: ' . Environment::getPublicPath() . '/');
            $mail->from(new Address($config['email']));
            $mail->to(new Address($config['email']));
            $body = '<html><head></head><body>
				<p>An unauthorized person from another country or with an IP address not in range tried to login to the backend:</p>
				<table>
				    <tr>
				        <td><b>Web server</b></td>
				        <td>' . $_SERVER['SERVER_NAME'] . '</td>
                    </tr>
                    <tr>
                        <td><b>Username</b></td>
                        <td>' . $this->login['uname'] . '</td>
                    </tr>
                    <tr>
                        <td><b>Password</b></td>
                        <td>' . (string)($this->login['uident_text'] ?? ''). '</td>
                    </tr>
					<tr>
						<td><b>IP address</b></td>
						<td>' . $remoteIp .'</td>
					</tr>';
            foreach ($this->logTest as $priority => $log) {
                $body .= '<tr>
                        <td><b>' . $log['testName'] . '</b></td>
                        <td>' . $log['country'] .'</td>
                    </tr>';
            }
            $body .= '</table>
			</body></html>';
            $mail->html(
                $body,
                'text/html'
            );
            
            $mail->send();
        }
        
        return -1;
    }


    /**
     * Convert ip string to array
     *
     * @param string $str ips as string
     *
     * @return array array of ips/networks
     */
    public function getIpsFromString(string $str): array
    {
        // Eintraege Komma oder Enter getrennt
        $ip_list = str_replace("\n", ',', trim($str));
        $ip_array = explode(',', $ip_list);
        
        $ips = array();
        
        foreach ($ip_array as $ip_entry) {
            $ip_vondbis = explode('-', $ip_entry);
            
            if (count($ip_vondbis) == 2) {
                $ip_von = trim($ip_vondbis[0]);
                $ip_bis = trim($ip_vondbis[1]);

                /* Idee fuer 141.39.240-255.* */
                $ip_von = str_replace('*', '0', $ip_von);
                $ip_bis = str_replace('*', '255', $ip_bis);
                $ip_von_array = explode('.', $ip_von);
                // Ergaenzen auf 4 bytes
                for ($i = count($ip_von_array); $i < 4; $i++) {
                    $ip_von_array[$i] = '0';
                }

                $ip_bis_array = explode('.', $ip_bis);
                $ip_bis_add = array();
                // Falls bis IP kürzer als 4 Stellen ist, Diese mit Werten von der Von IP ergaenzen
                for ($i = 0; $i < 4 - count($ip_bis_array); $i++) {
                    $ip_bis_add[] = $ip_von_array[$i];
                }
                $ip_bis_array = array_merge($ip_bis_add, $ip_bis_array);

                //echo( $ip_von . ' == ' . implode('.', $ip_von_array)  . '<br />' );

                if ($ip_vondbis[0] != '') {
                    $ip_von = ip2long(implode('.', $ip_von_array));
                } else {
                    $ip_von = null;
                }

                if ($ip_vondbis[1] != '') {
                    $ip_bis = ip2long(implode('.', $ip_bis_array));
                } else {
                    $ip_bis = null;
                }

                $ips[] = array('von' => $ip_von, 'bis' => $ip_bis, 'net' => null);
            } elseif ($ip_entry) {
                $ips[] = array('von' => null, 'bis' => null, 'net' => $ip_entry);
            }
        }

        return $ips;
    }

    /**
     * Test remote ip of client against allowed ips
     *
     * @param string $remoteIp remote ip of client
     *
     * @return bool test is successful
     */
    public function testIp(string $remoteIp)
    {
        $remoteIpLong = ip2long($remoteIp);

        foreach ($this->allowedIps as $allowedIp) {
            if (((!is_null($allowedIp['von']) && $allowedIp['von'] <= $remoteIpLong) && (is_null($allowedIp['bis']) || $allowedIp['bis'] >= $remoteIpLong))) {
                return true;
            }
            if (!is_null($allowedIp['net']) && GeneralUtility::cmpIP($remoteIp, $allowedIp['net'])) {
                return true;
            }
        }

        return false;
    }

    /**
     * Test Country against allowed Countries
     *
     * @param $country Country
     *
     * @return bool test is successful
     */
    public function testCountry(string $country): bool
    {
        if (!$country) {
            return false;
        }
    
        foreach ($this->allowedCountries as $nr => $c) {
            if ($country == strtoupper(trim($c))) {
                return true;
            }
        }
        
        return false;
    }
    /**
     * Log Test in Array, for e.g. email
     *
     * @param $testName TestName
     * @param $country  Country
     */
    public function logTest(string $testName, string $country): void
    {
        $this->logTest[]= [
            'testName' => $testName,
            'country' => $country,
        ];
    }

    /**
     * Test country from IP with GeoIP library
     *
     * @param $remoteIp IP Address
     *
     * @return bool is test successful
     */
    public function testCountryGeoip(string $remoteIp): bool
    {
        $country = null;
        if (file_exists($this->geoLite2CityFile) && class_exists(Reader::class)) {
            try {
                $reader = new Reader($this->geoLite2CountryFile);
                //$record = $reader->city($remoteIp);
                $record = $reader->country($remoteIp);
                $country = $record->country->isoCode; // $record->country->name;
                $this->logTest('GeoLite2', $country);
            } catch (\Exception $e) {
                // show error
            }
        }
        if (!$country) {
            $country = function_exists('geoip_country_code_by_name') ? geoip_country_code_by_name($remoteIp) : $this->geoLite2CityFile . ' missing or geoipupdate/geoip missing';
            $this->logTest('geoip', $country);
        }
        return $this->testCountry($country);
    }

    /**
     * Test country from IP with reverse DNS
     *
     * @param $remoteIp IP Address
     *
     * @return bool is test successful
     */
    public function testCountryReverseDns(string $remoteIp): bool
    {
        // reverse loopup to get the hostname
        $hostname = gethostbyaddr($remoteIp);
        $hostnameParts = explode('.', $hostname);
        $country = strtoupper($hostnameParts[count($hostnameParts) - 1]);
        $this->logTest('reverseDns', $country);
        return $this->testCountry($country);
    }
    
    /**
     * Test country from IP with RIPE DB
     *
     * @param $remoteIp IP Address
     *
     * @return bool is test successful
     */
    public function testCountryRipe(string $remoteIp): bool
    {
        //https://stat.ripe.net/data/rir-geo/data.json?resource=193.222.138.163
        $url = 'https://stat.ripe.net/data/rir-geo/data.json?resource=' . $remoteIp;
        try {
            $jsonData = @file_get_contents($url);
            if ($jsonData===false) {
                return false;
            }
            $data  = json_decode($jsonData, true);
        } catch (Exception $e) {
            return false;
        }

        if (!isset($data['data']['located_resources'])) {
            return false;
        }
        foreach ($data['data']['located_resources'] as $resource) {
            if (isset($resource['location'])) {
                $country=$resource['location'];
                $this->logTest('ripe', $country);
                $ret = $this->testCountry($country);
                if ($ret) {
                    return $ret;
                }
            }
        }
        return false;
    }
}
