<?php
namespace Internetgalerie\IgSimplesubscribe\Controller;

use Evoweb\Recaptcha\Services\CaptchaService;
use Internetgalerie\IgSimplesubscribe\Domain\Model\FrontendUser;
use Internetgalerie\IgSimplesubscribe\Domain\Repository\AddressChangesRepository;
use Internetgalerie\IgSimplesubscribe\Domain\Repository\CategoryRepository;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Property\PropertyMappingConfiguration;
use TYPO3\CMS\Extbase\Property\TypeConverter\PersistentObjectConverter;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Mail\FluidEmail;
use TYPO3\CMS\Core\Mail\Mailer;
use Psr\Http\Message\ServerRequestInterface;
use TYPO3\CMS\Core\Resource\File;
use TYPO3\CMS\Core\Resource\FileRepository;
use TYPO3\CMS\Core\Resource\StorageRepository;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Extbase\Persistence\Repository;
use Internetgalerie\IgSimplesubscribe\Event\BeforePersistenceEvent;
use Psr\EventDispatcher\EventDispatcherInterface;
/**
 * Newsletter User Controller
 */
class UserController extends \TYPO3\CMS\Extbase\Mvc\Controller\ActionController
{
    protected array $fieldConfigs = [];
    protected array $fieldNames = [];
    protected array $fieldErrors = [];
    protected array $mandatoryFieldNames = [];
    protected array $categoryUids = [];
    protected bool $hasError = false;
    protected string $tablename = 'tt_address';
    protected string $objectClass = \Internetgalerie\IgSimplesubscribe\Domain\Model\Address::class;
    protected Repository $dataRepository;
    protected ?CategoryRepository $categoryRepository;
    protected ?StorageRepository $storageRepository;
    protected ?FileRepository $fileRepository;
    protected ?AddressChangesRepository $addressChangesRepository;
    protected ?PersistenceManager $persistenceManager;
    protected CaptchaService $captchaService;

    public function __construct(
        CategoryRepository $categoryRepository,
        StorageRepository $storageRepository,
        FileRepository $fileRepository,
        AddressChangesRepository $addressChangesRepository,
        PersistenceManager $persistenceManager,
        CaptchaService $captchaService
    ) {
        $this->categoryRepository = $categoryRepository;
        $this->storageRepository = $storageRepository;
        $this->fileRepository = $fileRepository;
        $this->addressChangesRepository = $addressChangesRepository;
        $this->persistenceManager = $persistenceManager;
        $this->captchaService = $captchaService;
    }

    /**
     * Initialize the controller
     *
     * @return void
     */
    public function initializeAction(): void
    {
        if ($this->settings['fieldTypesPath']) {
            $this->fieldConfigs = require GeneralUtility::getFileAbsFileName($this->settings['fieldTypesPath']);
        }
        $this->fieldNames = $this->getFieldNames();
        $this->mandatoryFieldNames = $this->getMandatoryFieldNames();
        $this->categoryUids = $this->getCategoryUids();
        $this->tablename = $this->settings['flexform']['table'] ?? 'tt_address';
        $this->fieldErrors = [];
        if ($this->tablename == 'fe_users') {
            $this->dataRepository = GeneralUtility::makeInstance(\Internetgalerie\IgSimplesubscribe\Domain\Repository\FrontendUserRepository::class);
            $this->objectClass = \Internetgalerie\IgSimplesubscribe\Domain\Model\FrontendUser::class;
        } else {
            $this->dataRepository = GeneralUtility::makeInstance(\Internetgalerie\IgSimplesubscribe\Domain\Repository\AddressRepository::class);
            $this->objectClass = \Internetgalerie\IgSimplesubscribe\Domain\Model\Address::class;
        }

        //$this->fields = $this->initFields();

        // Fuer Kompatibilitaet - email immer am Schlussd hinzufuegen (falls nicht vorhanden)
        if ($this->tablename == 'tt_address') {
            if (!in_array('email', $this->fieldNames)) {
                $this->fieldNames[] = 'email';
            }
            if (!in_array('email', $this->mandatoryFieldNames)) {
                $this->mandatoryFieldNames[] = 'email';
            }
        }

        // Bugfix falls keine Kategorie ausgewählt ist
        if (isset($this->categoryUids[0]) && $this->categoryUids[0] == '') {
            unset($this->categoryUids[0]);
        }

        // Falls logo gesetzt ist Dateireferenz holen
        if (intval($this->settings['flexform']['logo'])) {
            //$this->settings['flexform']['logo'] = $this->fileRepository->findFileReferenceByUid(intval($this->settings['flexform']['logo']))->getOriginalFile();
        }

        // Hole CSS-Dateiinhalt für Mailtemplates (Wird inline in die Templates geschrieben)
        if ($this->settings['mailCssFile']) {
            $this->settings['mailCssFile'] = file_get_contents(GeneralUtility::getFileAbsFileName($this->settings['mailCssFile']));
        }
        $this->settings['hide_unsubscribe_link'] = $this->settings['flexform']['hide_unsubscribe_link'] ?? '';
    }

    /**
     * Shows the subscription form, saves the hidden address
     * and sends the confirmation mail if no errors occurred
     *
     * @return void
     */
    public function registerAction(): ResponseInterface
    {
        $categories = [];

        $fields = $this->getFieldsFromArguments();
        $arguments = $this->request->getArguments();
        $email = '';
        $accepted = true;
        if ($this->settings['needToAccept'] && $this->request->hasArgument('accepted')) {
            $accepted = $this->request->getArgument('accepted');
        } else if($this->settings['needToAccept'] && !$this->request->hasArgument('accepted')) {
            $accepted = false;
        }
        // Categories
        foreach ($this->categoryUids as $categoryUid) {
            $category = [];
            $category['object'] = $this->categoryRepository->findByUid($categoryUid);

            if ($this->isSubmitted() && is_array($arguments['categories'])) {
                $category['checked'] = in_array($categoryUid, $arguments['categories']);
            }

            $categories[] = $category;
        }

        if ($this->isSubmitted()) {
            if(!$this->hasErrors($fields) && (!(int)$this->settings['needToAccept'] || ((int)$this->settings['needToAccept'] && $accepted))) {
                if ($arguments['test'] != 3143558231) {
                    return $this->redirect('register');
                }
                $email = $fields[$this->getEmailFieldname()];
                if ($this->tablename == 'tt_address') {
                    $this->deleteOldEmails($email);
                }
                $ret = $this->sendConfirmationMail($email['value'], $fields, $categories, $accepted);
                if($ret instanceof ResponseInterface) {
                    return $ret;
                }
                return $this->redirect('registerMailInformation');
            }
        }
        // Validate input if the form has been submitted
        /*
        if(isset($arguments['submit'])) {
        return $this->forward('create');
        }
         */
        //var_dump($fields);die();
        $captchaError = false;
        $this->view->assignMultiple([
            'settings' => $this->settings,
            'fields' => $fields,
            'email' => $email,
            'categories' => $categories,
            'captchaError' => $captchaError,
            'submit' => 'signin',
            'accepted' => $accepted,
            'captchaConfiguration' => $this->captchaService->getConfiguration(),
        ]);
        
        return $this->htmlResponse();
    }
    
    /**
     * Shows the subscription form, saves the hidden address
     * and sends the confirmation mail if no errors occurred
     *
     * @return void
     */
    public function formAction(): ResponseInterface
    {
        $categories = [];
        // Categories
        foreach ($this->categoryUids as $categoryUid) {
            $categories[] = [
                'object' => $this->categoryRepository->findByUid($categoryUid),
            ];
        }
        $fields = $this->getFieldsFromArguments();
        $this->view->assignMultiple(
            [
                'settings' => $this->settings,
                'subscribePageUid' => $this->settings['flexform']['subscribePage'],
                'unsubscribePageUid' => $this->settings['flexform']['unsubscribePage'],
                'fields' => $fields,
                'email' => '',
                'categories' => $categories,
                'captchaError' => false,
                'submit' => 'signin',
                'accepted' => false,
            ]
        );

        return $this->htmlResponse();
    }

    /**
     * Shows Information that the user should check his mail
     *
     * @return void
     */
    public function registerMailInformationAction(): ResponseInterface
    {
        $this->view->assign('settings', $this->settings);
        $this->view->assign('settings', $this->settings);

        return $this->htmlResponse();
    }

    /**
     * Checks the parameters (addressUid and linkhash) of the link sent with the confirmation mail,
     * unsets the hidden flag of the address and resets the linkHash to a new hash and so the link invalidates
     *
     * @param int $addressUid
     * @param string $linkHash
     *
     * @return void
     */
    public function confirmRegistrationAction($addressUid, $linkHash): ResponseInterface
    {
        $address = $this->dataRepository->findHiddenByUid($addressUid);

        if ($address && $address->isHidden() && $this->checkLinkHash($linkHash, $address)) {
            $address->setHidden(false);
            $address->setLinkHash($this->generateLinkHash($address->getUid() . time()));

            $this->dataRepository->update($address);
            $this->persistenceManager->persistAll();

            // Admin mail
            if (intval($this->settings['flexform']['emailonsignin'])) {
                $this->sendAdminMail($address);
                $this->submitChanges($address->getUid(), 'subscribed', null, $address);
            }
        } else {
            return $this->redirect('register');
        }

        // If set in flexform, redirect to "thank you" page, else use this view
        if ($this->settings['flexform']['registerok']) {
            return $this->redirect(null, null, null, null, $this->settings['flexform']['registerok']);
        }

        $this->view->assign('address', $address);
        $this->view->assign('settings', $this->settings);

        return $this->htmlResponse();
    }

    /**
     * Shows input where user inserts his email address for editing his subscription settings
     *
     * @return void
     */
    public function editAction(): ResponseInterface
    {
        $initFieldNames = [$this->getEmailFieldname()];
        $initFieldConfigs = [$this->getEmailFieldname() => [
            'type' => 'text',
            'validate' => 'notEmpty,exists',
        ],
        ];
        $fields = $this->getFieldsFromArguments($initFieldNames, $initFieldNames, $initFieldConfigs);
        $email = '';
        if ($this->isSubmitted()) {
            if (!$this->hasErrors($fields)) {
                $email = $fields[$this->getEmailFieldname()];
                /*
                if($address && !$address->isHidden()) {
                } else {
                return $this->redirect('register'); // TODO: What do we do if the address does not already exist?
                }
                 */
                $obj = $this->dataRepository->findByEmail($email['value'])->getFirst();
                $obj->setLinkHash($this->generateLinkHash($obj->getUid() . time()));
                $this->dataRepository->update($obj);

                if(!$this->settings['singleOptIn']) {
                    $this->sendEditConfirmationMail($obj);
                    return $this->redirect('editMailInformation');
                } else {
                    return $this->redirect('editForm', null, null, ['address' => $obj->getUid(), 'linkHash' => $obj->getLinkHash()]);
                }
            }
        }

        /*
        $email = array();
        $email['address'] = $arguments['email'];
        $address = $this->dataRepository->findByEmail($email['address'])->getFirst();

        if(!$address) {
        $this->addError('', 'error.address.not_found');
        $email['error'] = 'address.not_found';
        }
        if(!$this->checkCaptcha($arguments['captcha'])) {
        $this->addError('captcha', 'error.captcha');
        $captchaError = true;
        }

        if(!$this->hasError) {
         */
        $captchaError = false;
        $this->view->assignMultiple([
            'settings' => $this->settings,
            'fields' => $fields,
            'email' => $email,
            'captchaError' => $captchaError,
            'submit' => 'submit',
        ]);
        $this->view->assign('settings', $this->settings);

        return $this->htmlResponse();
    }

    /**
     * Shows Information that the user should check his mail
     *
     * @return void
     */
    public function editMailInformationAction(): ResponseInterface
    {
        $this->view->assign('settings', $this->settings);
        $this->view->assign('settings', $this->settings);

        return $this->htmlResponse();
    }

    /**
     * Shows the editform for an address with a given linkHash/hash
     *
     * @param string $linkHash
     *
     * @return void
     */
    public function editFormAction($linkHash = ''): ResponseInterface
    {
        $userData = $this->request->hasArgument('address') ? $this->request->getArgument('address') : null; // $originalRequest = $this->request->getOriginalRequest();
        $arguments = $this->request->getArguments();
        /*
        $objUid=$address;
        if($objUid<=0) {
        return $this->redirect('edit');
        }
        $obj = $this->dataRepository->findByUid($objUid);
        if( !$obj ) {
        return $this->redirect('edit');
        }
        $address=$obj;

         */
        $categories = [];
         foreach ($this->categoryUids as $categoryUid) {
             $category = [];
             $category['object'] = $this->categoryRepository->findByUid($categoryUid);
             $categories[] = $category;
         }
        $propertyMappingConfiguration = $this->getPropertyMappingConfiguration(null, $userData);
        $propertyMapper = GeneralUtility::makeInstance(\TYPO3\CMS\Extbase\Property\PropertyMapper::class);
        $address = $propertyMapper->convert(
            $userData,
            $this->objectClass,
            $propertyMappingConfiguration
        );

        if(!$address) {
            return $this->redirect('register');
        }

        //return $objUid .':'. $obj->getLinkHash() .'='. $linkHash;
        if (!$address->isHidden() && $this->checkLinkHash($linkHash, $address)) {
            $oldData = clone $address;
            $fields = $this->getFieldsFromObject($address);

            // If the form has been submitted, fill the field values so it can be used in the view
            foreach ($fields as &$field) {
                $fieldNameCamelCase = GeneralUtility::underscoredToUpperCamelCase($field['name']);

                if ($this->isSubmitted()) {
                    if ($address->{'get' . $fieldNameCamelCase}() != $arguments[$field['name']]) {
                        if ($field['type'] == 'upload') {
                            $field['value'] = $arguments[$field['name']];
                            $field['fileUid'] = $this->uploadFile($field);

                            if ($field['fileUid']) {
                                $file = $this->fileRepository->findByUid($field['fileUid']);
                                $fileReference = GeneralUtility::makeInstance('Internetgalerie\\IgSimplesubscribe\\Domain\\Model\\FileReference');
                                $fileReference->setOriginalResource($file);
                                $fileReference->setUidForeign($address->getUid());

                                $address->{'add' . $fieldNameCamelCase}($fileReference);
                            }
                        } else {
                            $address->{'set' . $fieldNameCamelCase}($arguments[$field['name']]);
                        }
                    }

                    if ($field['mandatory'] && ($arguments[$field['name']] == '' || $arguments[$field['name']] == '0')) {
                        $this->addError($field['name'], 'error.' . $field['name']);
                        $field['error'] = true;
                    }
                }

                if ($field['type'] != 'upload') {
                    $field['value'] = $address->{'get' . $fieldNameCamelCase}();
                    $getter = 'get' . $fieldNameCamelCase;
                }

                //$fields[$field['name']] = $field;
            }
            foreach ($categories as &$category) {
                if ($address->hasSysDmailCategory($category['object'])) {
                    $category['checked'] = true;
                }

                if (isset($arguments['submit'])) {
                    if (in_array($categoryUid, $arguments['categories'])) {
                        if (!$address->hasSysDmailCategory($category['object'])) {
                            $category['checked'] = true;
                            $address->addSysDmailCategory($category['object']);
                        }
                    } elseif ($address->hasSysDmailCategory($category['object'])) {
                        $address->deleteSysDmailCategory($category['object']);
                        $category['checked'] = false;
                    }
                }
            }

            $email['address'] = $address->getEmail();

            if (isset($arguments['submit'])) {
                $email['address'] = $arguments['email'];

                if ($address->getEmail() != $arguments['email'] && $arguments['email'] != '') {
                    if ($this->dataRepository->findByEmail($email['address'])->count() > 0) {
                        $this->addError('email', 'error.address.already_exists');
                        $email['error'] = 'address.already_exists';
                    } elseif (!$this->checkEmail($arguments['email'])) {
                        $this->addError('email', 'error.email');
                        $email['error'] = 'email';
                    } else {
                        $address->setEmail($arguments['email']);
                        $email['address'] = $address->getEmail();
                    }
                } elseif ($arguments['email'] == '') {
                    $this->addError('email', 'error.address.not_found');
                    $email['error'] = 'address.not_found';
                }

                if (!$this->checkCaptcha($arguments['captcha'])) {
                    $this->addError('captcha', 'error.captcha');
                    $captchaError = true;
                }

                if (!$this->hasError) {
                    if ($this->settings['logIpAddress']) {
                        $address->setIpAddress($_SERVER['REMOTE_ADDR'] . ($_SERVER['HTTP_X_FORWARDED_FOR'] ? ' (' . $_SERVER['HTTP_X_FORWARDED_FOR'] . ')' : ''));
                    }
                    if ($this->settings['logBrowser']) {
                        $address->setBrowser($_SERVER['HTTP_USER_AGENT']);
                    }
                    $address->setLinkHash($this->generateLinkHash($address->getUid() . time()));
                    $this->dataRepository->update($address);
                    $this->persistenceManager->persistAll();
                    $this->submitChanges($address->getUid(), 'change', $oldData, $address);

                    // Admin mail
                    if (intval($this->settings['flexform']['emailonchange'])) {
                        $this->sendEditOrDeleteAdminMail($address, $fields, $categories);
                    }

                    return $this->redirect('confirmEdit');
                }
            }

            $this->view->assignMultiple([
                'settings' => $this->settings,
                'email' => $email,
                'address' => $address,
                'fields' => $fields,
                'categories' => $categories,
                'captchaError' => $captchaError,
                'submit' => 'change',
            ]);
        }

        $this->view->assign('settings', $this->settings);

        return $this->htmlResponse();
    }

    /**
     *
     *
     * @return void
     */
    public function confirmEditAction(): ResponseInterface
    {
        return $this->htmlResponse();
    }

    public function deleteFormAction(): ResponseInterface
    {
        $initFieldNames = [$this->getEmailFieldname()];
        $initFieldConfigs = [$this->getEmailFieldname() => [
            'type' => 'text',
            'validate' => 'notEmpty,exists',
        ],
        ];
        $fields = $this->getFieldsFromArguments($initFieldNames, $initFieldNames, $initFieldConfigs);
        $this->view->assign('fields', $fields);
        return $this->htmlResponse();
    }

    /**
     * Deletes an address by email
     *
     * @param  string $email
     * @return void
     */
    public function deleteAction($email): ResponseInterface
    {
        $address = $this->dataRepository->findOneByEmail($email);

        if ($address) {
            if ($this->settings['onlyDeactivateOnUnregister']) {
                $address->setHidden(true);
                $this->dataRepository->update($address);
            } else {
                $this->dataRepository->remove($address);
            }
            if (intval($this->settings['flexform']['emailonchange'])) {
                $fields = $this->getFieldsFromObject($address);
                $categories = $this->getCategories($address);
                $this->sendEditOrDeleteAdminMail($address, $fields, $categories, false);
            }
            $this->submitChanges($address->getUid(), 'delete', $address, null);
        }

        return $this->redirect('confirmUnregistration');
    }

    /**
     * Deletes an address with a given linkHash/hash
     *
     * @param integer $address
     * @param string $linkHash
     * @param boolean $showReallyUnregisterDialog
     *
     * @return void
     */
    public function unregisterAction($address = null, $linkHash = '', $showReallyUnregisterDialog = true): ResponseInterface
    {
        if($address === null) {
            $this->view->assign('noAddress', true);
            return $this->htmlResponse();
        }
        $objUid = $address;
        $userData = $this->request->hasArgument('address') ? $this->request->getArgument('address') : null; // $originalRequest = $this->request->getOriginalRequest();
        if (isset($userData['uid'])) {
            unset($userData['uid']);
        }
        $propertyMappingConfiguration = $this->getPropertyMappingConfiguration(null, $userData);
        $propertyMapper = GeneralUtility::makeInstance(\TYPO3\CMS\Extbase\Property\PropertyMapper::class);

        $address = null;
        try {
            $address = $propertyMapper->convert(
                $userData,
                $this->objectClass,
                $propertyMappingConfiguration
            );
        } catch(\Exception $e) {

        }
        if ($address === null) {
            $this->view->assign('addressDeleted', true);
            return $this->htmlResponse();
        }
        if ($address && !$address->isHidden() && $this->checkLinkHash($linkHash, $address)) {
            if ($showReallyUnregisterDialog) {
                $this->view->assignMultiple([
                    'address' => $address,
                ]);
            } else {
                if (intval($this->settings['flexform']['emailonchange'])) {
                    $fields = $this->getFieldsFromObject($address);
                    $categories = $this->getCategories($address);
                    $this->sendEditOrDeleteAdminMail($address, $fields, $categories, false);
                }

                $this->submitChanges($address->getUid(), 'delete', $address, null);

                if ($this->settings['onlyDeactivateOnUnregister']) {
                    $address->setHidden(true);
                    $this->dataRepository->update($address);
                } else {
                    $this->dataRepository->remove($address);
                }
                $this->persistenceManager->persistAll();
            }
        }

        if (!$showReallyUnregisterDialog) {
            return $this->redirect('confirmUnregistration');
        }
        $this->view->assign('settings', $this->settings);

        return $this->htmlResponse();
    }

    /**
     * Shows information that unregistration is complete or redirects to page set in flexform
     *
     * @return void
     */
    public function confirmUnregistrationAction(): ResponseInterface
    {
        // If set in flexform, redirect to "unsubscribed" page, else use this view
        if ($this->settings['flexform']['unregisterok']) {
            return $this->redirect(null, null, null, null, $this->settings['flexform']['unregisterok']);
        }
        $this->view->assign('settings', $this->settings);

        return $this->htmlResponse();
    }

    /**
     * Saves the changes in the tt_address_changes-Table
     *
     * @param int $uid The uid of the address
     * @param string $state The state to be submitted
     * @param \Internetgalerie\IgSimplesubscribe\Domain\Model\Address $old Old data
     * @param \Internetgalerie\IgSimplesubscribe\Domain\Model\Address $new New data
     *
     * @return void
     */
    public function submitChanges($uid, $state, $old = null, $new = null): void
    {
        $oldData = [];
        $newData = [];

        if (is_object($old)) {
            $oldData['email'] = $old->getEmail();
        }
        if (is_object($new)) {
            $newData['email'] = $new->getEmail();
        }

        $changesFields = GeneralUtility::trimExplode(',', $this->settings['changesFields']);

        if ($this->settings['needToAccept'] && !in_array('accepted', $changesFields)) {
            $changesFields[] = 'accepted';
        }

        if ($this->settings['logIpAddress'] && !in_array('ip_address', $changesFields)) {
            $changesFields[] = 'ip_address';
        }

        if ($this->settings['logBrowser'] && !in_array('browser', $changesFields)) {
            $changesFields[] = 'browser';
        }

        foreach ($changesFields as $fieldName) {
            $fieldNameCamelCase = GeneralUtility::underscoredToUpperCamelCase($fieldName);

            if (is_object($old)) {
                $oldData[$fieldName] = $old->{'get' . $fieldNameCamelCase}();
            }
            if (is_object($new)) {
                $newData[$fieldName] = $new->{'get' . $fieldNameCamelCase}();
            }
        }

        $addressChange = GeneralUtility::makeInstance('Internetgalerie\\IgSimplesubscribe\\Domain\\Model\\AddressChanges');
        $addressChange->setTstamp(time());
        $addressChange->setCrdate(time());
        $addressChange->setUserid($uid);
        $addressChange->setState($state);
        $addressChange->setOldData(utf8_encode(serialize($oldData)));
        $addressChange->setNewData(utf8_encode(serialize($newData)));

        $this->addressChangesRepository->add($addressChange);
        $this->persistenceManager->persistAll();
    }

    /**
     * Checks a mail address
     *
     * @param string $email
     *
     * @return bool
     */
    private function checkEmail($email): bool
    {
        return \TYPO3\CMS\Core\Utility\GeneralUtility::validEmail($email);
    }
    private function checkPassword($password): bool
    {
        return strlen($password) >= 6; // @todo
    }
    private function checkUnique($tablename, $fieldname, $value): bool
    {
        if ($tablename == 'tt_address' && $fieldname == 'email') {
            $address = $this->dataRepository->findByEmail($value)->getFirst();

            if ($address && !$address->isHidden()) {
                return false;
            }
            return true;
        }
        // Dieses Query ist ohne PID  - bei fe_users meist ok  @todo
        $queryBuilder = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance(\TYPO3\CMS\Core\Database\ConnectionPool::class)->getQueryBuilderForTable($tablename);
        $queryBuilder->getRestrictions();
        $rows = $queryBuilder->select($fieldname)->from($tablename)->where($queryBuilder->expr()->eq($fieldname, $queryBuilder->createNamedParameter($value)))->execute()->fetch();
        return $rows == false || count($rows) == 0;
    }

    private function checkExists($tablename, $fieldname, $value): bool
    {
        // Fuer TYpo3 7.6.x
        if ($tablename == 'tt_address' && $fieldname == 'email') {
            $address = $this->dataRepository->findByEmail($value)->getFirst();

            if ($address && !$address->isHidden()) {
                return true;
            }
            return false;
        }
        // PID fehlt???? @todo
        $queryBuilder = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance(\TYPO3\CMS\Core\Database\ConnectionPool::class)->getQueryBuilderForTable($tablename);
        $queryBuilder->getRestrictions();
        $rows = $queryBuilder->select($fieldname)->from($tablename)->where($queryBuilder->expr()->eq($fieldname, $queryBuilder->createNamedParameter($value)))->execute()->fetch();
        return $rows !== false && count($rows) == 1;
    }

    /**
     * Checks a captcha
     *
     * @param string $captcha
     *
     * @return bool
     */
    private function checkCaptcha($captcha): bool
    {
        if (intval($this->settings['flexform']['captcha']) && \TYPO3\CMS\Core\Utility\ExtensionManagementUtility::isLoaded('recaptcha')) {
            $versionInformation = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance(\TYPO3\CMS\Core\Information\Typo3Version::class);
            $validCaptcha = false;
            
            if ($versionInformation->getMajorVersion() > 11) {
                $response = $this->request->getParsedBody()['g-recaptcha-response'] ?? null;

                if($response) {
                    $captchaServiceValidation = $this->captchaService->validateReCaptcha($response);
                    if (isset($captchaServiceValidation['verified'])) {
                        if ($captchaServiceValidation['verified'] === true) {
                            $validCaptcha = true;
                        }
                    }
                }
            } else {
                $status = GeneralUtility::makeInstance(\Evoweb\Recaptcha\Services\CaptchaService::class)->validateReCaptcha();

                $validCaptcha = ($status == false || $status['error'] !== '') ? false : true;
            }

            if (!$validCaptcha) {
                return false;
            }
        }

        return true;
    }

    /**
     * Checks a linkHash/hash
     *
     * @param string $linkHash
     * @param \Internetgalerie\IgSimplesubscribe\Domain\Model\Address $address
     *
     * @return bool
     */
    public function checkLinkHash($linkHash, $address): bool
    {
        /* !!!!!!WE ALREADY HAVE THE SALTED PASSWORD SO WE ONLY NEED THIS AND COMPARE THE TWO HASHES!!!!!!
        if(\TYPO3\CMS\Core\Utility\ExtensionManagementUtility::isLoaded('saltedpasswords') && \TYPO3\CMS\Saltedpasswords\Utility\SaltedPasswordsUtility::isUsageEnabled('FE')) {
        $objSalt = \TYPO3\CMS\Saltedpasswords\Salt\SaltFactory::getSaltingInstance(null);

        if(is_object($objSalt)) {
        return $objSalt->checkLinkHash($password, $address->getLinkHash());
        }
        }*/
        return $linkHash == $address->getLinkHash();
    }

    /**
     * Sends mail with subscription link and shows infotext
     *
     * @param string $email
     * @param array $fields
     * @param array $categories
     *
     * @return void
     */
    protected function sendConfirmationMail($email, $fields = [], $categories = [], $accepted = false)
    {
        if ((int)$this->settings['subscriptionsPerHourLimit'] > 0) {
            $registrationCountInLastHour = $this->dataRepository->countInLastHour();
            if ($registrationCountInLastHour >= (int)$this->settings['subscriptionsPerHourLimit']) {
                return $this->redirect('register');
                exit();
            }
        }

        if ($email == '') {
            return true;
        }

        $context = GeneralUtility::makeInstance(Context::class);
        $languageUid = $context->getPropertyFromAspect('language', 'id');
        $event = $this->eventDispatcher->dispatch(
            new BeforePersistenceEvent($fields),
        );
        $fields = $event->getFields();
        if ($this->tablename == 'tt_address') {
            $address = $this->dataRepository->findByEmail($email)->getFirst();

            if ($address && !$address->isHidden()) {
                return $this->redirect('register'); // TODO: What do we do if the address exists already here?
                exit();
            }
            $new = false;

            if (!$address) {
                $new = true;
                $address = GeneralUtility::makeInstance('Internetgalerie\\IgSimplesubscribe\\Domain\\Model\\Address');
                $address->setEmail($email);
                $address->setHidden(true);
            }

            foreach ($categories as $category) {
                if ($category['checked'] && !$address->hasSysDmailCategory($category['object'])) {
                    $address->addSysDmailCategory($category['object']);
                } elseif ($address->hasSysDmailCategory($category['object'])) {
                    $address->deleteSysDmailCategory($category['object']);
                }
            }

            if ($this->settings['logIpAddress']) {
                $address->setIpAddress($_SERVER['REMOTE_ADDR'] . ($_SERVER['HTTP_X_FORWARDED_FOR'] ? ' (' . $_SERVER['HTTP_X_FORWARDED_FOR'] . ')' : ''));
            }
            if ($this->settings['logBrowser']) {
                $address->setBrowser($_SERVER['HTTP_USER_AGENT']);
            }
            if($this->settings['singleOptIn']) {
                $address->setHidden(false);
            }
            $address->setSendHtmlMail(true);

            $address->_setProperty('_languageUid', $languageUid);

            if ($new) {
                $this->dataRepository->add($address);
                $this->persistenceManager->persistAll();
            }
        }
        if ($this->tablename == 'fe_users') {
            // Missing
            $address = GeneralUtility::makeInstance(FrontendUser::class);
            $address->setEmail($email);
            $address->setUsername($email);
            $address->setDisable(true);
            $address->_setProperty('_languageUid', $languageUid);
            $this->dataRepository->add($address);
            $this->persistenceManager->persistAll();
        }

        $address->setLinkHash($this->generateLinkHash($address->getUid() . time()));

        if ($this->settings['needToAccept']) {
            $address->setAccepted($accepted);
        }

        $attachments = [];

        foreach ($fields as $field) {
            if (isset($field['value'])) {
                $fieldUpperCamelCase = GeneralUtility::underscoredToUpperCamelCase($field['name']);

                if ($field['type'] != 'upload') {
                    $address->{'set' . $fieldUpperCamelCase}($field['value']);
                } elseif ($field['fileUid']) {
                    $file = $this->fileRepository->findByUid($field['fileUid']);

                    $fileReference = GeneralUtility::makeInstance('Internetgalerie\\IgSimplesubscribe\\Domain\\Model\\FileReference');
                    $fileReference->setOriginalResource($file);
                    $fileReference->setUidForeign($address->getUid());

                    $address->{'add' . $fieldUpperCamelCase}($fileReference);

                    $attachments[] = $file->getPublicUrl();
                }
            }
        }

        $this->dataRepository->update($address);
        $this->persistenceManager->persistAll();

        $this->sendMail(
            $this->settings['flexform']['email'],
            $email,
            $this->settings['flexform']['emailtopic'],
            $this->settings['singleOptIn'] ? 'IgSimplesubscribe/SingleOptIn' : 'IgSimplesubscribe/Confirmation',
            [
                'settings' => $this->settings,
                'fields' => $fields,
                'address' => $address,
                'categories' => $categories,
            ]
        );

        // Admin mail
        if ($this->settings['singleOptIn'] && intval($this->settings['flexform']['emailonsignin'])) {
            $this->sendAdminMail($address);
            $this->submitChanges($address->getUid(), 'subscribed', null, $address);
        }

        return true;
    }

    /**
     * @param \Internetgalerie\IgSimplesubscribe\Domain\Model\Address $address
     *
     * @return void
     */
    protected function sendAdminMail($address): void
    {
        $fields = $this->getFieldsFromObject($address);
        $categories = $this->getCategories($address);

        $this->sendMail(
            $this->settings['flexform']['email'],
            $this->settings['flexform']['email'],
            $this->settings['flexform']['emailtopic'],
            'IgSimplesubscribe/Admin',
            [
                'settings' => $this->settings,
                'address' => $address,
                'fields' => $fields,
                'categories' => $categories,
            ]
        );
    }

    /**
     * Sends mail with edit link and shows infotext
     *
     * @return void
     */
    protected function sendEditConfirmationMail($address): void
    {
        $fields = $this->getFieldsFromObject($address);
        $categories = $this->getCategories($address);

        $this->sendMail(
            $this->settings['flexform']['email'],
            $address->getEmail(),
            $this->settings['flexform']['emailtopicedit'],
            'IgSimplesubscribe/Edit',
            [
                'settings' => $this->settings,
                'address' => $address,
                'fields' => $fields,
                'categories' => $categories,
            ]
        );
    }

    /**
     * Sends mail with edit link and shows infotext
     *
     * @param \Internetgalerie\IgSimplesubscribe\Domain\Model\Address $address
     * @param array $fields
     * @param array $categories
     *
     * @return void
     */
    protected function sendEditOrDeleteAdminMail($address, $fields, $categories, $edit = true): void
    {
        $email = $this->settings['flexform']['email'];
        if($edit && $this->settings['flexform']['emailEdit']) {
            $email = $this->settings['flexform']['emailEdit'];
        } else if(!$edit && $this->settings['flexform']['emailUnsubscribe']) {
            $email = $this->settings['flexform']['emailUnsubscribe'];
        }
        $this->sendMail(
            $email,
            $email,
            $this->settings['flexform']['emailtopicedit'],
            $edit ? 'IgSimplesubscribe/AdminEdit' : 'IgSimplesubscribe/AdminDelete',
            [
                'settings' => $this->settings,
                'address' => $address,
                'fields' => $fields,
                'categories' => $categories,
            ]
        );
    }

    /**
     * Send a mail with sender $this->settings['flexform']['email']
     *
     * @param string $from
     * @param mixed $to The recipient
     * @param string $subject Subject of the mail
     * @param string $body Mail content
     * @param array $attachments
     *
     * @return void
     */
    protected function sendMail($from, $to, $subject, $template, $variables = [], $attachments = []): void
    {
        $mail = GeneralUtility::makeInstance(FluidEmail::class);

        foreach ($attachments as $attachment) {
            $mail->attachFromPath($attachment);
        }

        if (!is_array($to)) {
            $to = [$to];
        }

        $mail->from($from)
            ->to(...$to)
            ->subject($subject)
            ->format('html')
            ->setTemplate($template)
            ->assignMultiple($variables);

        if($this->request) {
            $mail->setRequest($this->request);
        } else if (!$this->request && $GLOBALS['TYPO3_REQUEST'] instanceof ServerRequestInterface) {
            $mail->setRequest($GLOBALS['TYPO3_REQUEST']);
        }

        GeneralUtility::makeInstance(Mailer::class)->send($mail);
    }

    /**
     * Get the fields for an address
     *
     * @param \Internetgalerie\IgSimplesubscribe\Domain\Model\Address $address
     *
     * @return array
     */
    protected function getFieldsFromObject($address): array
    {
        $fields = $this->initFields();
        foreach ($fields as $fieldName => &$field) {
            $field['value'] = $address->{'get' . GeneralUtility::underscoredToUpperCamelCase($field['name'])}();
        }
        return $fields;
    }

    protected function getFieldsFromArguments($initFieldNames = null, $initMandatoryFieldNames = null, $initFieldConfigs = null): array
    {
        $fields = $this->initFields($initFieldNames, $initMandatoryFieldNames, $initFieldConfigs);
        foreach ($fields as $fieldName => &$field) {
            if ($this->request->hasArgument($field['name'])) {
                $field['value'] = $this->request->getArgument($field['name']);
            }
            if ($field['type'] == 'upload') {
                $field['fileUid'] = $this->uploadFile($field);
            }
        }
        return $fields;
    }

    /**
     * Get the categories for an address
     *
     * @param \Internetgalerie\IgSimplesubscribe\Domain\Model\Address $address
     *
     * @return array
     */
    protected function getCategories($address): array
    {
        $categories = [];

        foreach ($this->categoryUids as $categoryUid) {
            $category = [];
            $category['object'] = $this->categoryRepository->findByUid($categoryUid);
            $category['checked'] = $address->hasSysDmailCategory($category['object']);
            $categories[] = $category;
        }

        return $categories;
    }

    /**
     * Uploads a file by a field
     *
     * @param array $file
     * @param string $uploadDir
     *
     * @return \TYPO3\CMS\Core\Resource\File
     */
    protected function uploadFile(&$field): ?int
    {
        $uploadDir = $this->settings['uploadDir'];
        $pluginSignature = 'tx_igsimplesubscribe_igsimplesubscribe';
        $file = [];
        $file['name'] = $_FILES[$pluginSignature]['name'][$field['name']];
        $file['type'] = $_FILES[$pluginSignature]['type'][$field['name']];
        $file['tmp_name'] = $_FILES[$pluginSignature]['tmp_name'][$field['name']];
        $file['error'] = $_FILES[$pluginSignature]['error'][$field['name']];
        $file['size'] = $_FILES[$pluginSignature]['size'][$field['name']];
        $file['extension'] = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $file['directory'] = $uploadDir . '/' . $field['name'];

        if ($file['name'] == '' && !$field['mandatory']) {
            return null;
        } elseif ($file['name'] == '') {
            $this->addError($field['name'], 'error.' . $field['name']);
            $field['error'] = true;
            return null;
        }

        if ($field['extensions'] && !is_array($field['extensions']) && $field['extensions'] != '*'
            || is_array($field['extensions']) && !in_array($file['extension'], $field['extensions'])) {
            $this->addError($field['name'], 'error.extension.' . $field['name']);
            $field['error'] = true;
            return null;
        }

        $storage = $this->storageRepository->findByUid($this->settings['uploadStorageUid']);
        $parts = explode('/', $file['directory']);
        $path = '';

        foreach ($parts as $part) {
            $newPath = $path . $part . '/';
            if (!$storage->hasFolder($newPath)) {
                $storage->createFolder($newPath, $path != '' ? $storage->getFolder($path) : $storage->getRootLevelFolder());
            }
            $path = $newPath;
        }

        /** @var File */
        $newFile = $storage->addUploadedFile($file, $storage->getFolder($file['directory']), null, \TYPO3\CMS\Core\Resource\DuplicationBehavior::RENAME);
        if (!$newFile) {
            $this->addError($field['name'], 'error.upload.' . $field['name']);
            $field['error'] = true;
            return null;
        }

        return $newFile->getUid();
    }

    /**
     * Generate a hash for the edit and subscribe links in the mails
     *
     * @param string|null $str
     *
     * @return string
     */
    public function generateLinkHash($str): string
    {
        /*if (\TYPO3\CMS\Core\Utility\ExtensionManagementUtility::isLoaded('saltedpasswords') && \TYPO3\CMS\Saltedpasswords\Utility\SaltedPasswordsUtility::isUsageEnabled('FE')) {
            $objSalt = \TYPO3\CMS\Saltedpasswords\Salt\SaltFactory::getSaltingInstance(null);

            if (is_object($objSalt)) {
                return $objSalt->getHashedPassword($str);
            }
        }*/

        return md5($str . $GLOBALS['TYPO3_CONF_VARS']['SYS']['encryptionKey']);
    }

    /**
     * Get the field names from flexform
     *
     * @return array
     */
    protected function getFieldNames(): array
    {
        return $this->getExplodedFlexFormSetting('fields');
    }

    /**
     * Get the mandatory field names from flexform
     *
     * @return array
     */
    protected function getMandatoryFieldNames(): array
    {
        return $this->getExplodedFlexFormSetting('mandatory');
    }

    /**
     * Get the category uids from flexform
     *
     * @return array
     */
    protected function getCategoryUids(): array
    {
        return $this->getExplodedFlexFormSetting('category');
    }

    /**
     * Get an exploded flexform setting
     *
     * @param string $key Flexform settings key
     *
     * @return array
     */
    protected function getExplodedFlexFormSetting($key): array
    {
        if(!isset($this->settings['flexform'][$key])) {
            return [];
        }
        return GeneralUtility::trimExplode(',', $this->settings['flexform'][$key]);
    }

    protected function getFieldType($name): string
    {
        $type = $this->fieldConfigs[$name];
        return $type && $type['type'] ? $type['type'] : 'text';
    }

    protected function deleteOldEmails($email): void
    {
        $typo3Version = \TYPO3\CMS\Core\Utility\VersionNumberUtility::convertVersionStringToArray(\TYPO3\CMS\Core\Utility\VersionNumberUtility::getNumericTypo3Version());

        if ($email) {
            if ($typo3Version['version_main'] >= 8) {
                // Do Doctrine Delete
                $queryBuilder = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance(\TYPO3\CMS\Core\Database\ConnectionPool::class)
                    ->getQueryBuilderForTable('tt_address');

                $queryBuilder
                    ->delete('tt_address')
                    ->where($queryBuilder->expr()->eq('deleted', 1))
                    ->orWhere($queryBuilder->expr()->eq('hidden', 1))
                    ->andWhere($queryBuilder->expr()->eq('email', $queryBuilder->createNamedParameter(is_array($email) ? $email['value'] : $email)))
                    ->executeStatement();
            } else {
                // Do DatabaseConnection Delete
                $GLOBALS['TYPO3_DB']->exec_DELETEquery(
                    'tt_address',
                    'email = \'' . $email . '\' AND deleted = 1 OR hidden = 1'
                );
            }
        }
    }

    protected function initFields($initFieldNames = null, $initMandatoryFieldNames = null, $initFieldConfigs = null): array
    {
        $fields = [];
        // Fill the fields array
        $fieldNames = $initFieldNames === null ? $this->fieldNames : $initFieldNames;
        $mandatoryFieldNames = $initMandatoryFieldNames === null ? $this->mandatoryFieldNames : $initMandatoryFieldNames;
        $fieldConfigs = $initFieldConfigs === null ? $this->fieldConfigs : $initFieldConfigs;

        foreach ($fieldNames as $fieldName) {
            if ($fieldName == '') {
                continue;
            }

            $mandatory = in_array($fieldName, $mandatoryFieldNames);
            $field = [
                'name' => $fieldName,
                'type' => 'text',
                'mandatory' => $mandatory,
                'validate' => $mandatory ? 'notEmpty' : '',
                'value' => null,
                'error' => 0,
                'errorMessage' => '',
            ];
            // Default Condif kopieren
            foreach ($fieldConfigs[$field['name']] as $name => $value) {
                $field[$name] = $value;
            }

            $fields[$field['name']] = $field;
        }
        //var_dump($fields);exit(0);
        return $fields;
    }
    protected function getEmailFieldname()
    {
        return $this->tablename == 'fe_users' ? 'username' : 'email';
    }

    protected function isSubmitted(): bool
    {
        return $this->request->hasArgument('submit');
    }

    /**
     * Add an error message to the flashmessages
     *
     * @param string $messageKey
     * @param string $title
     *
     * @return void
     */
    protected function addError($attribute, $messageKey, $title = 'Error'): ?string
    {
        $this->hasError = true;
        $msg = LocalizationUtility::translate('LLL:' . $this->settings['langFile'] . ':' . ($attribute ? $attribute . '.' : '') . $messageKey);
        if (!$msg) {
            $msg = $messageKey;
        }

        if ($attribute) {
            $this->fieldErrors[$attribute] = true;
        }
        $this->addFlashMessage(
            $msg,
            $title,
            ContextualFeedbackSeverity::ERROR,
            false
        );
        // TODO muss bei Mail etc. schlaue Textmeldung geben
        return LocalizationUtility::translate('LLL:' . $this->settings['langFile'] . ':' . $messageKey);
    }

    protected function hasErrors(&$fields): bool
    {
        $arguments = $this->request->getArguments();
        $this->hasError = false;

        /*if(!isset($arguments['test']) && $arguments['test'] != 3143558231) {
        $this->hasError = true;
        }*/

        foreach ($fields as $fieldName => &$field) {
            $val = $field['value'];
            foreach (explode(',', $field['validate']) as $validate) {
                //echo($fieldName .' mit '. $validate .'<br />');
                if ($validate == 'notEmpty' && ($val === null || $val == '')) {
                    $field['error'] = 1;
                    $field['errorMessage'] = $this->addError($fieldName, $validate);
                    //echo('failed:' . $validate .'<br />');
                }
                if ($validate == 'email' && !$this->checkEmail($val)) {
                    $field['error'] = 1;
                    $field['errorMessage'] = $this->addError($fieldName, 'error.email');
                    //echo('failed:' . $validate .'<br />');
                }
                if ($validate == 'password' && !$this->checkPassword($val)) {
                    $field['error'] = 1;
                    $field['errorMessage'] = $this->addError($fieldName, 'error.password');
                    //echo('failed:' . $validate .'<br />');
                }
                if ($validate == 'unique' && !$this->checkUnique($this->tablename, $fieldName, $val)) {
                    $field['error'] = 1;
                    $field['errorMessage'] = $this->addError($fieldName, $fieldName . '.error.unique');
                    //echo('failed:' . $validate .'<br />');
                }
                if ($validate == 'exists' && !$this->checkExists($this->tablename, $fieldName, $val)) {
                    $field['error'] = 1;
                    $field['errorMessage'] = $this->addError($fieldName, 'error.exists');
                    //echo('failed:' . $validate .'<br />');
                }
            }
        }

        if (!$this->checkCaptcha($arguments['captcha'])) {
            $this->addError('captcha', 'error.captcha');
            $captchaError = true;
        }
        if ($this->settings['flexform']['category'] && isset($arguments['categories']) && empty($arguments['categories'])) {
            $this->addError('categories', 'error.categories');
            $this->view->assign('categoriesError', true);
        }
        if ($this->settings['needToAccept'] && isset($arguments['accepted']) && !$arguments['accepted']) {
            $this->addError('accepted', 'error.accepted');
            $this->view->assign('acceptedError', true);
        }
        return $this->hasError;
    }

    /**
     * @param PropertyMappingConfiguration $configuration
     * @param array $userData
     *
     * @return PropertyMappingConfiguration
     */
    protected function getPropertyMappingConfiguration(
        PropertyMappingConfiguration $configuration = null,
        $userData = []
    ): PropertyMappingConfiguration 
    {
        if (is_null($configuration)) {
            $configuration = GeneralUtility::makeInstance(
                PropertyMappingConfiguration::class
            );
        }

        $configuration->allowAllProperties();
        //$configuration->forProperty('usergroup')->allowAllProperties();
        //$configuration->forProperty('image')->allowAllProperties();
        $configuration->setTypeConverterOption(
            PersistentObjectConverter::class,
            PersistentObjectConverter::CONFIGURATION_CREATION_ALLOWED,
            true
        );

        /*
        $folder = $this->fileService->getTempFolder();
        $uploadConfiguration = [
        UploadedFileReferenceConverter::CONFIGURATION_ALLOWED_FILE_EXTENSIONS =>
        $GLOBALS['TYPO3_CONF_VARS']['GFX']['imagefile_ext'],
        UploadedFileReferenceConverter::CONFIGURATION_UPLOAD_FOLDER =>
        $folder->getStorage()->getUid() . ':' . $folder->getIdentifier(),
        ];

        $configuration->forProperty('image.0')
        ->setTypeConverterOptions(
        UploadedFileReferenceConverter::class,
        $uploadConfiguration
        );

        $configuration->forProperty('dateOfBirth')
        ->setTypeConverterOptions(
        DateTimeConverter::class,
        [
        DateTimeConverter::CONFIGURATION_USER_DATA => $userData,
        ]
        );
         */
        return $configuration;
    }
}
