<?php
declare(strict_types = 1);

namespace TYPO3\igconfig\DataProcessing;


use TYPO3\CMS\Core\Resource\FileRepository;
use TYPO3\CMS\Core\Resource\ResourceFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\VersionNumberUtility;
use TYPO3\CMS\Frontend\ContentObject\DataProcessorInterface;
use TYPO3\CMS\Frontend\ContentObject\ContentObjectRenderer;
use TYPO3\CMS\Frontend\Controller\TypoScriptFrontendController;

class PageMediaProcessor  implements DataProcessorInterface
{
    /**
     * The content object renderer
     *
     * @var ContentObjectRenderer
     */
    public $cObj;

    /**
     * The processor configuration
     *
     * @var array
     */
    protected $processorConfiguration;

    /**
     * @var string
     */
    protected $targetVariableName;
    /**
     * @var string
     */
    protected $fieldName;
    /**
     * @var bool
     */
    protected $slide;


    /**
     * @var array
     */
    protected $defaults = [
        'field' => 'media',
        'as' => 'pageImage',
        'random' => '0',
        'index' => '0',
        'count' => '1',
        'cssSizes' => '',
        'cssSelector' => 'body',
        'slide' => '1',
    ];
 
    /**
     * Get configuration value from processorConfiguration
     *
     * @param string $key
     * @return string
     */
    protected function getConfigurationValue(string $key): string
    {
        return $this->cObj->stdWrapValue($key, $this->processorConfiguration, $this->defaults[$key]);
    }

    /**
     * @param ContentObjectRenderer $cObj The data of the content element or page
     * @param array $contentObjectConfiguration The configuration of Content Object
     * @param array $processorConfiguration The configuration of this processor
     * @param array $processedData Key/value store of processed data (e.g. to be passed to a Fluid View)
     * @return array the processed data as key/value store
     */
    public function process(ContentObjectRenderer $cObj, array $contentObjectConfiguration, array $processorConfiguration, array $processedData): array
    {
        $this->cObj = $cObj;
        $this->processorConfiguration = $processorConfiguration;
        $this->slide = (bool)$this->getConfigurationValue('slide');
        // Get Configuration
        $this->targetVariableName = $this->getConfigurationValue('as');
        $this->fieldName = $this->getConfigurationValue('field');


        // references.data =  levelmedia:-1, slide

        //$request = $this->renderingContext->getAttribute(ServerRequestInterface::class);
        if (VersionNumberUtility::convertVersionNumberToInteger(VersionNumberUtility::getNumericTypo3Version()) >= 13000000) {
            $request = $GLOBALS['TYPO3_REQUEST'];
            $localRootLine = $request->getAttribute('frontend.page.information')->getLocalRootLine();
        } else {
            $tsfe = $this->getTypoScriptFrontendController();
            $localRootLine = $tsfe->tmpl->rootLine;
        }

        $numericKey = $cObj->getKey(-1, $localRootLine);
        $sysFileReferenceUidsString = $cObj->rootLineValue($numericKey, $this->fieldName, $this->slide);

        // Keine Bilder vorhandne
        if( !$sysFileReferenceUidsString ) {
            return $processedData;
        }

        $sysFileReferenceUids=explode(',', $sysFileReferenceUidsString);


        $index = intval($this->getConfigurationValue('index'));
        $count = intval($this->getConfigurationValue('count'));

        // verlangtes Bild nicht vorhanden
        if(!isset($sysFileReferenceUids[$index])) {
            return $processedData;
        }
        $random = $this->getConfigurationValue('random');
        if( intval($random) ) {
            shuffle($sysFileReferenceUids);
        }


        //$fileRepository = GeneralUtility::makeInstance(FileRepository::class);
        $resourceFactory = GeneralUtility::makeInstance(ResourceFactory::class);
        $images=[];
        if($count == 0 || $count > 1) {

            if($count == 0 || $count > count($sysFileReferenceUids)) {
                $count = count($sysFileReferenceUids);
            }

            for($i = 0; $i < $count; $i++) {
                $image = $resourceFactory->getFileReferenceObject((int)$sysFileReferenceUids[$i]);
                
                //$image = $fileRepository->findFileReferenceByUid($sysFileReferenceUids[$i]);
                if($this->addImage($image)) {
                    $images[] = $image;
                }
            }
        } else {
            $image = $resourceFactory->getFileReferenceObject((int)$sysFileReferenceUids[$index]);
            //$image = $fileRepository->findFileReferenceByUid($sysFileReferenceUids[$index]);
            if($this->addImage($image)) {
                $images[] = $image;
            }

            $cssSizes=trim($this->getConfigurationValue('cssSizes'));
            if( $cssSizes!='' ) {
                $cssSelector=$this->getConfigurationValue('cssSelector');
                $this->addCss( $image, $cssSizes, $cssSelector );
            }
        }

        $processedData[$this->targetVariableName]=$images;
        //[ 'fileReferences' => $images ];
        return $processedData;
    }

    protected function addImage($image)
    {
        return true;
    }

    protected function addCss( \TYPO3\CMS\Core\Resource\FileReference $image, string $cssSizes, string $cssSelector ) {
        $cssSizesArray=GeneralUtility::trimExplode(',',$cssSizes);
        sort($cssSizesArray);// smallest fist for mobile first
        $css='';
        $imageService=GeneralUtility::makeInstance(\TYPO3\CMS\Extbase\Service\ImageService::class);
        $lastSize=null;
        foreach($cssSizesArray as $size) {
            if($size=='full') {
                $css.='' . $cssSelector . '{background-image:url("'.$image->getPublicUrl().'");}';
            } else {
                $processingInstructions = [
                    'width' => intval($size) . 'w',
                ];

                $processedImage = $imageService->applyProcessingInstructions($image, $processingInstructions);
                $absoluteUri=true;
                $processedImageUri = $imageService->getImageUri($processedImage, $absoluteUri);
                if($lastSize===null) {
                    $css.='' . $cssSelector . '{ background-image:url("'.$processedImageUri.'");}';
                } else {
                    $css.='@media (min-width: '. ($lastSize+1).'px) { ' . $cssSelector . '{ background-image:url("'.$processedImageUri.'");}}';
                }
                $lastSize=intval($size);
            }

        }
        $pageRenderer=GeneralUtility::makeInstance(\TYPO3\CMS\Core\Page\PageRenderer::class);
        $pageRenderer->addCssInlineBlock( 'igconfig-pageMedia', $css );
    }

    /**
     * @return \TYPO3\CMS\Frontend\Controller\TypoScriptFrontendController
     */
    protected function getTypoScriptFrontendController(): TypoScriptFrontendController
    {
        return $GLOBALS['TSFE'];
    }

}
