<?php

declare(strict_types=1);

namespace Internetgalerie\IgsCrm\Database\Query;

use Internetgalerie\IgsCrm\Utility\ConfUtility;
use Internetgalerie\IgsCrm\Utility\SecurityUtility;
use PDO;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\Query\Expression\CompositeExpression;
use TYPO3\CMS\Core\Database\Query\QueryBuilder;
use TYPO3\CMS\Core\Database\Query\Restriction\HiddenRestriction;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;

class ContactQueryBuilder extends QueryBuilder
{
    protected $tablename = 'tx_igscrm_domain_model_contact';
    protected $contactverbandTablename = 'tx_igscrm_domain_model_contactverband';
    protected $contactverbandJoined = false;
    protected $membershipTablename = 'tx_igscrm_domain_model_mitgliedschaft';
    protected $membershipJoined = false;
    protected $mmPersonverbandTagTablename = 'tx_igscrm_contactverband_tag_mm';
    protected $tagTablename = 'tx_igscrm_domain_model_tag';

    protected $defaultOrderings = [
        'extraSorting' => QueryInterface::ORDER_ASCENDING,
        'meLastname' => QueryInterface::ORDER_ASCENDING,
        'meFirstname' => QueryInterface::ORDER_ASCENDING,
        'meCompanyname' => QueryInterface::ORDER_ASCENDING,
    ];

    
    /**
     * confUtility
     *
     * @var ConfUtility
     */
    protected $confUtility = null;

    /**
     * securityUtility
     *
     * @var SecurityUtility
     */
    protected $securityUtility = null;


    /**
     * use acl for queries
     *
     * @var bool
     */
    protected $useAcl = true;

    /**
     * Select
     *
     * @var Select
     */
    protected $sqlSelect = '*';

    private $activeYear = null;
    private $activeDate = null;

    /**
     * Inject the ConfUtility (for checking access to objects)
     */
    public function injectConfUtility(ConfUtility $confUtility): void
    {
        $this->confUtility = $confUtility;
    }

    /**
     * Inject the SecurityUtility (for checking access to objects)
     */
    public function injectSecurityUtility(SecurityUtility $securityUtility): void
    {
        $this->securityUtility = $securityUtility;
    }

    public function getUseAcl()
    {
        return $this->useAcl;
    }
    public function setUseAcl(bool $useAcl)
    {
        return $this->useAcl = $useAcl;
    }
    
    public function setDefaultOrderings(array $defaultOrderings)
    {
        return $this->defaultOrderings = $defaultOrderings;
    }
    
    public function setActiveYear($activeYear = null): void
    {
        if ($activeYear === null) {
            $this->activeYear = null;
        } else {
            $activeYear = intval($activeYear);
            /*
            if (strpos($activeYear, '-') > 0) {
                $this->activeDate = $activeYear;
            }
            */
            $this->activeYear = $activeYear > 0 ? $activeYear : null;
        }
    }
    public function getActiveYear()
    {
        return $this->activeYear;
    }
    public function getActiveStartNextYear()
    {
        return $this->activeYear > 0 ? intval($this->activeYear + 1) . '-01-01' : (1 + date('Y')) . '-01-01';
    }
    public function getActiveDateEnd()
    {
        return $this->activeYear > 0 ? intval($this->activeYear) . '-12-31' : date('Y') . '-12-31';
    }
    public function getActiveDateEndLastYear()
    {
        return $this->activeYear > 0 ? intval($this->activeYear - 1) . '-12-31' : date('Y') . '-12-31';
    }
    public function getActiveDateStart()
    {
        if ($this->activeDate !== null) {
            return $this->activeDate;
        }
        return $this->activeYear > 0 ? intval($this->activeYear) . '-01-01' : date('Y') . '-01-01';
    }
    public function getStartEndDate($dateMode)
    {
        $dates = [
            //'startDate' => null,
            //'endDate' => null,
        ];
        if ($dateMode == 1) {
            $dates['startDate'] = $this->getActiveDateEnd();
            $dates['endDate'] = $this->getActiveDateStart();
        } elseif ($dateMode == 'current') {
            $dates['startDate'] = date('Y-m-d');
            $dates['endDate'] = date('Y-m-d');
        } elseif ($dateMode == 'inYear') {
            $dates['startDate'] = $this->getActiveDateEnd();
            $dates['endDate'] = $this->getActiveDateStart();
        } elseif ($dateMode == 'startYear') {
            $dates['startDate'] = $this->getActiveDateEndLastYear();
            $dates['endDate'] = $this->getActiveDateStart();
        } elseif ($dateMode == 'endYear') {
            $dates['startDate'] = $this->getActiveDateEnd();
            $dates['endDate'] = $this->getActiveDateEnd();
        } elseif ($dateMode == 'nextYear') {
            $dates['startDate'] = $this->getActiveDateEnd();
            $dates['endDate'] = $this->getActiveStartNextYear();
        }
        return $dates;
    }
    public function getContactverbandTablename(): string
    {
        return $this->contactverbandTablename;
    }
    

    public function joinContactVerband(): void
    {
        if (!$this->contactverbandJoined) {
            $this->join(
                $this->tablename,
                $this->contactverbandTablename,
                $this->contactverbandTablename,
                $this->expr()
->eq($this->tablename . '.uid', $this->quoteIdentifier($this->contactverbandTablename . '.contact'))
            );
            $this->contactverbandJoined = true;
        }
    }
    public function joinMembership(): void
    {
        if (!$this->membershipJoined) {
            $this->joinContactVerband();
            $this->leftJoin(
                $this->contactverbandTablename,
                $this->membershipTablename,
                $this->membershipTablename,
                $this->expr()
->eq($this->contactverbandTablename . '.mitgliedschaft', $this->quoteIdentifier($this->membershipTablename . '.uid'))
            );
            $this->membershipJoined = true;
        }
    }
    
    public function addUserRestriction()
    {
        $constraints = [];
        if (!$this->securityUtility->hasRoleAdmin('crm.contact') && $this->useAcl) {
            $frontendUserGroupIds = $this->securityUtility->getFrontendUserGroupIds();

            $orConstraints = [$this->expr() ->eq('acl_owner', $this->securityUtility->getFrontendUserId())];
            foreach ($this->securityUtility->getFrontendUserGroupIds() as $userGroupId) {
                //$orConstraints[] =  "FIND_IN_SET(" . (int) $userGroupId . " ,acl_write_groups)";
                $orConstraints[] = $this->expr()->inSet('acl_read_groups', (string)$userGroupId);
            }

            // give permission to contacts where this frontend user is added or added with contacts
            // $uids = $this->securityUtility->getFrontendUserOwnerUids();
            // if (!empty($uids)) {
            //     $orConstraints[] = $this->expr()->in(
            //         $this->tablename . '.uid',
            //         $this->createNamedParameter($uids, Connection::PARAM_INT_ARRAY)
            //     );
            // }
            //var_dump($orConstraints);exit(0);

            if (!empty($orConstraints)) {
                $constraints[] = $this->expr()->or(...$orConstraints);
            }
        }
        if (!empty($constraints)) {
            $this->andWhere(...$constraints);
        }
        return $this;
    }

    public function resetSelect()
    {
        $this->sqlSelect = '*';
        return $this;
    }


    public function addSearch($search)
    {
        $constrains = [];
        if (isset($search['uid']) && $search['uid'] !== '') {
            $constrains[] = $this->expr()->eq($this->tablename . '.uid', trim((string) $search['uid']));
        }
        if ($search['person'] ?? false) {
            $constrains[] = $this->expr()->eq($this->tablename . '.uid', $search['person']);
        }
        if ($search['contact'] ?? false) {
            $constrains[] = $this->expr()->eq($this->tablename . '.uid', $search['contact']);
        }


        if ($search['type'] ?? false) {
            $constrains[] = $this->expr()->eq('type', $this->createNamedParameter($search['type']));
        }
        if ($search['hidden'] ?? false) {
            $this->getRestrictions()
->removeByType(HiddenRestriction::class);
            $constrains[] = $this->expr()->eq($this->tablename . '.hidden', true);
        }
        // Organisation
        if (isset($search['meCategoryOrganisation'])) {
            if ($search['meCategoryOrganisation'] > 0) {
                //$constrains[] = $this->expr()->eq('organisationCategory.uid', $search['meCategoryOrganisation']);
                $constrains[] = 'EXISTS (SELECT 1 FROM tx_igscrm_organisation_categoryorganisation_mm WHERE uid_local=' . $this->tablename . '.uid AND uid_foreign= ' . (int)$search['meCategoryOrganisation'] . ')';
            } elseif ($search['meCategoryOrganisation'] == -1) {
                $constrains[] = 'NOT EXISTS (SELECT 1 FROM tx_igscrm_organisation_categoryorganisation_mm WHERE uid_local=' . $this->tablename . '.uid)';
            }
        }

        // is the same like meCategoryOrganisation
        if (isset($search['category'])) {
            if ($search['category'] > 0) {
                //$constrains[] = $query->contains('organisation_category', $search['category']);
                $constrains[] = 'EXISTS (SELECT 1 FROM tx_igscrm_organisation_categoryorganisation_mm WHERE uid_local=' . $this->tablename . '.uid AND uid_foreign= ' . (int)$search['category'] . ')';
            } elseif ($search['category'] == -1) {
                $constrains[] = 'NOT EXISTS (SELECT 1 FROM tx_igscrm_organisation_categoryorganisation_mm WHERE uid_local=' . $this->tablename . '.uid)';
                //$constrains[] = $this->expr()->eq('organisation_category', 0); // @todo EXISTS only works in queryBuilder
            }
        }

        
        if (isset($search['meNoEmployees']) && $search['meNoEmployees'] != '') {
            $constrains[] = $this->expr()->eq('me_no_employees', $search['meNoEmployees']);
        }
        
        // Verband
        if (isset($search['verband']) && $search['verband'] !== '') {
            $verband = $search['verband'];
            $verbandUid = is_object($verband) ? $verband->getUid() : (int) $verband;
            $this->joinContactVerband();
            
            $constrains[] = $this->expr()->eq($this->contactverbandTablename . '.verband', $verbandUid);
            if (isset($search['hasMemberCard']) && $search['hasMemberCard'] !== '') {
                $constrains[] = $this->expr()->isNotNull($this->contactverbandTablename . '.mitgliedschaft');
            }
            if (isset($search['memberActive']) && $search['memberActive'] !== '' && $search['memberActive'] !== 'all') {
                if ($search['memberActive'] === '0' || $search['memberActive'] === 0) {
                    $constrains[] = $this->expr()->eq($this->contactverbandTablename . '.active', 0);
                } else {
                    if ($search['memberActive'] == 1) {
                        $constrains[] = $this->expr()->eq(
                            $this->contactverbandTablename . '.active',
                            intval($search['memberActive'])
                        );
                        $constrains[] = $this->expr()->eq('me_verstorben', 0);
                    }
                    $dates = $this->getStartEndDate($search['memberActive']);
                    if (!empty($dates)) {
                        //$constrains[] = $this->expr()->eq('meContactVerband.active', 1);
                        $constrains[] = $this->expr()->or(
                            $this->expr()->gte(
                                $this->contactverbandTablename . '.end_date',
                                $this->createNamedParameter($dates['endDate'])
                            ),
                            $this->expr()->isNull($this->contactverbandTablename . '.end_date')
                        );
                        $constrains[] = $this->expr()->or(
                            $this->expr()->lte(
                                $this->contactverbandTablename . '.start_date',
                                $this->createNamedParameter($dates['startDate'])
                            ),
                            $this->expr()->isNull($this->contactverbandTablename . '.start_date')
                        );
                    }
                }
            }



            //&& $search['memberDateFrom']!='dd.mm.yyy'
            if (isset($search['memberDateFrom'])) {
                $memberDateFrom = is_array(
                    $search['memberDateFrom']
                ) ? $search['memberDateFrom']['date'] : $search['memberDateFrom'];
                if ($memberDateFrom) {
                    $constrains[] = $this->expr()->gte(
                        $this->contactverbandTablename . '.start_date',
                        $this->createNamedParameter($memberDateFrom)
                    );
                }
            }
            if (isset($search['memberDateTo'])) {
                $memberDateTo = is_array(
                    $search['memberDateTo']
                ) ? $search['memberDateTo']['date'] : $search['memberDateTo'];
                if ($memberDateTo) {
                    $constrains[] = $this->expr()->lte(
                        $this->contactverbandTablename . '.start_date',
                        $this->createNamedParameter($memberDateTo)
                    );
                }
            }
        }


        if ($search['test'] ?? false) {
            $constrains[] = $this->expr()->eq('me_newsletter_test', $search['test']);
        }



        //$constrains[]=$this->expr()->eq('meContactVerband.active', 1);
        $advSearchConstrain = null;
        if ($search['duplicate'] ?? false) {
            $uids = $this->findUidsWithDuplicateName();
            $constrains[] = $this->expr()->in('uid', $uids);
        }

        if ($search['keyword'] ?? false) {
            $keywordWords = array_filter(explode(' ', (string) $search['keyword']));
            foreach ($keywordWords as $keyword) {
                $constrains[] = $this->expr()->or(
                    $this->expr()->like('me_lastname', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_firstname', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_shortname', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_addressid', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_email', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_account', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_companyname', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_addon', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('addon_intern', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_city', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_zip', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_address', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->like('me_remark', $this->createNamedParameter('%' . $keyword . '%')),
                    $this->expr()->eq($this->tablename . '.uid', $this->createNamedParameter($keyword, Connection::PARAM_INT))
                );
            }
        }
        if (isset($search['meName'])) {
            $meNameWords = array_filter(explode(' ', (string) $search['meName']));
            //$meNameSearchArray = [];
            foreach ($meNameWords as $meNameWord) {
                $constrains[] = $this->expr()->or(
                    $this->expr()->like('me_lastname', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('me_firstname', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('me_shortname', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('me_account', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('me_extra_firma', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('me_firma', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('me_addressid', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('me_addon', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('addon_intern', $this->createNamedParameter('%' . $meNameWord . '%')),
                    $this->expr()->like('me_email', $this->createNamedParameter('%' . $meNameWord . '%'))
                );
            }
        }

        if (isset($search['numberFrom']) && $search['numberFrom'] != '') {
            $constrains[] = $this->expr()->gte('me_addressid', $search['numberFrom']);
        }
        if (isset($search['numberTo']) && $search['numberTo'] != '') {
            $constrains[] = $this->expr()->lte('me_addressid', $search['numberTo']);
        }

        if (isset($search['meJahrgangVon']) && $search['meJahrgangVon'] > 0) {
            $constrains[] = $this->expr()->gte(
                'me_date_of_birth',
                $this->createNamedParameter($search['meJahrgangVon'] . '-01-01')
            );
        }
        if (isset($search['meJahrgangBis']) && $search['meJahrgangBis'] > 0) {
            $constrains[] = $this->expr()->lte(
                'me_date_of_birth',
                $this->createNamedParameter($search['meJahrgangBis'] . '-12-31')
            );
        }

        if (!empty($search['meJahrgang'])) {
            $searchSubConstrains = [];

            // Zahlen fuer ausgewaehlter Jahr
            $jahr = $search['jahr'] ?? date('Y');
            $jahr = (int) $jahr;
            $diffJahr = $jahr - date('Y');
            $this->crmJahrgaenge = $this->confUtility->getCrmJahrgaenge();
            foreach ($search['meJahrgang'] as $uid) {
                if (isset($this->crmJahrgaenge[$uid])) {
                    $jahrgaenge = $this->crmJahrgaenge[$uid];
                    if (!empty($jahrgaenge['f'])) {
                        $searchSubConstrains[] = $this->expr()->or(
                            // Rest
                            $this->expr()
->and(
    $this->expr()->neq('me_anrede_id', 2),
    $this->expr()->gt(
        'me_date_of_birth',
        $this->createNamedParameter(intval(date('Y') + $diffJahr - $jahrgaenge['max']) . '-01-01')
    ),
    $this->expr()->lte('me_date_of_birth', $this->createNamedParameter(
        intval(date('Y') + $diffJahr - $jahrgaenge['min']) . '-12-31'
    ))
),
                            // Frau: anderes AHV Alter
                            $this->expr()
->and(
    $this->expr()->eq('me_anrede_id', 2),
    $this->expr()->gt(
        'me_date_of_birth',
        $this->createNamedParameter(intval(date('Y') + $diffJahr - $jahrgaenge['f']['max']) . '-01-01')
    ),
    $this->expr()->lte('me_date_of_birth', $this->createNamedParameter(
        intval(date('Y') + $diffJahr - $jahrgaenge['f']['min']) . '-12-31'
    ))
)
                        );
                    } else {
                        $searchSubConstrains[] = $this->expr()->and(
                            $this->expr()->gt(
                                'me_date_of_birth',
                                $this->createNamedParameter(intval(
                                    date('Y') + $diffJahr - $jahrgaenge['max']
                                ) . '-01-01')
                            ),
                            $this->expr()->lte('me_date_of_birth', $this->createNamedParameter(
                                intval(date('Y') + $diffJahr - $jahrgaenge['min']) . '-12-31'
                            ))
                        );
                    }
                }
            }
            if (!empty($searchSubConstrains)) {
                $constrains[] = $this->expr()->or(...$searchSubConstrains);
            }
        }
        if (isset($search['meJahrgang30Bis65']) && $search['meJahrgang30Bis65'] > 0) {
            $constrains[] = $this->expr()->gte(
                'me_date_of_birth',
                $this->createNamedParameter(intval(date('Y') - 65) . '-01-01')
            );
            $constrains[] = $this->expr()->lte(
                'me_date_of_birth',
                $this->createNamedParameter(intval(date('Y') - 30) . '-12-31')
            );
        }

        if (isset($search['meJahrgangUnter30Ueber65']) && $search['meJahrgangUnter30Ueber65'] > 0) {
            $constrains[] = $this->expr()->or(
                $this->expr()->gt('me_date_of_birth', $this->createNamedParameter(intval(date('Y') - 30) . '-01-01')),
                $this->expr()->lt('me_date_of_birth', $this->createNamedParameter(intval(date('Y') - 65) . '-12-31'))
            );
        }
        if (isset($search['meAnredeId']) && $search['meAnredeId'] > 0) {
            $constrains[] = $this->expr()->eq('me_anrede_id', $search['meAnredeId']);
        }

        if (isset($search['certificate'])) {
            if (is_array($search['certificate'])) {
                $addConstrains = $this->getConstrainsMmIn($search['certificate'], 'tx_igscrm_domain_model_contactcertificate', 'contact', 'certificate');
                if ($addConstrains !== null) {
                    $constrains[] = '(' . $addConstrains . ')';
                }
            }
        }
        if (isset($search['meKanton'])) {
            if (is_array($search['meKanton'])) {
                $addConstrains = $this->getConstrainsEmptyOrIn('me_kanton_id', $search['meKanton']);
                if ($addConstrains !== null) {
                    $constrains[] = $addConstrains;
                }
            } else {
                if ($search['meKanton'] > 0) {
                    $constrains[] = $this->expr()->eq('me_kanton_id', $search['meKanton']);
                }
            }
        }
        if (isset($search['meLanguageid'])) {
            if (is_array($search['meLanguageid'])) {
                if (!empty($search['meLanguageid'])) {
                    $constrains[] = $this->expr()->in('me_languageid', $search['meLanguageid']);
                }
            } else {
                if ($search['meLanguageid'] > 0) {
                    $constrains[] = $this->expr()->eq('me_languageid', $search['meLanguageid']);
                }
            }
        }
        if (isset($search['meSpracheD']) && $search['meSpracheD'] > 0 || isset($search['meSpracheF']) && $search['meSpracheF'] > 0 || isset($search['meSpracheI']) && $search['meSpracheI'] || isset($search['meSpracheE']) && $search['meSpracheE'] > 0) {
            $sprache_query = [];
            if (isset($search['meSpracheD']) && $search['meSpracheD'] > 0) {
                $sprache_query[] = $this->expr()->eq('me_languageid', 1);
            }
            if (isset($search['meSpracheE']) && $search['meSpracheE'] > 0) {
                $sprache_query[] = $this->expr()->eq('me_languageid', 2);
            }
            if (isset($search['meSpracheF']) && $search['meSpracheF'] > 0) {
                $sprache_query[] = $this->expr()->eq('me_languageid', 3);
            }
            if (isset($search['meSpracheI']) && $search['meSpracheI'] > 0) {
                $sprache_query[] = $this->expr()->eq('me_languageid', 4);
            }

            $constrains[] = $this->expr()->or(...$sprache_query);
        }
        if (isset($search['verstorben'])) {
            if ($search['verstorben'] == '0' || $search['verstorben'] == '1') {
                $constrains[] = $this->expr()->eq('me_verstorben', $search['verstorben']);
            } elseif ($search['verstorben'] == 'endYear') {
                $compareEndDate = $this->getActiveDateEnd();
                $compareStartDate = $this->getActiveDateStart();
                $constrains[] = $this->expr()->gte('me_todestag', $this->createNamedParameter($compareStartDate));
                $constrains[] = $this->expr()->lte('me_todestag', $this->createNamedParameter($compareEndDate));
            }
        }

        if (isset($search['patent_vorhanden']) && $search['patent_vorhanden'] != '') {
            $constrains[] = $this->expr()->eq('me_patent_vorhanden', (int)$search['patent_vorhanden']);
        }

        if (isset($search['meStatus']) && $search['meStatus'] != '') {
            $constrains[] = $this->expr()->eq('me_status', $search['meStatus']);
        }
        if (isset($search['meBbf']) && $search['meBbf'] != '') {
            $constrains[] = $this->expr()->eq('me_bbf', $search['meBbf']);
        }
        if (isset($search['usedForStatistics'])) {
            $this->joinMembership();
            if ($search['usedForStatistics'] == 1) {
                $constrains[] = $this->expr()->or(
                    $this->expr()->eq($this->membershipTablename . '.no_person', 0),
                    $this->expr()->isNull($this->contactverbandTablename . '.mitgliedschaft'),
                    $this->expr()->eq($this->contactverbandTablename . '.mitgliedschaft', 0)
                );
            } elseif ($search['usedForStatistics'] === '0' || $search['usedForStatistics'] === 0) {
                $constrains[] = $this->expr()->eq($this->membershipTablename . '.no_person', 1);
            }
        }
        if (isset($search['noPerson'])) {
            $this->joinMembership();
            if ($search['noPerson'] == 'notSet') {
                $constrains[] = $this->expr()->or(
                    $this->expr()->eq($this->membershipTablename . '.no_person', 0),
                    $this->expr()->isNull($this->contactverbandTablename . '.mitgliedschaft'),
                    $this->expr()->eq($this->contactverbandTablename . '.mitgliedschaft', 0)
                );
            } else {
                $constrains[] = $this->expr()->eq($this->membershipTablename . '.no_person', (int)$search['noPerson']);
            }
        }

        if (isset($search['mitgliedschaft'])) {
            if (!empty($search['mitgliedschaft']) && is_array($search['mitgliedschaft'])) {
                $constrains[] = $this->expr()->in(
                    $this->contactverbandTablename . '.mitgliedschaft',
                    $search['mitgliedschaft']
                );
            } elseif ($search['mitgliedschaft'] > 0) {
                $constrains[] = $this->expr()->eq(
                    $this->contactverbandTablename . '.mitgliedschaft',
                    $search['mitgliedschaft']
                );
            } elseif ($search['mitgliedschaft'] === '0') {
                // Falls keine Mitgliedschaft ausgewählt ist, nur Mitgliedschaften welche Personen sindm bzw. gezählt werden sollen
                $this->joinMembership();
                $constrains[] = $this->expr()->or(
                    $this->expr()->eq($this->membershipTablename . '.no_person', 0),
                    $this->expr()->isNull($this->contactverbandTablename . '.mitgliedschaft'),
                    $this->expr()->eq($this->contactverbandTablename . '.mitgliedschaft', 0)
                );
            } elseif ($search['mitgliedschaft'] === '-1') {
                $constrains[] = $this->expr()->or(
                    $this->expr()->isNull($this->contactverbandTablename . '.mitgliedschaft'),
                    $this->expr()->eq($this->contactverbandTablename . '.mitgliedschaft', 0)
                );
            }
        }
        if (isset($search['mitgliedschaftExclude'])) {
            if (!empty($search['mitgliedschaftExclude'])) {
                $constrains[] = $this->expr()->notIn(
                    $this->contactverbandTablename . '.mitgliedschaft',
                    $search['mitgliedschaftExclude']
                );
            } elseif ($search['mitgliedschaftExclude'] > 0) {
                $constrains[] = $this->expr()->neq(
                    $this->contactverbandTablename . '.mitgliedschaft',
                    $search['mitgliedschaftExclude']
                );
            }
        }

        if (!empty($search['tags'])) {
            $searchSubConstrains = [];
            foreach ($search['tags'] as $name => $tags) {
                if ($name === 'settings') {
                    continue;
                }
                // search[tags][subtag-group][] -> use OR on entries of subtag and AND on subtag-groups
                if (is_array($tags)) {
                    $noTag = 0;
                    $inTags = [];
                    foreach ($tags as $tagUid) {
                        if ((int)$tagUid < 0) {
                            // if tagUid is negative, we want all entries with no tags in this tag group (tagUid = -tagGroupUid)
                            $noTag = abs((int)$tagUid);
                        } else {
                            $inTags [] = (int)$tagUid;
                        }
                    }
                    $subTagConstrains = [];
                    if ($noTag) {
                        $subTagConstrains[] = 'NOT EXISTS (SELECT 1 FROM ' . $this->mmPersonverbandTagTablename . ' JOIN ' . $this->tagTablename . ' ON ' . $this->tagTablename . '.uid =uid_foreign WHERE uid_local=' . $this->contactverbandTablename . '.uid  AND tagverband=' . $noTag . ')';
                    }
                    if ($inTags) {
                        $subTagConstrains[] = 'EXISTS (SELECT 1 FROM ' . $this->mmPersonverbandTagTablename . ' WHERE uid_local=' . $this->contactverbandTablename . '.uid AND uid_foreign IN (' . implode(
                            ',',
                            $inTags
                        ) . '))';
                    }
                    if (!empty($subTagConstrains)) {
                        $searchSubConstrains[] = $this->expr()->or(...$subTagConstrains);
                    }
                } else {
                    // search[tags][] -> use AND on all entries
                    if ($tags) {
                        // $constrains[] = $query->contains('meContactVerband.tags', $tags);
                        $searchSubConstrains[] = 'EXISTS (SELECT 1 FROM ' . $this->mmPersonverbandTagTablename . '  WHERE uid_local=' . $this->contactverbandTablename . '.uid AND uid_foreign=' . (int)$tags . ')';
                    }
                }
            }
            if (!empty($searchSubConstrains)) {
                $logicalOperator = $search['tags']['settings']['operator'] ?? 1;
                if ($logicalOperator == 1) {
                    $constrains[] = $this->expr()->and(...$searchSubConstrains);
                } else {
                    $constrains[] = $this->expr()->or(...$searchSubConstrains);
                }
            }
        }
        //var_dump($logicalOperator, $searchSubConstrains);die('a');
        /*
        if (isset($search['sektionen']) && $search['sektionen'] > 0) {
            $constrains[] = $this->expr()->eq($this->contactverbandTablename . '.sektionen', (int)$search['sektionen']);
        }
        */
        //var_dump($constrains);exit(0);


        if (isset($search['meZipCity']) && $search['meZipCity'] != '') {
            $constrains[] = $this->expr()->or(
                $this->expr()->like('me_zip', $this->createNamedParameter('%' . $search['meZipCity'] . '%')),
                $this->expr()->like('me_city', $this->createNamedParameter('%' . $search['meZipCity'] . '%'))
            );
        }
        // string like
        foreach ([
            'me_lastname' => 'meLastname',
            'me_firstname' => 'meFirstname',
            'me_zip' => 'meZip',
            'me_city' => 'meCity',
            'me_account' => 'meAccount',
        ] as $attribute => $searchField) {
            if (isset($search[$searchField]) && $search[$searchField] != '') {
                $constrains[] = $this->expr()->like(
                    $attribute,
                    $this->createNamedParameter('%' . $search[$searchField] . '%')
                );
            }
        }
        // equal
        foreach ([
            'me_bildungspass' => 'meBildungspass',
            'me_kanton_id' => 'meKantonId',
        ] as $attribute => $searchField) {
            if (isset($search[$searchField]) && is_array($search[$searchField])) {
                if (!empty($search[$searchField])) {
                    $constrains[] = $this->expr()->in($attribute, $search[$searchField]);
                }
            } else {
                if (isset($search[$searchField]) && $search[$searchField] != '') {
                    $constrains[] = $this->expr()->eq($attribute, $search[$searchField]);
                }
            }
        }

        if (isset($search['meIsSammelinvoice'])) {
            if ($search['meIsSammelinvoice'] === '0') {
                $constrains[] = $this->expr()->and(
                    $this->expr()->eq('me_is_sammelinvoice', 0),
                    $this->expr()->or($this->expr()->eq(
                        'me_is_family',
                        0
                    ), 'NOT EXISTS (SELECT 1 FROM ' . $this->tablename . ' AS p WHERE p.uid=' . $this->tablename . '.me_family AND p.me_invoice_to<>0)')
                );
            } elseif ($search['meIsSammelinvoice'] === '1') {
                $constrains[] = $this->expr()->or(
                    $this->expr()->eq('me_is_sammelinvoice', 1),
                    $this->expr()->and($this->expr()->eq(
                        'me_is_family',
                        1
                    ), 'EXISTS (SELECT 1 FROM ' . $this->tablename . ' AS p WHERE p.uid=' . $this->tablename . '.me_family AND p.me_invoice_to<>0)')
                );
            }
        }

        // boolean
        foreach ([
            'me_is_family' => 'meIsFamily',
            'me_is_lernende' => 'meIsLernende',
            //'me_is_sammelinvoice' => 'meIsSammelinvoice',
        ] as $attribute => $searchField) {
            if (isset($search[$searchField]) && $search[$searchField] != '') {
                $constrains[] = $this->expr()->eq($attribute, $search[$searchField]);
            }
        }

        foreach ([
            'me_family' => 'meFamilyIsSet',
            'me_invoice_to' => 'meInvoiceToIsSet',
        ] as $attribute => $searchField) {
            //die('p=' . $searchField .'='.$search[$searchField]);
            if (isset($search[$searchField]) && $search[$searchField] != '') {
                if ($search[$searchField] == '1') {
                    $constrains[] = $this->expr()->gt($attribute, 0);
                } else {
                    $constrains[] = $this->expr()->or(
                        $this->expr()->eq($attribute, 0),
                        $this->expr()->isNull($attribute)
                    );
                }
            }
        }
        if (isset($search['versand']) && $search['versand'] != '') {
            if ($search['versand'] == 'letter') {
                // $sql_where .= "( email IS NULL OR email NOT LIKE '%@%') AND main_uid=0 AND invoice_uid=0 AND abo_id IN (4,5,9)";

                $constrains[] = $this->expr()->or(
                    $this->expr()->isNull('me_email'),
                    $this->expr()->notLike('me_email', $this->createNamedParameter('%@%'))
                );
                // bringt z.B. bei brueckenweg nur 13 Eintraege d.h. falls Subelemente E-Mail haben nicht brief an Hauptadresse
                // AND tx_igscrm_domain_model_contact.uid NOT IN (select me_invoice_to from tx_igscrm_domain_model_contact WHERE me_email LIKE '%@%' AND me_invoice_to>0)
                // AND tx_igscrm_domain_model_contact.uid NOT IN (select me_family from tx_igscrm_domain_model_contact WHERE me_email LIKE '%@%' AND me_family>0)

                //$constrains[] = 'me_email IS NULL';
                $constrains[] = $this->expr()->or(
                    $this->expr()->isNull('me_family'),
                    $this->expr()->eq('me_family', 0)
                );
                $constrains[] = $this->expr()->or(
                    $this->expr()->isNull('me_invoice_to'),
                    $this->expr()->eq('me_invoice_to', 0)
                );

                //$this->expr()->eq('contactVerband.mitgliedschaft.uid', 4),
            } elseif ($search['versand'] == 'email') {
                $constrains[] = $this->expr()->like('me_email', '%@%');
            }
        }

        if (isset($search['jahresabo'])) {
            if ($search['jahresabo'] == '1') {
                $constrains[] = $this->expr()->in($this->contactverbandTablename . '.mitgliedschaft', [4, 5, 8, 9]);
            } elseif ($search['jahresabo'] == '0') {
                $constrains[] = $this->expr()->notIn($this->contactverbandTablename . '.mitgliedschaft', [4, 5, 8, 9]);
            }
        }
        if (isset($search['hasInvoice']) && $search['hasInvoice'] == 1) {
            $constrains[] = $this->expr()->or($this->expr()->isNull('me_family'), $this->expr()->eq('me_family', 0));
        }
        /* Organisation */
        if ($search['meRespPersonidIsNull'] ?? false) {
            //$constrains[] = $this->expr()->eq('meActive', true);
            $constrains[] = $this->expr()->isNull('parent');
            $constrains[] = $this->expr()->isNull('me_resp_personid');
            //$constrains[] = $this->expr()->eq('meRespPersonidIsNull', null);
        }
        if (isset($search['companyname']) && $search['companyname'] != '') {
            $constrains[] = $this->expr()->or(
                $this->expr()->like('me_companyname', $this->createNamedParameter('%' . $search['companyname'] . '%')),
                $this->expr()->like('me_addon', $this->createNamedParameter('%' . $search['companyname'] . '%'))
            );
        }
        if (isset($search['meErhaeltInvoice'])) {
            $constrains[] = $this->expr()->eq('me_erhaelt_invoice', $search['meErhaeltInvoice']);
        }
        if ($search['letter'] ?? false) {
            $constrains[] = $this->expr()->like('me_companyname', $this->createNamedParameter($search['letter'] . '%'));
        }
        if ($search['meRespPersonidIsNull'] ?? false) {
            $constrains[] = $this->expr()->isNull('parent');
        }
        if ($search['meRespPersonidIsNull'] ?? false) {
            $constrains[] = $this->expr()->isNull('me_resp_personid');
        }
        $this->addSearchOrderBy($search);
        
        if (!empty($constrains)) {
            $this->andWhere(...$constrains);
        }
        return $this;
    }
    public function addArrayOrderBy(array $orderBys)
    {
        foreach ($orderBys as $attribute => $order) {
            if ($order == 'meAddressidWithMeFamiliyMeInvoiceTo') {
                $this->concreteQueryBuilder->addOrderBy(
                    'CASE WHEN me_invoice_to IS NOT NULL AND me_invoice_to>0 THEN (SELECT me_addressid FROM  tx_igscrm_domain_model_contact p WHERE p.uid=tx_igscrm_domain_model_contact.me_invoice_to) WHEN me_family IS NOT NULL AND me_family>0 THEN (SELECT CASE WHEN me_invoice_to IS NULL OR me_invoice_to<=0 THEN  me_addressid  ELSE  (SELECT me_addressid FROM  tx_igscrm_domain_model_contact pi WHERE pi.uid=p.me_invoice_to) END FROM  tx_igscrm_domain_model_contact p WHERE p.uid=tx_igscrm_domain_model_contact.me_family) ELSE me_addressid END', null);
            } else {
                $this->addOrderBy($attribute, $order);
            }
        }
        return $this;
    }
    public function addSearchOrderBy($search)
    {
        if (isset($search['type']) && $search['type'] == 'Organisation') {
            $orderBy = [
                'me_companyname' => QueryInterface::ORDER_ASCENDING,
                'me_city' => QueryInterface::ORDER_ASCENDING,
            ];
            $this->addArrayOrderBy($orderBy);
        }
        return $this;
    }
    /*
    public function getConstrainsMmEmptyOrIn(array $items, ): array
    {
        if (empty($items)) {
            return [];
        }
        // use no items constrint
        $noItem = false;
        $inItems = [];
        foreach ($items as $item) {
            if ((int)$item < 0) {
                $noItem = true;
            } else {
                $inItems [] = (int)$item;
            }
        }
        $subConstrains = [];
        if ($noItem) {
            $subConstrains[] = 'NOT EXISTS (SELECT 1 FROM ' . $this->mmPersonverbandTagTablename  . ' JOIN ' . $this->tagTablename . ' ON ' . $this->tagTablename . '.uid =uid_foreign WHERE uid_local=' . $this->contactverbandTablename . '.uid  AND tagverband=' . $noItem . ')';
        }
        if ($inItems) {
            $subConstrains[] = 'EXISTS (SELECT 1 FROM ' . $this->mmPersonverbandTagTablename  . ' WHERE uid_local=' . $this->contactverbandTablename . '.uid AND uid_foreign IN (' . implode(',', $inItems) . '))';
        }
        return $subConstrains;
    }
    */

    // return is null|CompositeExpression|string
    public function getConstrainsEmptyOrIn(string $attribute, array $items)
    {
        if (empty($items)) {
            return null;
        }
        // use no items constrint
        $noItem = false;
        $inItems = [];
        foreach ($items as $item) {
            if ((int)$item < 0) {
                $noItem = true;
            } else {
                $inItems [] = (int)$item;
            }
        }
        $subConstrains = [];
        if ($noItem) {
            $subConstrains[] = $this->expr()->isNull($attribute);
            $subConstrains[] = $this->expr()->eq($attribute, '0');
        }
        if ($inItems) {
            $subConstrains[] = $this->expr()->in($attribute, $inItems);
        }
        if (empty($subConstrains)) {
            return null;
        }
        if (count($subConstrains) == 1) {
            return $subConstrains[0];// string
        }
        return $this->expr()
            ->or(...$subConstrains);
    }
    public function getConstrainsMmIn(array $items, string $tablename, string $localAttribute = 'uid_local', string $foreignAttribute = 'uid_foreign')
    {
        if (empty($items)) {
            return null;
        }
        $notInItems = [];
        $inItems = [];
        foreach ($items as $itemUid) {
            if ((int)$itemUid < 0) {
                // if itemUid is negative, we want all entries with no tags in this tag group (tagUid = -tagGroupUid)
                $notInItems = abs((int)$itemUid);
            } else {
                $inItems[] = (int)$itemUid;
            }
        }
        $subItemConstrains = [];
        if (!empty($notInItems)) {
            $subItemConstrains[] = 'NOT EXISTS (SELECT 1 FROM ' . $this->quoteIdentifier($tablename) . ' WHERE ' . $this->quoteIdentifier($localAttribute) . '=' . $this->tablename . '.uid AND ' . $this->quoteIdentifier($foreignAttribute) . '  IN (' . implode(',', $notInItems) . '))';
        }
        if (!empty($inItems)) {
            $subItemConstrains[] = 'EXISTS (SELECT 1 FROM ' . $this->quoteIdentifier($tablename) . ' WHERE ' . $this->quoteIdentifier($localAttribute) . '=' . $this->tablename . '.uid AND ' . $this->quoteIdentifier($foreignAttribute) . '  IN (' . implode(',', $inItems) . '))';
            //return 'EXISTS (SELECT 1 FROM ' . $this->quoteIdentifier($tablename) . ' WHERE ' . $this->quoteIdentifier($localAttribute) . '=' . $this->tablename . '.uid AND ' . $this->quoteIdentifier($foreignAttribute) . '  IN (' . implode(',', $inItems) . '))';
        }
        if (empty($subItemConstrains)) {
            return null;
        }
        if (count($subItemConstrains) == 1) {
            return $subItemConstrains[0];// string
        }
        return $this->expr()->or(...$subItemConstrains);
    }


}
