<?php

namespace Internetgalerie\IgsCrm\Domain\Repository;

use DateTime;
use Internetgalerie\IgsCrm\Database\Query\ContactQueryBuilder;
use Internetgalerie\IgsCrm\Domain\Model\Person;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\LazyLoadingProxy;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;
use TYPO3\CMS\Extbase\Persistence\Generic\Storage\Typo3DbQueryParser;
use TYPO3\CMS\Extbase\Persistence\Generic\Typo3QuerySettings;
use TYPO3\CMS\Extbase\Persistence\ObjectStorage;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use TYPO3\CMS\Extbase\Reflection\ObjectAccess;
use TYPO3\CMS\Extbase\Reflection\ReflectionService;

/**
 * The repository for Persons
 */
class ContactRepository extends AbstractCrmRepository
{
    protected $tablename = 'tx_igscrm_domain_model_contact';
    protected $objectType = Contact::class;
    protected $type = null;
    protected $useAcl = true;
    
    protected $apiAttributes = [
        'uid',
        'me_lastname',
        'me_firstname',
        'me_addon',
        'me_address',
        'me_zip',
        'me_city',
        'phone',
        'me_email',
    ];

    protected $defaultOrderings = [
        'extraSorting' => QueryInterface::ORDER_ASCENDING,
        'meLastname' => QueryInterface::ORDER_ASCENDING,
        'meFirstname' => QueryInterface::ORDER_ASCENDING,
        'meCompanyname' => QueryInterface::ORDER_ASCENDING,
    ];

    /**
     * @var DataMapper
     **/
    protected $dataMapper;

    private $activeYear = null;
    private $activeDate = null;



    public function initializeObject(): void
    {
        $querySettings = GeneralUtility::makeInstance(Typo3QuerySettings::class);
        $querySettings->setRespectStoragePage(false);
        $this->setDefaultQuerySettings($querySettings);
    }
    public function injectDataMapper(DataMapper $dataMapper): void
    {
        $this->dataMapper = $dataMapper;
    }

    public function setStoragePids(array $storagePids): void
    {
        $this->initDefaultQuerySettings();
        $this->defaultQuerySettings->setStoragePageIds($storagePids);
    }
    /**
     * return a queryBuilder for contacts
     *
     * @return ContactQueryBuilder
     */
    public function createContactQueryBuilder()
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable($this->tablename);
        $queryBuilder = GeneralUtility::makeInstance(ContactQueryBuilder::class, $conn);
        $queryBuilder->injectSecurityUtility($this->securityUtility);
        $queryBuilder->injectConfUtility($this->confUtility);
        $queryBuilder->setDefaultOrderings($this->defaultOrderings);
        $queryBuilder->setActiveYear($this->activeYear);
        $queryBuilder->setUseAcl($this->useAcl);
        return $queryBuilder;
    }

    
    public function setActiveYear($activeYear = null): void
    {
        if (strpos((string) $activeYear, '-') > 0) {
            $this->activeDate = $activeYear;
            $activeYear = intval($activeYear);
        }
        $this->activeYear = $activeYear > 0 ? $activeYear : null;
    }
    public function getActiveYear()
    {
        return $this->activeYear;
    }
    public function getActiveStartNextYear()
    {
        return $this->activeYear > 0 ? intval($this->activeYear + 1) . '-01-01' : (1 + date('Y')) . '-01-01';
    }
    public function getActiveDateEnd()
    {
        return $this->activeYear > 0 ? intval($this->activeYear) . '-12-31' : date('Y') . '-12-31';
    }
    public function getActiveDateEndLastYear()
    {
        return $this->activeYear > 0 ? intval($this->activeYear - 1) . '-12-31' : date('Y') . '-12-31';
    }
    public function getActiveDateStart()
    {
        if ($this->activeDate !== null) {
            return $this->activeDate;
        }
        return $this->activeYear > 0 ? intval($this->activeYear) . '-01-01' : date('Y') . '-01-01';
    }
    public function getStartEndDate($dateMode)
    {
        $dates = [
            //'startDate' => null,
            //'endDate' => null,
        ];
        if ($dateMode == 1) {
            $dates['startDate'] = $this->getActiveDateEnd();
            $dates['endDate'] = $this->getActiveDateStart();
        } elseif ($dateMode == 'current') {
            $dates['startDate'] = date('Y-m-d');
            $dates['endDate'] = date('Y-m-d');
        } elseif ($dateMode == 'inYear') {
            $dates['startDate'] = $this->getActiveDateEnd();
            $dates['endDate'] = $this->getActiveDateStart();
        } elseif ($dateMode == 'startYear') {
            $dates['startDate'] = $this->getActiveDateEndLastYear();
            $dates['endDate'] = $this->getActiveDateStart();
        } elseif ($dateMode == 'endYear') {
            $dates['startDate'] = $this->getActiveDateEnd();
            $dates['endDate'] = $this->getActiveDateEnd();
        } elseif ($dateMode == 'nextYear') {
            $dates['startDate'] = $this->getActiveDateEnd();
            $dates['endDate'] = $this->getActiveStartNextYear();
        }
        return $dates;
    }
    
    public function sqlActive($dateMode = 'endYear')
    {
        $where = [];
        if ($this->activeYear === null) {
            $dates = [
                'startDate' => date('Y-m-d'), //'CURRENT_DATE';
                'endDate' => date('Y') . '-01-01',
            ];
            $where[] = 'me_todestag IS NULL';
        } else {
            $dates = $this->getStartEndDate($dateMode);
            //$sqlAndStartDate=" OR  start_date<=" . $sqlEndDate . "";
            $where[] = "(start_date<='" . $dates['startDate'] . "')"; // start_date IS NULL OR
            //$where[] = "(me_todestag IS NULL OR me_todestag>'" . intval($this->activeYear) . "-12-31')" ; // nicht im Auswertungsjahr gestorben
        }
        if ($this->type) {
            $andType = " AND limit_to_type='" . $this->type . "'";
            $where[] = "type='" . $this->type . "'";
        } else {
            $andType = '';
        }
        //echo("(tx_igscrm_domain_model_contact.deleted=0 AND tx_igscrm_domain_model_contactverband.deleted=0 AND active=1 AND start_date>'0000-00-00' " . $sqlAndStartDate . " AND  start_date<=" . $sqlEndDate . " AND (end_date>=" . $sqlEndDate . " OR end_date IS NULL OR end_date='0000-00-00') AND (tx_igscrm_domain_model_contactverband.mitgliedschaft=0 OR tx_igscrm_domain_model_contactverband.mitgliedschaft<>5))");exit(0);
        // active=1 AND
        return '(tx_igscrm_domain_model_contact.deleted=0 AND tx_igscrm_domain_model_contact.hidden=0 AND tx_igscrm_domain_model_contactverband.deleted=0 AND ' . implode(
            ' AND ',
            $where
        ) . " AND (end_date>='" . $dates['endDate'] . "' OR end_date IS NULL) AND (tx_igscrm_domain_model_contactverband.mitgliedschaft=0 OR tx_igscrm_domain_model_contactverband.mitgliedschaft IS NULL OR tx_igscrm_domain_model_contactverband.mitgliedschaft NOT IN (SELECT uid FROM tx_igscrm_domain_model_mitgliedschaft WHERE no_person=1 " . $andType . ')))';
        // tx_igscrm_domain_model_contactverband.mitgliedschaft<>5 JOIN und no_person=false
    }
    public function queryActiveYear($query, $jahr)
    {
        return $query->logicalAnd(
            //$query->equals('meContactVerband.active', 1),
            $query->logicalOr(
                $query->lessThanOrEqual('meContactVerband.startDate', $jahr . '-12-31'),
                $query->equals('meContactVerband.startDate', null),
                //$query->equals('meContactVerband.startDate', '0000-00-00')
            ),
            $query->logicalOr(
                $query->greaterThanOrEqual('meContactVerband.endDate', $jahr . '-01-01'),
                $query->equals('meContactVerband.endDate', null),
                //$query->equals('meContactVerband.endDate', '0000-00-00')
            )
        );
    }

    // Query fuer Mitglieder die im Jahr $jahr geboren wurden
    public function queryBornInYear($query, $jahr)
    {
        return $query->logicalOr(
            $query->logicalAnd(
                $query->greaterThanOrEqual('meDateOfBirth', $jahr . '-01-01'),
                $query->lessThanOrEqual('meDateOfBirth', $jahr . '-12-31')
            ),
            $query->logicalAnd($query->equals('meYearOfBirth', $jahr))
        );
    }
    // Query fuer Mitglieder die im oder vor den Jahr $jahr geboren wurden
    public function queryBornInOrBeforeYear($query, $jahr)
    {
        return $query->logicalOr(
            $query->logicalAnd(
                $query->greaterThanOrEqual('meDateOfBirth', '1900-01-01'),
                $query->lessThanOrEqual('meDateOfBirth', $jahr . '-12-31')
            ),
            $query->logicalAnd(
                $query->greaterThanOrEqual('meYearOfBirth', 1900),
                $query->lessThanOrEqual('meYearOfBirth', $jahr)
            )
        );
    }

    public function getUidByNumber(int $number): ?int
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($this->tablename);
        $res = $queryBuilder
             ->select('uid')
             ->from($this->tablename)
             ->where($queryBuilder->expr() ->eq('me_addressid', $number))
             ->executeQuery();
        if ($d = $res->fetchAssociative()) {
            return $d['uid'];
        }
        return null;
    }

    
    public function personRoleFindByVerbandRaw(int $tag, $verband = null, array $search = [], $orderBy = [])
    {
        if ($verband !== null) {
            $search['verband'] = is_object($verband) ? $verband->getUid() : (int)$verband;
        }
        $queryBuilder = $this->findBySearch($search, $orderBy);
        $queryBuilder->joinContactVerband();

        /*
          $queryBuilder->join(
          $this->tablename,
          'tx_igscrm_domain_model_contactverband',
          'tx_igscrm_domain_model_contactverband',
          'tx_igscrm_domain_model_contactverband.contact = ' . $this->tablename  . '.uid'
          );
        */
        $queryBuilder->join(
            'tx_igscrm_domain_model_contactverband',
            'tx_igscrm_contactverband_tag_mm',
            'contactverband_tag_mm',
            'contactverband_tag_mm.uid_local = tx_igscrm_domain_model_contactverband.uid AND contactverband_tag_mm.uid_foreign = ' . (int) $tag
        );
        $queryBuilder->leftJoin(
            'contactverband_tag_mm',
            'tx_igscrm_domain_model_contacttag',
            'contacttag_mm',
            'contactverband_tag_mm.uid_foreign = contacttag_mm.tag AND contacttag_mm.contact=' . $this->tablename . '.uid'
        );
        $queryBuilder->leftJoin(
            'contacttag_mm',
            'tx_igscrm_domain_model_tagrole',
            'tx_igscrm_domain_model_tagrole',
            'tx_igscrm_domain_model_tagrole.uid = contacttag_mm.tagrole'
        );
        $queryBuilder->add(
            'orderBy',
            'CASE WHEN tx_igscrm_domain_model_tagrole.sorting IS NOT NULL THEN tx_igscrm_domain_model_tagrole.sorting ELSE ~0 END,me_lastname, me_firstname'
        );
        //$queryBuilder->orderBy('tx_igscrm_domain_model_tagrole.sorting', 'ASC')->addOrderBy('tx_igscrm_domain_model_contact.me_lastname', 'ASC')->addOrderBy('me_firstname');

        //return $queryBuilder;
        return $queryBuilder->executeQuery()
->fetchAllAssociative();
    }

    public function personRoleFindByVerband(int $tag, $verband = null, array $search = [], $orderBy = [])
    {
        $rows = $this->personRoleFindByVerbandRaw($tag, $verband, $search, $orderBy);
        if (!empty($rows)) {
            $persons = $this->dataMapper->map(Person::class, $rows);
            foreach ($persons as &$person) {
                $contactTagRepository = GeneralUtility::makeInstance(ContactTagRepository::class);
                $contactTag = $contactTagRepository->findOneByContactTag($person, $tag);
                if (is_object($contactTag)) {
                    $person->setContactTag($contactTag);
                }
            }
            return $persons;
        }
        return false;
    }

    public function findByVerband($verband = null, array $search = [], $orderBy = [])
    {
        if ($verband !== null) {
            $search['verband'] = is_object($verband) ? $verband->getUid() : (int)$verband;
        }
        $queryBuilder = $this->findBySearch($search, $orderBy);
        $rows = $queryBuilder->executeQuery()
            ->fetchAllAssociative();
        return $this->dataMapper->map($this->objectType, $rows);
        //return $this->findBySearch($search, $orderBy);
    }

    public function findBySearch(array $search = [], array $orderBy = [])
    {
        $queryBuilder = $this->createContactQueryBuilder();
        $queryBuilder->addUserRestriction();
        $queryBuilder->select($this->tablename . '.*');
        $queryBuilder->from($this->tablename);
        $queryBuilder->addSearch($search);
        $queryBuilder->setActiveYear($this->activeYear);



        if (!empty($orderBy)) {
            $queryBuilder->addArrayOrderBy($orderBy);
        } else {
            if ($search['sorting'] ?? false) {
                $sorting = GeneralUtility::trimExplode(',', $search['sorting'], true);
                if (!empty($sorting)) {
                    foreach ($sorting as $sortingPart) {
                        [$sortingProperty, $sortingOrder] = GeneralUtility::trimExplode(' ', $sortingPart);
                        $sortingAttribute = GeneralUtility::camelCaseToLowerCaseUnderscored($sortingProperty);
                        $queryBuilder->orderBy($sortingAttribute, $sortingOrder == 'desc' ? 'DESC' : 'ASC');
                    }
                }
            }
        }
        return $queryBuilder;
    }
    public function findBySearchExecute(array $search = [], $orderBy = [])
    {
        $queryBuilder = $this->findBySearch($search, $orderBy);
        $rows = $queryBuilder->executeQuery()
->fetchAllAssociative();
        return $this->dataMapper->map($this->objectType, $rows);
    }
    public function findByVerbandMmGroupBy($verband, $group, $attribute = 'name')
    {
        $verbandUid = $verband->getUid();
        $tablename = 'tx_igscrm_domain_model_' . $group;
        $tableMm = 'tx_igscrm_contactverband_' . $group . '_mm';

        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );


        $sql = 'SELECT ' . $tablename . '.' . $attribute . ' AS x,sum(1) AS anzahl FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contact.uid=contact AND verband=' . $verbandUid . ' AND ' . $this->sqlActive() . ') JOIN ' . $tableMm . ' ON (uid_local=tx_igscrm_domain_model_contactverband.uid) JOIN ' . $tablename . ' ON (uid_foreign=' . $tablename . '.uid AND ' . $tablename . '.deleted=0 AND ' . $tablename . '.hidden=0) GROUP BY uid_foreign';
        //echo('<h4>'.$sql.'</h4>');
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = $res->fetchAllAssociative();
        return $rows;
    }


    public function findByVerbandTagsGroupBy($verband, $tagverbandUid)
    {
        $attribute = 'name';
        $verbandUid = $verband->getUid();
        $tablename = 'tx_igscrm_domain_model_tag';
        $tableMm = 'tx_igscrm_contactverband_tag_mm';

        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );


        $sql = 'SELECT ' . $tablename . '.' . $attribute . ' AS x,sum(1) AS anzahl FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contact.uid=contact AND verband=' . $verbandUid . ' AND ' . $this->sqlActive() . ') JOIN ' . $tableMm . ' ON (uid_local=tx_igscrm_domain_model_contactverband.uid) JOIN ' . $tablename . ' ON (uid_foreign=' . $tablename . '.uid AND ' . $tablename . '.deleted=0 AND ' . $tablename . '.hidden=0) JOIN tx_igscrm_domain_model_tagverband ON (tagverband=tx_igscrm_domain_model_tagverband.uid) WHERE tagverband= ' . intval(
            $tagverbandUid
        ) . ' GROUP BY uid_foreign';
        //echo('<h4>'.$sql.'</h4>');
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = $res->fetchAllAssociative();
        return $rows;
    }



    public function findByVerbandJoinPersonGroupBy($verband, $tablename, $joinAttribute, $attribute)
    {
        $verbandUid = $verband->getUid();

        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );


        $sql = 'SELECT ' . $tablename . '.' . $attribute . ' AS x,sum(1) AS anzahl FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contact.uid=contact AND verband=' . $verbandUid . ' AND ' . $this->sqlActive() . ') LEFT OUTER JOIN ' . $tablename . ' ON (' . $tablename . '.uid=tx_igscrm_domain_model_contact.' . $joinAttribute . ' AND ' . $tablename . '.deleted=0 AND ' . $tablename . '.hidden=0) GROUP BY ' . $tablename . '.' . $attribute;
        // echo('<h4>'.$sql.'</h4>');
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = $res->fetchAllAssociative();
        return $rows;
    }
    public function findByVerbandJoinContactVerbandGroupBy($verband, $tablename, $joinAttribute, $attribute)
    {
        $verbandUid = $verband->getUid();

        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );


        $sql = 'SELECT ' . $tablename . '.uid AS xuid,' . $tablename . '.' . $attribute . ' AS x,sum(1) AS anzahl FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contact.uid=contact AND verband=' . $verbandUid . ' AND ' . $this->sqlActive() . ') JOIN ' . $tablename . ' ON (' . $tablename . '.uid=tx_igscrm_domain_model_contactverband.' . $joinAttribute . ' AND ' . $tablename . '.deleted=0 AND ' . $tablename . '.hidden=0) GROUP BY ' . $tablename . '.' . $attribute . ',' . $tablename . '.uid';
        //echo('<h4>'.$sql.'</h4>');
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = $res->fetchAllAssociative();
        return $rows;
    }

    public function countActiveInYearByVerband($verband, $dateMode = 'endYear')
    {
        $verbandUid = $verband->getUid();

        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );
        //        $sqlAndWhere = $this->type ? " AND type='" . $this->type. "'" : '';
        //  . $sqlAndWhere
        //var_dump($this->type, $sqlAndWhere);exit(0);
        $sql = 'SELECT count(1) AS anzahl FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contact.uid=contact AND verband=' . $verbandUid . ' AND ' . $this->sqlActive(
            $dateMode
        ) . ')';// JOIN tx_igscrm_domain_model_mitgliedschaft ON (tx_igscrm_domain_model_mitgliedschaft.uid=tx_igscrm_domain_model_contactverband.mitgliedschaft AND tx_igscrm_domain_model_mitgliedschaft.deleted=0 AND tx_igscrm_domain_model_mitgliedschaft.hidden=0 AND tx_igscrm_domain_model_mitgliedschaft.no_person=0)';
        //if ($this->type=='Organisation') {            die($sql);        }

        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        //$row = $stmt->fetchAssociative();

        $count = $res->fetchOne();
        //var_dump($count);        die('<h4>'.$sql.'</h4>');
        return $count;
    }



    public function findByVerbandJoinPersonOrganisationVerbandGroupBy($verband, $attribute)
    {
        $verbandUid = $verband->getUid();

        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );


        $sql = 'SELECT ' . $attribute . ' AS x,sum(1) AS anzahl FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contact.uid=contact AND verband=' . $verbandUid . ' AND ' . $this->sqlActive() . ') JOIN tx_igscrm_domain_model_organisation ON (tx_igscrm_domain_model_organisation.uid=tx_igscrm_domain_model_contactverband.organisation_id AND tx_igscrm_domain_model_organisation.deleted=0 AND tx_igscrm_domain_model_organisation.hidden=0) JOIN tx_igscrm_domain_model_kanton ON (tx_igscrm_domain_model_kanton.uid=of_kanton_id) GROUP BY ' . $attribute;
        //echo('<h4>'.$sql.'</h4>');
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = $res->fetchAllAssociative();
        return $rows;
    }


    public function findByVerbandGroupBy($verband, $group)
    {
        $verbandUid = $verband->getUid();
        $attribute = $group;
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );


        $sql = 'SELECT ' . $attribute . ' AS x,sum(1) AS anzahl FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contact.uid=contact AND verband=' . $verbandUid . ' AND ' . $this->sqlActive() . ')  GROUP BY x ORDER BY x';
        //echo('<h4>'.$sql.'</h4>');
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = $res->fetchAllAssociative();
        return $rows;
    }


    public function findByVerbandAlterskategorieGroupBy($verband, int $mitgliedschaftUid = null)
    {
        $verbandUid = $verband->getUid();

        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );
        $andWhere = '';
        if ($mitgliedschaftUid) {
            $andWhere = ' AND tx_igscrm_domain_model_mitgliedschaft.uid = ' . $mitgliedschaftUid;
        }
        // get and prepare mitgliedschaften
        $sql = 'SELECT tx_igscrm_domain_model_mitgliedschaft.uid,name,alter_in_jahr_von,alter_in_jahr_bis,an_name,tx_igscrm_domain_model_anrede.uid as an_uid FROM tx_igscrm_domain_model_preise JOIN  tx_igscrm_domain_model_mitgliedschaft ON (mitgliedschaften=tx_igscrm_domain_model_mitgliedschaft.uid) LEFT OUTER JOIN tx_igscrm_domain_model_anrede ON tx_igscrm_domain_model_anrede.uid = anrede WHERE verband=' . $verbandUid . ' ' . $andWhere . ' ORDER BY alter_in_jahr_von,alter_in_jahr_bis;';
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $sqlMitgliedschaft = [];
        // min max geht nur mit einer Mitgliedschaft
        $min = []; // 0 => 200
        $max = [];// 0 => 0
        $maxJahre = 200;
        $mitgliedschaftenUid = [];
        while ($row = $res->fetchAssociative()) {
            $anredeUid = (int) $row['an_uid'];
            if ($row['alter_in_jahr_bis'] > 0 && $row['alter_in_jahr_bis'] < $maxJahre) {
                if (!isset($min[$anredeUid])) {
                    $min[$anredeUid] = $row['alter_in_jahr_bis'] + 1;
                } else {
                    $min[$anredeUid] = min($min[$anredeUid], $row['alter_in_jahr_bis'] + 1);
                }
            }
            if ($row['alter_in_jahr_von'] > 0 && $row['alter_in_jahr_von'] < $maxJahre) {
                if (!isset($min[$anredeUid])) {
                    $max[$anredeUid] = $row['alter_in_jahr_von'] - 1;
                } else {
                    $max[$anredeUid] = max($max[$anredeUid], $row['alter_in_jahr_von'] - 1);
                }
            }
            if ($anredeUid >= 1) {
                $anredeSql = ' AND me_anrede_id=' . intval($anredeUid);
                $anredeText = ', ' . $row['an_name'] . '';
            } else {
                $anredeSql = '';
                $anredeText = '';
            }
            $sqlMitgliedschaft[] = 'WHEN tx_igscrm_domain_model_contactverband.mitgliedschaft=' . $row['uid'] . $anredeSql . ' AND YEAR(CURRENT_DATE)-YEAR(me_date_of_birth) BETWEEN ' . intval(
                $row['alter_in_jahr_von']
            ) . ' AND ' . intval(
                $row['alter_in_jahr_bis']
            ) . " THEN '" . ($row['alter_in_jahr_von'] > 0 ? intval(
                $row['alter_in_jahr_von']
            ) : '') . ' bis ' . ($row['alter_in_jahr_bis'] < 200 ? intval(
                $row['alter_in_jahr_bis']
            ) : '') . ' (' . $row['name'] . $anredeText . ")'";
            $mitgliedschaftenUid[$row['uid']] = $row['name'];
        }
        if (count($sqlMitgliedschaft) == 0) {
            return [];
        }
        $minText = implode('/', $min);
        $maxText = implode('/', $max);
        foreach ($mitgliedschaftenUid as $uid => $name) {
            $sqlMitgliedschaft[] = 'WHEN tx_igscrm_domain_model_contactverband.mitgliedschaft=' . $uid . "  THEN '" . $minText . ' bis ' . $maxText . ' (' . $name . ")'";
        }
        //var_dump($sqlMitgliedschaft);
        //$sqlAlter='CASE WHEN me_date_of_birth IS NOT NULL THEN YEAR(me_date_of_birth) ELSE me_year_of_birth END';
        // '".($min+1)." bis ".($max-1)."' oder 'Kein Datum'
        //$sqlAlter="CASE WHEN me_date_of_birth IS NULL OR YEAR(me_date_of_birth)=0 THEN '".($min+1)." bis ".($max-1)."'
        $sqlAlter = 'CASE ' . implode(' ', $sqlMitgliedschaft) . "
ELSE '" . $minText . ' bis ' . $maxText . "'
END";

        $sql = 'SELECT ' . $sqlAlter . ' AS x,sum(1) AS anzahl FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contact.uid=contact AND verband=' . $verbandUid . ' AND ' . $this->sqlActive() . ') JOIN tx_igscrm_domain_model_mitgliedschaft ON (tx_igscrm_domain_model_mitgliedschaft.uid=tx_igscrm_domain_model_contactverband.mitgliedschaft) AND tx_igscrm_domain_model_contactverband.mitgliedschaft IN (' . implode(
            ',',
            array_keys($mitgliedschaftenUid)
        ) . ') GROUP BY ' . $sqlAlter . ' ORDER BY x';
        //echo($this->sqlActive() .'<br />');        echo('<h4>'.$sql.'</h4>');        exit(0);
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = $res->fetchAllAssociative();
        return $rows;
    }



    public function findEintritteByVerband($verband, $jahr)
    {
        $query = $this->createQuery();
        $query->matching(
            $query->logicalAnd(
                $query->equals('meContactVerband.verband', $verband),
                $query->equals('meContactVerband.active', 1),
                $query->logicalAnd(
                    $query->greaterThanOrEqual('meContactVerband.startDate', $jahr . '-01-01'),
                    $query->lessThanOrEqual('meContactVerband.startDate', $jahr . '-12-31')
                )
            )
        );
        $return = $query->execute();
        return $return;
    }
    public function findAustritteByVerband($verband, $jahr)
    {
        $query = $this->createQuery();
        $query->matching(
            $query->logicalAnd(
                $query->equals('meContactVerband.verband', $verband),
                $query->equals('meContactVerband.active', 0),
                $query->logicalAnd(
                    $query->greaterThanOrEqual('meContactVerband.endDate', $jahr . '-01-01'),
                    $query->lessThanOrEqual('meContactVerband.endDate', $jahr . '-12-31')
                ),
                $query->logicalOr(
                    $query->equals('meVerstorben', 0),
                    $query->logicalNot($query->equals('meTodestag', null))
                )
            )
        );
        $return = $query->execute();
        return $return;
    }

    public function findVerstorbenByVerband($verband, $jahr, $noPerson = 0)
    {
        $query = $this->createQuery();
        $query->matching(
            $query->logicalAnd(
                $query->equals('meContactVerband.verband', $verband),
                $query->equals('meContactVerband.mitgliedschaft.noPerson', $noPerson),
                $this->queryActiveYear($query, $jahr),
                $query->logicalAnd(
                    $query->greaterThanOrEqual('meTodestag', $jahr . '-01-01'),
                    $query->lessThanOrEqual('meTodestag', $jahr . '-12-31')
                )
            )
        );
        $orderBy = [
            'meLastname' => QueryInterface::ORDER_ASCENDING,
            'meFirstname' => QueryInterface::ORDER_ASCENDING,
        ];
        $query->setOrderings($orderBy);
        $return = $query->execute();
        return $return;
    }

    public function findVeteranenByVerband($verband, $jahr, $veteranenJahr)
    {
        $query = $this->createQuery();
        $query->matching(
            $query->logicalAnd(
                $query->equals('meContactVerband.verband', $verband),
                $this->queryActiveYear($query, $jahr),
                $this->queryBornInYear($query, $veteranenJahr)
            )
        );
        $orderBy = [
            'meLastname' => QueryInterface::ORDER_ASCENDING,
            'meFirstname' => QueryInterface::ORDER_ASCENDING,
        ];
        $query->setOrderings($orderBy);
        $return = $query->execute();
        return $return;
    }


    public function getMaxAddressNo(): int
    {
        $query = $this->createQuery();
        $query->matching($query->equals('type', $this->type));
        $lastEntry = $query->setOrderings([
            'meAddressid' => QueryInterface::ORDER_DESCENDING,
        ])->setLimit(1)->execute()->getFirst();
        return is_object($lastEntry) ? (int)$lastEntry->getMeAddressid() : 0;
    }
    public function findIgsOrganisation($organisationUidArray, $personUidsArray)
    {
        //var_dump(implode(',',$organisationUidArray), $personUidsArray);
        $query = $this->createQuery();
        $constrains = [];
        foreach ($organisationUidArray as $o) {
            $constrains[] = $query->equals('meContactVerband.organisationId', $o);
        }
        foreach ($personUidsArray as $m) {
            $constrains[] = $query->equals('uid', $m);
        }
        $query->matching($query->logicalAnd(
            $query->logicalOr(
                $query->greaterThanOrEqual('meContactVerband.endDate', $this->getActiveDateStart()),
                $query->equals('meContactVerband.endDate', null),
            ),
            $query->equals('meContactVerband.verband', 1),
            $query->logicalOr(...$constrains)
        ));
        //echo('meContactVerband.endDate' . $this->getActiveDateStart());


        return $query->execute();
    }
    /*
      public function findByAccount( $username ) {
      }*/



    public function findHistory($organisationUidArray, $personUidsArray)
    {
        $query = $this->createQuery();
        $constrains = [];
        $constrains[] = $query->equals('meNewsletterGeosummit', 1);
        $constrains[] = $query->greaterThan('tstamp', 0);
        //	  $constrains[] = $query->like('meEmail', '%@%');

        $query->matching($query->logicalAnd(...$constrains));
        return $query->setOrderings([
            'tstamp' => QueryInterface::ORDER_DESCENDING,
        ])->setLimit(100)->execute();
    }

    public function getExternEntry($meExternIdNamespace, $meExternId)
    {
        if (!$meExternId || !$meExternIdNamespace) {
            return null;
        }

        $query = $this->createQuery();
        $query->matching(
            $query->logicalAnd(
                $query->equals('meExternIdNamespace', $meExternIdNamespace),
                $query->equals('meExternId', $meExternId)
            )
        );
        $res = $query->execute();
        if ($res->count() > 1) {
            return false; // Fehler
        }
        if ($res->count() == 1) {
            return $res->getFirst();
        }
        return null;
    }


    public function findExportKurse($search, $selectFields)
    {
        $query = $this->createQuery();
        $constrains = [];
        if ($search['meIgsActive'] != '') {
            $constrains[] = $query->equals('meIgsActive', intval($search['meIgsActive']));
        }
        if ($search['meGeosuisseActive'] != '') {
            $constrains[] = $query->equals('meGeosuisseActive', intval($search['meGeosuisseActive']));
        }
        $constrains[] = $query->equals('meVerstorben', 0);

        if (!empty($constrains)) {
            $query->matching($query->logicalAnd(...$constrains));
        }
        $kursteilnehmerSql = '';
        if ($search['jahre']) {
            $kursteilnehmerSql = " AND kt_startdatum>='" . intval($search['jahre']) . "-01-01'";
        }

        $queryParser = GeneralUtility::makeInstance(Typo3DbQueryParser::class);
        $queryBuilder = $queryParser->convertQueryToDoctrineQueryBuilder($query);
        $queryBuilder->andWhere(
            'uid IN (SELECT kt_person FROM tx_igscrm_domain_model_kursteilnehmer where deleted=0' . $kursteilnehmerSql . ')'
        );
        $queryBuilder->select(...$selectFields);
        return $queryBuilder->executeQuery()
->fetchAllAssociative();

        //return $query->setOrderings(array('tstamp' => \TYPO3\CMS\Extbase\Persistence\QueryInterface::ORDER_DESCENDING))->executeQuery();
    }

    public function findUidsMeOkOnNew()
    {
        $sql = 'SELECT uid from tx_igscrm_domain_model_contact AS main where me_ok=0 AND not exists (select 1 FROM tx_igscrm_domain_model_contact AS old where me_ok=1 AND trim(main.me_lastname)=trim(old.me_lastname) AND trim(main.me_firstname)=trim(old.me_firstname) );';
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();

        $uids = [];
        while ($row = $res->fetchAssociative()) {
            $uids[] = $row['uid'];
        }
        return $uids;
    }
    public function findWithNameByMeOk($meOk)
    {
        $query = $this->createQuery();
        $searchConstrains = [];
        $searchConstrains[] = $query->equals('meOk', $meOk);
        $searchConstrains[] = $query->logicalNot($query->equals('meLastname', ''));
        $searchConstrains[] = $query->logicalNot($query->equals('meFirstname', ''));
        if (!empty($searchConstrains)) {
            $query->matching($query->logicalAnd(...$searchConstrains));
        }
        $return = $query->execute();
        return $return;
    }
    public function updateMeOkOnNew()
    {
        $uids = $this->findUidsMeOkOnNew();
        if (count($uids) > 0) {
            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
                'tx_igscrm_domain_model_contact'
            );
            $queryBuilder->update('tx_igscrm_domain_model_contact')
->where($queryBuilder->expr() ->in('uid', $uids))->set('me_ok', 1)
                         ->executeStatement();
        }
        return count($uids);
    }
    public function updateMeOkAll(): void
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            'tx_igscrm_domain_model_contact'
        );
        $queryBuilder->update('tx_igscrm_domain_model_contact')
->where($queryBuilder->expr() ->in('me_ok', 0))->set('me_ok', 1)
                     ->executeStatement();
    }

    public function findSimilar(Person $person)
    {
        $query = $this->createQuery();
        $searchConstrains = $this->getQueryConstraintsWithAcl($query);
        $searchConstrains[] = $query->equals('meOk', 1);
        $searchConstrains[] = $query->equals('meLastname', $person->getMeLastname());
        $searchConstrains[] = $query->equals('meFirstname', $person->getMeFirstname());
        $searchConstrains[] = $query->logicalOr(
            $query->equals('meZip', $person->getMeZip()),
            $query->equals('meCity', $person->getMeCity())
        );
        $query->matching($query->logicalAnd(...$searchConstrains));
        $query->setLimit(4);
        return $query->execute();

        //$sql="SELECT * FROM tx_igscrm_domain_model_contact WHERE me_ok=1 AND me_lastname=".$person->getMeLastname()." AND me_firstname=".$person->getMeFirstname()." AND (me_zip=".$person->getMeZip()." OR me_city=".$person->getMeCity()." OR me_email=".$person->getMeEmail().");";
    }


    public function findUidsWithDuplicateName()
    {
        $sql = "SELECT uid FROM tx_igscrm_domain_model_contact WHERE (me_lastname,me_firstname) IN (SELECT me_lastname,me_firstname FROM tx_igscrm_domain_model_contact WHERE deleted=0 AND me_lastname<>'' group by  me_lastname,me_firstname  having sum(1)>1 order by sum(1));";
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();

        $uids = [];
        while ($row = $res->fetchAssociative()) {
            $uids[] = $row['uid'];
        }
        return $uids;
    }


    public function findSimilarWeak(Person $person)
    {
        $query = $this->createQuery();
        $searchConstrains = $this->getQueryConstraintsWithAcl($query);
        $searchConstrains[] = $query->equals('meOk', 1);
        $searchConstrains[] = $query->equals('meLastname', $person->getMeLastname());
        $searchConstrains[] = $query->equals('meFirstname', $person->getMeFirstname());
        $query->matching($query->logicalAnd(...$searchConstrains));
        $query->setLimit(4);
        return $query->execute();

        //$sql="SELECT * FROM tx_igscrm_domain_model_contact WHERE me_ok=1 AND me_lastname=".$person->getMeLastname()." AND me_firstname=".$person->getMeFirstname()." AND (me_zip=".$person->getMeZip()." OR me_city=".$person->getMeCity()." OR me_email=".$person->getMeEmail().");";
    }

    public function mergeUids($mergeUids)
    {
        die('inactive... mergeUids not available in the moment.....');
        $changes = [];
        $reflectionService = GeneralUtility::makeInstance(ReflectionService::class);
        $objectAccess = GeneralUtility::makeInstance(ObjectAccess::class);
        $properties = $reflectionService->getClassPropertyNames(Person::class);
        $mergeProperties = [];
        foreach ($properties as $property) {
            if (str_starts_with($property, 'me')) {
                $mergeProperties[] = $property;
            }
        }

        foreach ($mergeUids as $newUid => $oldUid) {
            $html = '';
            if ($newUid <= 0 || $oldUid <= 0) {
                continue;
            }
            $old = $this->findByUid($oldUid);
            $new = $this->findByUid($newUid);
            $html .= '<h3>Bestehend: ' . $oldUid . ', Neu:' . $newUid . '</h3>';
            foreach ($mergeProperties as $attribute) {
                $func = 'get' . ucfirst($attribute);
                $newValue = $new->{$func}();
                $oldValue = $old->{$func}();
                $newClass = $newValue::class;
                $pr = $objectAccess->getProperty($new, $attribute);
                //$n=$reflectionService->getClassSchema($newValue);
                //echo('<br />' . $attribute.':'. $newClass );
                //if($newValue===null || $newValue=='' || ($newClass!=''  && $newClass!=\DateTime::class))
                if ($newValue === null || $newValue == '' || (str_ends_with($newClass, 'Storage'))) {
                    continue;
                }
                //echo(' - COPY');
                if ($newValue::class == DateTime::class) {
                    $newValue = $newValue->format('Y-m-d');
                }
                if ($oldValue::class == DateTime::class) {
                    $oldValue = $oldValue->format('Y-m-d');
                }
                if (!$old->{$func}()) {
                    if ($newValue instanceof LazyLoadingProxy) {
                        $newValue = $newValue->_loadRealInstance();
                    }
                    $html .= '+ ' . $attribute . ':' . $newValue . '<br />';
                    $old->{ 'set' . ucfirst($attribute) }($newValue);
                } else {
                    if ($oldValue != $newValue) {
                        $html .= '- ' . $attribute . ':' . $oldValue . ($newValue ? ' (Neu wäre: ' . $newValue . ')' : '') . '<br />';
                    }
                }
            }
            //exit(0);
            // Telefon
            if (count($new->getMePhoneNumberId()) > 0 && count($old->getMePhoneNumberId()) == 0) {
                foreach ($new->getMePhoneNumberId() as $e) {
                    $html .= '+ RELATION Phone:' . $e->getPnPhonenumber() . '<br />';
                    $old->addMePhoneNumberId($e);
                }
                $new->setMePhoneNumberId(new ObjectStorage());
            }
            // E-Mail
            if (count($new->getMeEmailaddressId()) > 0 && count($old->getMeEmailaddressId()) == 0) {
                foreach ($new->getMeEmailaddressId() as $e) {
                    $html .= '+ RELATION E-Mail:' . $e->getEaEmailaddress() . '<br />';
                    $old->addMeEmailaddressId($e);
                }
                // Preven the new entries from deletion
                $new->setMePhoneNumberId(new ObjectStorage());
            }
            $this->update($old);
            $this->remove($new);
            $changes[] = $html;
        }
        return $changes;
    }


    public function findAjax($keyword, $offset = 0, $limit = 250)
    {
        $query = $this->createQuery();

        $constrainsKeyword = [];
        
        $constrainsKeyword[] = $query->like('meCompanyname', '%' . $keyword . '%');
        $constrainsKeyword[] = $query->like('meLastname', '%' . $keyword . '%');
        $constrainsKeyword[] = $query->like('meFirstname', '%' . $keyword . '%');
        $constrainsKeyword[] = $query->like('meCity', '%' . $keyword . '%');

        if ((int) $keyword > 0) {
            $constrainsKeyword[] = $query->equals('meAddressid', intval($keyword));
            $constrainsKeyword[] = $query->equals('meZip', intval($keyword));
        }
        $searchConstrains = $this->getQueryConstraintsWithAcl($query);
        $searchConstrains[] = $query->logicalNot(
            $query->logicalAnd(
                $query->equals('meCompanyname', ''),
                $query->equals('meLastname', ''),
                $query->equals('meFirstname', ''),
                $query->equals('meCity', '')
            )
        );
        if ($this->type) {
            $searchConstrains[] = $query->equals('type', $this->type);
        }
        $searchConstrains[] = $query->logicalOr(...$constrainsKeyword);
        $query->matching($query->logicalAnd(...$searchConstrains));
        $query->setOrderings($this->defaultOrderings);


        $queryParser = GeneralUtility::makeInstance(Typo3DbQueryParser::class);
        $queryBuilder = $queryParser->convertQueryToDoctrineQueryBuilder($query);
        $queryBuilder->select(
            'uid',
            'me_lastname',
            'me_firstname',
            'me_zip',
            'me_city',
            'me_companyname',
            'me_addressid',
            'type'
        );
        $queryBuilder->setMaxResults($limit)
->setFirstResult($offset);
        return $queryBuilder->executeQuery()
->fetchAllAssociative();
        return $query->executeQuery();
    }
    
    public function findBugsByVerband($verband, $settings = [])
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );

        $andType = $this->type ? " AND type='" . $this->type . "'" : '';
        $sqlSelectBase = 'tx_igscrm_domain_model_contact.uid,active,start_date,end_date,me_todestag,me_firstname,me_lastname,me_companyname,type';
        $sqlOrderBy = 'type,me_lastname,me_firstname,me_companyname';
        
        // Fehlende Austrittsdatum
        $sql = 'SELECT ' . $sqlSelectBase . ",'Austrittsdatum fehlt' AS errormessage from tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contactverband.contact=tx_igscrm_domain_model_contact.uid AND tx_igscrm_domain_model_contactverband.deleted=0 AND tx_igscrm_domain_model_contact.deleted=0) WHERE tx_igscrm_domain_model_contactverband.active=0 AND end_date is null AND start_date IS NOT NULL AND verband=" . intval(
            $verband
        ) . $andType . ' ORDER BY ' . $sqlOrderBy;
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = $res->fetchAllAssociative();


        // ist noch aktiv
        $sql = 'SELECT ' . $sqlSelectBase . ",'ist noch aktiv' AS errormessage from tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contactverband.contact=tx_igscrm_domain_model_contact.uid AND tx_igscrm_domain_model_contactverband.deleted=0 AND tx_igscrm_domain_model_contact.deleted=0) WHERE tx_igscrm_domain_model_contactverband.active=1 AND ((end_date IS NOT NULL AND end_date<'" . date(
            'Y'
        ) . "-01-01') OR me_verstorben=1 OR me_todestag IS NOT NULL) AND verband=" . intval(
            $verband
        ) . $andType . ' ORDER BY ' . $sqlOrderBy;
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = array_merge($rows, $res->fetchAllAssociative());


        // Fehlende Eintrittssdatum
        $sql = 'SELECT ' . $sqlSelectBase . ",'Eintrittsdatum fehlt' AS errormessage from tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contactverband.contact=tx_igscrm_domain_model_contact.uid AND tx_igscrm_domain_model_contactverband.deleted=0 AND tx_igscrm_domain_model_contact.deleted=0) WHERE tx_igscrm_domain_model_contactverband.active=1 AND (start_date is null OR start_date IS NULL)  AND verband=" . intval(
            $verband
        ) . $andType . ' ORDER BY ' . $sqlOrderBy;
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = array_merge($rows, $res->fetchAllAssociative());

        // bei welchen typen sind Mitgliedschaft verwendet
        $sql = 'SELECT limit_to_type FROM tx_igscrm_domain_model_mitgliedschaft WHERE verband=' . intval(
            $verband
        ) . $andType . ' AND deleted=0 AND hidden=0 GROUP BY limit_to_type';
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $mitgliedschaften = [];
        while ($mitgliedschaft = $res->fetchOne()) {
            $mitgliedschaften[] = $mitgliedschaft;
        }
        if (empty($mitgliedschaften)) {
            $mitgliedschaftAndType = '';
        } else {
            $mitgliedschaftAndType = " AND type IN ('" . implode("', '", $mitgliedschaften) . "')";
        }

        // Mitgliedschaft fehlt
        $sql = 'SELECT ' . $sqlSelectBase . ",'Mitgliedschaft fehlt' AS errormessage from tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contactverband.contact=tx_igscrm_domain_model_contact.uid AND tx_igscrm_domain_model_contactverband.deleted=0 AND tx_igscrm_domain_model_contact.deleted=0) WHERE tx_igscrm_domain_model_contactverband.active=1 AND (mitgliedschaft=0 OR mitgliedschaft IS NULL) AND start_date IS NOT NULL AND verband=" . intval(
            $verband
        ) . $mitgliedschaftAndType . ' ORDER BY ' . $sqlOrderBy;
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = array_merge($rows, $res->fetchAllAssociative());


        // Todestag fehlt
        $sql = 'SELECT ' . $sqlSelectBase . ",'Todesdatum/Verstorben/Firma Aufgelöst fehlt' AS errormessage from tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contactverband.contact=tx_igscrm_domain_model_contact.uid AND tx_igscrm_domain_model_contactverband.deleted=0 AND tx_igscrm_domain_model_contact.deleted=0) WHERE tx_igscrm_domain_model_contactverband.active=0 AND ( (me_verstorben=1 AND (me_todestag IS null OR me_todestag IS NULL)) OR (me_verstorben=0 AND me_todestag IS NOT NULL)) AND start_date IS NOT NULL AND verband=" . intval(
            $verband
        ) . $andType . ' ORDER BY ' . $sqlOrderBy;
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = array_merge($rows, $res->fetchAllAssociative());


        // Todestag falsch
        $sql = 'SELECT ' . $sqlSelectBase . ",'Fehler: Todesjahr kleiner Austrittsdatum' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON (tx_igscrm_domain_model_contactverband.contact=tx_igscrm_domain_model_contact.uid AND tx_igscrm_domain_model_contactverband.deleted=0 AND tx_igscrm_domain_model_contact.deleted=0) WHERE tx_igscrm_domain_model_contactverband.active=0 AND (me_verstorben=1 AND YEAR(me_todestag)<YEAR(end_date) AND end_date IS NOT NULL) AND start_date IS NOT NULL AND verband=" . intval(
            $verband
        ) . $andType . ' ORDER BY ' . $sqlOrderBy;
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        $rows = array_merge($rows, $res->fetchAllAssociative());


        if ($settings['showPersonFamily'] || $settings['showPersonInvoice']) {
            $sqls = [
                // Sammelrechnung mit inaktiver Hauptadresse
                'SELECT ' . $sqlSelectBase . ",'Fehler: Sammelrechnung mit inaktiver Hauptadresse' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid = contact WHERE active AND me_invoice_to IN (SELECT p.uid FROM  tx_igscrm_domain_model_contact p JOIN tx_igscrm_domain_model_contactverband ON p.uid = contact WHERE active=FALSE) AND verband=" . intval(
                    $verband
                ) . $andType . ' ORDER BY ' . $sqlOrderBy,
                // Sammelrechnung mit Inaktive Familienadresse
                'SELECT ' . $sqlSelectBase . ",'Fehler: Sammelrechnung mit Inaktive Familienadresse' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid = contact WHERE active AND me_family IN (SELECT p.uid FROM  tx_igscrm_domain_model_contact p JOIN tx_igscrm_domain_model_contactverband ON p.uid = contact WHERE active=FALSE) AND verband=" . intval(
                    $verband
                ) . $andType . ' ORDER BY ' . $sqlOrderBy,
                // Rechnungs- bzw. Familienadresse ist eigene Adresse
                'SELECT ' . $sqlSelectBase . ",'Fehler: Rechnungs- bzw. Familienadresse ist eigene Adresse' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid = contact WHERE active AND (tx_igscrm_domain_model_contact.uid=me_invoice_to OR tx_igscrm_domain_model_contact.uid = me_family) AND verband=" . intval(
                    $verband
                ) . $andType . ' ORDER BY ' . $sqlOrderBy,
                // Familie nicht angewählt aber keine Familie
                'SELECT ' . $sqlSelectBase . ",'Fehler: Familie nicht angewählt aber keine Familie' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid = contact WHERE active AND me_is_family=FALSE AND (me_family<>0 OR EXISTS (SELECT 1 FROM tx_igscrm_domain_model_contact AS p WHERE tx_igscrm_domain_model_contact.uid=p.me_family)) AND verband=" . intval(
                    $verband
                ) . $andType . ' ORDER BY ' . $sqlOrderBy,
                // Familie und Sammelrechnung (2021 nur: 6375=Walther-Gipp)
                'SELECT ' . $sqlSelectBase . ",'Fehler: Familie und Sammelrechnung' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid = contact WHERE active AND (me_invoice_to<>0 OR me_is_sammelinvoice) AND (me_family<>0 OR me_is_family) AND verband=" . intval(
                    $verband
                ) . $andType . ' ORDER BY ' . $sqlOrderBy,
                // Familie aber nicht TUTTI (Rosette Rohrbach Gyger (5226))
                'SELECT ' . $sqlSelectBase . ",'Fehler: Familie aber nicht TUTTI' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid = contact WHERE active AND me_is_family AND mitgliedschaft<>9  AND verband=" . intval(
                    $verband
                ) . $andType . ' ORDER BY ' . $sqlOrderBy,
                // Keine Sammelrechnung aber Hauptadresse (Rechnung an) eingetragen
                'SELECT ' . $sqlSelectBase . ",'Fehler: Keine Sammelrechnung aber Hauptadresse (Rechnung an) eingetragen' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid = contact WHERE active AND (me_invoice_to<>0 AND me_is_sammelinvoice=FALSE) AND verband=" . intval(
                    $verband
                ) . $andType . ' ORDER BY ' . $sqlOrderBy,
                // Keine Sammelrechnung, aber Mitglieder haben diese Adresse als Hauptadresse definiert
                'SELECT ' . $sqlSelectBase . ",'Fehler: Keine Sammelrechnung, aber Mitglieder haben diese Adresse als Hauptadresse definiert' AS errormessage FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid = contact WHERE active AND me_is_sammelinvoice=FALSE AND (me_invoice_to<>0 OR EXISTS (SELECT 1 FROM tx_igscrm_domain_model_contact AS persub WHERE tx_igscrm_domain_model_contact.uid=persub.me_invoice_to)) AND verband=" . intval(
                    $verband
                ) . $andType . ' ORDER BY ' . $sqlOrderBy,

            ];
            foreach ($sqls as $sql) {
                $stmt = $conn->prepare($sql);
                $res = $stmt->executeQuery();
                $rows = array_merge($rows, $res->fetchAllAssociative());
            }
        }
            
        return $rows;
    }


    public function findSimilarByString($meLastname, $meFirstname, $meCompanyname, $meZip, $meCity, $meAddress, $type)
    {
        if (
            ($type == 'Organisation' && !$meCompanyname)
            || ($type != 'Organisation' && (!$meLastname || !$meFirstname))
        ) {
            return false;
        }
        $query = $this->createQuery();
        $searchConstrains = $this->getQueryConstraintsWithAcl($query);
        $searchConstrains[] = $query->equals('type', $type);
        if ($type == 'Organisation') {
            $searchConstrains[] = $query->equals('meCompanyname', $meCompanyname);
        } else {
            $searchConstrains[] = $query->equals('meLastname', $meLastname);
            $searchConstrains[] = $query->equals('meFirstname', $meFirstname);
        }
        $searchConstrains[] = $query->equals('meAddress', $meAddress);
        $searchConstrains[] = $query->logicalOr($query->equals('meZip', $meZip), $query->equals('meCity', $meCity));
        $query->matching($query->logicalAnd(...$searchConstrains));
        $query->setLimit(1);
        //echo("find: $meLastname $meFirstname $meZip $meCity mit $meEmail<br />");
        return $query->execute();

        //$sql="SELECT * FROM tx_igscrm_domain_model_contact WHERE me_ok=1 AND me_lastname=".$person->getMeLastname()." AND me_firstname=".$person->getMeFirstname()." AND (me_zip=".$person->getMeZip()." OR me_city=".$person->getMeCity()." OR me_email=".$person->getMeEmail().");";
    }


    public function findByMailAndName(
        ?string $meLastname,
        ?string $meFirstname,
        ?string $meCompanyname,
        string $meEmail,
        string $type
    ) {
        if (
            ($type == 'Organisation' && !$meCompanyname)
            || ($type != 'Organisation' && (!$meLastname || !$meFirstname))
        ) {
            return false;
        }
        $query = $this->createQuery();
        $searchConstrains = $this->getQueryConstraintsWithAcl($query);
        $searchConstrains[] = $query->equals('meEmail', $meEmail);
        if ($type == 'Organisation') {
            $searchConstrains[] = $query->equals('meCompanyname', $meCompanyname);
        } else {
            $searchConstrains[] = $query->equals('meLastname', $meLastname);
            $searchConstrains[] = $query->equals('meFirstname', $meFirstname);
        }
        $searchConstrains[] = $query->equals('type', $type);
        $query->matching($query->logicalAnd(...$searchConstrains));
        $query->setLimit(1);
        return $query->execute();

        //$sql="SELECT * FROM tx_igscrm_domain_model_contact WHERE me_ok=1 AND me_lastname=".$person->getMeLastname()." AND me_firstname=".$person->getMeFirstname()." AND (me_zip=".$person->getMeZip()." OR me_city=".$person->getMeCity()." OR me_email=".$person->getMeEmail().");";
    }
    public function findByMailType(string $email, string $type)
    {
        $query = $this->createQuery();
        $searchConstrains = $this->getQueryConstraintsWithAcl($query);
        $searchConstrains[] = $query->equals('meEmail', $email);
        $searchConstrains[] = $query->equals('type', $type);
        $query->matching($query->logicalAnd(...$searchConstrains));
        $query->setLimit(1);
        return $query->execute();
    }
    public function findFamilyMembers($meFamily, $uid)
    {
        $query = $this->createQuery();
        $query->matching($query->logicalOr($query->equals('meFamily', $uid), $query->equals('uid', $meFamily)));
        return $query->execute();
    }
    public function findInvoiceToMembers($meInvoiceTo, $uid)
    {
        $query = $this->createQuery();
        $query->matching(
            $query->logicalOr($query->equals('meInvoiceTo', $uid), $query->equals('uid', $meInvoiceTo))
        );
        return $query->execute();
    }
    public function findBySearchDoctrine($search)
    {
        // Create a Query Object
        $search['type'] = $this->type;
        $queryBuilder = $this->findBySearch($search);

        $constraints = [];

        $select = $this->apiAttributes;


        $alias = 'tx_igscrm_domain_model_contact';
        $prefixedSelect = [];
        foreach ($select as $value) {
            $prefixedSelect[] = $alias . '.' . $value;
        }
        $select = $prefixedSelect;
        $queryBuilder->select(...$select);//->from('tx_igscrm_domain_model_contact', 'o');
        $rows = $queryBuilder->executeQuery()
                             ->fetchAllAssociative();

        return $rows;
    }
    public function findForAuth($login)
    {
        if (empty($login['username'] ?? '') || empty($login['password'] ?? '')) {
            return false;
        }

        // Create a Query Object
        $login['type'] = $this->type;
        $query = $this->createQuery();
        $loginConstrains = [];
        if ($login['type']) {
            $loginConstrains[] = $query->equals('type', $login['type']);
        }
        if (isset($login['verband'])) {
            $verband = (int)$login['verband'];
            $loginConstrains[] = $query->equals('meContactVerband.verband', (int)$login['verband']);
            $loginConstrains[] = $query->equals('meContactVerband.active', 1);
            $loginConstrains[] = $this->queryActiveYear($query, date('Y'));
        } else {
            $verband = 0;
        }
        $loginConstrains[] = $query->equals('me_account', $login['username']);
        $loginConstrains[] = $query->equals('me_password', $login['password']);
        $query->matching($query->logicalAnd(...$loginConstrains));
        $res = $query->execute(true);

        if (!isset($res[0]['uid'])) {
            return false;
        }
        $contact = $res[0];

        $conf = [
            'uid' => 'uid',
            'type' => 'type',
            'status' => 'me_status',
            'username' => 'me_account',
            'lastname' => 'me_lastname',
            'firstname' => 'me_firstname',
            'salutation' => 'me_briefanrede',
            'title' => 'me_titel',
            'addressid' => 'me_addressid',
            'shortname' => 'me_shortname',
            'addon' => 'me_addon',
            'address' => 'me_address',
            'address2' => 'me_address2',
            'address3' => 'me_address3',
            'pobox' => 'me_pobox',
            'zip' => 'me_zip',
            'city' => 'me_city',
            'country' => 'me_country_abrev',
            'email' => 'me_email',
            'phone' => 'phone',
            'fax' => 'fax',
            'url' => 'me_url',
            'dateOfBirth' => 'me_date_of_birth',
            'languageid' => 'me_languageid',
            'fallbackLanguage' => 'fallback_language',
            'anredeId' => 'me_anrede_id',
            'kantonId' => 'me_kanton_id',
            'companyname' => 'me_companyname',
            'bfsUid' => 'bfs_uid',
        ];
        $data = [];
        foreach ($conf as $name => $attribute) {
            $data[$name] = $contact[$attribute];
        }

        $groups = [];

        
        // contactverband
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            'tx_igscrm_domain_model_contactverband'
        );
        $result = $queryBuilder->select('uid')
->from('tx_igscrm_domain_model_contactverband')
->where('deleted=0 AND contact=' . (int)$contact['uid'] . ' AND verband=' . (int)$verband)->executeQuery();
        $contactVerbandUid = $result->fetchOne();

        // Tag Kategorien
        //$sql = 'SELECT uid,name FROM tx_igscrm_domain_model_tagverband WHERE verband=2 AND deleted=0';
        // tags
        //$sql = 'SELECT uid,tagverband,name,usergroup FROM tx_igscrm_domain_model_tag WHERE deleted=0 AND uid IN (SELECT uid_foreign FROM tx_igscrm_contactverband_tag_mm WHERE uid_local =525)';

        // tags inkl. Kategorien
        $sql = 'SELECT tx_igscrm_domain_model_tag.uid,tx_igscrm_domain_model_tag.name,usergroup,tagverband AS tagverbandUid,tx_igscrm_domain_model_tagverband.name AS tagverbandName FROM tx_igscrm_domain_model_tag JOIN tx_igscrm_domain_model_tagverband ON tagverband=tx_igscrm_domain_model_tagverband.uid AND tx_igscrm_domain_model_tagverband.deleted=0 WHERE tx_igscrm_domain_model_tag.deleted=0 AND tx_igscrm_domain_model_tag.uid IN (SELECT uid_foreign FROM tx_igscrm_contactverband_tag_mm WHERE uid_local = ' . (int) $contactVerbandUid . ') ORDER BY tagverbandUid,tx_igscrm_domain_model_tag.uid';
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_tag'
        );
        $stmt = $conn->executeQuery($sql);
        while ($row = $stmt->fetchAssociative()) {
            $groups[] = $row;
        }
        $data['groups'] = $groups;
        return $data;
    }

    public function findOneForPasswordRecovery($emailOrUsername)
    {
        // Create a Query Object
        $query = $this->createQuery();
        $loginConstrains = [];
        $loginConstrains[] = $query->logicalOr(
            $query->equals('me_account', $emailOrUsername),
            $query->equals('me_email', $emailOrUsername)
        );
        $query->matching($query->logicalAnd(...$loginConstrains));
        $res = $query->execute(true);

        if (!isset($res[0]['uid'])) {
            return false;
        }
        $contact = $res[0];

        $conf = [
            'uid' => 'uid',
            'type' => 'type',
            'status' => 'me_status',
            'username' => 'me_account',
            'lastname' => 'me_lastname',
            'firstname' => 'me_firstname',
            'salutation' => 'me_briefanrede',
            'title' => 'me_titel',
            'addressid' => 'me_addressid',
            'shortname' => 'me_shortname',
            'addon' => 'me_addon',
            'address' => 'me_address',
            'address2' => 'me_address2',
            'address3' => 'me_address3',
            'pobox' => 'me_pobox',
            'zip' => 'me_zip',
            'city' => 'me_city',
            'country' => 'me_country_abrev',
            'email' => 'me_email',
            'phone' => 'phone',
            'fax' => 'fax',
            'url' => 'me_url',
            'dateOfBirth' => 'me_date_of_birth',
            'languageid' => 'me_languageid',
            'fallbackLanguage' => 'fallback_language',
            'anredeId' => 'me_anrede_id',
            'kantonId' => 'me_kanton_id',
            'companyname' => 'me_companyname',
            'bfsUid' => 'bfs_uid',
        ];
        $data = [];
        foreach ($conf as $name => $attribute) {
            $data[$name] = $contact[$attribute];
        }
        return $data;
    }
    public function getDataList(int $verbandUid, int $listId)
    {
        // first line must be the uid, second type, third is title for link, all other columns are shown behind
        $lists = [
            [
                'name' => 'Aktive Organisationen (nur Hauptsitze) mit Anzahl zugehörigen Personen beim Verband',
                'sql' => 'SELECT tx_igscrm_domain_model_contact.uid,type, me_companyname,me_zip, me_city, (SELECT count(1) FROM tx_igscrm_domain_model_contactverband AS pv WHERE pv.deleted=0 AND (organisation_id=tx_igscrm_domain_model_contact.uid OR EXISTS (SELECT 1 FROM tx_igscrm_contactverband_organisation_mm WHERE uid_local=pv.uid AND uid_foreign=tx_igscrm_domain_model_contact.uid )) AND active AND (end_date IS NULL  OR end_date=0 OR end_date>CURRENT_TIMESTAMP) AND verband=' . $verbandUid . ") AS anzahl_personen FROM tx_igscrm_domain_model_contact JOIN tx_igscrm_domain_model_contactverband ON tx_igscrm_domain_model_contact.uid=contact  WHERE tx_igscrm_domain_model_contact.deleted=0 AND tx_igscrm_domain_model_contactverband.deleted=0 AND tx_igscrm_domain_model_contact.hidden=0 AND type='Organisation' AND active =true AND (end_date IS NULL OR end_date=0 OR end_date>CURRENT_TIMESTAMP) AND verband=" . $verbandUid . ' AND EXISTS (SELECT 1 FROM tx_igscrm_organisation_categoryorganisation_mm WHERE uid_local=tx_igscrm_domain_model_contact.uid AND uid_foreign=1) ORDER BY anzahl_personen DESC;',
                // EXISTS (SELECT 1 FROM tx_igscrm_organisation_categoryorganisation_mm WHERE uid_local=tx_igscrm_domain_model_contact.uid AND uid_foreign=1)
                // (parent=0 OR parent IS NULL)
                'headers' => ['Name', 'PLZ', 'Ort', 'Anzahl Personen'],
            ],
        ];
        $list = $lists[$listId];
        $sql = $list['sql'];
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(
            'tx_igscrm_domain_model_contact'
        );
        $stmt = $conn->executeQuery($sql);
        $rows = [];
        while ($row = $stmt->fetchAssociative()) {
            $rows[] = $row;
        }
        $list['rows'] = $rows;
        return $list;
    }


    public function findHiddenByUid(int $uid)
    {
        $query = $this->createQuery();
        $querySettings = $query->getQuerySettings();
        $querySettings->setIgnoreEnableFields(true);
        $querySettings->setEnableFieldsToBeIgnored(['disabled']);
        $query->matching($query->equals('uid', (int)$uid));
        return $query->execute()
->getFirst();
    }

    public function countHidden(array $search = [])
    {
        $query = $this->createQuery();
        $querySettings = $query->getQuerySettings();
        $querySettings->setIgnoreEnableFields(true);
        $querySettings->setEnableFieldsToBeIgnored(['disabled']);
        //$querySettings->setEnableFieldsToBeIgnored(['hidden']);
        $searchConstrains = $this->getQueryConstraintsWithAcl($query);

        $searchConstrains[] = $query->equals('hidden', true);
        if (!empty($searchConstrains)) {
            $query->matching($query->logicalAnd(...$searchConstrains));
        }
        $stmt = $query->execute();
        return $stmt->count();
    }

    public function findOneByFrontendUserUsername($username)
    {
        $query = $this->createQuery();
        
        $query->matching($query->equals('frontendUser.username', $username));

        return $query->execute()
->getFirst();
    }

    public function findByFrontendUser(int $frontendUserUid, bool $returnRawQueryResult = false)
    {
        $query = $this->createQuery();
        
        $query->matching($query->equals('frontendUser', $frontendUserUid));

        return $query->execute($returnRawQueryResult);
    }
    public function findByNameAndAddress(
        string $name,
        string $address,
        string $zip,
        string $city,
        ?string $country = null,
        int $tenantId = null
    ) {
        $nameParts = explode(' ', $name, 2);
        $lastname = $nameParts[0] ?? '';
        $firstname = $nameParts[1] ?? '';

        $queryBuilder = $this->createContactQueryBuilder();
        $queryBuilder->addUserRestriction();
        $queryBuilder->select($this->tablename . '.*');
        $queryBuilder->from($this->tablename);
        $queryBuilder->joinContactVerband();
        $constrains = [];
        $constrains[] = $queryBuilder->expr()->eq($queryBuilder->getContactverbandTablename() . '.verband', $tenantId);
        $constrains[] = $queryBuilder->expr()->or(
            $queryBuilder->expr()->eq('me_address', $queryBuilder->createNamedParameter($address)),
            $queryBuilder->expr()->eq('me_address', $queryBuilder->createNamedParameter(
                str_replace('strasse', 'str.', $address)
            ))
        );
        $constrains[] = $queryBuilder->expr()->or(
            $queryBuilder->expr()->eq('me_zip', $queryBuilder->createNamedParameter($zip)),
            $queryBuilder->expr()->eq('me_city', $queryBuilder->createNamedParameter($city))
        );
        $constrains[] = $queryBuilder->expr()->or($queryBuilder->expr()->eq(
            'me_companyname',
            $queryBuilder->createNamedParameter($name)
        ), $queryBuilder->expr()->or(
            $queryBuilder->expr()
->and(
    $queryBuilder->expr()->eq('me_lastname', $queryBuilder->createNamedParameter($lastname)),
    $queryBuilder->expr()->eq('me_firstname', $queryBuilder->createNamedParameter($firstname))
),
            // firstname/lastname maybe switched
            $queryBuilder->expr()
->and(
    $queryBuilder->expr()->eq('me_firstname', $queryBuilder->createNamedParameter($lastname)),
    $queryBuilder->expr()->eq('me_lastname', $queryBuilder->createNamedParameter($firstname))
)
        ));
        $queryBuilder->where(...$constrains);
        $rows = $queryBuilder->executeQuery()
->fetchAllAssociative();
        $objects = $this->dataMapper->map($this->objectType, $rows);
        return $objects[0] ?? null;
        /*
        $query = $this->createQuery();
        $searchConstrains = $this->getQueryConstraintsWithAcl($query);
        //$searchConstrains[] = $query->equals('meOk', 1);
        $searchConstrains[] = $query->logicalOr(
            $query->equals('meCompanyname', $name),
            $query->logicalAnd(
                $query->equals('meLastname', $lastname),
                $query->equals('meFirstname', $firstname),
            )
        );
        $searchConstrains[] = $query->logicalOr(
            $query->equals('meZip', $zip),
            $query->equals('meCity', $city)
        );
        $searchConstrains[] = $query->equals('meAddress', $address);
        $query->matching(
            $query->logicalAnd(...$searchConstrains)
        );
        //var_dump($searchConstrains);exit(0);
        return $query->execute()->getFirst();
        */
    }

    protected function initDefaultQuerySettings()
    {
        if ($this->defaultQuerySettings === null) {
            $this->defaultQuerySettings = GeneralUtility::makeInstance(Typo3QuerySettings::class);
        }
    }
}
