<?php

namespace Internetgalerie\IgsCrm\Domain\Repository;

use Internetgalerie\IgsCrm\Domain\Model\Verband;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\Typo3QuerySettings;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;

/**
 * The repository for Verbands
 */
class VerbandRepository extends AbstractCrmRepository
{
    protected static $requiredRole = 'crm.tenant';
    protected $tablename = 'tx_igscrm_domain_model_verband';
    /**
     * @var array
     */
    protected $defaultOrderings = [
        'sorting' => QueryInterface::ORDER_ASCENDING,
    ];

    
    public function initializeObject(): void
    {
        $querySettings = GeneralUtility::makeInstance(Typo3QuerySettings::class);
        $querySettings->setRespectStoragePage(false);
        $this->setDefaultQuerySettings($querySettings);
    }

    /**
     * Replaces an existing object with the same identifier by the given object
     *
     * @param Verband $verband The modified object
     */
    public function add($verband)
    {
        if (!$this->securityUtility->hasPermissionWrite($verband->getParent(), 'crm.tenant')) {
            return false;
        }
        $ret = parent::add($verband);
        $this->persistenceManager->persistAll();
        $this->update($verband);
        return $ret;
    }

    /**
     * Replaces an existing object with the same identifier by the given object
     *
     * @param Verband $verband The modified object
     */
    public function update($verband)
    {
        if (!$this->securityUtility->hasPermissionWrite($verband, 'crm.tenant')) {
            return false;
        }
        $pathUids = [];
        $pathVerband = $verband;
        do {
            $pathUids[] = $pathVerband->getUid();
            $pathVerband = $pathVerband->getParent();
        } while ($pathVerband);
        $pathUids = array_reverse($pathUids);
        $pathUidsString = implode(',', $pathUids);

        $verband->setPathUids($pathUidsString);
        // @todo change children
        return parent::update($verband);
    }

    public function findDefault()
    {
        return $this->findByUid(1);
    }


    public function findAll($returnRawQueryResult = false)
    {
        $query = $this->createQuery();
        $constraints = $this->getQueryConstraintsWithAcl($query);
        if (!empty($constraints)) {
            $query->matching($query->logicalAnd(...$constraints));
        }
        return $query->execute($returnRawQueryResult);
    }

    public function findAllWithAcl(bool $returnRawQueryResult = false)
    {
        return $this->findAll($returnRawQueryResult);
    }

    public function findWithUids(array $uids = [], $returnRawQueryResult = false)
    {
        $query = $this->createQuery();
        $constraints = $this->getQueryConstraintsWithAcl($query);

        $constraints[] = $query->in('uid', $uids);

        if (!empty($constraints)) {
            $query->matching($query->logicalAnd(...$constraints));
        }
        
        return $query->execute($returnRawQueryResult);
    }

    public function findWithParent(int $parent, $returnRawQueryResult = false)
    {
        $query = $this->createQuery();
        $constraints = $this->getQueryConstraintsWithAcl($query);

        $constraints[] = $query->equals('parent', $parent);
        if (!empty($constraints)) {
            $query->matching($query->logicalAnd(...$constraints));
        }
        
        return $query->execute($returnRawQueryResult);
    }

    public function findOneWithUid(int $uid, $returnRawQueryResult = false)
    {
        $query = $this->createQuery();
        $constraints = $this->getQueryConstraintsWithAcl($query);

        $constraints[] = $query->equals('uid', $uid);
        if (!empty($constraints)) {
            $query->matching($query->logicalAnd(...$constraints));
        }
        
        $ret = $query->execute($returnRawQueryResult);
        return $ret->getFirst();
    }


    public function findBySearch(array $search, $returnRawQueryResult = false)
    {
        $query = $this->createQuery();
        $constraints = $this->getQueryConstraintsWithAcl($query);

        if (isset($search['parent']) && $search['parent'] !== null & $search['parent'] !== '') {
            $constraints[] = $query->equals('parent', (int) $search['parent']);
        }
        if (isset($search['uids']) && $search['uids'] !== null & $search['uids'] !== '') {
            $constraints[] = $query->in('uid', $search['uids']);
        }
        
        if (!empty($constraints)) {
            $query->matching($query->logicalAnd(...$constraints));
        }
        return $query->execute($returnRawQueryResult);
    }


    public function findAllRoots()
    {
        if ($this->securityUtility->hasRoleAdmin(self::$requiredRole)) {
            return null;
        }
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($this->tablename);
        $queryBuilder
            ->select('uid', 'parent')
            ->from($this->tablename);
        $constraints = $this->getQueryBuilderConstraintsWithAcl($queryBuilder);

        $queryBuilder->where(...$constraints);
        $result = $queryBuilder->executeQuery();
        $allEntries = $result->fetchAllAssociative();
        $uids = [];
        foreach ($allEntries as $entry) {
            $uids[] = $entry['uid'];
        }
        foreach ($allEntries as $index => $entry) {
            $allEntries[$index]['found'] = in_array($entry['parent'], $uids);
        }
        $ret = [];
        foreach ($allEntries as $entry) {
            if (!$entry['found']) {
                $ret[] = $entry['uid'];
            }
        }
        return $ret;
    }
}
