<?php

namespace Internetgalerie\IgsCrm\Utility;

use Internetgalerie\IgsCrm\Exception\TableNotFoundException;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use TYPO3\CMS\Core\Resource\Exception as CoreException;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

class ExcelUtility
{
    public function export(
        $headerArray,
        $tableDoubleArray,
        $baseFilename = 'export',
        $sheetName = '',
        $colsFormatCodes = []
    ): void {
        $spreadsheet = new Spreadsheet();
 
        // Sheetname setzen
        if ($sheetName) {
            $spreadsheet->removeSheetByIndex(0);
            $sheet = new Worksheet($spreadsheet, $sheetName);
            $spreadsheet->addSheet($sheet, 0);
        } else {
            $sheet = $spreadsheet->getActiveSheet();
        }
 
        // Daten abfüllen
        $sheet->fromArray($headerArray, null, 'A1');
        $sheet->fromArray($tableDoubleArray, null, 'A2');
 
        // FormatCodes anwenden für spezielle Typen (z.B. Datum/Zeit)
        if (!empty($colsFormatCodes)) {
            foreach ($colsFormatCodes as $formatCol => $formatCode) {
                $sheet->getStyle(
                    $formatCol . '2:' . $formatCol . intval(count($tableDoubleArray) + 1)
                )->getNumberFormat()->setFormatCode(
                    $formatCode
                );
            }
        }
 
        // Erste Zeile Fett
        $sheet->getStyle('A1:' . $sheet->getHighestDataColumn() . '1')->getFont()->setBold(true);

        $maxColumnNumber = Coordinate::columnIndexFromString($sheet->getHighestDataColumn());
        //die ('A1:' . $sheet->getHighestDataColumn() . '= '  . $maxColumnNumber);
        // Spaltenbreiten automatisch setzen
        for ($col = 1; $col <= $maxColumnNumber; $col++) {
            $sheet->getColumnDimension(Coordinate::stringFromColumnIndex($col))->setAutoSize(true);
        }

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        //header('Content-Type: application/force-download');
        //header('Content-Description: File Transfer' );
 
        //header('Content-Type: application/octet-stream' );
        //header('Content-Type: application/download');
        //////header('Content-Transfer-Encoding: binary' );
        header('Content-Disposition: attachment; filename=' . trim((string) $baseFilename) . '.xlsx');
        header('Connection: Keep-Alive');
        header('Expires: 0');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Pragma: public');

        $writer = new Xlsx($spreadsheet);
        $filename = tempnam('', 'tempfile');
        $writer->save($filename);
        
        //header('Content-Length: '.filesize($filename));
        unlink($filename);
        $writer->save('php://output');
        exit(0);
    }

    public function getConfigFromHeader(array $header, array $attributes, array $exampleLines = [], $aliases = [])
    {
        $config = [];
        foreach ($header as $rowNumber => $title) {
            $matchLabel = null;
            $matchAttribute = null;
            $matchFilter = null;
            $matchQualitiy = 0;
            $title = trim((string) $title);
            // Exact Match
            foreach ($attributes as $attribute => $label) {
                $label = trim((string) $label);
                if ($title == $label) {
                    $matchAttribute = $attribute;
                    $matchQualitiy = 100;
                    $matchLabel = $label;
                    break;
                }
            }
            // Lowercase Exact Match
            if ($matchAttribute === null) {
                $compTitle = strtolower($title);
                foreach ($attributes as $attribute => $label) {
                    if ($compTitle == strtolower((string) $label)) {
                        $matchAttribute = $attribute;
                        $matchQualitiy = 99;
                        $matchLabel = $label;
                        break;
                    }
                    if (!empty($aliases[$attribute]['attributes'])) {
                        foreach ($aliases[$attribute]['attributes'] as $aliasAttribute) {
                            if ($compTitle == strtolower((string) $aliasAttribute)) {
                                $matchAttribute = $attribute;
                                $matchFilter = $aliases[$attribute]['filter'] ?? null;
                                $matchQualitiy = 100;
                                $matchLabel = $label;
                                break 2;
                            }
                        }
                    }
                }
            }
            // Lowercase with removal of specialchars exact Match
            if ($matchAttribute === null) {
                $specialChars = [
                    '-',
                    ' ',
                    '.',
                    '_',
                    "\t",
                    ',',
                    '+',
                    '*',
                    '"',
                    "\'",
                    '/',
                    '(',
                    ')',
                    '=',
                    '?',
                    '^',
                    '~',
                    '[',
                    ']',
                ];
                $compTitle = str_replace($specialChars, '', strtolower($title));
                foreach ($attributes as $attribute => $label) {
                    if ($compTitle == str_replace($specialChars, '', strtolower((string) $label))) {
                        $matchAttribute = $attribute;
                        $matchQualitiy = 90;
                        $matchLabel = $label;
                        break;
                    }
                }
            }
            $firstLines = [];
            foreach ($exampleLines as $line) {
                $firstLines[] = $line[$rowNumber];
            }
            $config[] = [
                'title' => $title,
                'label' => $matchLabel,
                'attribute' => $matchAttribute,
                'filter' => $matchFilter,
                'quality' => $matchQualitiy,
                'firstLines' => $firstLines,
            ];
        }
        return $config;
    }
 
    public function config(string $table, string $filePath, array $aliases = [], array $exclude = [])
    {
        if (!is_file($filePath)) {
            throw new CoreException\FileDoesNotExistException();
        }
        $ext = pathinfo($filePath, PATHINFO_EXTENSION);
        $ext = strtolower($ext);

        if (empty($GLOBALS['TCA'][$table]['columns'])) {
            throw new TableNotFoundException($table);
        }

        $tca = $GLOBALS['TCA'][$table]['columns'];
        $attributes = [];
        $privateTypo3Names = array_merge($exclude, [
            'uid', 'pid', 'sys_language_uid', 'l10n_parent', 'l10n_diffsource', 't3ver_label', 'hidden', 'tstamp', 'crdate', 'cruser_id', 'deleted', 'starttime', 'endtime', 't3ver_oid', 't3ver_id',
        ]);
        
        foreach ($tca as $name => $config) {
            if (in_array($name, $privateTypo3Names)) {
                continue;
            }
            $value = $config['label'] ? LocalizationUtility::translate($config['label'], 'IgsCrm') : $name;
            if ($value !== null) {
                $attributes[$name] = $value;
            }
        }
        //var_dump($attributes);
        
        if ($ext == 'csv' || $ext == 'txt') {
            $fp = fopen($filePath, 'r');
            $firstLine = fgetcsv($fp);//, 0, $separator,  $enclosure,  $escape);
            $exampleLines = [];
            for ($i = 0;$i < 3; $i++) {
                $exampleLines[] = fgetcsv($fp);
            }
            //$firstLine[1] = 'nach-name';
            $config = $this->getConfigFromHeader($firstLine, $attributes, $exampleLines, $aliases);
            fclose($fp);
            //var_dump($config, $attributes);
            return [
                'config' => $config,
                'attributes' => $attributes,
            ];
        } elseif ($ext == 'xls' || $ext == 'xlsx' || $ext == 'ods') {
        } else {
            throw new CoreException\IllegalFileExtensionException('Used File Extension: ' . $ext);
        }


        $extReader = ucfirst($ext); // Xlsx,Xls,Ods,Gnumeric,Xml

        
        var_dump($ext);
        die('File=' . $filePath . '<br />table=' . $table);
    }
}
