<?php

namespace InternetGalerie\Igshop2\Command;

use Exception;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use TYPO3\CMS\Core\Core\Bootstrap;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\DataHandling\DataHandler;
use TYPO3\CMS\Core\Resource\DefaultUploadFolderResolver;
use TYPO3\CMS\Core\Resource\Exception\OnlineMediaAlreadyExistsException;
use TYPO3\CMS\Core\Resource\File;
use TYPO3\CMS\Core\Resource\OnlineMedia\Helpers\OnlineMediaHelperRegistry;
use TYPO3\CMS\Core\Resource\ResourceFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;

/**
 * Command for ugrade video
 * copy entries from tx_igshop2_domain_model_video to sys_file
 */
class UpgradeVideoCommand extends Command
{
    protected ?SymfonyStyle $io = null;

    protected int $newCount = 0;

    public function __construct(
        protected readonly ResourceFactory $resourceFactory,
        protected readonly DefaultUploadFolderResolver $uploadFolderResolver,
        protected readonly OnlineMediaHelperRegistry $onlineMediaHelperRegistry,
    ) {
        parent::__construct();
    }

    /**
     * Defines the allowed options for this command
     */
    protected function configure()
    {
        $this
          ->setDescription('upgrade tx_igshop2_domain_model_video')
          ->setHelp('copy entries from tx_igshop2_domain_model_video to sys_file')
          ->addArgument('targetFolderIdentifier', InputArgument::REQUIRED, 'target folder e.g. 1:video');
    }

    protected function folderExists(string $targetFolderIdentifier): bool
    {
        try {
            $targetFolder = $this->resourceFactory->getFolderObjectFromCombinedIdentifier($targetFolderIdentifier);
        } catch (Exception) {
            $targetFolder = null;
        }

        return $targetFolder !== null;
    }

    protected function addMediaFromUrl(
        string $url,
        ?string $targetFolderIdentifier,
        array $allowedExtensions = []
    ): ?File {
        $targetFolder = null;
        if ($targetFolderIdentifier) {
            try {
                $targetFolder = $this->resourceFactory->getFolderObjectFromCombinedIdentifier($targetFolderIdentifier);
            } catch (Exception) {
                $targetFolder = null;
            }
        }

        if ($targetFolder === null) {
            die($targetFolderIdentifier . ' was removed during migration');
            //$targetFolder = $this->uploadFolderResolver->resolve($this->getBackendUser());
        }

        return $this->onlineMediaHelperRegistry->transformUrlToFile($url, $targetFolder, $allowedExtensions);
    }

    protected function addFileReference(int $fileUid, array $oldVideo): bool
    {
        $productUid = (int)$oldVideo['product'];
        $tablename = 'tx_igshop2_domain_model_product';
        $fieldname = 'videos';
        $pid = (int)$oldVideo['pid'];
        ++$this->newCount;
        $newId = 'NEW' . $this->newCount;
        // Prepare data for the FileReference record
        $date = [];
        $data = [
            'sys_file_reference' => [
                $newId => [
                    'table_local' => 'sys_file',
                    'uid_local' => $fileUid,
                    'tablenames' => $tablename,
                    'uid_foreign' => $productUid,
                    'fieldname' => $fieldname,
                    'pid' => $pid,
                ],
            ],
        ];

        $data[$tablename][$productUid] = [
            $fieldname => $newId,
        ];
        $dataHandler = GeneralUtility::makeInstance(DataHandler::class);
        $dataHandler->start($data, []);
        $dataHandler->process_datamap();
        if (!empty($dataHandler->errorLog)) {
            foreach ($dataHandler->errorLog as $log) {
                $this->io->error($log);
            }

            return false;
        }

        return true;
    }

    protected function getOldVideos(): array
    {
        // fetch videos
        $connectionPool = GeneralUtility::makeInstance(ConnectionPool::class);
        $queryBuilder = $connectionPool->getQueryBuilderForTable('tx_igshop2_domain_model_video');
        $result = $queryBuilder
                ->select('*')
                ->from('tx_igshop2_domain_model_video')
                //->where('product=161')
                ->executeQuery();
        return $result->fetchAllAssociative();
    }

    /**
     * Installs an extension
     *
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        // Make sure the _cli_ user is loaded
        Bootstrap::initializeBackendAuthentication();

        $this->io = new SymfonyStyle($input, $output);
        $targetFolderIdentifier = $input->getArgument('targetFolderIdentifier');
        if (!$this->folderExists($targetFolderIdentifier)) {
            $this->io->error(
                'Eror target folder not found "' . $targetFolderIdentifier . '" (use e.g. "1:shop/video")'
            );
            return 1;
        }

        $oldVideos = $this->getOldVideos();
        $oldVideosCount = count($oldVideos);
        $this->io->writeln('<comment>' . $oldVideosCount . ' Videos (Youtube or File) found</comment>');
        $youtubeCount = 0;
        $fileCount = 0;
        $errorFound = false;

        foreach ($oldVideos as $oldVideo) {
            if ($oldVideo['youtube_video']) {
                ++$youtubeCount;
                $url = $oldVideo['youtube_video'];
                if (strpos('://', (string) $url) !== null) {
                    $url = 'https://www.youtube.com/watch?v=' . $url;
                }

                // add new media or use present meda if it is presen in folder
                try {
                    $file = $this->addMediaFromUrl($url, $targetFolderIdentifier);
                } catch (OnlineMediaAlreadyExistsException $e) {
                    // Ignore this exception since the endpoint is called e.g. in inline context, where the
                    // folder is not relevant and the same asset can be attached to a record multiple times.
                    $file = $e->getOnlineMedia();
                }

                $success = $this->addFileReference($file->getUid(), $oldVideo);
                if ($success) {
                    $this->io->writeln($url . ' added as sys_file.uid=' . $file->getUid());
                } else {
                    $this->io->eror('ERROR FileReference in ' . $url . ' added as sys_file.uid=' . $file->getUid());
                    $errorFound = true;
                }
            } else {
                ++$fileCount;
                $success = $this->addFileReference((int)$oldVideo['video'], $oldVideo);
                if ($success) {
                    $this->io->writeln('new FileReference for sys_file.uid=' . $oldVideo['video']);
                } else {
                    $this->io->eror('ERROR FileReference for sys_file.uid=' . $oldVideo['video']);
                    $errorFound = true;
                }
            }
        }

        $this->io->writeln('<comment>' . $youtubeCount . ' Youtube Videos found</comment>');
        $this->io->writeln('<comment>' . $fileCount . ' Local Files found</comment>');
        if ($errorFound) {
            $this->io->error('Eror in migration videos');
            return 1;
        }

        $this->io->success('videos successfully migrated to sys_file.');
        return 0;
    }
}
