<?php

namespace InternetGalerie\Igshop2\Controller;

use Ig\IgPayment\Utility\PaymentUtility;
use InternetGalerie\Igshop2\Domain\Model\Cart;
use InternetGalerie\Igshop2\Domain\Model\Order;
use InternetGalerie\Igshop2\Domain\Repository\CartRepository;
use InternetGalerie\Igshop2\Domain\Repository\CountryRepository;
use InternetGalerie\Igshop2\Domain\Repository\OrderRepository;
use InternetGalerie\Igshop2\Utility\CartUtility;
use InternetGalerie\Igshop2\Utility\ProductCacheUtility;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;

/**
 * PaymentController
 */
class PaymentController extends ActionController
{
    public function __construct(
        protected PaymentUtility $paymentUtility,
        protected CartUtility $cartUtility,
        protected CountryRepository $countryRepository,
        protected CartRepository $cartRepository,
        protected OrderRepository $orderRepository
    ) {
    }

    public function paymentAction(Order $order): ResponseInterface
    {
        if($order->getSubmitted()) {
            return $this->redirect('show', 'Order', 'Igshop2', null, $this->settings['cartPid']);
        }
        $language = $this->request->getAttribute('language');
        $locale = $language->getLocale()->getName();

        if (!$order->getPaymentMethod() || $order->getPaymentMethod()->getCode() == '' || $order->getTotalProductsCount() == 0) {
            return $this->redirect('show', 'Order', 'Igshop2', null, $this->settings['cartPid']);
        }



        if ($order->getCheckoutReference() == 0) {
            $order->setCheckoutReference(time());
            $this->cartUtility->update(false);
        }

        //$this->cartUtility->setCurrentOrderToPayment();
        $paymentService = $this->paymentUtility->setCurrentService(
            $order->getPaymentMethod()->getCode(),
            $order->getCurrency()?->getShort(),
            $order->getTotalWithPorto(),
            $order->getUid(),
        );

        //$paymentService = $this->paymentUtility->getServiceByCode($order->getPaymentMethod()->getCode());

        if (!$paymentService || !$order->canReserve()) {
            return $this->redirect('show', 'Order', 'Igshop2', null, $this->settings['cartPid']);
        }

        $order->reserve();
        $this->orderRepository->update($order);
        GeneralUtility::makeInstance(PersistenceManager::class)->persistAll();
        GeneralUtility::makeInstance(ProductCacheUtility::class)->flushByOrder($order); // flush cache where products are used
        //$order->flushProductCaches();

        $reference = $order->getCheckoutReference();
        $checkTransactionSuccessUrl = $this->uriBuilder->reset()->setCreateAbsoluteUri(true)->uriFor('checkTransactionSuccess', ['order' => $order->getUid()]);
        $failUrl = $this->uriBuilder->reset()->setCreateAbsoluteUri(true)->setTargetPageUid($this->settings['cartPid'])->uriFor('paymentError', null, 'Order');
        $paymentService->setProducts($order->getProductsForPayment());
        if($order->getPorto() && !$order->hasProductWithoutPorto()) {
            $paymentService->setShipping($order->getShippingForPayment());
        }

        $orderInfo = $order->getOrderinfo();
        $shippingAddress = $this->getAddressArray($orderInfo);
        $billingAddress = $shippingAddress;

        if ($orderInfo['isCalcAddress'] || $orderInfo['extraBillingAddress']) {
            $billingAddress = $this->getAddressArray($orderInfo['calcAddress']);
        }
        
        $paymentService->setShippingAddress($shippingAddress);
        $paymentService->setBillingAddress($billingAddress);

        $paymentStatus = $paymentService->invokePayment(
            $order->getUid(),
            $order->getCurrency()->getShort(),
            $order->getTotalWithPorto(),
            $reference,
            $checkTransactionSuccessUrl,
            $failUrl
        );
        
        if($reference != $order->getCheckoutReference()) {
            $order->setCheckoutReference($reference);
            $this->cartUtility->update(false);
        }

        if ($paymentStatus->isPaid()) {
            return $this->redirect('checkTransactionSuccess', null, null, [
                'order' => $order->getUid(),
                'externalId' => $paymentService->getExternalId()
            ]);
        }

        if($paymentStatus->getRedirect() !== null) {
            return $paymentStatus->getRedirect();
        }

        $this->view->assignMultiple($paymentService->getAssigns());
        $this->view->assign('order', $order);
        $this->view->assign('locale', $locale);
        return $this->htmlResponse();
    }

    public function checkTransactionSuccessAction(Order $order, ?string $externalId = null): ResponseInterface
    {
        $paymentService = $this->paymentUtility->getServiceByCode($order->getPaymentMethod()->getCode());
        if($paymentService && !$externalId) {
            $externalId = $paymentService->getExternalId();
        } else if($paymentService && $externalId) {
            $paymentService->setExternalId($externalId);
        }

        if (!$paymentService) {
            return $this->redirect('paymentError', 'Order', null, null, $this->settings['cartPid']);
        }

        $paymentStatus = $paymentService->checkTransactionSuccess(
            $order->getCurrency()->getShort(),
            $order->getTotalWithPorto()
        );

        if($paymentStatus->getOrderChanged()) {
            $order->setCheckoutReference(0);
            $this->cartUtility->update(false);
            return $this->redirect('paymentError', 'Order', null, null, $this->settings['cartPid']);
        }

        if ($paymentStatus->getTransactionSuccess()) {
            return $this->redirect('submitPayment', 'Order', null, [
                'complus' => 1,
                'order' => $order->getUid(),
                'ordertosubmit' => $order->getUid(),
            ], $this->settings['cartPid']);
        }

        $order->setCheckoutReference(0);
        $this->cartUtility->update(false);
        return $this->redirect('paymentError', 'Order', null, null, $this->settings['cartPid']);
    }

    protected function getAddressArray(array $data = []): array
    {
        $salutation = '';
        $country = null;
        $countryName = '';

        if (isset($data['salutation']) && $data['salutation']) {
            $salutation = $data['salutation'] == 1 ? 'm' : 'f';
        }

        if (isset($data['country']) && $data['country']) {
            $country = $this->countryRepository->findByUid($data['country']);
            if ($country && $country->getDataTrigger()) {
                $countryName = $data['foreignCountry'];
            } elseif ($country) {
                $countryName = $country->getName();
            }
        }

        $address = [
            'gender' => $salutation,
            'given_name' => $data['firstName'] ?? null,
            'family_name' => $data['lastName'] ?? null,
            'organization_name' => $data['company'] ?? null,
            'street' => $data['address'] ?? null,
            'postcode' => $data['zip'] ?? null,
            'city' => $data['city'] ?? null,
            'country' => $countryName,
            'phone_number' => $data['phone'] ?? null,
            'email_address' => $data['mail'] ?? null,
        ];

        return $address;
    }
}
