<?php

declare(strict_types=1);

namespace InternetGalerie\Igshop2\DataProcessing;

use Doctrine\DBAL\ParameterType;
use PDO;
use Ig\IgMenu\Domain\Model\MenuItem;
use Ig\IgMenu\Source\DataProcessorInterface;
use InternetGalerie\Igshop2\Domain\Repository\CategoryRepository;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;

use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Mvc\Web\Routing\UriBuilder;

class CategoriesProcessor implements DataProcessorInterface, SingletonInterface
{
    protected CategoryRepository $categoryRepository;

    public function __construct()
    {
        $storagePid = $this->getStoragePid();
        $this->categoryRepository = GeneralUtility::makeInstance(CategoryRepository::class);

        $querySettings = $this->categoryRepository->createQuery()
->getQuerySettings();
        $querySettings->setStoragePageIds([$storagePid]);

        $this->categoryRepository->setDefaultQuerySettings($querySettings);
    }

    /**
     * Process first call $menuItem is a page
     *
     * @param MenuItem $menuItem - the current menuItem
     */
    public function process(MenuItem $menuItem): array
    {
        //$shopUid = 1;
        $parentCategoryUid = (int) $menuItem->getIgMenuSourceOption('parentCategory');
        $categories = $this->categoryRepository->getChildren($parentCategoryUid);
        return $this->convertCategories($menuItem->getUid(), $categories);
    }

    /**
     * Process subitems - $menuItem is a category
     *
     * @param MenuItem $menuItem - the current menuItem
     */
    public function getChildren(MenuItem $menuItem): array
    {
        //$shopUid = 1;
        $parentCategory = $menuItem->getData();
        $categories = $this->categoryRepository->getChildren($parentCategory->getUid());
        return $this->convertCategories($menuItem->getUid(), $categories);
    }

    /**
     * convert to menuItems
     */
    public function convertCategories(int $pageUid, mixed $categories): array
    {
        $menuitems = [];
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        foreach ($categories as $category) {
            $menuitem = GeneralUtility::makeInstance(MenuItem::class);
            $menuitem->setTitle($category->getName());
            $menuitem->setUid($pageUid);
            $menuitem->setData($category);
            $menuitem->setActiveKey('category-' . $category->getUid());
            $uriBuilder
                ->reset()
                ->setTargetPageUid($pageUid);
            //->setArguments(['category' => $category->getUid()]);
            $menuitem->setLink($uriBuilder->uriFor(
                'list', // actionName
                [
                    'category' => $category->getUid(),
                ], //controllerArguments
                'Product', // controllerName
                'igshop2', // extensionName
                'Feshop' // pluginName
            ));

            $menuitems[] = $menuitem;
        }

        return $menuitems;
    }

    public function getStoragePid(): int
    {
        $configurationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
        return (int)$configurationManager->getConfiguration(
            ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        )['plugin.']['tx_igshop2.']['settings.']['storagePid'];
    }

    /**
     * Get Active Entries (Parameters with values)
     */
    public function getActiveKeys(int $pageUid, array $queryParams): array
    {
        $activeKeys = [];
        if (isset($queryParams['tx_igshop2_feshop']['category'])) {
            $category = (int) $queryParams['tx_igshop2_feshop']['category'];
            $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
                'tx_igshop2_domain_model_category'
            );
            $limit = 10;
            while ($category > 0 && $limit > 0) {
                --$limit;
                $activeKeys[] = 'category-' . $category;
                $result = $queryBuilder
                        ->select('parent')
                        ->from('tx_igshop2_domain_model_category')
                        ->where(
                            $queryBuilder->expr()
->eq('uid', $queryBuilder->createNamedParameter($category, ParameterType::INTEGER))
                        )
                        ->executeQuery()
->fetchAssociative();
                $category = (int) $result['parent'];
            }
        }

        return $activeKeys;
    }
}
