<?php

namespace InternetGalerie\Igshop2\Domain\Model;

use DateTime;
use TYPO3\CMS\Extbase\Annotation\ORM\Cascade;
use TYPO3\CMS\Extbase\Annotation\ORM\Lazy;
use TYPO3\CMS\Extbase\Persistence\ObjectStorage;

class Order extends AbstractOrder
{
    protected string $productClass = OrderProduct::class;

    /**
     * products
     *
     * @var ObjectStorage<OrderProduct>
     * @Cascade
     * @Lazy
     */
    protected $products = null;

    /**
     * cart
     * 
     * @var Cart
     */
    protected $cart = null;

    protected ?Shipping $shipping;
    protected float $shippingAmount = 0.0;
    protected bool $reserved = false;
    protected bool $canceled = false;
    protected ?DateTime $reservedUntil = null;


    /**
     * Returns the products
     *
     * @return ObjectStorage<OrderProduct> $products
     */
    public function getProducts()
    {
        return $this->products;
    }

    /**
     * Adds a OrderProduct
     * 
     * @param OrderProduct $product
     */
    public function addProduct($product): void
    {
        $this->products->attach($product);
        $this->mustCalculateTotal = true;
    }

    /**
     * Removes a OrderProduct
     *
     * @param OrderProduct $productToRemove The OrderProduct to be removed
     */
    public function removeProduct($productToRemove): void
    {
        $this->products->detach($productToRemove);
        $this->mustCalculateTotal = true;
    }

    /**
     * @return Cart
     */
    public function getCart()
    {
        return $this->cart;
    }

    /**
     * @param Cart $cart 
     * @return void 
     */
    public function setCart($cart)
    {
        $this->cart = $cart;
    }

    public function getShipping(): ?Shipping
    {
        return $this->shipping;
    }

    public function setShipping(?Shipping $shipping): void
    {
        $this->shipping = $shipping;
    }

    public function getShippingAmount(): float
    {
        return $this->shippingAmount;
    }

    public function setShippingAmount(float $shippingAmount): void
    {
        $this->shippingAmount = $shippingAmount;
    }

    /**
     * Returns whether the order is reserved
     */
    public function getReserved(): bool
    {
        return $this->reserved;
    }

    /**
     * Sets whether the order is reserved
     */
    public function setReserved(bool $reserved): void
    {
        $this->reserved = $reserved;
    }

    public function canReserve(): bool
    {
        foreach ($this->products as $orderProduct) {
            $product = $orderProduct->getProduct();
            if (
                $product->hasStock()
                && !$product->allowsBackorder()
                && $orderProduct->getAmount() > $product->getAvailableStock($this->getCart()->getUid(), $this->getUserid())
            ) {
                return false;
            }
        }
        return true;
    }
    public function reserve(int $minutes = 15): void
    {
        $this->reserved = true;
        $this->reservedUntil = new DateTime('+' . $minutes . ' minutes');
    }

    public function releaseReservation(): void
    {
        $this->reserved = false;
        $this->reservedUntil = null;
    }

    public function isReserved(): bool
    {
        return $this->reserved && $this->reservedUntil > new DateTime();
    }    

    /**
     * Returns whether the order is canceled
     *
     * @return bool
     */
    public function getCanceled(): bool
    {
        return $this->canceled;
    }

    /**
     * Sets whether the order is canceled
     */
    public function setCanceled(bool $canceled): void
    {
        $this->canceled = $canceled;
    }

    public function getReservedUntil(): ?DateTime
    {
        return $this->reservedUntil;
    }

    public function setReservedUntil(?DateTime $reservedUntil): void
    {
        $this->reservedUntil = $reservedUntil;
    }

    public function calculateTotalWithPorto(): void
    {
        if($this->getShippingAmount() > 0) {
            $this->totalWithPorto = $this->getTotal() + $this->getShippingAmount();
        } elseif ($this->getPorto() != null && !$this->hasProductWithoutPorto()) {
            $this->totalWithPorto = $this->getTotal() + $this->getPorto()
                ->getCurrentPrice($this->getTotalProductsPrice(), $this->getTotalProductsCount());
        } else {
            $this->totalWithPorto = $this->getTotal();
        }
    }
    
    
}