<?php

namespace InternetGalerie\Igshop2\Domain\Model;

use TYPO3\CMS\Extbase\DomainObject\AbstractEntity;
use DateTime;
use Ig\IgPayment\Utility\PaymentUtility;
use TYPO3\CMS\Extbase\Persistence\ObjectStorage;
use TYPO3\CMS\Extbase\Annotation\ORM\Cascade;
use TYPO3\CMS\Extbase\Annotation\ORM\Lazy;
use TYPO3\CMS\Extbase\Domain\Model\FileReference;
use InternetGalerie\Igshop2\Domain\Repository\CountryRepository;
use InternetGalerie\Igshop2\Domain\Repository\CurrencyRepository;
use InternetGalerie\Igshop2\Domain\Repository\PaymentMethodRepository;
use InternetGalerie\Igshop2\Domain\Repository\ShippingRepository;
use InternetGalerie\Igshop2\Utility\ShippingUtility;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Mvc\Web\Routing\UriBuilder;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;

/**
 * Order
 */
abstract class AbstractOrder extends AbstractEntity
{
    protected string $productClass = '';
    /**
     * tstamp
     *
     * @var int
     */
    protected $tstamp;

    /**
     * userid
     *
     * @var integer
     */
    protected $userid = 0;

    /**
     * orderdate
     *
     * @var DateTime
     */
    protected $orderdate = null;

    /**
     * total
     *
     * @var float
     */
    protected $total = 0.0;

    /**
     * total
     *
     * @var float
     */
    protected $totalMembers = 0.0;

    /**
     * customername
     *
     * @var string
     */
    protected $customername = '';

    /**
     * ordermail
     *
     * @var string
     */
    protected $ordermail = '';

    /**
     * orderinfo
     *
     * @var string
     */
    protected $orderinfo = '';

    /**
     * paymentmode
     *
     * @var string
     */
    protected $paymentmode = '';

    /**
     * paymentMethod
     *
     * @var PaymentMethod
     */
    protected $paymentMethod = null;

    /**
     * submitted
     *
     * @var integer
     */
    protected $submitted = 0;

    /**
     * done
     *
     * @var string
     */
    protected $done = '';

    /**
     * shippingType
     *
     * @var string
     */
    protected $shippingType = '';

    /**
     * shippingStatus
     *
     * @var string
     */
    protected $shippingStatus = '';

    /**
     * isMember
     *
     * @var boolean
     */
    protected $isMember = false;

    /**
     * totalWithPorto
     *
     * @var float
     */
    protected $totalWithPorto = 0.0;

    /**
     * plzort
     *
     * @var string
     */
    protected $plzort = '';

    /**
     * strasse
     *
     * @var string
     */
    protected $strasse = '';

    /**
     * paid
     *
     * @var boolean
     */
    protected $paid = false;

    /**
     * isOrder
     *
     * @var boolean
     */
    protected $isOrder = false;

    /**
     * hasBillingAddress
     *
     * @var boolean
     */
    protected $hasBillingAddress = false;

    /**
     * checkoutReference
     *
     * @var int
     */
    protected $checkoutReference = 0;

    /**
     * checkoutId
     *
     * @var string
     */
    protected $checkoutId = '';

    /**
     * porto
     *
     * @var Porto
     */
    protected $porto = null;

    /**
     * country
     *
     * @var Country
     */
    protected $country = null;

    /**
     * countryText
     *
     * @var string
     */
    protected $countryText = '';

    /**
     * currency
     *
     * @var Currency
     */
    protected $currency = null;

    /**
     * deliveryPeriod
     * @var DeliveryPeriod
     */
    protected $deliveryPeriod = null;

    /**
     * mustCalculateTotal
     *
     * @var boolean
     */
    protected $mustCalculateTotal = true;

    /**
     * mustCalculateTotal
     *
     * @var boolean
     */
    protected $mustCalculateTotalMembers = true;

    /**
     * survey
     *
     * @var ObjectStorage<Survey>
     * @Lazy
     */
    protected $survey = null;

    /**
     * confirmation
     *
     * @var FileReference
     * @Cascade
     * @Lazy
     */
    protected $confirmation = null;

    /**
     * The delivery date
     *
     * @var DateTime
     */
    protected $deliveryDate = null;

    /**
     * products
     *
     * @var ObjectStorage<AbstractOrderProduct>
     * @Cascade
     */
    protected $products = null;

    protected ?ShippingService $shippingService;

    public abstract function getProducts();
    public abstract function addProduct($product): void;
    public abstract function removeProduct($productToRemove): void;
    public abstract function calculateTotalWithPorto(): void;


    /**
     * __construct
     */
    public function __construct()
    {
        //Do not remove the next line: It would break the functionality
        $this->initStorageObjects();
    }

    protected function getProductPosition(
        Product $product,
        $prodLimit,
        $options = []
    ) {
        if ($options['amount'] <= 0) {
            //$this->forward("delete", "", "", array("productUid" => $product->getUid()));
        }

        $newProduct = GeneralUtility::makeInstance($this->productClass);
        //$newProduct = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance("InternetGalerie\\Igshop2\\Domain\\Model\\OrderProduct");
        if ($prodLimit > 0) {
            $newProduct->setAmount(min($prodLimit, $options['amount']));
        } else {
            $newProduct->setAmount($options['amount']);
        }

        if ($product->getMaxAmount() > 0 && $newProduct->getAmount() > $product->getMaxAmount()) {
            $newProduct->setAmount($product->getMaxAmount());
        }

        unset($options['amount']);
        $newProduct->setProdOptions(serialize($options));
        $newProduct->setProduct($product);
        $newProduct->calcPrice($this->currency, $this->isMember);
        return $newProduct;
    }

    /**
     * Returns the OrderProduct with the given uid from this order
     * @param int $orderProductUid
     * @return mixed
     */
    public function getProductPositionByUid($orderProductUid)
    {
        foreach ($this->products as $orderProduct) {
            if ($orderProduct->getUid() == $orderProductUid) {
                return $orderProduct;
            }
        }

        return null;
    }

    /**
     * Returns a Product with the same Options
     * @param int $productUid
     * @param string $prodOptions
     * @return mixed
     */
    public function getProduct($productUid, $prodOptions)
    {
        foreach ($this->products as $product) {
            if ($product->getProduct()->getUid() == $productUid && $product->getProdOptions() == $prodOptions) {
                return $product;
            }
        }

        return false;
    }

    /**
     * Function getSimilarproduct
     *
     * Searches a product who has the same UID and the same options as the given Orderproduct
     * @return mixed Product or false
     */
    public function getSimiliarProduct(AbstractOrderProduct $product)
    {
        foreach ($this->getProducts() as $orderProduct) {
            if ($orderProduct->getProduct() && $product->getProduct() && $orderProduct->getProduct()->getUid() == $product->getProduct()->getUid() && $orderProduct->getProdOptions() == $product->getProdOptions()) {
                return $orderProduct;
            }
        }

        return false;
    }

    /**
     * Sets the products
     */
    public function setProducts(ObjectStorage $products): void
    {
        $this->products = $products;
        $this->mustCalculateTotal = true;
    }

    /**
     * Adds a Product (entweder neues orderproduct oder falls schon bestehend Anzahl dazu addieren)
     */
    public function addNewProduct(Product $product, $prodLimit, $options = []): void
    {
        $newProduct = $this->getProductPosition($product, $prodLimit, $options);
        $similiarProduct = $this->getSimiliarProduct($newProduct);
        $canBeOrdered = true;
        if (isset($GLOBALS['TYPO3_CONF_VARS']['SC_OPTIONS']['igshop2_hooks']) && is_array(
            $GLOBALS['TYPO3_CONF_VARS']['SC_OPTIONS']['igshop2_hooks']['hooks']
        )) {
            foreach ($GLOBALS['TYPO3_CONF_VARS']['SC_OPTIONS']['igshop2_hooks']['hooks'] as $classRef) {
                $hookObj = GeneralUtility::makeInstance($classRef);
                if (method_exists($hookObj, 'checkProductState')) {
                    $canBeOrdered = !$hookObj->checkProductState($product);
                }
            }
        }

        if (!$canBeOrdered) {
            return;
        }

        // Produkt ist neu
        if ($similiarProduct === false) {
            $this->addProduct($newProduct);
        } else {
            // Produkt schon in Bestellung vorhanden
            // dynamisches limit, gegeben durch input form
            $simAmount = $similiarProduct->getAmount();
            $newProdAmount = $newProduct->getAmount();
            if ($prodLimit) {
                $newAmount = min($prodLimit, $simAmount + $newProdAmount);
            } else {
                $newAmount = $simAmount + $newProdAmount;
            }

            if ($product->getMaxAmount() > 0) {
                $newAmount = min($product->getMaxAmount(), $newAmount);
            }

            $similiarProduct->setAmount($newAmount);
            $similiarProduct->calcPrice($this->currency, $this->isMember);
        }

        $this->setDeliveryPeriod($this->getMaxDeliveryPeriod());
    }

    /**
     * change a Product
     *
     * @return boolean product was deleted
     */
    public function changeProductQuantity(
        $orderProduct,
        $newQuantity
    ) {
        $deleteProduct = false;
        if ($newQuantity <= 0) {
            $deleteProduct = true;
            $this->removeProduct($orderProduct);
        } else {
            $orderProduct->setAmount($newQuantity);
            $orderProduct->calcPrice($this->currency, $this->isMember);
            $this->mustCalculateTotal = true;
        }

        return $deleteProduct;
    }

    /**
     * @return int
     */
    public function getTstamp()
    {
        return $this->tstamp;
    }

    /**
     * Returns the userid
     *
     * @return integer
     */
    public function getUserid()
    {
        return $this->userid;
    }

    /**
     * Sets the userid
     *
     * @param integer $userid
     */
    public function setUserid($userid): void
    {
        $this->userid = $userid;
    }

    /**
     * Returns the orderdate
     *
     * @return DateTime
     */
    public function getOrderdate()
    {
        return $this->orderdate;
    }

    /**
     * Sets the orderdate
     */
    public function setOrderdate(DateTime $orderdate): void
    {
        $this->orderdate = $orderdate;
    }

    /**
     * Returns the total
     *
     * @return float
     */
    public function getTotal()
    {
        if (!$this->submitted || !$this->total) {
            if ($this->mustCalculateTotal) {
                $this->calculateTotal();
            }
        }

        return $this->total;
    }

    /**
     * Returns the total
     *
     * @return float
     */
    public function getTotalMembers()
    {
        if (!$this->submitted || !$this->total) {
            if ($this->mustCalculateTotal) {
                $this->calculateTotalMembers();
            }
        }

        return $this->totalMembers;
    }

    /**
     * Sets the total
     *
     * @param float $total
     */
    public function setTotal($total): void
    {
        $this->total = $total;
    }

    /**
     * Returns the customername
     *
     * @return string
     */
    public function getCustomername()
    {
        return $this->customername;
    }

    /**
     * Sets the customername
     *
     * @param string $customername
     */
    public function setCustomername($customername): void
    {
        $this->customername = $customername;
    }

    /**
     * Returns the ordermail
     *
     * @return string
     */
    public function getOrdermail()
    {
        return $this->ordermail;
    }

    /**
     * Sets the ordermail
     *
     * @param string $ordermail
     */
    public function setOrdermail($ordermail): void
    {
        $this->ordermail = $ordermail;
    }

    /**
     * Returns the orderinfo
     *
     * @return array
     */
    public function getOrderinfo()
    {
        $address = unserialize($this->orderinfo);
        return $address;
    }

    /**
     * Sets the orderinfo
     *
     * @param array $orderinfo
     */
    public function setOrderinfo($orderinfo): void
    {
        $country = $orderinfo['country'];
        if (is_object($country)) {
            $orderinfo['country'] = $country->getUid();
        }

        if ($orderinfo['calcAddress'] ?? false) {
            $calcAddressCountry = $orderinfo['calcAddress']['country'];
            if (is_object($calcAddressCountry)) {
                $orderinfo['calcAddress']['country'] = $calcAddressCountry->getUid();
            }
        }

        $this->orderinfo = serialize($orderinfo);
    }

    /**
     * Returns the paymentmode
     *
     * @return string
     */
    public function getPaymentmode()
    {
        return $this->paymentmode;
    }

    /**
     * Sets the paymentmode
     *
     * @param string $paymentmode
     */
    public function setPaymentmode($paymentmode): void
    {
        $this->paymentmode = $paymentmode;
        //$this->mustCalculateTotal=true;
    }

    /**
     * Returns the paymentMethod
     *
     * @return PaymentMethod
     */
    public function getPaymentMethod()
    {
        return $this->paymentMethod;
    }

    /**
     * Sets the paymentMethod
     */
    public function setPaymentMethod(PaymentMethod $paymentMethod): void
    {
        $this->paymentMethod = $paymentMethod;
    }




    /**
     * Returns the submitted
     *
     * @return integer
     */
    public function getSubmitted()
    {
        return $this->submitted;
    }

    /**
     * Sets the submitted
     *
     * @param integer $submitted
     */
    public function setSubmitted($submitted): void
    {
        $this->submitted = $submitted;
    }

    /**
     * Returns the done
     *
     * @return string
     */
    public function getDone()
    {
        return $this->done;
    }

    /**
     * Sets the done
     *
     * @param string $done
     */
    public function setDone($done): void
    {
        $this->done = $done;
    }

    /**
     * Returns the shippingType
     *
     * @return string
     */
    public function getShippingType()
    {
        return $this->shippingType;
    }

    /**
     * Sets the shippingType
     *
     * @param string $shippingType
     */
    public function setShippingType($shippingType): void
    {
        $this->shippingType = $shippingType;
        //$this->mustCalculateTotal=true;
    }

    /**
     * Returns the shippingStatus
     *
     * @return string
     */
    public function getShippingStatus()
    {
        return $this->shippingStatus;
    }

    /**
     * Sets the shippingStatus
     *
     * @param string $shippingStatus
     */
    public function setShippingStatus($shippingStatus): void
    {
        $this->shippingStatus = $shippingStatus;
    }

    /**
     * Returns the isMember
     *
     * @return boolean
     */
    public function getIsMember()
    {
        return $this->isMember;
    }

    /**
     * Sets the isMember
     *
     * @param boolean $isMember
     */
    public function setIsMember($isMember): void
    {
        if ($this->isMember != $isMember) {
            $this->mustCalculateTotal = true;
        }

        $this->isMember = $isMember;
    }

    /**
     * Returns the boolean state of isMember
     *
     * @return boolean
     */
    public function isIsMember()
    {
        return $this->isMember;
    }

    /**
     * Returns the totalWithPorto
     *
     * @return float
     */
    public function getTotalWithPorto()
    {
        if (!$this->submitted || !$this->totalWithPorto) {
            $this->calculateTotalWithPorto();
        }

        return $this->totalWithPorto;
    }

    /**
     * Sets the totalWithPorto
     *
     * @param float $totalWithPorto
     */
    public function setTotalWithPorto($totalWithPorto): void
    {
        $this->totalWithPorto = $totalWithPorto;
    }

    /**
     * Returns the plzort
     *
     * @return string
     */
    public function getPlzort()
    {
        return $this->plzort;
    }

    /**
     * Sets the plzort
     *
     * @param string $plzort
     */
    public function setPlzort($plzort): void
    {
        $this->plzort = $plzort;
    }

    /**
     * Returns the strasse
     *
     * @return string
     */
    public function getStrasse()
    {
        return $this->strasse;
    }

    /**
     * Sets the strasse
     *
     * @param string $strasse
     */
    public function setStrasse($strasse): void
    {
        $this->strasse = $strasse;
    }

    /**
     * Returns the paid
     *
     * @return boolean
     */
    public function getPaid()
    {
        return $this->paid;
    }

    /**
     * Sets the paid
     *
     * @param boolean $paid
     */
    public function setPaid($paid): void
    {
        $this->paid = $paid;
    }

    /**
     * Returns the boolean state of paid
     *
     * @return boolean
     */
    public function isPaid()
    {
        return $this->paid;
    }

    /**
     * Returns the isOrder
     *
     * @return boolean
     */
    public function getIsOrder()
    {
        return $this->isOrder;
    }

    /**
     * Sets the isOrder
     *
     * @param boolean $isOrder
     */
    public function setIsOrder($isOrder): void
    {
        $this->isOrder = $isOrder;
    }

    /**
     * Returns the boolean state of isOrder
     *
     * @return boolean
     */
    public function isIsOrder()
    {
        return $this->isOrder;
    }

    /**
     * Returns the hasBillingAddress
     *
     * @return boolean
     */
    public function getHasBillingAddress()
    {
        return $this->hasBillingAddress;
    }

    /**
     * Sets the hasBillingAddress
     *
     * @param boolean $hasBillingAddress
     */
    public function setHasBillingAddress($hasBillingAddress): void
    {
        $this->hasBillingAddress = $hasBillingAddress;
    }

    /**
     * Returns the boolean state of hasBillingAddress
     *
     * @return boolean
     */
    public function isHasBillingAddress()
    {
        return $this->hasBillingAddress;
    }

    /**
     * Returns the number of products
     *
     * @return int $totalProducts
     */
    public function getTotalProductsCount()
    {
        $totalProducts = 0;
        foreach ($this->products as $product) {
            $totalProducts += $product->getAmount();
        }

        return $totalProducts;
    }

    public function getTotalProductsPrice()
    {
        return $this->getTotal();
    }




    /**
     * Returns the porto
     *
     * @return Porto
     */
    public function getPorto()
    {
        return $this->porto;
    }

    /**
     * is the current porto for country $country
     *
     * @return bool
     */
    public function isPortoValidWithCountry($country)
    {
        return $this->porto && $country && $this->porto->hasCountry($country);
    }

    /**
     * Returns the porto.price
     *
     * @return string
     */
    public function getPortoPrice()
    {
        //echo($this->getCurrency() .','. $this->getTotalProductsPrice() .','.$this->getTotalProductsCount());
        if (is_object($this->porto) && !$this->hasProductWithoutPorto()) {
            if ($this->getCurrency()->getShort() == 'CHF') {
                return $this->porto->getCurrentPrice($this->getTotalProductsPrice(), $this->getTotalProductsCount());
            }

            return $this->porto->getCurrentPriceEuro(
                $this->getTotalProductsPrice(),
                $this->getTotalProductsCount()
            );
        }

        return '';
    }

    /**
     * Sets the porto
     */
    public function setPorto(Porto $porto): void
    {
        $this->porto = $porto;
    }

    /**
     * Removes the porto
     */
    public function removePorto(): void
    {
        $this->porto = null;
    }

    /**
     * Returns the porto.price
     *
     * @return string
     */
    public function getPortoDescription()
    {
        if (is_object($this->porto)) {
            //echo($this->getCurrency() .','. $this->getTotalProductsPrice() .','.$this->getTotalProductsCount());
            $search = ['{currency.short}', '{currency.name}', '{currency.symbol}', '{order.portoPricesDiff}'];
            $replace = [
                $this->getCurrency()->getShort(),
                $this->getCurrency()->getName(),
                $this->getCurrency()->getSymbol(),
                $this->getPortoPricesDiff(),
            ];

            // @TODO Euro Preise bei amount_min und amount_max
            return str_replace(
                $search,
                $replace,
                $this->porto->getCurrentDescription($this->getTotalProductsPrice(), $this->getTotalProductsCount())
            );
        }

        return '';
    }

    /**
     * Returns the Betrag bis nächste Porto Stufe erreicht wird
     *
     * @return string
     */

    public function getPortoPricesDiff()
    {
        if (is_object($this->porto)) {
            $next_price = $this->porto->getNextPrice($this->getTotalProductsPrice(), $this->getTotalProductsCount());
            $current_price = $this->getTotalProductsPrice();
            if ($next_price && $next_price > $current_price) {
                return number_format($next_price - $current_price, 2);
            }
        }

        return '';
    }

    /**
     * Returns the country
     *
     * @return Country
     */
    public function getCountry()
    {
        return $this->country;
    }

    /**
     * Sets the country
     */
    public function setCountry(Country $country): void
    {
        $this->country = $country;
        //$this->mustCalculateTotal=true;
    }

    /**
     * Returns the countryText
     *
     * @return string
     */
    public function getCountryText()
    {
        return $this->countryText;
    }

    /**
     * Returns the country object (entweder direkt Objekt oder passendes zu countryText - sonst null)
     *
     * @param boolean $countryAsUid Country Objekt verwenden sonst Text
     * @return string
     */
    public function getCountryObject($countryAsUid)
    {
        if ($countryAsUid) {
            return $this->country;
        }

        // really old.....
        $country = null;
        $countryText = $this->getCountryText();

        // @TODO DA - muss weg
        if (GeneralUtility::makeInstance(Context::class)->getPropertyFromAspect('language', 'id') == 2) {
            if ($countryText == 'Schweiz') {
                $countryText = 'Suisse';
            }

            if ($countryText == 'Deutschland') {
                $countryText = 'Allemagne';
            }
        } else {
            if ($countryText == 'Suisse') {
                $countryText = 'Schweiz';
            }

            if ($countryText == 'Allemagne') {
                $countryText = 'Deutschland';
            }
        }

        $countryRepository = GeneralUtility::makeInstance(CountryRepository::class);
        $countryRes = $countryRepository->findByName($countryText);
        if (count($countryRes) >= 1) {
            $country = $countryRes->getFirst();
        }

        return $country;
    }

    public function getCalcCountry()
    {
        $countryRepository = GeneralUtility::makeInstance(CountryRepository::class);
        $country = $countryRepository->findByUid($this->getOrderinfo()['calcAddress']['country']);
        return $country;
    }

    /**
     * Sets the countryText
     *
     * @param string $countryText
     */
    public function setCountryText($countryText): void
    {
        $this->countryText = $countryText;
    }


    /**
     * Returns the currency
     *
     * @return Currency
     */
    public function getCurrency()
    {
        return $this->currency;
    }

    /**
     * Sets the currency
     */
    public function setCurrency(Currency $currency): void
    {
        if ($this->currency != $currency) {
            $this->currency = $currency;
            $this->mustCalculateTotal = true;
            // @TODO muss weg
            foreach ($this->getProducts()as $op) {
                $op->calcPrice($this->currency, $this->getIsMember());
            }
        }
    }

    /**
     * @return DeliveryPeriod
     */
    public function getDeliveryPeriod()
    {
        return $this->deliveryPeriod;
    }


    public function setDeliveryPeriod($deliveryPeriod): void
    {
        $this->deliveryPeriod = $deliveryPeriod;
    }

    public function getMaxDeliveryPeriod($productToRemove = null)
    {
        $maxDeliveryPeriod = $this->getDeliveryPeriod();
        $maxPriority = $this->getDeliveryPeriod() ? $this->getDeliveryPeriod()
->getPriority() : 0;

        if ($productToRemove) {
            $maxDeliveryPeriod = null;
            $maxPriority = 0;
        }

        foreach ($this->getProducts() as $orderProduct) {
            if ($productToRemove && $orderProduct->getUid() == $productToRemove->getUid()) {
                continue;
            }

            $product = $orderProduct->getProduct();

            if ($product && $product->getDeliveryPeriod()) {
                $maxPriority = max($maxPriority, $product->getDeliveryPeriod()->getPriority());
            }

            if ($orderProduct->getOptionWithPrice() && $orderProduct->getOptionWithPrice()->getDeliveryPeriod()) {
                $maxPriority = max(
                    $maxPriority,
                    $orderProduct->getOptionWithPrice()
->getDeliveryPeriod()
->getPriority()
                );
            }

            if ($product && $product->getDeliveryPeriod() && $maxPriority == $product->getDeliveryPeriod()->getPriority()) {
                $maxDeliveryPeriod = $product->getDeliveryPeriod();
            } elseif ($orderProduct->getOptionWithPrice() && $orderProduct->getOptionWithPrice()->getDeliveryPeriod() && $maxPriority == $orderProduct->getOptionWithPrice()->getDeliveryPeriod()->getPriority()) {
                $maxDeliveryPeriod = $orderProduct->getOptionWithPrice()
->getDeliveryPeriod();
            }
        }

        return $maxDeliveryPeriod;
    }


    /**
     * Sets the currency per Uid (from session)
     */
    public function setCurrencyByUid($currencyUid)
    {
        if (intval($currencyUid) > 0) {
            $currencyRepository = GeneralUtility::makeInstance(CurrencyRepository::class);
            $currency = $currencyRepository->findOneByUid($currencyUid);
            //$currencyRes=$this->currencyRepository->findByUid($currencyUid);
            //$currency=$currencyRes->getFirst();
            if ($currency) {
                $this->setCurrency($currency);
                return $currency;
            }
        }

        return false;
    }




    /**
     * Returns the total amount of products
     *
     * @return int
     */
    public function getTotalProductsAmount()
    {
        return $this->getTotalProductsCount();
    }


    public function saveAddress($address, $settings): void
    {
        if ($address['country']) {
            if ($settings['countryAsUid']) {
                $this->setCountry($address['country']);
            } else {
                $this->setCountryText($address['country']);
            }
        }

        if ($address['firstName']) {
            $this->setCustomername($address['firstName']);
        }

        if ($address['mail']) {
            $this->setOrdermail($address['mail']);
        }

        if ($address['address']) {
            $this->setStrasse($address['address']);
        }

        if ($address['zip'] && $address['city']) {
            $this->setPlzort($address['zip'] . ' ' . $address['city']);
        }

        if (is_array($address)) {
            if (key_exists('calcAddress', $address)) {
                $calcAddress = $address['calcAddress'];
                unset($address['calcAddress']);
            }

            $orderInfo = unserialize($this->orderinfo);
            foreach ($address as $field => $value) {
                $orderInfo[$field] = $address[$field];
            }

            $orderInfo['extraBillingAddress'] = 0;
            if (isset($orderInfo['isCalcAddress']) && $orderInfo['isCalcAddress']) {
                $address['extraBillingAddress'] = true;
                $orderInfo['extraBillingAddress'] = 1;
            }

            if (isset($address['extraBillingAddress']) && $address['extraBillingAddress']) {
                $orderInfo['isCalcAddress'] = 1;
            } else {
                $orderInfo['isCalcAddress'] = 0;
            }

            $this->setHasBillingAddress($this->extraBillingAddress($address, $settings));
            if (isset($calcAddress) && is_array($calcAddress)) {
                foreach ($calcAddress as $field => $value) {
                    $orderInfo['calcAddress'][$field] = $calcAddress[$field];
                }
            }

            $this->setOrderinfo($orderInfo);
        }

        if (isset($address['payment']) && $address['payment']) {
            $paymentMethodRepository = GeneralUtility::makeInstance(PaymentMethodRepository::class);
            $paymentMethod = $paymentMethodRepository->findByUid($address['payment']);
            $this->setPaymentMethod($paymentMethod);
        }
    }

    public function hasProductWithoutPorto()
    {
        foreach ($this->products as $orderProduct) {
            $product = $orderProduct->getProduct();
            if ($product && $product->isNoPorto()) {
                return true;
            }
        }

        return false;
    }


    /**
     * Returns the total amount of products
     *
     * @return int
     */




    /**
     * Sets the price for the order for all ordered Products
     */
    public function calcAllOrderProductPrices(): void
    {
        foreach ($this->getProducts() as $product) {
            $product->calcPrice($this->getCurrency(), $this->isMember);
        }
    }

    /**
     * Sets the price for the order for all ordered Products
     */
    public function calcAllOrderProductMemberPrices(): void
    {
        foreach ($this->getProducts() as $product) {
            $product->calcPrice($this->getCurrency(), true);
        }
    }

    // Extra Rechnungsadresse angewählt
    public function extraBillingAddress($address, $settings)
    {
        return $settings['differentAddresses'] && (isset($address['extraBillingAddress']) && $address['extraBillingAddress']) || (isset($address['sameBillingAddress']) && !$address['sameBillingAddress']) && $address['payment'] != 'PayPal';
        // $this->settings['differentAddresses'] &&
    }

    // @TODO noch sinnvoller machen
    public function hasPorto()
    {
        return $this->getPorto();
    }

    /**
     * Adds a Survey
     */
    public function addSurvey(Survey $survey): void
    {
        $this->survey->attach($survey);
    }

    /**
     * Removes a Survey
     *
     * @param Survey $surveyToRemove The Survey to be removed
     */
    public function removeSurvey(Survey $surveyToRemove): void
    {
        $this->survey->detach($surveyToRemove);
    }

    /**
     * Returns the survey
     *
     * @return ObjectStorage<Survey> $survey
     */
    public function getSurvey()
    {
        return $this->survey;
    }

    /**
     * Sets the survey
     *
     * @param ObjectStorage<Survey> $survey
     */
    public function setSurvey(ObjectStorage $survey): void
    {
        $this->survey = $survey;
    }


    /**
     * Adds a confirmation
     */
    /*public function addConfirmation(\TYPO3\CMS\Extbase\Domain\Model\FileReference $confirmation) {
          $this->confirmation->attach($confirmation);
      }*/
    /**
     * Removes a confirmation
     */
    /*public function removeConfirmation(\TYPO3\CMS\Extbase\Domain\Model\FileReference $confirmationToRemove) {
          $this->confirmation->detach($confirmationToRemove);
      }*/
    /**
     * Returns the confirmation
     *
     * @return FileReference
     */
    public function getConfirmation()
    {
        return $this->confirmation;
    }

    /**
     * Sets the confirmation
     */
    public function setConfirmation(FileReference $confirmation): void
    {
        $this->confirmation = $confirmation;
    }

    /**
     * Returns the
     */
    public function getVat()
    {
        $total = $this->getTotal();
        return $total - ($total / 1.077);
    }

    /**
     * Get the value of The delivery date
     *
     * @return DateTime
     */
    public function getDeliveryDate()
    {
        return $this->deliveryDate;
    }

    /**
     * Set the value of The delivery date
     *
     * @return self
     */
    public function setDeliveryDate(DateTime $deliveryDate = null)
    {
        $this->deliveryDate = $deliveryDate;

        return $this;
    }

    /**
     * @return int
     */
    public function getCheckoutReference()
    {
        return $this->checkoutReference;
    }

    /**
     * @param int $checkoutReference
     */
    public function setCheckoutReference($checkoutReference): void
    {
        $this->checkoutReference = $checkoutReference;
    }

    /**
     * @return string
     */
    public function getCheckoutId()
    {
        return $this->checkoutId;
    }

    /**
     * @param string $checkoutId
     */
    public function setCheckoutId($checkoutId): void
    {
        $this->checkoutId = $checkoutId;
    }

    /**
     * Initializes all ObjectStorage properties
     * Do not modify this method!
     * It will be rewritten on each save in the extension builder
     * You may modify the constructor of this class instead
     */
    protected function initStorageObjects()
    {
        $this->products = new ObjectStorage();
        $this->survey = new ObjectStorage();
    }

    protected function calculateTotal()
    {
        $total = 0;
        $this->calcAllOrderProductPrices();
        foreach ($this->products as $orderProduct) {
            $options = unserialize($orderProduct->getProdOptions());
            $product = $orderProduct->getProduct();

            if ($product) {
                $orderProductPrice = $product->getFinalPrice(
                    $this->getCurrency(),
                    $options['optionWithPrice'] ?? null,
                    $orderProduct->getAmount(),
                    false,
                    $this->isMember
                );

                $total += $orderProductPrice * $orderProduct->getAmount();
            }
        }

        $this->total = $total;
    }

    protected function calculateTotalMembers()
    {
        $total = 0;
        $this->calcAllOrderProductMemberPrices();
        foreach ($this->products as $orderProduct) {
            $options = unserialize($orderProduct->getProdOptions());
            $product = $orderProduct->getProduct();
            if ($product) {
                $orderProductPrice = $product->getFinalPrice(
                    $this->getCurrency(),
                    $options['optionWithPrice'] ?? null,
                    $orderProduct->getAmount(),
                    false,
                    true
                );
                $total += $orderProductPrice * $orderProduct->getAmount();
            }
        }

        $this->totalMembers = $total;
    }

    public function updatePayment()
    {
        if ($this->getPaymentMethod() && $this->getCheckoutReference()) {
            $configruationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
            $configuration = $configruationManager->getConfiguration(ConfigurationManager::CONFIGURATION_TYPE_FRAMEWORK);
            $cartPid = $configuration['settings']['cartPid'];
            $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
            $checkTransactionSuccessUrl = $uriBuilder->reset()->setCreateAbsoluteUri(true)->setTargetPageUid($cartPid)->uriFor('checkTransactionSuccess', ['order' => $this->getUid()], 'Payment', 'Igshop2', 'Fecart');

            $paymentUtility = GeneralUtility::makeInstance(PaymentUtility::class);
            foreach($paymentUtility->getServices() as $paymentService) {
                $paymentService->updatePayment(
                    $this->getUid(),
                    $this->getCurrency()->getShort(),
                    $this->getTotalWithPorto(),
                    $this->getCheckoutReference(),
                    $checkTransactionSuccessUrl,
                    $this->getPaymentMethod()->getCode()
                );
            }
            $this->setCheckoutReference(0);
        }
    }

    public function getProductsForPayment()
    {
        $products = [];
        foreach ($this->getProducts() as $orderProduct) {
            $orderProduct->calcPrice($this->getCurrency());
            $products[] = [
                'id' => $orderProduct->getProduct()->getUid(),
                'name' => $orderProduct->getProduct()->getName(),
                'price' => $orderProduct->getPrice(),
                'amount' => $orderProduct->getAmount()
            ];
        }
        return $products;
    }

    public function getShippingForPayment()
    {
        return [
            'id' => $this->getPorto()->getUid(),
            'name' => $this->getPorto()->getName(),
            'price' => $this->getPorto()->getPrice(),
            'amount' => 1
        ];
    }

    public function getShippingService(): ?ShippingService
    {
        return $this->shippingService;
    }

    public function setShippingService(?ShippingService $shippingService): void
    {
        $this->shippingService = $shippingService;
    }

    public function getTotalShippingWeight(): float
    {
        $weight = 0.0;

        foreach ($this->products as $orderProduct) {
            $product = $orderProduct->getProduct();
            if ($product->getShippingWeight() !== null) {
                $weight += $product->getShippingWeight();
            }
        }

        return $weight;
    }

    public function getAllShippingLabels(): array
    {
        $allShippingLabels = [];
        foreach($this->getProducts() as $orderProduct) {
            $product = $orderProduct->getProduct();
            $shippingLabels = $product->getShippingLabels();
            foreach($shippingLabels as $shippingLabel) {
                if($shippingLabel && !isset($allShippingLabels[$shippingLabel->getUid()])) {
                    $allShippingLabels[$shippingLabel->getUid()] = $shippingLabel;
                }
            }
        }

        return $allShippingLabels;
    }

    public function getShippingByService(): ?Shipping
    {
        if($this->shippingService) {
            $shippingUtility = GeneralUtility::makeInstance(ShippingUtility::class);
            return $shippingUtility->getShipping($this->shippingService, [
                'weight' => $this->getTotalShippingWeight(),
                'quantity' => $this->getTotalProductsCount(),
                'price' => $this->getTotalProductsPrice(),
                'shippingLabels' => $this->getAllShippingLabels()
            ]);
        }
        return null;
    }
}
