<?php

namespace InternetGalerie\Igshop2\Utility;

use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Extbase\Service\ImageService;
use TYPO3\CMS\Core\Resource\FileReference;
use TYPO3\CMS\Core\Resource\ResourceFactory;
use TYPO3\CMS\Core\Resource\Exception\ResourceDoesNotExistException;

class AjaxUtility implements SingletonInterface
{
    public function __construct(
        private readonly ResourceFactory $resourceFactory,
        private readonly ImageService $imageService,
    ) {}

    public function getImageIds($productUid, $fieldName, $tablename)
    {
        // Fetch image IDs from the database using the product UID and field name
        $connection = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getConnectionForTable('sys_file_reference');

        $queryBuilder = $connection->createQueryBuilder();
        $rows = $queryBuilder
            ->select('uid_local')
            ->from('sys_file_reference')
            ->where(
                $queryBuilder->expr()->eq('uid_foreign', $queryBuilder->createNamedParameter($productUid, \PDO::PARAM_INT)),
                $queryBuilder->expr()->eq('tablenames', $queryBuilder->createNamedParameter($tablename, \PDO::PARAM_STR)),
                $queryBuilder->expr()->eq('fieldname', $queryBuilder->createNamedParameter($fieldName, \PDO::PARAM_STR))
            )
            ->execute()
            ->fetchAll();

        // Extract image IDs
        $imageIds = array_column($rows, 'uid_local');
        return $imageIds;
    }
    
    public function getImagePaths(array $imageIds, ?int $width = null, ?int $height = null)
    {
        $imagePaths = [];
        $processingInstructions = [];
        if ($width > 0) {
            $processingInstructions['width'] = $width . 'c'; // c = crop if necessary
        }
        if ($height > 0) {
            $processingInstructions['height'] = $height . 'c'; // c = crop if necessary
        }

        
        foreach ($imageIds as $imageId) {
            $file = $this->resourceFactory->getFileObject($imageId);
            if ($file) {
                if ($width > 0) {
                    $processedImage = $this->imageService->applyProcessingInstructions($file, $processingInstructions);
                    $filePath = $this->imageService->getImageUri($processedImage);
                } else {
                    $filePath = $file->getPublicUrl();
                }
                $imagePaths[] = $filePath;
            }
        }

        return $imagePaths;
    }

    /**
     * get categories from a product
     */
    public function getCategoriesByProductUid(int $productUid, int $limit = 0): array
    {
        $categories = [];
        $connection = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getConnectionForTable('tx_igshop2_domain_model_category');

        $queryBuilder = $connection->createQueryBuilder();
        $attributes = ['uid', 'name', 'description', 'path_segment', 'list_page_uid', 'detail_page_uid']; 
        $prefixedAttributes = array_map(function($attribute) {
            return 'tx_igshop2_domain_model_category.' . $attribute;
        }, $attributes);
        $queryBuilder
            //->select('tx_igshop2_domain_model_category.*')
            ->select(...$prefixedAttributes)
            ->from('tx_igshop2_domain_model_category')
              ->join(
                  'tx_igshop2_domain_model_category',
                  'tx_igshop2_product_category_mm',
                  'tx_igshop2_product_category_mm',
                  $queryBuilder->expr()->eq('uid', $queryBuilder->quoteIdentifier('uid_foreign') )
              )
            ->where(
                $queryBuilder->expr()->eq('uid_local', $queryBuilder->createNamedParameter($productUid, \PDO::PARAM_INT)),
            )
            ->orderBy('tx_igshop2_product_category_mm.sorting'); // or sorting_foreign
        if ($limit > 0) {
            $queryBuilder->setMaxResults($limit);
        }
        return $queryBuilder->execute()->fetchAllAssociative();
    }
    
    /**
     * get detail link from raw product data
     */
    public function getDetailPageUid(int $productUid, int $productDetailPageUid, int $fallbackPageUid): int
    {
        // detail page of product
        if ($productDetailPageUid > 0) {
            return $productDetailPageUid;
        }
        // detail page of first category
        $categories = $this->getCategoriesByProductUid($productUid, 1);
        $categoryDetailPageUid = $categories[0]['detail_page_uid'] ?? 0;
        if ($categoryDetailPageUid > 0) {
            return $categoryDetailPageUid;
        }
        // fallback (flexform, current page)
        return $fallbackPageUid;
    }

    /*
    public function getImagePaths($imageIds)
    {
        $imagePaths = [];
        
        foreach ($imageIds as $imageId) {
            try {
                $fileReference = $this->resourceFactory->getFileReferenceObject($imageId);
                if ($fileReference) {
                    $filePath = $fileReference->getPublicUrl();
                    $imagePaths[] = $filePath;
                }
            } catch (ResourceDoesNotExistException $e) {
                $imagePaths[] = $imageId;
            }
        }

        return $imagePaths;
    }
    */
    public static function http_build_url(array $parsed_url): string
    {
        $scheme   = isset($parsed_url['scheme']) ? $parsed_url['scheme'] . '://' : '';
        $host     = isset($parsed_url['host']) ? $parsed_url['host'] : '';
        $port     = isset($parsed_url['port']) ? ':' . $parsed_url['port'] : '';
        $user     = isset($parsed_url['user']) ? $parsed_url['user'] : '';
        $pass     = isset($parsed_url['pass']) ? ':' . $parsed_url['pass']  : '';
        $pass     = ($user || $pass) ? "$pass@" : '';
        $path     = isset($parsed_url['path']) ? $parsed_url['path'] : '';
        $query    = isset($parsed_url['query']) ? '?' . $parsed_url['query'] : '';
        $fragment = isset($parsed_url['fragment']) ? '#' . $parsed_url['fragment'] : '';
        return "$scheme$user$pass$host$port$path$query$fragment";
    }

}