<?php

namespace InternetGalerie\Igshop2\Controller;

use InternetGalerie\Igshop2\Domain\Model\Product;
use InternetGalerie\Igshop2\Domain\Model\ProductComparison;
use InternetGalerie\Igshop2\Domain\Model\Shop;
use InternetGalerie\Igshop2\Domain\Repository\CurrencyRepository;
use InternetGalerie\Igshop2\Domain\Repository\FeUserRepository;
use InternetGalerie\Igshop2\Domain\Repository\ProductComparisonRepository;
use InternetGalerie\Igshop2\Domain\Repository\ProductRepository;
use InternetGalerie\Igshop2\Domain\Repository\ShopRepository;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Frontend\Controller\ErrorController;
use TYPO3\CMS\Core\Http\ImmediateResponseException;

/**
 * ProductComparisonController
 */
class ProductComparisonController extends ActionController
{
    protected Shop $shop;

    protected ?ProductComparison $productComparison = null;

    public function __construct(
        protected ShopRepository $shopRepository,
        protected ProductComparisonRepository $productComparisonRepository,
        protected FeUserRepository $feUserRepository,
        protected CurrencyRepository $currencyRepository,
        protected ProductRepository $productRepository
    ) {
    }

    public function listAction(
        ?ProductComparison $productComparison = null,
        string $compare = null
    ): ResponseInterface
    {
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:igshop2/Resources/Public/JavaScript/Products/igShop_Products.js');
        //$pageRenderer->addJsFooterFile('EXT:igshop2/Resources/Public/JavaScript/ProductComparison/sticky.js');
        if ($compare !== null) {
            $productUids = array_filter(GeneralUtility::trimExplode(',', $compare));

            $productsToRemove = [];
            foreach ($this->productComparison->getProducts() as $product) {
                if (!in_array($product->getUid(), $productUids)) {
                    $productsToRemove[] = $product;
                }
            }

            foreach ($productsToRemove as $product) {
                $this->productComparison->removeProduct($product);
            }

            foreach ($productUids as $productUid) {
                $product = $this->productRepository->findByUid($productUid);
                if (!is_null($product) && !$this->productComparison->hasProduct($product)) {
                    $this->productComparison->addProduct($product);
                }
            }

            $this->productComparisonRepository->update($this->productComparison);
        } elseif ($productComparison) {
            $this->productComparison = $productComparison;
        }

        $this->view->assign('productComparison', $this->productComparison);
        $this->view->assign('currencies', $this->shop->getCurrenciesAssociative());
        $this->view->assign('currency', $this->currencyRepository->getCurrency());
        $this->view->assign('compare', $compare);

        return $this->htmlResponse();
    }

    /**
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("product")
     */
    public function addProductAction(?Product $product = null): ResponseInterface
    {
        $data = [];
        if ($product === null) {
            $response = GeneralUtility::makeInstance(ErrorController::class)
                      ->pageNotFoundAction($this->request, 'The requested page does not exist');
            throw new ImmediateResponseException($response, 1591428020);
        }
        if ($this->productComparison->hasProduct($product)) {
            $this->productComparison->removeProduct($product);
            $data['status'] = 'removed';
        } else {
            $this->productComparison->addProduct($product);
            $data['status'] = 'success';
        }

        $this->productComparisonRepository->update($this->productComparison);

        $compare = '';

        foreach ($this->productComparison->getProducts() as $p) {
            if ($compare != '') {
                $compare .= ',';
            }

            $compare .= $p->getUid();
        }

        $compareLink = $this->uriBuilder
            ->reset()
            ->setTargetPageUid($this->settings['productComparison']['pageUid'])
            ->build();


        $data['compareUrl'] = $compare ? $compareLink . '?tx_igshop2_productcomparison[compare]=' . $compare : $compareLink;

        $this->view->assign('status', $data['status']);
        $this->view->assign('product', $product);
        $this->view->assign('productComparison', $this->productComparison);
        $this->view->assign('currencies', $this->shop->getCurrenciesAssociative());
        $this->view->assign('currency', $this->currencyRepository->getCurrency());
        $this->view->assign('compareUrl', $data['compareUrl']);

        $data['productComparisonUid'] = $this->productComparison->getUid();
        $data['productCount'] = $this->productComparison->getProducts()->count();
        $data['html'] = $this->view->render();


        return $this->jsonResponse(json_encode($data));
    }

    public function fetchProductsAction(): ResponseInterface
    {
        $this->view->assign('productComparison', $this->productComparison);

        $compare = '';

        foreach ($this->productComparison->getProducts() as $product) {
            if ($compare != '') {
                $compare .= ',';
            }

            $compare .= $product->getUid();
        }

        $compareLink = $this->uriBuilder
                            ->reset()
                            ->setTargetPageUid($this->settings['productComparison']['pageUid'])
                            ->build();

        $data = [
            'status' => 'success',
            'productComparisonUid' => $this->productComparison->getUid(),
            'productCount' => $this->productComparison->getProducts()
->count(),
            'html' => $this->view->render(),
            'compareUrl' => $compare ? $compareLink . '?tx_igshop2_productcomparison[compare]=' . $compare : $compareLink,
        ];
        return $this->jsonResponse(json_encode($data));
    }

    public function removeProductAction(Product $product): ResponseInterface
    {
        if ($this->productComparison->hasProduct($product)) {
            $this->productComparison->removeProduct($product);
            $this->productComparisonRepository->update($this->productComparison);
        }

        $data = [
            'status' => 'success',
            'productComparisonUid' => $this->productComparison->getUid(),
            'productCount' => $this->productComparison->getProducts()
->count(),
            'html' => $this->view->render(),
        ];

        return $this->redirect('list');
    }

    protected function initializeAction(): void
    {
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addCssFile('EXT:igshop2/Resources/Public/Css/Products/igShop_ProductComparison.scss');
        $pageRenderer->addJsFooterFile(
            'EXT:igshop2/Resources/Public/JavaScript/ProductComparison/productComparison.js'
        );
        if ($this->settings['shopUid']) {
            $this->shop = $this->shopRepository->findByUid($this->settings['shopUid']);
        }

        $context = GeneralUtility::makeInstance(Context::class);
        $feUserId = $context->getPropertyFromAspect('frontend.user', 'id');
        $feUser = null;

        if ($feUserId) {
            $feUser = $this->feUserRepository->findByUid($feUserId);
        }

        $productComparisonUid = null;
        if ($this->request->hasArgument('productComparison')) {
            $productComparisonUid = $this->request->getArgument('productComparison');
        }

        $this->productComparison = $this->productComparisonRepository->getCurrentProductComparison(
            $productComparisonUid,
            $feUser
        );
    }
}
