<?php

namespace InternetGalerie\Igshop2\Domain\Repository;

use Doctrine\DBAL\ParameterType;
use InternetGalerie\Igshop2\Database\Query\ProductQueryBuilder;
use TYPO3\CMS\Extbase\Persistence\Repository;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use InternetGalerie\Igshop2\Domain\Model\Currency;
use InternetGalerie\Igshop2\Domain\Model\Product;
use InternetGalerie\Igshop2\Utility\RankingUtility;
use PDO;
use InternetGalerie\Igshop2\Domain\Model\Category;
use InternetGalerie\Igshop2\Event\ModifyProductSearchEvent;
use Psr\EventDispatcher\EventDispatcherInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\Restriction\DeletedRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\HiddenRestriction;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;
use TYPO3\CMS\Extbase\Persistence\Generic\Storage\Typo3DbQueryParser;
use TYPO3\CMS\Extbase\Persistence\QueryResultInterface;

/**
 * The repository for Products
 */
class ProductRepository extends Repository
{
    protected $defaultOrderings = [
        //                        'category.parent.sorting'=> \TYPO3\CMS\Extbase\Persistence\QueryInterface::ORDER_ASCENDING,
        //'category.sorting'=> \TYPO3\CMS\Extbase\Persistence\QueryInterface::ORDER_ASCENDING,
        'sorting' => QueryInterface::ORDER_ASCENDING,
    ];

    protected EventDispatcherInterface $eventDispatcher;



    public function __construct(
        protected DataMapper $dataMapper,
        protected Typo3DbQueryParser $queryParser,
    ) {
        parent::__construct();
    }

    public function injectEventDispatcher(EventDispatcherInterface $eventDispatcher): void
    {
        $this->eventDispatcher = $eventDispatcher;
    }

    /*
      protected $settings = [];

      public function setSettings($settings) {
      $this->settings = $settings;
      }
    */

    /**
     * better findByUid Function with language Support
     *
     * @return array
     */
    /*    public function findByUidAndLang($uid){

          $query = $this ->createQuery();
          $query->getQuerySettings()->setRespectSysLanguage(TRUE);
          $query->matching(
          $query->logicalOR(
          $query->equals('uid', $uid),
          $query->equals('l10n_parent', $uid)
          )
          );
          return $query->execute()->toArray();
          }
    */

    public function initializeObject(): void
    {
        if (GeneralUtility::makeInstance(Context::class)->getPropertyFromAspect('language', 'id') == 0) {
            $this->defaultOrderings['category.sorting'] = QueryInterface::ORDER_ASCENDING;
        }
    }

    public function createProductQueryBuilder()
    {
        return ProductQueryBuilder::create();
    }

    public function findBySearch(array $search=[], Currency $currency, array $orderBy = [])
    {
        $queryBuilder =  $this->createProductQueryBuilder();
        //$queryBuilder->addUserRestriction();
        $queryBuilder->select(ProductQueryBuilder::$tablename  . '.*');
        $queryBuilder->distinct();
        $queryBuilder->from(ProductQueryBuilder::$tablename);
        $queryBuilder->addSearch($search, $currency);
        /*
        if (!empty($orderBy)) {
            $queryBuilder->addArrayOrderBy($orderBy);
        } else {
            if ($search['sorting'] ?? false) {
                $sorting = GeneralUtility::trimExplode(',', $search['sorting'], true);
                if (!empty($sorting)) {
                    foreach ($sorting as $sortingPart) {
                        [$sortingProperty, $sortingOrder] = GeneralUtility::trimExplode(' ', $sortingPart);
                        $sortingAttribute = GeneralUtility::camelCaseToLowerCaseUnderscored($sortingProperty);
                        $queryBuilder->orderBy($sortingAttribute, strtolower($sortingOrder) == 'desc' ? 'DESC' : 'ASC');
                    }
                }
            }
        }
        */
        return $queryBuilder;
    }

    public function findBySearchExecute(array $search=[], Currency $currency, $orderBy = [], $limit = 0, $disableCatSort = false)
    {
        $queryBuilder = $this->findBySearch($search, $currency, $orderBy);
        if ($limit) {
            $queryBuilder->setMaxResults($limit);
        } else {
            $queryBuilder->setMaxResults(96);
        }

        $rows = $queryBuilder->executeQuery()->fetchAllAssociative();
        return $this->dataMapper->map($this->objectType, $rows);
    }
    public function queryBuilderBySearchAjax(array $search=[], Currency $currency, $orderBy = [])
    {
        $queryBuilder = $this->findBySearch($search, $currency, $orderBy);

        $attributes = ['uid', 'name', 'productid', 'price', 'description', 'image', 'site_id', 'product_of_month', 'discount']; //  , 'discount_start', 'discount_end', 'image_thumb',
        $prefixedAttributes = array_map(function($attribute) {
            return ProductQueryBuilder::$tablename . '.' . $attribute;
        }, $attributes);
        $queryBuilder->select(...$prefixedAttributes);
        return $queryBuilder;
    }
    public function findBySearchAjax(array $search=[], Currency $currency, $orderBy = [], $limit = 0, $disableCatSort = false)
    {
        $queryBuilder = $this->queryBuilderBySearchAjax($search, $currency, $orderBy); 
        $queryBuilder->setMaxResults(5);
        $rows = $queryBuilder->executeQuery()->fetchAllAssociative();
        return $rows;
    }
    
    /**
     * function to find Products by a given String
     *
     * @param string $flexUids
     * @param mixed $catNavUids
     * @param mixed $sorting
     * @param string $sortWay
     * @param array $filter
     * @param Currency $currency
     * @param string $searchString
     * @return Product
     */
    public function findAll(
        $flexUids = null,
        $catNavUids = null,
        $sorting = 'sorting',
        $sortWay = 'asc',
        $filter = [],
        $currency = null,
        $searchString = null,
        $limit = 0,
        $statusses = null,
        $disableCatSort = false,
        $properties = []
    ) {
        $query = $this->createQuery();
        $querySettings = $query->getQuerySettings();
        //$querySettings->setLanguageOverlayMode(false);

        $constraints = [];


        $constraints[] = $query->logicalOr(
            $query->equals('stock', null),
            $query->logicalNot($query->equals('out_of_stock_behavior', 'hide')),
            $query->greaterThan('stock', 0)
        );
            
        
        // Filter for Colors
        if (key_exists('colors', $filter) && is_array($filter['colors'])) {
            $colorConstraint = [];
            foreach ($filter['colors'] as $color) {
                $colorConstraint[] = $query->contains('optionsColor', $color);
            }

            $constraints[] = $query->logicalOr(...$colorConstraint);
        }

        // Filter for Sizes
        if (key_exists('sizes', $filter) && is_array($filter['sizes'])) {
            $sizeConstraint = [];
            foreach ($filter['sizes'] as $size) {
                $sizeConstraint[] = $query->contains('optionsSize', $size);
            }

            $constraints[] = $query->logicalOr(...$sizeConstraint);
        }

        // Filter for Product Classes
        if (key_exists('productClasses', $filter) && is_array($filter['productClasses'])) {
            $constraints[] = $query->in('productClass', $filter['productClasses']);
        }

        // Select only Products where Price is not null
        $query->matching($query->logicalAnd($query->logicalNot($query->equals('price', null))));


        $minPrice = null;
        $maxPrice = null;
        if (key_exists('priceRange', $filter)) {
            $minPrice = $filter['priceRange']['from'] ?? null;
            $maxPrice = $filter['priceRange']['to'] ?? null;
        } else {
            if (key_exists('minPrice', $filter) && $filter['minPrice'] > 0) {
                $minPrice = $filter['minPrice'];
            }

            if (key_exists('maxPrice', $filter) && $filter['minPrice'] < $filter['maxPrice']) {
                $maxPrice = $filter['maxPrice'];
            }
        }


        // Filter for Prices - minPrice
        if ($minPrice !== null && $minPrice > 0) {
            if (is_object($currency) && $currency->getShort() == 'EUR') {
                $constraints[] = $query->logicalAnd($query->greaterThanOrEqual('price_euro', $minPrice));
            } else {
                $constraints[] = $query->logicalAnd($query->greaterThanOrEqual('price', $minPrice));
            }
        }

        // Filter for Prices - maxPrice
        if ($maxPrice !== null && $minPrice < $maxPrice) {
            if (is_object($currency) && $currency->getShort() == 'EUR') {
                $constraints[] = $query->logicalAnd($query->lessThanOrEqual('price_euro', $maxPrice));
            } else {
                $constraints[] = $query->logicalAnd($query->lessThanOrEqual('price', $maxPrice));
            }
        }

        if (isset($filter['mandant']) && $filter['mandant'] > 0) {
            $constraints[] = $query->equals('mandantUid', (int) $filter['mandant']);
        }

        $categoryUids = [];
        if (isset($filter['categories'])) {
            if (is_array($filter['categories'])) {
                $filter['categories'] =  array_map('intval', array_filter($filter['categories']));
            } else {
                $filter['categories'] = GeneralUtility::intExplode(',', (string)$filter['categories'], true);
            }
            $categoryUids = $filter['categories'];
        } 

        if (isset($filter['category']) && $filter['category'] > 0) {
            $categoryUids[] = (int)$filter['category'];
        }

        if ($flexUids) {
            $flexformCategoryUids = GeneralUtility::intExplode(',', $flexUids, true); 
            if (empty( $categoryUids)) {
                $categoryUids = $flexformCategoryUids;
            } else {
                $categoryUids = array_values(array_intersect($categoryUids, $flexformCategoryUids));
                if (empty($categoryUids)) {
                    $categoryUids = $flexformCategoryUids;
                }
            }
        }
        if (!empty($categoryUids)) {
            $categoryConstraints = [];
            foreach ($categoryUids as $categoryUid) {
                $categoryConstraints[] = $query->contains('category', $categoryUid);
            }

            $constraints[] = $query->logicalAnd(...$categoryConstraints);
            //$qs = $this->createQuery();
            /*$qs->statement('SELECT uid FROM tx_igshop2_domain_model_category WHERE parent IN (' . implode(',', $category) . ')');
              $qsRes = $qs->execute(true);
              foreach ($qsRes as $r) {
              $category[] = $r['uid'];
              }
              $q = $this->createQuery();
              $q->statement('SELECT uid_local FROM tx_igshop2_product_category_mm WHERE uid_foreign IN (' . implode(',', $category) . ')');
              $qRes = $q->execute(true);

              $qArray = [];
              $qArray[] = -1;
              foreach ($qRes as $r) {
              $qArray[] = $r['uid_local'];
              }
              $constraints[] = $query->logicalOr(
              $query->in('uid', $qArray),
              $query->in('l10n_parent', $qArray)
              );*/
        }

        $rankingCase = '';
        if (isset($filter['keyword'])) {
            $searchString = $filter['keyword'];
        }

        // Filter for SearchStringz
        if ($searchString !== null) {
            $searchArray = GeneralUtility::trimExplode(' ', $searchString, true);
            $constrains = [];
            $searchStringConstraint = [];
            if ($searchArray !== []) {
                foreach ($searchArray as $index => $word) {
                    $searchStringConstraint[] = $query->logicalOr(
                        $query->like('name', '%' . $word . '%'),
                        $query->like('productid', '%' . $word . '%'),
                        $query->like('pharmacode', '%' . $word . '%'),
                        $query->like('description', '%' . $word . '%'),
                        $query->like('searchString', '%' . $word . '%')
                    );
                    //                    $searchStringConstraint[] = $query -> like('category.name', '%' . $word . '%');
                    //$searchStringConstraint[] = $query -> like('category.parent.name', '%' . $word . '%');

                    if (strlen($word) >= 3) {
                        $rankingCase .= ($index > 0 ? ' + ' : '') . RankingUtility::getRankingCase(
                            'tx_igshop2_domain_model_product',
                            'name',
                            $word
                        );
                    }
                }
            }

            if ($searchStringConstraint !== []) {
                $constraints[] = $query->logicalAnd(...$searchStringConstraint);
            }
        }

        if ($statusses) {
            $constraints[] = $query->in('state', $statusses);
        }

        /*if(!empty($properties)) {
          $propertyConstraints = [];

          foreach($properties as $property => $value) {
          $propertyValueConstraints = [];
          $propertyValues = array_filter(GeneralUtility::trimExplode(' ', $value));
          foreach($propertyValues as $propertyValue) {
          $propertyValueConstraints[] = $query->like('properties.propertyValue', '%' . $propertyValue . '%');
          }

          if(!empty($propertyValueConstraints)) {
          $propertyConstraints[] = $query->logicalAnd(
          $query->equals('properties.property.fieldName', $property),
          ...$propertyValueConstraints
          );
          }
          }

          $constraints[] = $query->logicalAnd(...$propertyConstraints);
          }*/

        $modifyProductSearchEvent = $this->eventDispatcher->dispatch(
            new ModifyProductSearchEvent($query, $constraints)
        );
        $query = $modifyProductSearchEvent->getQuery();
        $constraints = $modifyProductSearchEvent->getConstraints();

        if (count($constraints) > 0) {
            $query->matching($query->logicalAnd(...$constraints));
        }

        if ($sortWay == 'asc') {
            if ($sorting != 'sorting') {
                $orderings = [];
            }

            foreach (explode(',', (string) $sorting) as $sort) {
                $orderings[$sort] = QueryInterface::ORDER_ASCENDING;
            }

            $query->setOrderings($orderings);
        }

        // Bilder zuerst
        //CASE WHEN image>0 THEN 1 ELSE 0 END
        //$query->setOrderings (['image' => \TYPO3\CMS\Extbase\Persistence\QueryInterface::ORDER_DESCENDING,'sorting' => \TYPO3\CMS\Extbase\Persistence\QueryInterface::ORDER_ASCENDING]);

        if ($sortWay == 'desc') {
            if ($sorting != 'sorting') {
                $orderings = [];
            }

            foreach (explode(',', (string) $sorting) as $sort) {
                $orderings[$sort] = QueryInterface::ORDER_DESCENDING;
            }

            $query->setOrderings($orderings);
        }



        $queryBuilder = $this->queryParser->convertQueryToDoctrineQueryBuilder($query);

        if ($limit) {
            $queryBuilder->setMaxResults($limit);
            //$query->setLimit($limit);
        } else {
            $queryBuilder->setMaxResults(96);
            //$query->setLimit(96);
        }

        if (count($categoryUids) == 1 && $sorting == 'sorting' && !$disableCatSort) {
            // Eine Kategorie ausgewählt -> sortieren nach dieser
            $queryBuilder->leftJoin(
                'tx_igshop2_domain_model_product',
                'tx_igshop2_product_category_mm',
                'product_category_mm',
                $queryBuilder->expr()
                             ->and(
                                 $queryBuilder->expr()
                                              ->eq('uid_foreign', $queryBuilder->createNamedParameter($categoryUids[0], ParameterType::INTEGER)),
                                 $queryBuilder->expr()
                                              ->or(
                                                  $queryBuilder->expr()
                                                               ->and(
                                                                   $queryBuilder->expr()
                                                                                ->eq('product_category_mm.uid_local', $queryBuilder->quoteIdentifier('tx_igshop2_domain_model_product.uid')),
                                                                   $queryBuilder->expr()
                                                                                ->eq('tx_igshop2_domain_model_product.l10n_parent', $queryBuilder->createNamedParameter(0, ParameterType::INTEGER))
                                                               ),
                                                  $queryBuilder->expr()
                                                               ->and(
                                                                   $queryBuilder->expr()
                                                                                ->eq('product_category_mm.uid_local', $queryBuilder->quoteIdentifier('tx_igshop2_domain_model_product.l10n_parent')),
                                                                   $queryBuilder->expr()
                                                                                ->gt('tx_igshop2_domain_model_product.l10n_parent', $queryBuilder->createNamedParameter(0, ParameterType::INTEGER))
                                                               )
                                                  //$queryBuilder->expr()->eq('product_category_mm.uid_local', $queryBuilder->quoteIdentifier('tx_igshop2_domain_model_product.uid'))
                                                  //$queryBuilder->expr()->eq('product_category_mm.uid_local', $queryBuilder->quoteIdentifier('tx_igshop2_domain_model_product.l10n_parent'))
                                              )
                             )
            )
                         ->andWhere($queryBuilder->expr()->in('tx_igshop2_domain_model_product.sys_language_uid', [
                             $queryBuilder->createNamedParameter(-1, ParameterType::INTEGER),
                             $queryBuilder->createNamedParameter(
                                 GeneralUtility::makeInstance(Context::class)->getPropertyFromAspect('language', 'id'),
                                 ParameterType::INTEGER
                             ),
                         ]));

            if ($rankingCase) {
                $queryBuilder->addSelectLiteral($rankingCase . ' AS ranking_case')
                             ->addOrderBy('ranking_case', 'DESC')
                             ->addOrderBy('sorting', 'ASC');
                //$queryBuilder->add('orderBy', $rankingCase . ' DESC, sorting ASC');
            } else {
                //$queryBuilder->orderBy('product_category_mm.sorting', ($sortWay == 'asc' ? 'ASC' : 'DESC'));
                $queryBuilder->orderBy('product_category_mm.sorting_foreign', ($sortWay == 'asc' ? 'ASC' : 'DESC'));
                $queryBuilder->addOrderBy(
                    'tx_igshop2_domain_model_product.sorting',
                    ($sortWay == 'asc' ? 'ASC' : 'DESC')
                );
            }

            $sortedRes = $queryBuilder->executeQuery()
                                      ->fetchAllAssociative();
            return $this->dataMapper->map(Product::class, $sortedRes);
        } elseif (GeneralUtility::makeInstance(Context::class)->getPropertyFromAspect(
            'language',
            'id'
        ) > 0) { // empty($category) &&
            // Ohne Kategorie oder mehrere Kategorien sortieren nach Standard Sprache sorting von Produkt
            $queryBuilder->leftJoin(
                'tx_igshop2_domain_model_product',
                'tx_igshop2_domain_model_product',
                'product_mm',
                $queryBuilder->expr()
                             ->or(
                                 $queryBuilder->expr()
                                              ->and(
                                                  $queryBuilder->expr()
                                                               ->eq('product_mm.uid', $queryBuilder->quoteIdentifier('tx_igshop2_domain_model_product.uid')),
                                                  $queryBuilder->expr()
                                                               ->eq('tx_igshop2_domain_model_product.l10n_parent', $queryBuilder->createNamedParameter(0, ParameterType::INTEGER))
                                              ),
                                 $queryBuilder->expr()
                                              ->and(
                                                  $queryBuilder->expr()
                                                               ->eq('product_mm.uid', $queryBuilder->quoteIdentifier('tx_igshop2_domain_model_product.l10n_parent')),
                                                  $queryBuilder->expr()
                                                               ->gt('tx_igshop2_domain_model_product.l10n_parent', $queryBuilder->createNamedParameter(0, ParameterType::INTEGER))
                                              )
                             )
            );

            if ($rankingCase) {
                $queryBuilder->addSelectLiteral($rankingCase . ' AS ranking_case')
                    ->addOrderBy('ranking_case', 'DESC')
                    ->addOrderBy('sorting', 'ASC');
                //$queryBuilder->add('orderBy', $rankingCase . ' DESC, sorting ASC');
            } else {
                $queryBuilder->orderBy('product_mm.sorting', ($sortWay == 'asc' ? 'ASC' : 'DESC'));
            }

            // CASE WHEN product_mm.sorting THEN product_mm.sorting ELSE tx_igshop2_domain_model_product.sorting END

            $sortedRes = $queryBuilder->executeQuery()
                                      ->fetchAllAssociative();
            return $this->dataMapper->map(Product::class, $sortedRes);
        }

        if ($rankingCase) {
            $queryBuilder->addSelectLiteral($rankingCase . ' AS ranking_case')
                ->addOrderBy('ranking_case', 'DESC')
                ->addOrderBy('sorting', 'ASC');
            //$queryBuilder->add('orderBy', $rankingCase . ' DESC, sorting ASC');

            if ($limit) {
                $queryBuilder->setMaxResults($limit);
            } else {
                $queryBuilder->setMaxResults(150);
            }

            //die($queryBuilder->getSQL());
            $sortedRes = $queryBuilder->executeQuery()
                                      ->fetchAllAssociative();

            return $this->dataMapper->map(Product::class, $sortedRes);
        }

        return $query->execute();
    }

    public function getMaxPrice($currency)
    {
        $query = $this->createQuery();
        $products = $query->execute()
                          ->toArray();
        $maxPrice = 0;
        foreach ($products as $product) {
            $prodPrice = $product->getFinalPrice($currency);
            if ($prodPrice > $maxPrice) {
                $maxPrice = $prodPrice;
            }
        }

        return $maxPrice;
    }

    /*
      public function findAllExceptCategory($category = null)
      {
      $query = $this->createQuery();

      if (is_object($category)) {
      $query->matching($query->logicalNot($query->contains('category', $category)));
      }

      return $query->execute();
      }
    */
    public function findAllForExport()
    {
        $query = $this->createQuery();
        $query->matching($query->equals('category.hideInExport', false));
        return $query->execute();
    }

    /**
     * Function to search minimum Price of given list by category
     */
    public function getMinPrice($flexUids, $catNavUids, $currency)
    {
        /** @var QueryResultInterface|array */
        $products = $this->findAll($flexUids, $catNavUids);

        if (!is_array($products)) {
            $products = $products->toArray();
        }

        $minPrice = '';
        $isFirst = true;
        foreach ($products as $product) {
            if ($isFirst) {
                $isFirst = false;
                if ($currency->getShort() == 'EUR' && $product->getPriceEuro() != null) {
                    $minPrice = $product->getPriceEuro();
                } else {
                    $minPrice = $product->getPrice() * $currency->getFaktor();
                }
            } else {
                if ($currency->getShort() == 'EUR' && $product->getPriceEuro() != null) {
                    if ($product->getPriceEuro() < $minPrice) {
                        $minPrice = $product->getPriceEuro();
                    }
                } else {
                    if ($product->getPrice() < $minPrice) {
                        $minPrice = $product->getPrice() * $currency->getFaktor();
                    }
                }
            }
        }

        return $minPrice;
    }

    public function getProductsWithActionPrice($sorting = 'sorting', $sortWay = 'asc')
    {
        $query = $this->createQuery();

        $query->matching($query->logicalAnd($query->logicalNot($query->equals('price', null))));
        $query->matching($query->logicalAnd($query->equals('prices.isSpecialPrice', true)));

        // if (isset($_GET['debug']) && $_GET['debug']) {
        //     var_dump($sorting);
        //     var_dump($sortWay);
        // }

        if ($sortWay == 'asc') {
            if ($sorting == 'price') {
                $query->setOrderings([
                    'prices.price' => QueryInterface::ORDER_ASCENDING,
                ]);
            } else {
                $query->setOrderings([
                    $sorting => QueryInterface::ORDER_ASCENDING,
                ]);
            }
        }

        if ($sortWay == 'desc') {
            if ($sorting == 'price') {
                $query->setOrderings([
                    'prices.price' => QueryInterface::ORDER_DESCENDING,
                ]);
            } else {
                $query->setOrderings([
                    $sorting => QueryInterface::ORDER_DESCENDING,
                ]);
            }
        }

        return $query->execute();
    }

    /**
     * function to find Products by a given String
     *
     * @param string $searchString
     * @return Product
     */
    public function findProducts($searchString = 0)
    {
        $query = $this->createQuery();
        //$query->getQuerySettings()->setRespectSysLanguage(TRUE);

        $searchArray = explode(' ', $searchString);
        $constrains = [];

        foreach ($searchArray as $word) {
            $constrains[] = $query->like('name', '%' . $word . '%');
            $constrains[] = $query->like('description', '%' . $word . '%');
            $constrains[] = $query->like('searchString', '%' . $word . '%');
        }

        $query->matching($query->logicalAnd($query->logicalNot($query->equals('price', null))));
        $query->matching($query->logicalOr(...$constrains));
        return $query->execute();
    }

    /**
     * @deprecated
     */
    public function getProductByCategory($sorting, $category = null, $sortWay = 'asc', $filter = [], $currency = null)
    {
        $query = $this->createQuery();
        $constraints = [];

        if (key_exists('colors', $filter) && is_array($filter['colors'])) {
            $constraints[] = $query->logicalAnd($query->in('optionsColor.uid', $filter['colors']));
        }

        if (key_exists('sizes', $filter) && is_array($filter['sizes'])) {
            $constraints[] = $query->logicalAnd($query->in('optionsSize.uid', $filter['sizes']));
        }

        if (key_exists('minPrice', $filter) && $filter['minPrice'] > 0) {
            if (is_object($currency) && $currency->getShort() == 'EUR') {
                $constraints[] = $query->logicalAnd($query->greaterThanOrEqual('price_euro', $filter['minPrice']));
            } else {
                $constraints[] = $query->logicalAnd($query->greaterThanOrEqual('price', $filter['minPrice']));
            }
        }

        if (key_exists('maxPrice', $filter) && $filter['minPrice'] < $filter['maxPrice']) {
            if (is_object($currency) && $currency->getShort() == 'EUR') {
                $constraints[] = $query->logicalAnd($query->lessThanOrEqual('price_euro', $filter['maxPrice']));
            } else {
                $constraints[] = $query->logicalAnd($query->lessThanOrEqual('price', $filter['maxPrice']));
            }
        }

        if ($category != null) {
            $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_igshop2_product_category_mm');
            $res = null;
            if (is_array($category)) {
                $res = $conn->executeQuery(
                    'SELECT uid_local FROM tx_igshop2_product_category_mm WHERE uid_foreign IN (' . implode(
                        ',',
                        $category
                    ) . ')'
                );
            } else {
                $res = $conn->executeQuery(
                    'SELECT uid_local FROM tx_igshop2_product_category_mm WHERE uid_foreign =' . $category . ''
                );
            }

            $qRes = $res->fetchAllAssociative();
            $qArray = [];
            $qArray[] = -1;
            foreach ($qRes as $r) {
                $qArray[] = $r['uid_local'];
            }

            $constraints[] = $query->logicalOr($query->in('uid', $qArray), $query->in('l10n_parent', $qArray));
        }

        if ($sortWay == 'asc') {
            $orderings = [];
            foreach (explode(',', (string) $sorting) as $sort) {
                $orderings[$sort] = QueryInterface::ORDER_ASCENDING;
            }

            $query->setOrderings($orderings);
        }

        if ($sortWay == 'desc') {
            $orderings = [];
            foreach (explode(',', (string) $sorting) as $sort) {
                $orderings[$sort] = QueryInterface::ORDER_DESCENDING;
            }

            $query->setOrderings($orderings);
        }

        //$query -> matching($query->logicalAnd($query->logicalNot($query -> equals('price', null))));
        $constraints[] = $query->logicalNot($query->equals('price', null));
        $query->matching($query->logicalAnd(...$constraints));
        return $query->execute();
    }

    /**
     * Selects the Products for the list view, depending on the given uid lists for categories
     * If the uid's come from FlexForm, it will select all products with those categories,
     * if it is empty, it searches with the uid's from Category Navigation
     *
     * @param string $flexUids
     * @param string $catNavUids
     * @param string $sorting
     * @param string $sortWay
     * @param array $filter
     * @param Currency $currency
     * @param string $searchString
     * @return Product
     * @deprecated
     */
    public function getProducts(
        $flexUids,
        $catNavUids,
        $sorting = 'sorting',
        $sortWay = 'asc',
        $filter = [],
        $currency = null,
        $searchString = null
    ) {
        if ($catNavUids) {
            return $this->getProductByCategory($sorting, $catNavUids, $sortWay, $filter, $currency);
        } elseif ($flexUids) {
            return $this->getProductByCategory($sorting, explode(',', $flexUids), $sortWay, $filter, $currency);
        }

        return $this->findAll($sorting, $sortWay, $filter, $currency, $searchString);
    }

    /**
     * Find by multiple uids using, seperated string. Found on blog.teamgeist-medien.de
     *
     * @param string $uids String containing uids
     * @return Product Matching model records
     */
    public function findByUids($uids)
    {
        $uidArray = explode(',', $uids);
        $query = $this->createQuery();
        foreach ($uidArray as $key => $value) {
            $constraints[] = $query->equals('uid', $value);
        }

        $query->matching($query->logicalAnd($query->logicalNot($query->equals('price', null))));
        return $query->matching(
            $query->logicalAnd(
                $query->logicalOr(...$constraints),
                $query->equals('hidden', 0),
                $query->equals('deleted', 0)
            )
        )->execute();
    }

    public function findByCategories(array $categories)
    {
        $query = $this->createQuery();

        $query->matching($query->contains('category', $categories));

        return $query->execute();
    }

    /**
     * Selects all Product from given Product, exludes given Product
     * in result
     */
    public function getByCategoryExcludeProduct(Product $product)
    {
        $query = $this->createQuery();
        $query->matching($query->logicalAnd(
            $query->in('category', $product->getCategory()),
            $query->logicalNot($query->equals('uid', $product->getUid())),
        ));

        return $query->execute();
    }

    /**
     * Find all entries in a category joined together with the product-category intermediate table
     *
     * @param Category $category
     * @return array
     */
    public function findBySearchAndCategoryDoctrine($searchword = '', $category = null, $storagePid = null)
    {
        $tableName = 'tx_igshop2_domain_model_product';
        $query = $this->createQuery();
        //$queryBuilder = $this->queryParser->convertQueryToDoctrineQueryBuilder($query);
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tableName);
        $queryBuilder->getRestrictions()
                     ->removeAll();
        $queryBuilder->getRestrictions()
                     ->add(GeneralUtility::makeInstance(DeletedRestriction::class));
        /*
          $queryBuilder->getRestrictions()->removeByType(HiddenRestriction::class);
          $queryBuilder->getRestrictions()->add(GeneralUtility::makeInstance(DeletedRestriction::class));
          $queryBuilder->getRestrictions()->add(GeneralUtility::makeInstance(StartTimeRestriction::class));
          $queryBuilder->getRestrictions()->add(GeneralUtility::makeInstance(EndTimeRestriction::class));
        */

        $queryBuilder->select('*')
                     ->from($tableName)
                     ->where($queryBuilder->expr()->in(
                         $tableName . '.pid',
                         !is_null($storagePid) ? [$storagePid] : $query->getQuerySettings()->getStoragePageIds()
                     ))
                     ->andWhere($queryBuilder->expr()->in($tableName . '.sys_language_uid', [
                         $queryBuilder->createNamedParameter(-1, ParameterType::INTEGER),
                         $queryBuilder->createNamedParameter(0, ParameterType::INTEGER),
                     ]));

        if ($searchword) {
            $queryBuilder->andWhere(
                $queryBuilder->expr()
                             ->or(
                                 $queryBuilder->expr()
                                              ->like(
                                                  $tableName . '.name',
                                                  $queryBuilder->createNamedParameter('%' . $queryBuilder->escapeLikeWildcards($searchword) . '%')
                                              ),
                                 $queryBuilder->expr()
                                              ->like(
                                                  $tableName . '.productid',
                                                  $queryBuilder->createNamedParameter('%' . $queryBuilder->escapeLikeWildcards($searchword) . '%')
                                              )
                             )
            );
        }

        if ($category) {
            $queryBuilder->andWhere($queryBuilder->expr()->eq(
                'product_category.uid_foreign',
                $queryBuilder->createNamedParameter($category->getUid(), ParameterType::INTEGER)
            ))
                         ->leftJoin(
                             $tableName,
                             'tx_igshop2_product_category_mm',
                             'product_category',
                             $queryBuilder->expr()
                                          ->eq('product_category.uid_local', $queryBuilder->quoteIdentifier($tableName . '.uid'))
                         )
                         ->orderBy('product_category.sorting_foreign', 'ASC');
        } else {
            $queryBuilder->orderBy('sorting', 'ASC');
        }

        return $queryBuilder->executeQuery()
                            ->fetchAllAssociative();
    }

    /**
     * Finds a product-category relation (intermediate table entry)
     *
     * @param int $productUid
     * @param int $categoryUid
     * @return array
     */
    public function findProductCategoryRelationDoctrine($productUid, $categoryUid)
    {
        $tableName = 'tx_igshop2_domain_model_product';
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tableName);
        $queryBuilder->getRestrictions()
                     ->removeByType(HiddenRestriction::class);
        return $queryBuilder->select('product_category.*')
                            ->from($tableName)
                            ->where($queryBuilder->expr()->eq(
                                $tableName . '.uid',
                                $queryBuilder->createNamedParameter($productUid, ParameterType::INTEGER)
                            ))
                            ->andWhere($queryBuilder->expr()->in($tableName . '.sys_language_uid', [
                                $queryBuilder->createNamedParameter(-1, ParameterType::INTEGER),
                                $queryBuilder->createNamedParameter(0, ParameterType::INTEGER),
                            ]))
                            ->andWhere($queryBuilder->expr()->eq(
                                'product_category.uid_foreign',
                                $queryBuilder->createNamedParameter($categoryUid, ParameterType::INTEGER)
                            ))
                            ->leftJoin(
                                $tableName,
                                'tx_igshop2_product_category_mm',
                                'product_category',
                                $queryBuilder->expr()
                                             ->eq('product_category.uid_local', $queryBuilder->quoteIdentifier($tableName . '.uid'))
                            )
                            ->orderBy('product_category.sorting_foreign', 'ASC')
                            ->executeQuery()
                            ->fetchAssociative();
    }

    /**
     * Finds a product-category relation (intermediate table entry)
     *
     * @param int $categoryUid
     * @return array
     */
    public function findCategoryRelationDoctrine($categoryUid)
    {
        $tableName = 'tx_igshop2_domain_model_product';
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tableName);
        $queryBuilder->getRestrictions()
                     ->removeByType(HiddenRestriction::class);
        // 'tx_igshop2_domain_model_product.uid',
        return $queryBuilder->select('product_category.*')
                            ->from($tableName)
                            ->where($queryBuilder->expr()->in($tableName . '.sys_language_uid', [
                                $queryBuilder->createNamedParameter(-1, ParameterType::INTEGER),
                                $queryBuilder->createNamedParameter(0, ParameterType::INTEGER),
                            ]))
                            ->andWhere($queryBuilder->expr()->eq(
                                'product_category.uid_foreign',
                                $queryBuilder->createNamedParameter($categoryUid, ParameterType::INTEGER)
                            ))
                            ->leftJoin(
                                $tableName,
                                'tx_igshop2_product_category_mm',
                                'product_category',
                                $queryBuilder->expr()
                                             ->eq('product_category.uid_local', $queryBuilder->quoteIdentifier($tableName . '.uid'))
                            )
                            ->orderBy('product_category.sorting_foreign', 'ASC')
                            ->executeQuery()
                            ->fetchAllAssociative();
    }

    /**
     * Update sorting foreign of a product$
     *
     * @param array $product
     */
    public function updateProductCategorySortingDoctrine($product): void
    {
        $tableName = 'tx_igshop2_product_category_mm';
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($tableName);
        $queryBuilder->update($tableName)
                     ->set('sorting_foreign', $product['sorting_foreign'])
                     ->where($queryBuilder->expr()->eq(
                         $tableName . '.uid_local',
                         $queryBuilder->createNamedParameter($product['uid_local'], ParameterType::INTEGER)
                     ))
                     ->andWhere($queryBuilder->expr()->eq(
                         $tableName . '.uid_foreign',
                         $queryBuilder->createNamedParameter($product['uid_foreign'], ParameterType::INTEGER)
                     ))
                     ->executeStatement();
    }

    /**
     * find with mandaten_uid and search keyword
     *
     * @param int $mandantUid
     * @param string $keyword
     */
    public function findWidthMandantUidAndKeyword($mandantUid, $keyword)
    {
        $query = $this->createQuery();
        $query->matching(
            $query->logicalAnd(
                $query->equals('mandant_uid', $mandantUid),
                $query->logicalOr(
                    $query->like('name', '%' . $keyword . '%'),
                    $query->like('description', '%' . $keyword . '%')
                )
            )
        );
        $query->setLimit(1);
        return $query->execute();
    }
}
